/******************************************************************************
	  CCCC	    A	  BBBBB	  L	EEEEE  N     N	EEEEE	TTTTTTT
	 C    C    A A	  B    B  L	E      NN    N	E	   T
	C	  A   A	  B    B  L	E      N N   N	E	   T
	C	 AAAAAAA  BBBBB	  L	EEEEE  N  N  N	EEEEE	   T
	C        A     A  B    B  L	E      N   N N	E	   T
	 C    C  A     A  B    B  L	E      N    NN  E 	   T
	  CCCC	 A     A  BBBBB	  LLLL	EEEEE  N     N	EEEEE	   T
*******************************************************************************

CableNet Source Module:
	Copyright 1994-1995 (C) CableNet Limited. All Rights Reserved.

    Module Name:		$RCSfile: AppArgs.c,v $
    Module Description:	application argument handling access functions


Description: 


Edit History:

	$Log: AppArgs.c,v $
 * Revision 1.1  1995/01/09  18:21:23  V
 * Initial revision
 *


*/

/* RCS identification string (for "what" program) */
static char moduleRCSid[] = "@(#) $Id: AppArgs.c,v 1.1 1995/01/09 18:21:23 V Exp $";

/* must come first header files */
#include "V.h"			/* virtual header file */
#include "Vport.h"		/* port header file */


#include <stdio.h>
#include <ctype.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>

#ifdef VANSI_1
#include <stdarg.h>
#else
#include <varargs.h>
#endif

#include "Vlib.h"



/*++ ***************  function VappArgCHAR ****************/
/*
    Purpose:  determine a char pointer value for a named argument

    Prototype:
    char    *VappArgCHAR( char *arg )

    Globals changed: none

    Parameter Description:
    arg      == named argument to return

    Return Values: char *
    ! NULL      == success, arg was given on the command line
    NULL        == error, arg was not given on the command line

    Dependencies:
    _FindArgInfoShort, _FindArgInfoLong

    Notes:

*/
/************************************************************************ ++**/

#ifdef VANSI_1
char           *
VappArgCHAR(char *arg)
#else
char           *
VappArgCHAR(arg)
char           *arg;

#endif
{
    VargInfoPtr     va;

    /*-- find the arg info item for the name given */
    if (strlen(arg) > 1)
	va = _FindArgInfoLong(arg);
    else
	va = _FindArgInfoShort(*arg);

    if (va == NULL)
	return NULL;

    /*-- if the name is a long name */
    if (strlen(arg) > 1) {
	/*-- and there is an optarg associated with this arg info item
	  then return the value of the optarg item */
	if (va->optarg != NULL)
	    return va->optarg->value;
	else
	    /*-- else return "TRUE" to indicate that the arg existed */
	    return "TRUE";
    } else {
	/*-- else for a short name */
	/*-- if the arg info item value is not NULL then return it */
	if (va->value != NULL)
	    return va->value;
	/*-- else if there is a optarg item for this arg then
	  return it's value */
	else if (va->optarg != NULL)
	    return va->optarg->value;
	else
	    /*-- else return "TRUE" to indicate the arg's existence */
	    return "TRUE";
    }

}

/*++ ***************  function  VappArgPosCHAR ****************/
/*
    Purpose: determine the char pointer value for a positional argument

    Prototype:
    char *   VappArgPosCHAR( int pos )

    Globals changed: none

    Parameter Description:
    pos     == ordinal position of argument in the argument list
               pos >= 0  position is from argv[1] onwards
	       pos < 0   position is from argv[argc] backwards

    Return Values: char *
    ! NULL      == success
    NULL        == error, position is out of range of number of args

    Dependencies:

    Notes:

*/
/************************************************************************ ++**/

#ifdef VANSI_1
char           *
VappArgPosCHAR(int pos)
#else
char           *
VappArgPosCHAR(pos)
int             pos;

#endif
{
    VargInfoPtr     va;
    LLnode     *node;
    LLlist      *arglist;

    /*-- get a handle on the arg info item list */
    arglist = VappArgList();

    /*-- if the position number is positive or 0 */
    if (pos >= 0) {
	/*-- then start from the head of the arg info item list */
	if ((node = arglist->head) == NULL) {
	    return NULL;
	}
	/*-- for each node on the list */
	while (node) {
	    if ((va = (VargInfoPtr) node->data) == NULL)
		continue;

	    /*-- if the index of the arg info item == the
	      position number then stop searching */
	    if (va->argindex == pos)
		break;

	    node = node->next;
	}
    } else {
	/*-- else start searching from the tail of the list */
	if ((node = arglist->tail) == NULL) {
	    return NULL;
	}
	/*-- skip back up the list incrementing the negative position
	  number as we go until the position number == 0 */
	while (node && (++pos < 0)) {
	    node = node->prev;
	}

	if ((va = (VargInfoPtr) node->data) == NULL) {
	    return NULL;
	}
    }
    /*-- if we ran off the end of the list return NULL */
    if (node == NULL) {
	return NULL;
    }
    /*-- if there is an optarg for this named argument then return
      the optarg's value */
    if (va->optarg != NULL) {
	return va->optarg->value;
    }
    /*-- else return the arg info item's value if it is non NULL */
    else if (va->value != NULL) {
	return va->value;
    }
    /*-- or "TRUE" if it is NULL */
    else {
	return "TRUE";
    }

}



/*++ ***************  function  VappMutExclArgs ****************/
/*
    Purpose:  indicate a list of mutually exclusive arguments

    Prototype:
    int  VappMutExclArgs( char *arg1, ...., NULL );

    Globals changed: none

    Parameter Description:
    a variable argument list, which must be NULL terminated

    Return Values: int
    0       success
    <0      error

    Dependencies:
    _FindArgInfoShort, _AddNewArgInfo, _FindArgInfoLong

    Notes:

*/
/************************************************************************ ++**/

#ifdef VANSI_1
int
VappMutExclArgs(char *arg,...)
#else
int
VappMutExclArgs(arg, va_alist)
char           *arg;

va_dcl
#endif
{
    va_list         args;
    char           *parg;
    VargInfoPtr     va;
    static int      mutexId = 1;

    /*-- start processing variable argument list */
#ifdef VANSI_1
    va_start(args, arg);
#else
    va_start(args);
#endif

    if (arg == NULL)
	return -1;

    /*-- while there are more argument identifiers */
    while ((parg = va_arg(args, char *)) != NULL) {
	/*-- if the identifer is not a single char then find the arg info item
	  using the long name */
	if (strlen(parg) > 1) {
	    /*-- if it doesn't exist then add a new one */
	    if ((va = _FindArgInfoLong(parg)) == NULL)
		va = _AddNewArgInfo(parg, '\0');
	}
	/*-- otherwise find arg info item using single letter */
	else {
	    if ((va = _FindArgInfoShort(*parg)) == NULL)
		va = _AddNewArgInfo(NULL, *parg);
	}
	/*-- indicate the arg info item is part of this
	  the mutually exclusive set */
	va->mutexid = mutexId;
    }

    va_end(args);

    mutexId++;

    return 0;
}


/*++ ***************  function  VsetupArg ****************/
/*
    Purpose:
       set up an arg that this program is interested in

    Prototype:

    int     VappSetupArg( char * arg,           // long version
                          char key,             // short version
                          char *desc,           // desciption
                          char *range,          // argument range
                          BOOLEAN *(func)(),    // validation function
                          int exval,            // exit value
                          BitMask   flags       // control flags
                        );

    Globals changed: none

    Parameter Description:
    arg         == long version of argument which cannot be NULL
    key         == short version which may be '\0' (but shouldn't be)
    desc        == a description for usage printout which may be NULL
    range       == an valid numeric range which may be NULL
    func        == a validation function BOOLEAN *(func)() which may be NULL
    exval       == exit value which may be 0
    flags       == control flags BitMask which may be 0

    Return Values: int
    0    success        setup arg object OK
    <0   error          did not setup arg object
                -1    already exists
                -2    both range and func given (they are mutually exclusive)
                -3    exit value given but flags VARG_EXITFAIL bit not set
                -4    VARG_PRNUSAGE bit set but desc is NULL

    Dependencies:
    Memset, CopyString, LLappend, _FindArgInfoShort, _FindArgInfoLong
    VappArgList

    Notes:
    range is a char * string of the form n1,n2 where n1 is the minimum
    integer value for arg and n2 is the maximum integer value for arg.
    if n1 >= 0 and n2 == -1 then there is no upper limit for validation
    purposes.

    Example Use:

    err = VappSetupArg( "startno",      // setup the startno arg
                'S',                    // recognise '-S' for -startno
                "starting file number", // description for usage printout
                "1,200",                // valid range for arg
                IsValidStartNo,         // validation function
                exit_value,             // if arg causes program to exit
                VARG_MANDATORY_SET |    // control flags
                VARG_FAILFALSE | VARG_EXITFAIL
              );

*/
/************************************************************************ ++**/

#ifdef VANSI_1
int
VappSetupArg(char *arg,		/* long version */
    char key,			/* short version */
    char *desc,			/* desciption */
    char *range,		/* argument range */
    BOOLEAN(*func) (),		/* validation function */
    int exval,			/* exit value */
    BitMask flags		/* control flags */
)
#else
int
VappSetupArg(arg, key, desc, range, func, exval, flags)
char           *arg;		/* long version */
char            key;		/* short version */
char           *desc;		/* desciption */
char           *range;		/* argument range */

BOOLEAN(*func) ();		/* validation function */
int             exval;		/* exit value */
BitMask         flags;		/* control flags */

#endif
{
    VargInfoPtr     va = NULL;
    LLlist      *arglist;

    if (!arg && !key)
	return -1;

    /*-- get a handle on the arg info item list */
    arglist = VappArgList();

    /*-- see if this arg info item already exists */
    if (arg) {
	if ((va = _FindArgInfoLong(arg)) != NULL)
	    return -1;
    } else {
	if ((va = _FindArgInfoShort(key)) != NULL)
	    return -1;
    }

    /*-- validate the arguments to this function */
    /*-- both range and validation function cannot be given */
    if (range && func)
	return -2;

    /*-- exit value must also have exit fail bit set in flags */
    if (exval > 0 && !(flags & VARG_EXITFAIL))
	return -3;

    /*-- print usage bit set must also have a description */
    if ((flags & VARG_PRNUSAGE) && (desc == NULL))
	return -4;

    va = (VargInfo *)Malloc(sizeof(VargInfo));

    /*-- zero initialise the new item */
    Memset(va, 0, sizeof(VargInfo));

    /*-- set the long and short names, and the description */
    if (arg != NULL)
	va->longname = CopyString(arg);

    if (key != '\0')
	va->shortname = key;

    if (desc != NULL)
	va->desc = CopyString(desc);

    /*-- set the range or validation function */
    if (range != NULL)
	va->range = CopyString(range);

    if (func != NULL)
	va->func = func;

    /*-- set the exit value and control flags */
    va->exval = exval;

    va->flags = flags;

    /*-- add a new element to the list */
    LLappend(arglist, va );

    return 0;
}


/*++ ***************  function  VappArgPosition ****************/
/*
    Purpose: determine the ordinal position on the command line
             of a particular argument

    Prototype:
    int     VappArgPosition( char *arg, BOOLEAN flag )

    Globals changed: none

    Parameter Description:
    arg      == named argument to determine position of
    flag     == indicate whether to count from 1st or last arg
                0 == start from head !0 == start from tail

    Return Values: int
    success     == positive or negative value depending on flag
                   and errno set to 0
    error       == named argument not found errno set to -1
                   return value indeterminate

    Dependencies:

    Notes:

*/
/************************************************************************ ++**/

#ifdef VANSI_1
int
VappArgPosition(char *arg, BOOLEAN flag)
#else
int
VappArgPosition(arg, flag)
char           *arg;
BOOLEAN         flag;

#endif
{
    VargInfoPtr     va;
    LLnode     *node;
    LLlist      *arglist;
    int             i;

    errno = 0;

    /*-- get a handle on the argument list */
    arglist = VappArgList();

    /*-- if the flag is 0 then count from the front */
    if (flag == 0) {
	i = 0;
	if ((node = arglist->head) == NULL) {
	    errno = -1;
	    return -1;
	}
	/*-- for each node on the list */
	while (node) {
	    if ((va = (VargInfoPtr) node->data) == NULL) {
		errno = -1;
		return -1;
	    }
	    /*-- if the named arg compares with the long or short
	      name then stop searching */
	    if (strlen(arg) > 1 && va->longname != NULL) {
		if (strncmp(va->longname, arg, strlen(arg)) == 0)
		    break;
	    } else if (va->shortname != '\0') {
		if (va->shortname == *arg)
		    break;
	    }
	    i++;
	    node = node->next;
	}

    }
    /*-- else start counting from the tail */
    else {
	i = -1;
	if ((node = arglist->tail) == NULL) {
	    errno = -1;
	    return -1;
	}
	/*-- for each node on the list */
	while (node) {
	    if ((va = (VargInfoPtr) node->data) == NULL) {
		errno = -1;
		return -1;
	    }
	    /*-- if the named arg compares with the long or short
	      name then stop searching */
	    if (strlen(arg) > 1 && va->longname != NULL) {
		if (strncmp(va->longname, arg, strlen(arg)) == 0)
		    break;
	    } else if (va->shortname != '\0') {
		if (va->shortname == *arg)
		    break;
	    }
	    i--;
	    node = node->prev;
	}
    }
    /*-- if we ran off the end of the list (front or back) then
      set errno to -1 */
    if (node == NULL) {
	errno = -1;
	return -1;
    } else
	return i;

}


/*++ ***************  function  VappArgListStart ****************/
/*
    Purpose:  determine a pointer to the start of a VargInfo list
              for a named argument

    Prototype:
    VargInfo    *VappArgListStart( char *name )

    Globals changed: none

    Parameter Description:
    name     == name of argument to determine

    Return Values: VargInfo *
    success == a non NULL pointer which points to a valid arginfo item
    error   == NULL argument not on command line

    Dependencies:
    _FindArgInfoShort, _FindArgInfoLong

    Notes:

*/
/************************************************************************ ++**/

#ifdef VANSI_1
VargInfoPtr
VappArgListStart(char *name)
#else
VargInfoPtr
VappArgListStart(name)
char           *name;

#endif
{
    VargInfoPtr     va = NULL;

    if (!*name)
	return NULL;

    /*-- if the identifier is a not a single char then find the arg
      using the long name otherwise find the arg for the letter */
    if (strlen(name) > 1) {
	va = _FindArgInfoLong(name);
    } else {
	va = _FindArgInfoShort(*name);
    }

    return va;
}



/*++ ***************  function  VappArgListScan ****************/
/*
    Purpose:  determine char * value for a VargInfo item

    Prototype:
    char    *VappArgListScan( VargInfoPtr *ptr )

    Globals changed: none

    Parameter Description:
    ptr     == address of VargInfoPtr which is updated with address of next
               VargInfo item

    Return Values: int
    success == char * which may be NULL
    error   == errno is set to -1, return value indeterminate

    Dependencies:

    Notes:

*/
/************************************************************************ ++**/

#ifdef VANSI_1
char           *
VappArgListScan(VargInfoPtr * arg)
#else
char           *
VappArgListScan(arg)
VargInfoPtr    *arg;

#endif
{
    char           *tmp;

    if (!*arg)
	return NULL;

    /*-- if the arg item has an optional argument and it's value in non NULL
      then return this value otherwise return the arg item's value */
    if ((*arg)->optarg != NULL && (*arg)->optarg->value != NULL)
	tmp = (*arg)->optarg->value;
    else
	tmp = (*arg)->value;

    /*-- point to the next arg info item in the list before returning */
    *arg = (*arg)->next;

    return tmp;
}



/*++ ***************  function  VappArgINT ****************/
/*
    Purpose:  determine an integer value for a named argument

    Prototype:
    int     VappArgINT( char *arg )

    Globals changed: none

    Parameter Description:
    arg     == long name of argument

    Return Values: int
    success == atoi value of argument, errno is set to 0
    error   == errno is set to -1, return value indeterminate

    Dependencies:
    _FindArgInfoShort, _FindArgInfoLong

    Notes:

*/
/************************************************************************ ++**/

#ifdef VANSI_1
int
VappArgINT(char *arg)
#else
int
VappArgINT(arg)
char           *arg;

#endif
{
    VargInfoPtr     va;

    /*-- get an arg info item for the identifier */
    if (strlen(arg) > 1)
	va = _FindArgInfoLong(arg);
    else
	va = _FindArgInfoShort(*arg);

    errno = 0;

    if (va == NULL)
	return -1;

    /*-- if the identifier is a long name */
    if (strlen(arg) > 1) {
	/*-- if there is an optional argument */
	if (va->optarg != NULL) {
	    /*-- if the value of the optional argument is not a digit */
	    if (!isdigit(*(va->optarg->value)) &&
		*(va->optarg->value) != '-') {
		/*-- then return error condition -1 in errno */
		errno = -1;
		return -1;
	    } else {
		/*-- otherwise return the atoi value of the opt arg */
		return atoi(va->optarg->value);
	    }
	}
	/*-- otherwise return an error condition -1 in errno */
	else {
	    errno = -1;
	    return -1;
	}
    }
    /*-- otherwise it is a short name */
    else {
	/*-- if the value item is non NULL */
	if (va->value != NULL) {
	    /*-- if it does not have leading digits return an error
	      condition -1 in errno */
	    if (!isdigit(*(va->value)) &&
		*(va->value) != '-') {
		errno = -1;
		return -1;
	    }
	    /*-- otherwise return the atoi value of the value item */
	    else {
		return atoi(va->value);
	    }
	}
	/*-- otherwise if there is an optional argument */
	else if (va->optarg != NULL) {
	    /*-- if it does not have leading digits then return an error
	     condition in errno */
	    if (!isdigit(*(va->optarg->value)) &&
		*(va->optarg->value) != '-') {
		errno = -1;
		return -1;
	    }
	    /*-- otherwise return the atoi value of the optioanl argument's
	      value item */
	    else {
		return atoi(va->optarg->value);
	    }
	}
	/*-- otherwise return an error condition -1 in errno */
	else {
	    errno = -1;
	    return -1;
	}
    }

}



/*++ ***************  function  VappArgPosINT ****************/
/*
    Purpose:  determine an integer value for a positional argument

    Prototype:
    int    VappArgPosINT( int pos )

    Globals changed: none

    Parameter Description:
    pos     == ordinal position of argument in the argument list
               pos >= 0  position is from argv[1] onwards
	       pos < 0   position is from argv[argc] backwards

    Return Values: int
    success     -  atoi value of corresponding arg
    error       -  return value is indeterminate and errno is set to;
                   -1   position outside of range of number of args

    Dependencies:

    Notes:

*/
/************************************************************************ ++**/

#ifdef VANSI_1
int
VappArgPosINT(int pos)		/* get positional arg as an integer */
#else
int
VappArgPosINT(pos)		/* get positional arg as an integer */
int             pos;

#endif
{
    VargInfoPtr     va;
    LLnode     *node;
    LLlist      *arglist;

    arglist = VappArgList();

    if (pos >= 0) {
	if ((node = arglist->head) == NULL) {
	    errno = -1;
	    return -1;
	}
	while (node) {
	    if ((va = (VargInfoPtr) node->data) == NULL)
		continue;

	    if (pos == va->argindex)
		break;

	    node = node->next;
	}

    } else {
	if ((node = arglist->tail) == NULL) {
	    errno = -1;
	    return -1;
	}
	while (node && (++pos < 0)) {
	    node = node->prev;
	}

	if ((va = (VargInfoPtr) node->data) == NULL) {
	    errno = -1;
	    return -1;
	}
    }

    if (node == NULL) {
	errno = -1;
	return -1;
    }
    if (va->value != NULL) {
	if (!isdigit(*(va->value)) &&
	    *(va->value) != '-') {
	    errno = -1;
	    return -1;
	} else {
	    return atoi(va->value);
	}
    } else if (va->optarg != NULL) {
	if (!isdigit(*(va->optarg->value)) &&
	    *(va->optarg->value) != '-') {
	    errno = -1;
	    return -1;
	} else {
	    return atoi(va->optarg->value);
	}
    } else {
	errno = -1;
	return -1;
    }

}
