/******************************************************************************
	  CCCC	    A	  BBBBB	  L	EEEEE  N     N	EEEEE	TTTTTTT
	 C    C    A A	  B    B  L	E      NN    N	E	   T
	C	  A   A	  B    B  L	E      N N   N	E	   T
	C	 AAAAAAA  BBBBB	  L	EEEEE  N  N  N	EEEEE	   T
	C        A     A  B    B  L	E      N   N N	E	   T
	 C    C  A     A  B    B  L	E      N    NN  E 	   T
	  CCCC	 A     A  BBBBB	  LLLL	EEEEE  N     N	EEEEE	   T
*******************************************************************************

CableNet Source Module:
	Copyright 1994-1995 (C) CableNet Limited. All Rights Reserved.

    Module Name:		$RCSfile: DfltsIntface.c,v $
    Module Description:	Defaults library interface functions

Description:

Edit History:

	$Log: DfltsIntface.c,v $
 * Revision 2.4  1996/07/03  14:46:17  damian
 * reset value each call to DfltGet*()
 *
 * Revision 2.3  1996/04/28  01:54:54  damian
 * fix SetString()
 *
 * Revision 2.2  1996/02/15  11:40:52  damian
 * allow comments in defaults files
 *
 * Revision 2.1  1996/01/04  18:23:20  V
 * Version 2
 * add DfltGetBool()
 *
 * Revision 1.4  1995/02/27  17:40:04  V
 * cleaned up unused warnings
 *
 * Revision 1.3  1995/02/24  20:45:20  V
 * add DfltGetIntInSet
 *
 * Revision 1.2  1995/02/09  19:01:02  V
 * don't make the set unless we are setting the value
 *
 * Revision 1.1  1995/02/03  18:27:53  V
 * Initial revision
 *


*/

/* RCS identification string (for "what" program) */
static char moduleRCSid[] = "@(#) $Id: DfltsIntface.c,v 2.4 1996/07/03 14:46:17 damian Exp $";

/* must come first header files */
#include "V.h"			/* virtual header file */
#include "Vport.h"		/* port header file */


/*
 *   system header files
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <dirent.h>
#include <sys/param.h>
#include <sys/stat.h>
#include <memory.h>
#include <time.h>

#include "Vlib.h"

/*
 *  local application/library header files
 */

#include "dfltsP.h"


/*++ ***************  function DfltSetString ****************/
/*
	Purpose:        set a string in the current or specified group
	                to a value
	
			can handle tag & value strings of up to 4095
			combined total

	Globals changed: none

	Parameters:
	char     *set    == set to use or NULL for current set
	char     *group  == group to use or NULL for current group
	char     *tag    == tag identifier
	char     *val    == string value for tag

	Return Values:  int
	0       success
	< 0     error

	Dependencies:
	StripLTBlanks(), CopyString(), DfltMakeSet(), DfltDeleteGroup()
	DfltSetCurrentGroup(), DfltSetCurrentSet(), TerminateAtNextChar()

*/
/************************************************************************ ++**/

#ifdef VANSI_1
int
DfltSetString (DfltsInfo * di)
#else
int
DfltSetString (di)
     DfltsInfo *di;
#endif
{
  char buf[4096];
  char namebuf[MAXPATHLEN];
  char *str, *defsdir;
  char *ftag, *t, *p;
  FILE *fp, *tempfp;
  int found = FALSE;

  /*-- if a set was specified then set the current set to this set */
  if (di->set)
    DfltSetCurrentSet (di);

  DfltMakeSet (DfltGetCurrentSet (di->base));

  /*-- if a group was specified then set the current group to this group */
  if (di->group)
    DfltSetCurrentGroup (di);

  /*-- if the tag wasn't specified but the group was then delete the entire
      group */
  if (!di->tag && di->group)
    {				/* delete the entire group */
      DfltDeleteGroup (DfltGetCurrentPath (di->base));
      return 0;
    }

  if ((defsdir = UserDefaults ()) == NULL)
    if ((defsdir = LocalDefaults ()) == NULL)
      if ((defsdir = GlobalDefaults ()) == NULL)
	defsdir = "./";

  sprintf (namebuf, "%s.tmp", DfltGetCurrentPath (di->base) );

  if ((p = strrchr(namebuf, '/')) != NULL) {
      *p = '\0';
      DfltMakeSet (namebuf);	/* ensure directory exists */
      *p = '/';
  }

  if ((tempfp = fopen (namebuf, "w")) == NULL)
    {
      fprintf (stderr, "%s couldn't open temporary file %s\n",
	       ProgramName (), namebuf);
      return -1;
    }

  /*-- copy all but the tag line from group file to temp file */
  if ((fp = fopen (DfltGetCurrentPath (di->base), "r")) != NULL)
    {
      if (remove (DfltGetCurrentPath (di->base)) != 0)
	/* disable access by other processes */
	fprintf (stderr, "%s remove %s failed\n", ProgramName (),
		 DfltGetCurrentPath (di->base));
      /*-- while there are more lines in the group file */
      while ((str = fgets (buf, 4095, fp)) != NULL)
	{
	  t = CopyString (str);
	  ftag = t;

	  /*-- determine if this is a 'tag = value' line */
	  if (buf[0] != '#' && TerminateAtNextChar (t, "=") != NULL)
	    {
	      StripLTBlanks (ftag);

	      /*--if the tag matches and a value was specified */
	      if (strcmp (di->tag, ftag) == 0)
		{
		  found = TRUE;
		  if (di->val)
		    /*-- replace the value */
		    fprintf (tempfp, "%s = %s\n", di->tag, di->val);
		}
	      else
		/*-- else copy the line to the temp file */
		fputs (str, tempfp);
	    }
	  else
	    /*-- if there was no tag then copy the line to the temp file */
	    fputs (str, tempfp);
	  Free (t);
	}
    }

  /*-- if the tag didn't exist in the group file then add it to the end */
  if ((fp == NULL) || (found == FALSE))
    if (di->val)
      fprintf (tempfp, "%s = %s\n", di->tag, di->val);

  /*-- close both files */
  if (fp)
    fclose (fp);

  fclose (tempfp);

  /*-- and move the temp file to the group file */
  if (rename (namebuf, DfltGetCurrentPath (di->base)) != 0)
    {
      fprintf (stderr, "%s couldn't rename tempfile %s to groupfile %s",
	       ProgramName (), namebuf, DfltGetCurrentPath (di->base));
      return -1;
    }

  return 0;
}



static char *value = NULL;

/*++ **********************  function  DfltGetStringInSet ***************

	Purpose:  get a string value from a particular set

	Return Values: char *

******************************************************************* ++*/
#ifdef VANSI_1
static char *
DfltGetStringInSet (DfltsInfo * di)
#else
static char *
DfltGetStringInSet (di)
     DfltsInfo *di;
#endif
{
  char buf[4096], fbuf[1024];
  char *str;
  char *ftag = NULL, *fval = NULL;
  FILE *fp;

  if (!di)
    return NULL;

  sprintf(fbuf, "%s", DfltGetCurrentPath (di->base) );

  /*-- open the current group file given by current path */
  if ((fp = fopen (fbuf,"r")) == NULL)
    return NULL;

  if ( value )
      Free(value);

  value = NULL;

  /*-- read each line in the file */
  while ((str = fgets (buf, 4095, fp)) != NULL)
    {

      if ( buf[0] == '#' )  /* a comment line */
	  continue;

      /*-- seperate into tag and value fields */
      if (SeperateTagAndValue (str, &ftag, &fval) < 0)
	continue;

      /*-- remove leading and trailing blanks from tag & value */
      StripLTBlanks (ftag);
      StripLTBlanks (fval);
      StripLTBlanks (ftag);
      StripLTBlanks (fval);

      /*-- if this left tag or value as empty strings then continue */
      if (!ftag[0] || !fval[0])
	continue;

      /*-- if this is the tag we are looking for then set the return value */
      if (strcmp (di->tag, ftag) == 0)
	{
	  value = CopyString (fval);
	  break;
	}
    }

  /*-- close group file and return the found value or default string */
  fclose (fp);

  return value;

}


/*++ ***************  function DfltGetString ****************/
/*
	Purpose:        get a string value for a tag from a set/group
	
	Globals changed: none

	Parameters:
	DfltInfo *info structure which contains
	   char     *set    == set to use or NULL for current set
	   char     *group  == group to use or NULL for current group
	   char     *tag    == tag identifier
	   char     *default    == string to return if tag-value not found
	   int      base     == which set to use global/local/user

	Return Values:
	char     *val     ==   string value corresponding to tag        or
	char     *default ==   string user passed as a default

	Dependencies:
	StripLTBlanks(), DfltSetCurrentGroup(), Free(), DfltSetCurrentSet()

*/
/************************************************************************ ++**/
#ifdef VANSI_1
char *
DfltGetString (DfltsInfo * di)
#else
char *
DfltGetString (di)
     DfltsInfo di;
#endif
{
  char *val = NULL;

  if (value)
    Free (value);

  value = NULL;

  /*-- if set &/or group specified set up current set/group */
  if (di->set)
    DfltSetCurrentSet (di);

  if (di->group)
    DfltSetCurrentGroup (di);

  /* if base is any base set then search all sets */
  if (di->base == ANY_DFLT_SET)
    {

      di->base = USER_DFLT_SET;
      do
	{
	  val = DfltGetStringInSet (di);
	  di->base--;
	}
      while (val == NULL && di->base > ANY_DFLT_SET);

      return val;
    }
  else
    /* otherwise search a particular base set */
    val = DfltGetStringInSet (di);

  return val;
}


/*++ ***************  function DfltSetInt ****************/
/*
	Purpose:         set a tag value to an number
	
	Globals changed: none

	Parameters:
	char     *set    == set to use or NULL for current set
	char     *group  == group to use or NULL for current group
	char     *tag    == tag identifier
	int      val     == integer value for the tag

	Return Values: int
	0       success
	< 0     error

	Dependencies:
	DfltSetString()

*/
/************************************************************************ ++**/

#ifdef VANSI_1
int
DfltSetInt (DfltsInfo * di)
#else
int
DfltSetInt (di)
     DfltsInfo *di;
#endif
{
  char buf[32];

  sprintf (buf, "%d", di->num);

  di->val = buf;

  return DfltSetString (di);

}



/*++ ***************  function DfltGetInt ****************/
/*
	Purpose:     get an integer value for a tag from a set/group
	
	Globals changed: none

	Parameters:
	char     *set    == set to use or NULL for current set
	char     *group  == group to use or NULL for current group
	char     *tag    == tag identifier
	int      default == integer to return if tag-value not found

	Return Values:
	int      val    ==   integer value corresponding to tag        or
	int      default ==  integer value user passed as a default

	Dependencies:
	StripLTBlanks(), DfltSetCurrentGroup(), DfltSetCurrentSet()

*/
/************************************************************************ ++**/

#ifdef VANSI_1
int
DfltGetInt (DfltsInfo * di)
#else
int
DfltGetInt (di)
     DfltsInfo *di;
#endif
{
  char *val;
  int fint;

  fint = di->num;

  /*-- if set &/or group specified then set them up */
  if (di->set)
    DfltSetCurrentSet (di);

  if (di->group)
    DfltSetCurrentGroup (di);

  /* if base is any base set then search all sets */
  if (di->base == ANY_DFLT_SET)
    {

      di->base = USER_DFLT_SET;
      do
	{
	  val = DfltGetStringInSet (di);
	  di->base--;
	}
      while (val == NULL && di->base > ANY_DFLT_SET);

      if (val == NULL)
	return di->num;
      else
	return atoi (val);
    }
  else
    /* otherwise search a particular base set */
    val = DfltGetStringInSet (di);

  if (val != NULL)
    return atoi (val);
  else
    return di->num;

}


/*++ ***************  function DfltGetBool ****************/
/*
	Purpose:     get an boolean value for a tag from a set/group
	
	Globals changed: none

	Parameters:
	     DfltsInfo *di;

	Return Values:
	0   == FALSE  (value not defined as TRUE or YES)
	1   == TRUE  (value defined as TRUE or YES)

	Dependencies:
	StripLTBlanks(), DfltSetCurrentGroup(), DfltSetCurrentSet()

*/
/************************************************************************ ++**/

#ifdef VANSI_1
int
DfltGetBool (DfltsInfo * di)
#else
int
DfltGetBool (di)
     DfltsInfo *di;
#endif
{
  char *val;

  /*-- if set &/or group specified then set them up */
  if (di->set)
    DfltSetCurrentSet (di);

  if (di->group)
    DfltSetCurrentGroup (di);

  /* if base is any base set then search all sets */
  if (di->base == ANY_DFLT_SET)
    {

      di->base = USER_DFLT_SET;
      do
	{
	  val = DfltGetStringInSet (di);
	  di->base--;
	}
      while (val == NULL && di->base > ANY_DFLT_SET);

      if (val == NULL)
	return FALSE;
      else {
	  if ( strncasecmp(val, "TRUE", 4) == 0 ||
	      strncasecmp(val, "YES", 3) == 0 ||
	      strncasecmp(val, "1", 1) == 0 )
	      return TRUE;
	  else
	      return FALSE;
      }
    }
  else
    /* otherwise search a particular base set */
    val = DfltGetStringInSet (di);


  if (val == NULL)
      return FALSE;
  else {
      if ( strncasecmp(val, "TRUE", 4) == 0 ||
	  strncasecmp(val, "YES", 3) == 0 ||
	  strncasecmp(val, "1", 1) == 0 )
	  return TRUE;
  }

  return FALSE;

}


/*++ ***************  function DfltGetSet ****************/
/*
	Purpose:        get a comma seperated list of all entries in a set
	
	Globals changed: none

	Parameters:
	char        *set        == set to list

	Return Values:
	char     *list   == allocated buffer of list of set entries

	Dependencies:
	Malloc(), DfltSetCurrentSet()

*/
/************************************************************************ ++**/

#ifdef VANSI_1
char *
DfltGetSet (DfltsInfo * di)
#else
char *
DfltGetSet (di)
     DfltsInfo *di;
#endif
{
  char *t, buf[MAXPATHLEN];
  DIR *dir;
  struct dirent *dent;
  int tsize = 0;

  /*-- if a set specified then make it the new current set
      else use the current set */
  if (di->set)
    DfltSetCurrentSet (di);

  /*-- allocate some space for the list */
  t = (char *) Malloc (1024);
  tsize = 1024;
  memset (t, 0, 1024);

  sprintf (buf, "%s/%s", DfltGetBaseSet (di->base),
	   DfltGetCurrentSet (di->base));

  /*-- open the set */
  if ((dir = opendir (buf)) == NULL)
    {
      return NULL;
    }
  else
    {
      /*-- read successive entries in the set */
      while ((dent = readdir (dir)) != NULL)
	{
	  int newsize = 0;

	  /*-- if this entry is not "." or ".." directories */
	  if (strcmp (dent->d_name, ".") == 0)
	    continue;
	  if (strcmp (dent->d_name, "..") == 0)
	    continue;

	  /*-- calculate the new list length */
	  if (*t)
	    newsize = strlen (t) + strlen (dent->d_name) + 2;
	  else
	    newsize = strlen (dent->d_name) + 2;

	  /*-- and if it's longer than current space length
	      realloc the space to add another 1024 bytes */
	  if (newsize > tsize)
	    {
	      t = (char *) Realloc (t, (uint) tsize + 1024);
	      tsize += 1024;
	    }
	  /*-- add the name to the list */
	  if (*t)
	    {
	      strcat (t, ",");
	      strcat (t, dent->d_name);
	    }
	  else
	    strcpy (t, dent->d_name);
	}
    }
  /*-- return the list */
  return t;
}


/*++ ***************  function  DfltGetGroup ****************/
/*
	Purpose:     determine a list of tags in a group
	
	Globals changed: none

	Parameters:
	char        *set        == set name for group
	char        *group      == group name to list

	Return Values:
	char        *list       == allocated buffer of list of group entries

	Dependencies:

*/
/************************************************************************ ++**/

#ifdef VANSI_1
char *
DfltGetGroup (DfltsInfo * di)
#else
char *
DfltGetGroup (di)
     DfltsInfo *di;
#endif
{
  char buf[4096];
  char *str, *t;
  char *ftag, *fval;
  FILE *fp;
  int tsize;

  /*-- if set &/or group specified then set up new current set/group */
  if (di->set)
    DfltSetCurrentSet (di);

  if (di->group)
    DfltSetCurrentGroup (di);

  /*-- open the current/new path group file */
  if ((fp = fopen (DfltGetCurrentPath (di->base), "r")) == NULL)
    {
      fprintf (stderr, "%s invalid group %s\n", ProgramName (),
	       DfltGetCurrentPath (di->base));
      return NULL;
    }

  /*-- allocate some space */
  t = (char *) Malloc (1024);
  tsize = 1024;
  memset (t, 0, 1024);

  /*-- read successive lines from the group file */
  while ((str = fgets (buf, 4095, fp)) != NULL)
    {
      int newsize = 0;

      /*-- seperate into tag and value fields */
      if (SeperateTagAndValue (str, &ftag, &fval) < 0)
	continue;

      /*-- remove leading & trailing blanks from tag */
      StripLTBlanks (ftag);

      /*-- and add it to the list */
      if (ftag)
	{
	  if (*t)
	    newsize = strlen (t) + strlen (ftag) + 2;
	  else
	    newsize = strlen (ftag) + 2;

	  /*-- reallocate the list buffer space if addition goes past the end
	      of the buffer space */
	  if (newsize > tsize)
	    {
	      t = (char *) Realloc (t, (uint) tsize + 1024);
	      tsize += 1024;
	    }
	  if (*t)
	    {
	      strcat (t, ",");
	      strcat (t, ftag);
	    }
	  else
	    strcpy (t, ftag);
	}
    }
  /*-- return tag list */
  return t;
}
