/******************************************************************************
	  CCCC	    A	  BBBBB	  L	EEEEE  N     N	EEEEE	TTTTTTT
	 C    C    A A	  B    B  L	E      NN    N	E	   T
	C	  A   A	  B    B  L	E      N N   N	E	   T
	C	 AAAAAAA  BBBBB	  L	EEEEE  N  N  N	EEEEE	   T
	C        A     A  B    B  L	E      N   N N	E	   T
	 C    C  A     A  B    B  L	E      N    NN  E 	   T
	  CCCC	 A     A  BBBBB	  LLLL	EEEEE  N     N	EEEEE	   T
*******************************************************************************

CableNet Source Module:
	Copyright 1994-1995 (C) CableNet Limited. All Rights Reserved.

    Module Name:		$RCSfile: HashLList.c,v $
    Module Description:	handle linked lists with hashing


Description:


Edit History:

	$Log: HashLList.c,v $
 * Revision 1.5  1996/04/17  15:19:30  damian
 * exclude malloc.h for bsdi
 *
	Revision 1.4  1995/03/14 01:33:45  V
	if user destroy function exists use that else use free

 * Revision 1.3  1995/02/27  17:40:04  V
 * cleaned up unused warnings
 *
 * Revision 1.2  1995/02/08  00:10:04  V
 * replace LLmalloc with Malloc
 *
 * Revision 1.1  1995/01/09  18:21:23  V
 * Initial revision
 *


*/

/* RCS identification string (for "what" program) */
static char moduleRCSid[] = "@(#) $Id: HashLList.c,v 1.5 1996/04/17 15:19:30 damian Exp $";

/* must come first header files */
#include "V.h"			/* virtual header file */
#include "Vport.h"		/* port header file */


/*
 *  system Include files
 */

#include <stdio.h>

#if ! defined(Vbsdi)
#include <malloc.h>
#endif

#include <sys/types.h>
#include <memory.h>

/*
 *   public project include files
 */

#include "Vllist.h"		/* public header file */
#include "Vlib.h"


/*
 * Function: 	int LLaction(LLlist *list, LLActFunc actfunc);
 *
 * Description: This will traverse the entire linked list pointed to by
 *              list and call actfunc (with a pointer to the userdata)
 *              to perform some function for each element.
 *
 *              If actfunc() returns non-zero then the traverse is aborted.
 *
 * Parameters:	LLlist *list     -  ptr to head of list.
 *		LLActFunc actfunc  -  ptr to function to call
 *
 * Returns:	0 if got to end of list or return code from actfunc.
 *
 * Notes:       actfunc() must not change the list in any way or the behavior
 *              is undefined.
 */
#ifdef VANSI_1
int
LLaction (LLlist * list, LLActFunc actfunc)
#else
int
LLaction (list, actfunc)
     LLlist *list;
     LLActFunc actfunc;
#endif
{
  void *udp;
  int res;
  LLnode *node;

  node = list->head;

  while (node)
    {
      udp = node->data;
      if ((res = actfunc (udp)) != 0)
	return (res);
      node = node->next;
    }
  return (0);

}

/*
 * Function: 	LLnode *LLinsert(LLlist *list, LLnode *after, void *udp);
 *
 * Description: Add an item into a linked list after
 *              a specified item.
 *
 * Parameters:
 *	LLlist		*list - pointer to linked list
 *	LLnode 		after  - item to link after.
 *	void 		*udp - user data ptr to put in item.
 *
 * Returns:     Pointer to new item or NULL if could not add.
 *
 */

#ifdef VANSI_1
LLnode *
LLinsert (LLlist * list, LLnode * after, void *udp)
#else
LLnode *
LLinsert (list, after, udp)
     LLlist *list;
     LLnode *after;
     void *udp;
#endif
{
  LLnode *newnode;

  if (list == NULL)
    return NULL;

  list->count++;

  if ((newnode = (LLnode *) Malloc (sizeof (LLnode))) == NULL)
    return (NULL);

  newnode->data = udp;

  /*-- if the 'after' node is null prepend the new node to the list */
  if (after == (LLnode *) NULL)
    {
      /* if the list is currently empty */
      if ((newnode->next = list->head) == NULL)
	{
	  list->tail = newnode;
	}
      else
	{
	  newnode->next->prev = newnode;
	}

      list->head = newnode;
      newnode->prev = (LLnode *) NULL;
    }
  else
    {
      /*-- otherwise insert new node after the given list node  */
      if ((newnode->next = after->next) == (LLnode *) NULL)
	{
	  /* the new node is being appended to the list */
	  list->tail = newnode;
	}
      else
	{
	  newnode->next->prev = newnode;
	}

      newnode->prev = after;
      after->next = newnode;
    }

  return (newnode);

}

/*
 * Function: 	LLnode *LLinsertB(LLlist *list, LLnode *before, void *udp);
 *
 * Description: Add an item into a linked list before
 *              a specified item.
 *
 * Parameters:
 *	LLlist		*list - pointer to linked list
 *	LLnode 		before  - item to link before.
 *	void 		*udp - user data ptr to put in item.
 *
 * Returns:     Pointer to new item or NULL if could not add.
 *
 */

#ifdef VANSI_1
LLnode *
LLinsertB (LLlist * list, LLnode * before, void *udp)
#else
LLnode *
LLinsertB (list, before, udp)
     LLlist *list;
     LLnode *before;
     void *udp;
#endif
{
  LLnode *prev;


  if (list == NULL)
    return NULL;

  /*-- if the 'before' node is null append the new node to the list */
  if (before == (LLnode *) NULL)
    {
      /* append new node to list */
      prev = list->tail;

    }
  else
    {
      /*-- otherwise insert new node before the given list node  */
      prev = before->prev;

    }

  return LLinsert (list, prev, udp);

}

#ifdef VANSI_1
LLnode *
NextHashNode (LLlist * list, void *udp)
#else
LLnode *
NextHashNode (list, udp)
     LLlist *list;
     void *udp;

#endif
{
  LLnode *node;
  int i, val = list->hash (udp);

  if (val + 1 < list->numhash)
    node = list->nodes[val + 1];
  else
    node = NULL;

  if (node == NULL)
    {
      for (i = val + 1; i < list->numhash; i++)
	if ((node = list->nodes[i]) != NULL)
	  break;
    }

  return node;
}

#ifdef VANSI_1
LLnode *
PrevHashNode (LLlist * list, void *udp)
#else
LLnode *
PrevHashNode (list, udp)
     LLlist *list;
     void *udp;

#endif
{
  LLnode *node;
  int i, val = list->hash (udp);

  if (val + 1 < list->numhash)
    node = list->nodes[val + 1];
  else
    node = NULL;

  if (node == NULL)
    {
      for (i = val; i >= 0; i--)
	if ((node = list->nodes[i]) != NULL)
	  break;
    }

  return node;
}

/*
 * Function: 	 LLnode *LLadd(LLlist *list, void *udp);
 *
 * Description: Add an item into a linked list anywhere in the list
 *
 * Parameters:	LLlist	*list	- list pointer
 *	        void *udp - user data ptr to put in item.
 *
 * Returns:     Pointer to new item or NULL if could not add.
 *
 */

#ifdef VANSI_1
LLnode *
LLadd (LLlist * list, void *udp)

#else
LLnode *
LLadd (list, udp)
     LLlist *list;
     void *udp;

#endif
{
  int val;
  LLnode *hashnode, *nexthash, *node, *prevhash;

  /*-- if there is no hash function append the item and return */
  if (list->hash == NULL)
    return LLappend (list, udp);
  else
    /*-- otherwise determine the hash value */
    val = list->hash (udp);

  /*-- make sure it is a valid hash value */
  if (val < 0 || val >= list->numhash)
    return NULL;

  /*-- if the list is empty then append the item and setup it's hash
      node pointer and return */
  if (LLcount (list) == 0)
    {
      node = LLappend (list, udp);

      list->nodes[val] = node;

      return node;
    }

  hashnode = list->nodes[val];

  nexthash = NextHashNode (list, udp);

  prevhash = PrevHashNode (list, udp);

  /*-- if this is the first node with this hash value */
  if (hashnode == NULL)
    {

      /*-- if there is a greater hash node
	  insert the item before the next hash node */
      if (nexthash != NULL)
	{
	  node = LLinsertB (list, nexthash, udp);

	  list->nodes[val] = node;

	  return node;
	}
      else
	{

	  /*-- otherwise append it to the list */

	  node = LLappend (list, udp);

	  list->nodes[val] = node;

	  return node;
	}
    }

  /*-- this node is not the first with this hash value so find it's
       place in the list starting from hashnode */
  if (list->comp)
    {
      node = hashnode;

      while (node != nexthash)
	{
	  if (list->comp (node->data, udp) >= 0)
	    {
	      return LLinsert (list, node->prev, udp);
	      break;
	    }
	  node = node->next;
	}

      if (node == nexthash)
	if (node == NULL)
	  {
	    return LLappend (list, udp);
	  }
	else
	  {
	    return LLinsert (list, node->prev, udp);
	  }
    }
  else
    {
      if (nexthash != NULL)
	{
	  return LLinsert (list, nexthash->prev, udp);
	}
      else
	return LLappend (list, udp);
    }

  return NULL;
}


/*
 * Function: 	  LLnode *LLappend(LLlist *list, void *udp);
 *
 * Description: Add an item into a linked list anywhere in the list
 *
 * Parameters:	LLlist	*list	- list pointer
 *	        void *udp - user data ptr to put in item.
 *
 * Returns:     Pointer to new item or NULL if could not add.
 *
 */

#ifdef VANSI_1
LLnode *
LLappend (LLlist * list, void *udp)
#else
LLnode *
LLappend (list, udp)
     LLlist *list;
     void *udp;
#endif
{
  return (LLinsert (list, list->tail, udp));
}

/*
 * Function: 	  LLnode *LLpush(LLlist *list, void *udp );
 *
 * Description: Add an item onto the front of a linked list
 *
 * Parameters:	LLlist	*list	- list pointer
 *	        void *udp - user data ptr to put in item.
 *
 * Returns:     Pointer to new item or NULL if could not add.
 *
 */

#ifdef VANSI_1
LLnode *
LLpush (LLlist * list, void *udp)
#else
LLnode *
LLpush (list, udp)
     LLlist *list;
     void *udp;
#endif
{
  return LLinsertB (list, list->head, udp);
}


/*
 * Function: 	void *LLpop(LLlist *list);
 *
 * Description: pop an item from a linked list
 *
 * Parameters:	LLlist	*list	- list pointer
 *
 * Returns:     void *data pointer to user data item
 *
 */
#ifdef VANSI_1
void *
LLpop (LLlist * list)
#else
void *
LLpop (list)
     LLlist *list;
#endif
{
  LLnode *node, *hash;
  void *vdp;
  int val;

  node = list->head;
  vdp = node->data;

  list->count--;
  list->head = list->head->next;

  /*-- unlikely but if the head is also a hashed node then move
      the hash node pointer to the next node */
  if (list->hash)
    {
      if ((val = list->hash (vdp)) >= 0 &&
	  (val < list->numhash))
	{
	  hash = list->nodes[val];

	  if (hash == node)
	    list->nodes[val] = hash->next;
	}

    }

  free (node);

  return (vdp);

}

/*++ ****************  function LLcreate *********************/
/*
    Purpose:    Create a new linked list.

    Parameters: None

    Return Values:    LLlist *
    list     = a pointer to the head of a new linked list        or
    NULL    if could not create

    Dependencies:


*/
/*********************************************************************** ++*/

LLlist *
LLcreate (int hash (), int comp (), void dest (), int numhash)
{
  LLlist *list;
  unsigned extsize;

  extsize = (unsigned) sizeof (LLlist) + (sizeof (LLnode *) * numhash);

  if ((list = (LLlist *) malloc (extsize)) == NULL)
    return (NULL);		/* No memory */

  memset (list, 0, sizeof (LLlist) + (sizeof (LLnode *) * (numhash)));

  if (comp != NULL)
    list->comp = comp;

  if (hash != NULL)
    list->hash = hash;

  if (dest != NULL)
    list->dest = dest;

  list->numhash = numhash;

  return (list);
}

/*++ ****************  function LLdelete *********************/
/*
    Purpose:    Delete an item from a linked list

    Prototype: 	void LLdelete(LLlist *list)

    Parameters:
    LLlist	*list		pointer to linked list
    LLnode	*node		pointer to node to delete

    Return Values:

*/
/*********************************************************************** ++*/

#ifdef VANSI_1
void
LLdelete (LLlist * list, LLnode * node)
#else
void
LLdelete (list, node)
     LLlist *list;
     LLnode *node;

#endif
{
  LLnode *hash = NULL;
  int val;

  /*-- if deleting the current node move the current node pointer to
      the next node */
  if (list->current == node)
    list->current = node->next;

  /*-- likewise for the head pointer */
  if (list->head == node)
    list->head = node->next;
  else
    node->prev->next = node->next;


  /*-- and to the previous node for the tail pointer */
  if (list->tail == node)
    list->tail = node->prev;
  else
    node->next->prev = node->prev;

  /*-- determine if the deleted node is pointed to by any hash node
      pointers */
  if (list->hash != NULL)
    {
      /*-- determine the hash value, and check it is within range */
      if (((val = list->hash (node->data)) >= 0) &&
	  (val < list->numhash))
	{
	  /*-- get the hash node pointer */
	  hash = list->nodes[val];

	  /*-- if we are deleting the hash node then point the hash node
	      pointer to the next node */
	  if (hash == node)
	    {
	      list->nodes[val] = hash->next;
	    }
	}
    }

  list->count--;
  LLfree (node);

}


/*++
 *  Name:       LLdestroy_list
 *
 *  Purpose:	local function to actually do the destroying
 *
 *  Globals changed: none
 *
 *  Parameter Description:
 *	LLlist		*list	- list pointer
 *	int		all	- destroy head flag
 *
 *  Return Values: void
 *
 *  Dependencies: none
 *
 *  Notes:
 *
++**/
#ifdef VANSI_1
static void
LLdestroy_list (LLlist * list, int all)
#else
static void
LLdestroy_list (list, all)
     LLlist *list;
     int all;
#endif
{
  void *udp;
  LLnode *node, *next;

  node = list->head;
  while (node)
    {
      next = node->next;
      udp = node->data;

      /*-- if a user defined destroy function was provided then call it */
      if (list->dest)
	list->dest (udp);
      else
	free (udp);		/* free the user data item */

      LLdelete (list, node);

      node = next;
    }

  /*-- if the all flag is true then also delete the list header */
  if (all == 1)
    LLfree (list);

}


/*++ ****************  function LLdestroy *********************/
/*
    Purpose:   Delete every item on a linked list and the linked list
               header

    Prototype:  void LLdestroy(LLlist *list)

    Parameters:
    LLlist   *list         -   ptr to linked list

    Return Values: void

*/
/*********************************************************************** ++*/

#ifdef VANSI_1
void
LLdestroy (LLlist * list)
#else
void
LLdestroy (list)
     LLlist *list;
#endif
{
  LLdestroy_list (list, 1);
}


/*++
 *  Name:       LLclear
 *
 *  Purpose:	clear a linked list leaving the header intact but initialised
 *
 *  Globals changed: none
 *
 *  Parameter Description:
 *	LLlist			*list	- linked list to clear
 *
 *  Return Values: void
 *
 *  Dependencies: LLdestroy_list
 *
 *  Notes:
 *
++**/

#ifdef VANSI_1
void
LLclear (LLlist * list)
#else
void
LLclear (list)
     LLlist *list;
#endif
{
  int idx;

  LLdestroy_list (list, 0);

  /*-- initialise the list header pointers and node hashlist */
  list->head = list->tail = list->current = NULL;
  list->count = 0;

  for (idx = 0; idx < list->numhash; idx++)
    list->nodes[idx] = NULL;

}



#ifdef VANSI_1
void
LLfreeBuff (LLlist * list)
#else
void
LLfreeBuff (list)
     LLlist *list;

#endif
{
  LLnode *node, *next;

  if (!list)
    return;

  node = list->head;
  while (node)
    {
      next = node->next;
      free (node->data);
      LLdelete (list, node);	/* node is pointless after freeing data so
				 * delete it */
      node = next;
    }
}

int
LLremove (LLlist * list, void *udp, int val)
{
  LLnode *node, *next;
  void *nudp;
  int deleted = 0;

  /*-- if the list or it's compare function or destroy function
      is null then return NULL */
  if (!list || !list->comp || !list->dest)
    return -1;

  /*-- start from the hash node or the head of the list */
  if ((node = LLhashNode (list, udp)) == NULL)
    node = list->head;

  /*-- for each node on the remainder */
  while (node)
    {
      next = node->next;
      nudp = node->data;

      /*-- if the compare function return value matches the parameter
	  value then remove this node */
      if (list->comp (nudp, udp) == val)
	{
	  list->dest (nudp);
	  LLdelete (list, node);
	  deleted++;
	}
      node = next;
    }

  return deleted;
}


#ifdef VANSI_1
void *
LLgetNth (LLlist * list, int n)
#else
void *
LLgetNth (list, n)
     LLlist *list;
     int n;
#endif
{
  LLnode *node;

  if (n > LLcount (list) || n < 0)
    return NULL;

  node = list->head;

  while (node && (n > 0))
    {
      n--;
      node = node->next;
    }

  return node->data;

}



/*
 * Function: 	void *LLfind(LLlist *list, void *udp, int val);
 *
 * Description: find an item in a linked list
 *
 * Parameters:
 *	LLlist		*list - pointer to linked list
 *	void 		*udp - user data ptr to look for.
 *	int		val - return value from comp function for match
 *
 * Returns:    void * 	Pointer to item or NULL if could not find.
 *
 */

#ifdef VANSI_1
void *
LLfind (LLlist * list, void *udp, int val)
#else
void *
LLfind (list, udp, val)
     LLlist *list;
     void *udp;
     int val;
#endif
{
  LLnode *node;

  if ((node = LLfindNode (list, udp, val)) != NULL)
    return node->data;
  else
    return NULL;
}


/*++
 *  Function:		LLfindNode( LLlist *list, void *udp, int val )
 *
 *  Purpose:	find a node for a user data pointer
 *
 *  Globals changed: none
 *
 *  Parameter Description:
 *	LLlist		*list  == list to search
 *	void		*udp   == user item to look for
 *	int		val    == val from compare function for a match
 *
 *  Return Values:  LLnode *
 *
 *  Dependencies:
 *
 *  Notes:
 *
++**/

LLnode *
LLfindNode (LLlist * list, void *udp, int val)
{
  LLnode *node = NULL;

  /*-- if the list or it's compare function is null return NULL */
  if (list == NULL || list->comp == NULL)
    return NULL;

  /*-- if the list is hashed then start searching from the hash node */
  /*-- otherwise start searching from the list head */
  if ((node = LLhashNode (list, udp)) == NULL)
    node = list->head;

  /*-- foreach node on the remainder of the list */
  while (node)
    {

      /*-- call the list compare function with the user item parameter
	  and this user item from the list
	  if the value returned from the compare function matches the
	  value parameter then we have found a match so return the node */

      if (list->comp (node->data, udp) == val)
	break;

      node = node->next;
    }

  /*-- otherwise return null if we got to the end of the list */
  return node;

}


#ifdef VANSI_1
int
LLdumpNode (LLnode * node)
#else
int
LLdumpnode (node)
     LLnode *node;
#endif
{

  if (node == NULL)
    return -1;

  /*-- dump a linked list item pointer to stdout */
  printf ("0x%p -- to 0x%p LLnode data:0x%p next:0x%p prev:0x%p\n",
	  (char *) node, (char *) node + sizeof (LLnode),
	  node->data, node->next, node->prev);

  return 0;
}

#ifdef VANSI_1
int
LLdumpList (LLlist * list)
#else
int
LLdumpList (list)
     LLlist *list;

#endif
{
  LLnode *node;

  if (list == NULL)
    return -1;

  node = list->head;
  while (node)
    {
      LLdumpNode (node);
      node = node->next;
    }

  return 0;
}


#ifdef VANSI_1
int
LLcount (LLlist * list)
#else
int
LLcount (list)
     LLlist *list;

#endif
{

  return list->count;

}


/*
 * Function: 	void *LLmerge(LLlist *list1, LLlist *list2, int val);
 *
 * Description: merge two linked lists merge list2 into list1, removing
 *		items from list2 which go into list1
 *
 * Parameters:
 *	LLlist		*list1, *list2 - pointer to linked lists
 *      int		val	== comp function return value when items match
 *
 * Returns: int
 *
 */

#ifdef VANSI_1
int
LLmerge (LLlist * list1, LLlist * list2, int val)
#else
int
LLmerge (list, list2, val)
     LLlist *list1;
     LLlist *list2;
     int val;
#endif
{
  LLnode *fnode, *fnext;
  void *fudp = NULL, *tudp = NULL;

  if (list1 == NULL || list1->comp == NULL)
    return -1;

  fnode = list2->head;

  while (fnode)
    {
      fnext = fnode->next;
      fudp = fnode->data;

      /*-- if the value exists on the dest list then don't move it across */
      if ((tudp = LLfind (list1, fudp, val)) == NULL)
	{
	  /*-- not found on the dest list so move it */
	  LLadd (list1, fudp);
	  LLdelete (list2, fnode);
	}

      fnode = fnext;
    }

  return 0;

}



#ifdef VANSI_1
LLnode *
LLhashNode (LLlist * list, void *udp)
#else
LLnode *
LLhashNode (list, udp)
     LLlist *list;
     void *udp;
#endif
{
  int idx;

  if (list->hash == NULL)
    return NULL;

  if ((idx = list->hash (udp)) < 0 ||
      (idx >= list->numhash))
    return NULL;

  return list->nodes[idx];
}

#ifdef VANSI_1
void
LLinitScan (LLlist * list, int direction)
#else
void
LLinitScan (list, direction)
     LLlist *list;
     int direction;
#endif
{
  if (direction == 0)
    list->current = list->head;
  else
    list->current = list->tail;
}

/*
 * Function: 	void *LLscan(LLlist *list)
 *
 * Description: Scan a linked list returning user data pointer
 *              for each succesive item after each call.
 *
 * Parameters:	LLlist	*list  list to scan
 *
 * Returns:	User data pointer from item or NULL if no more items.
 *
 * Example code:
 *
 * Given list points to linked list and we wish to search this
 * list for a user defined data item with a value
 *
 * char *FindAddress( LLlist *list , char *name)
 * {
 *   	MyrecType	*mrec;
 *
 *   	LLinitScan(list, 0);
 *
 *   	while ((mrec = (MyrecType *)LLscan(list)) != NULL)
 *   	{
 *          if ( strcmp(mrec->name, name) == 0)
 *           	break;        # found value we were looking for
 *   	}
 *   	if (mrec == NULL) {
 *	    printf( "name not found\n" );
 *	    return NULL;
 *	} else
 *	    return mrec->address;
 */

#ifdef VANSI_1
void *
LLscan (LLlist * list)
#else
void *
LLscan (list)
     LLlist *list;

#endif
{
  void *vdp;

  if (list->current == NULL)
    return NULL;

  /*-- get a pointer to the user data for the current node */
  vdp = list->current->data;

  /*-- advance the current pointer to the next node */
  list->current = list->current->next;

  /*-- if this is equal to the head then we have reached the end
      so reset the current pointer so that the next call to
      this function will terminate the scan */
  if (list->current == list->head)
    list->current = NULL;

  return vdp;
}



/*
 * Function: 	void *LLrScan(LLlist *list)
 *
 * Description: Scan a linked list in reverse order returning user data pointer
 *              for each succesive item after each call.
 *
 * Parameters:	LLlist	*list pointer to list to scan
 *
 * Returns:	User data pointer from item or NULL if no more items.
 *              (cpos will point to LLITEM for which data ptr returned)
 *
 * Example code: - see LLscan()
 */
#ifdef VANSI_1
void *
LLrScan (LLlist * list)
#else
void *
LLrScan (list)
     LLlist *list;

#endif
{

  void *vdp;

  if (list->current == NULL)
    return NULL;

  /*-- get a pointer to the user data for the current node */
  vdp = list->current->data;

  /*-- advance the current pointer to the next node */
  list->current = list->current->prev;

  /*-- if this is equal to the tail then we have reached the end
      so reset the current pointer so that the next call to
      this function will terminate the scan */
  if (list->current == list->tail)
    list->current = NULL;

  return vdp;

}
