/******************************************************************************
	  CCCC	    A	  BBBBB	  L	EEEEE  N     N	EEEEE	TTTTTTT
	 C    C    A A	  B    B  L	E      NN    N	E	   T
	C	  A   A	  B    B  L	E      N N   N	E	   T
	C	 AAAAAAA  BBBBB	  L	EEEEE  N  N  N	EEEEE	   T
	C        A     A  B    B  L	E      N   N N	E	   T
	 C    C  A     A  B    B  L	E      N    NN  E 	   T
	  CCCC	 A     A  BBBBB	  LLLL	EEEEE  N     N	EEEEE	   T
*******************************************************************************

CableNet Source Module:
	Copyright 1994-1995 (C) CableNet Limited. All Rights Reserved.

    Module Name:		$RCSfile: SimpleLList.c,v $
    Module Description:	simple linked list handling

Description: 

Edit History:

	$Log: SimpleLList.c,v $
 * Revision 1.2  1996/04/17  15:19:30  damian
 * exclude malloc.h for bsdi
 *
	Revision 1.1  1995/01/09 18:21:23  V
	Initial revision



*/

/* RCS identification string (for "what" program) */
static char moduleRCSid[] = "@(#) $Id: SimpleLList.c,v 1.2 1996/04/17 15:19:30 damian Exp $";

/* must come first header files */
#include "V.h"			/* virtual header file */
#include "Vport.h"		/* port header file */

/*
 *   system header files
 */

#include <stdio.h>
#include <stdlib.h>

#if ! defined(Vbsdi)
#include <malloc.h>
#endif

#include <sys/types.h>


/*
 *   public project/department header files
 */

#include "Vlib.h"		/* public header file */

/*
 *  local application/library header files
 */


LLmallocFunc    LLmalloc = (LLmallocFunc) malloc;
LLfreeFunc      LLfree = (LLfreeFunc) free;


/*++ ****************  function LLCreate *********************/
/*
    Purpose:    Create a head to a new linked list.

    Parameters: None

    Return Values:    LLITEMPTR
    list     = a pointer to the head of a new linked list        or
    NULL    if could not create

    Dependencies:
       LLInitHead()

*/
/*********************************************************************** ++*/

LLITEMPTR 
LLCreate()
{
    LLITEMPTR       head;

    if ((head = (LLITEMPTR) LLmalloc(sizeof(LLITEM))) == NULL)
	return (NULL);		/* No memory */

    LLInitHead(head);
    return (head);
}

/*++ ****************  function LLInitHead *********************/
/*
    Purpose:    Initialise a linked list item to be a head to an empty
                list.

    Prototype:  void LLInitHead(LLITEMPTR head)

    Parameters:
    LLITEMPTR head -  ptr to head of linked list.

    Return Values: void

*/
/*********************************************************************** ++*/

#ifdef VANSI_1
void 
LLInitHead(LLITEMPTR head)
#else
void 
LLInitHead(head)
LLITEMPTR       head;

#endif
{
    head->next = head->prev = head;
    head->datap = NULL;
}

/*++ ****************  function LLDelete *********************/
/*
    Purpose:    Delete an item from a linked list (but not the list head)

    Prototype: 	void LLDelete(LLITEMPTR item)

    Parameters:
    LLITEMPTR item  - ptr to item to delete.

    Return Values:

*/
/*********************************************************************** ++*/

#ifdef VANSI_1
void 
LLDelete(LLITEMPTR item)
#else
void 
LLDelete(item)
LLITEMPTR       item;

#endif
{
    item->next->prev = item->prev;
    item->prev->next = item->next;
    LLfree(item);
}

/*++ ****************  function LLDestroy *********************/
/*
    Purpose:   Delete every item on a linked list and optionally the head as
               well.

    Prototype:  void LLDestroy(LLITEMPTR head, int all)

    Parameters:
    LLITEMPTR   head         -   ptr to head of linked list
    int         all          -   non-zero if delete the head aswell.

    Return Values: void

*/
/*********************************************************************** ++*/

#ifdef VANSI_1
void 
LLDestroy(LLITEMPTR head, int all)
#else
void 
LLDestroy(head, all)
LLITEMPTR       head;
int             all;

#endif
{
    LLITEMPTR       cpos = head;

    while (1) {
	if (LLScan(&cpos) == NULL)
	    break;		/* end of list reached */
	LLDelete(cpos);
	cpos = head;		/* reset  as may not be valid anymore */
    }

    if (all != 0)
	LLfree(head);

}

/*
 * Function: 	LLITEMPTR LLAdd(LLITEMPTR after,caddr_t userdata)
 *
 * Description: Add an item into a linked list after
 *              a specified item.
 *
 * Parameters:	LLITEMPTR after  - item to link after.
 *	            caddr_t userdata - user data ptr to put in item.
 *
 * Returns:     Pointer to new item or NULL if could not add.
 *
 */
#ifdef VANSI_1
LLITEMPTR LLAdd(LLITEMPTR after, void *userdata)
#else
LLITEMPTR LLAdd(after, userdata)
LLITEMPTR       after;
void           *userdata;
#endif
{
    LLITEMPTR       newitem;

    if ((newitem = (LLITEMPTR) LLmalloc(sizeof(LLITEM))) == NULL)
	return (NULL);

    newitem->datap = userdata;
    newitem->prev = after;

    newitem->next = after->next;
    after->next->prev = newitem;
    after->next = newitem;
    return (newitem);

}

/*
 * Function: 	LLITEMPTR LLAddEnd(LLITEMPTR head,caddr_t userdata)
 *
 * Description: Add an item into a linked list at end
 *
 * Parameters:	LLITEMPTR head - ptr to head of list
 *		caddr_t userdata - user data ptr to put in item.
 *
 * Returns:	Pointer to new item or NULL if could not add.
 *
 *
 */
#ifdef VANSI_1
LLITEMPTR LLAddEnd(LLITEMPTR head, void *userdata)
#else
LLITEMPTR LLAddEnd(head, userdata)
LLITEMPTR       head;
void           *userdata;
#endif
{
    return (LLAdd(head->prev, userdata));
}




#ifdef VANSI_1
void *LLGetNth(LLITEMPTR head, int n)
#else
void *LLGetNth(head, n)
LLITEMPTR       head;
int             n;
#endif
{
    LLITEMPTR       cpos;
    void           *userptr;
    int             num = 0;

    cpos = head;
    while ((userptr = LLScan(&cpos)) != NULL) {
	if (num == n)
	    break;
	num++;
    }

    return userptr;


}


/*
 * Function: 	caddr_t LLScan(LLITEMPTR *cpos)
 *
 * Description: Scan a linked list returning user data pointer
 *              for each succesive item after each call.
 *
 * Parameters:	LLITEMPTR cpos - used to keep track of position. Set *cpos to
 *				 head of list at start.
 *
 * Returns:	User data pointer from item or NULL if no more items.
 *              (cpos will point to LLITEM for which data ptr returned)
 *
 * Example code:
 *
 * Given alist points to head of linked list and we wish to search this
 * list for a user pointer which points to the number 42.
 *
 *   LLITEMPTR cpos;
 *   int *userptr;
 *
 *   cpos = alist;    # set up start condition
 *   while ((userptr = LLScan(&cpos)) != NULL)
 *   {
 *       if (*userptr == 42)
 *           break;        # found value we were looking for
 *   }
 *   if (userptr == NULL)
 *       printf("No match found\n");     # did not find match
 *
 *
 */

#ifdef VANSI_1
void           *
LLScan(LLITEMPTR *cpos)
#else
void           *
LLScan(cpos)
LLITEMPTR      *cpos;

#endif
{
    /* get the next data pointer so that the calling function gets a pointer
     * and its object cpos points to the ll item for datap */

    /* cpos point to next item on list */
    *cpos = (*cpos)->next;

    /* now get it's data pointer */
    return ((*cpos)->datap);	/* this will be null for list tail(==head) */
}



/*
 * Function: 	caddr_t LLRScan(LLITEMPTR *cpos)
 *
 * Description: Scan a linked list in revers returning user data pointer
 *              for each succesive item after each call.
 *
 * Parameters:	LLITEMPTR cpos - used to keep track of position. Set *cpos to
 *				 head of list at start.
 *
 * Returns:	User data pointer from item or NULL if no more items.
 *              (cpos will point to LLITEM for which data ptr returned)
 *
 * Example code: - see LLScan()
 */
#ifdef VANSI_1
void           *
LLRScan(LLITEMPTR *cpos)
#else
void           *
LLRScan(cpos)
LLITEMPTR      *cpos;

#endif
{

    *cpos = (*cpos)->prev;
    return ((*cpos)->datap);	/* this will be null for list tail(==head) */

}


#ifdef VANSI_1
void
LLFreeBuff (LLITEMPTR list)
#else
void
LLFreeBuff (list)
     LLITEMPTR list;

#endif
{
  extern void Free ();
  LLITEMPTR cpos;
  void *up = NULL;

  if (!list)
    return;


  /* we could use LLAction to call Free for each item on list */
  /* but Free is a void function and LLActFunc is an int function */
  /* so instead program it here */
  cpos = list;
  while ((up = (void *) LLScan (&cpos)) != NULL)
    {
      Free (up);
      LLDelete (cpos);		/* LLITEM is pointless after freeing data so
				 * delete */
    }
}



#ifdef VANSI_1
int DumpLLitemPtr(LLITEMPTR cpos)
#else
int DumpLLitemPtr(cpos)
LLITEMPTR       cpos;
#endif
{

    /*-- dump a linked list item pointer to stdout */
    printf("0x%p -- to 0x%p LLITEM data:0x%p next:0x%p prev:0x%p\n",
	(char *) cpos, (char *) cpos + sizeof(LLITEM),
	cpos->datap, cpos->next, cpos->prev);

    return 0;

}



#ifdef VANSI_1
int 
LLCount(LLITEMPTR head)
#else
int 
LLCount(head)
LLITEMPTR       head;

#endif
{

    LLITEMPTR       cpos;
    void           *userptr;
    int             num = 0;

    cpos = head;
    while ((userptr = LLScan(&cpos)) != NULL)
	num++;

    return num;

}


/*
 * Function: 	int LLAction(LLITEMPTR *lptr, int actfunc(caddr_t arg)())
 *
 * Description: This will traverse the entire linked list pointed to by
 *              lptr and call actfunc (with a pointer to the userdata)
 *              to perform some function for each element.
 *
 *              If actfunc() returns non-zero then the traverse is aborted.
 *
 * Parameters:	LLITEMPTR lptr     -  ptr to head of list.
 *		LLActFunc actfunc  -  ptr to function to call to dump contents
 *
 * Returns:	0 if got to end of list or return code from actfunc.
 *
 * Notes:       actfunc() must not change the list in any way or the behavior
 *              is undefined.
 */
#ifdef VANSI_1
int LLAction(LLITEMPTR lptr, LLActFunc actfunc)
#else
int LLAction(lptr, actfunc)
LLITEMPTR       lptr;
LLActFunc       actfunc;
#endif
{
    caddr_t         userdata;
    int             res;

    while ((userdata = LLScan(&lptr)) != NULL) {
	if ((res = actfunc(userdata)) != 0)
	    return (res);
    }
    return (0);

}
