#include <stdio.h>
#include <getopt.h>
#include "/usr/local/msql/include/msql.h"
#include "strutil.h"

#define BUFFER_SIZE 4096		/* size of the query buffer */
#define MAXFIELDS   64                  /* max. nubmers of fields   */
#define MAXFIELDSIZE 512                /* max. size of fields in chars */
#define STRSIZE 256                     /* size of strings          */
#define INDEXLEN 4                      /* length of index field (iop) */

#define HLPMSG 0                        /* help called              */
#define SQLERR 1                        /* msql call error          */
#define OVFERR 2                        /* query buffer overflow    */
#define CMDERR 3                        /* error in commandline     */
#define SEPERR 4                        /* wrong seperator          */
#define FRMERR 5                        /* wrong format             */
#define FLDERR 6                        /* field overflow           */

#define max(x,y) ((x)>(y)) ? (x) : (y)

/********************
** vars
*/

int sock;
m_result *result;
m_row cursor;
int fields;
m_field *field[MAXFIELDS];
int field_len[MAXFIELDS];

int nop = 0;                                    /* name of fields         */
int qop = 0;                                    /* pipe query too = false */
int fop = 0;                                    /* output not formated    */
int uop = 0;                                    /* underline names = no   */
int iop = 0;                                    /* print line nr (index)  */

int lop = 0;                                    /* print length of fields */
int kop = 0;                                    /* print key info         */
int top = 0;                                    /* print type info        */
int aop = 0;                                    /* print name off table   */

char sep[STRSIZE];                              /* seperator              */
char lbegin[STRSIZE];                           /* begin of line          */
char lend[STRSIZE];                             /* end of line            */

char *usage = "usage: sql [-fiaktlnqub] [-b line_begin] [-s seperator] [-e line_end] [-h host] table";

int width;                                      /* width of output        */

/**********************
** print error messages
*/

void error(int nr) {
  switch(nr) {
    case SQLERR: fprintf(stderr, "%s\n", msqlErrMsg); break;
    case OVFERR: fprintf(stderr, "query buffer overflow, use ';'\n"); break;
    case CMDERR: fprintf(stderr, "cmd-line err: %s\n", usage); break;
    case SEPERR: fprintf(stderr, "error in seperator\n"); break;
    case FRMERR: fprintf(stderr, "wrong output format\n"); break;
    case FLDERR: fprintf(stderr, "too many fields in output\n"); break;
    case HLPMSG: printf("%s\n", usage); break;
  }
  exit(nr);
}

/***********************
** print filler
*/

void fill(char fill, int count) {
  for(;count>0; count--) printf("%c", fill);
}

/***********************
** underline
*/

void underline() {
  fill('-', width);
}

/************************
** print field
*/

void print_field(char *str, int nr) {

  if (fop) {
    printf("%s", str);
    fill(' ', field_len[nr] - strlen(str));
  } else {
    printf("%s", str); 
  }
  if(nr<fields) printf("%s", sep);
}


/*********************************************************************
** main
*/

int main(int argc, char *argv[]) {

  int i, c, go = 0, line, k;
  char query[BUFFER_SIZE];
  char tmp[STRSIZE];

  char *host = NULL;
  char *dbname = "";                              /* name of database       */

  sep[0] = ':';                                   /* default seperator      */
  sep[1] = '\0';
  lbegin[0] = '\0';
  lend[0] = '\n';
  lend[1] = '\0';

  /***********************************************
  ** parse commandline
  */
  
  while((c = getopt(argc, argv, "?fiaktlnqub:e:h:s:")) != -1) {
    switch(c) {
      case 'f': fop = -1; break;      
      case 'i': iop = -1; break;
      case 't': top = -1; break;
      case 'a': aop = -1; break;
      case 'k': kop = -1; break;
      case 'n': nop = -1; break;
      case 'q': qop = -1; break;
      case 'u': uop = -1; break;
      case 'l': lop = -1; break;
      case 'b': if(!strconv(lbegin, optarg)) error(SEPERR); break;
      case 'e': if(!strconv(lend, optarg)) error(SEPERR); break;
      case 'h': host = optarg; break;
      case 's': if(!strconv(sep, optarg)) error(SEPERR);
	        break;
      case '?': if(optopt=='?') {
	          error(HLPMSG);
                } else {
	          error(CMDERR);
		}
	        break;
      }
  }
  if( optind<argc ) dbname = argv[optind];
    else error(CMDERR); /* no return */
     
  /***********************************************
  ** establish database connection
  */
  
  if((sock = msqlConnect(host))<0) error(SQLERR);
  if( msqlSelectDB(sock, dbname) < 0 ) error(SQLERR);

  /*********************************************
  ** print header
  */


  while(!feof(stdin)) {

    /****************************************************************
    ** read stdin into the query buffer
    */

    while((c = getchar())==' ' || c=='\n');     /* skip spaces and newlines */
    ungetc(c, stdin);

    i = 0;
    while(i<BUFFER_SIZE-1 && !go && (query[i] = getchar()) != EOF) {
      if(i>0 && query[i-1]=='\\' && query[i]=='g') {
        query[i-1] = 0; 
        go = -1;
      }
      i++;
    }

    if(go) {
      go = 0;
    } else {
      query[i-1] = 0;
    }

    if( i==BUFFER_SIZE-1 ) error(OVFERR);

    while((c = getchar())==' ' || c=='\n');     /* skip spaces and newlines */
    ungetc(c, stdin);
    
    if(qop) {
        printf(">query\n%s\n>result\n", query);
    }

    /***************************************************
    ** read answer
    */

    if(msqlQuery(sock, query) < 0) error(SQLERR);
    result = msqlStoreResult();
    fields = msqlNumFields(result);
    if(iop) {
      field_len[0] = INDEXLEN;
    }
    if( fields >= MAXFIELDS ) error(FLDERR); /* no return */

    width = strlen(lbegin) + strlen(lend) - 1;
    for(i = 1; i<=fields; i++) {
      field[i] = msqlFetchField(result);
      field_len[i] = field[i]->length;
      if(kop) field_len[i] = max(field_len[i], 2);
      if(top) field_len[i] = max(field_len[i], 5);
      if(lop) field_len[i] = max(field_len[i], 4);
      if(nop) field_len[i] = max(field_len[i], strlen(field[i]->name));
      if(aop) field_len[i] = max(field_len[i], strlen(field[i]->table));
      width += field_len[i] + strlen(sep);
    }
    if(iop) {
      width += INDEXLEN;
    } else {
      width -= strlen(sep);
    }

    if( (top || nop || kop || lop || aop) && uop) {
      underline();
      printf("\n");
    }

    /****************************************
    ** print name of fields
    */

    if(nop) {
      printf(lbegin);
      if(iop) {
        sprintf(tmp, "%i", -5);
        print_field(tmp, 0);
      }
      for(i=1; i<=fields; i++) print_field(field[i]->name, i);
      printf(lend);
    }

    /****************************************
    ** print length of fields
    */

    if(lop) {
      printf(lbegin);
      if(iop) {
        sprintf(tmp, "%i", -4);
        print_field(tmp, 0);
      }
      for(i=1; i<=fields; i++) { 
        sprintf(tmp, "%i", field[i]->length); 
	print_field(tmp, i);
      }
      printf(lend);
    }

    /****************************************
    ** print type of fields
    */

    if(top) {
      printf(lbegin);
      if(iop) {
	sprintf(tmp, "%i", -3);
        print_field(tmp, 0);
      }
      for(i=1; i<=fields; i++) { 
        switch(field[i]->type) {
          case INT_TYPE: strcpy(tmp, "INT"); break;
          case CHAR_TYPE: strcpy(tmp, "CHAR"); break;
          case REAL_TYPE: strcpy(tmp, "REAL"); break;
          case IDENT_TYPE: strcpy(tmp, "IDENT"); break;
          case NULL_TYPE: strcpy(tmp, "NULL"); break;
        } 
	print_field(tmp, i);
      }
      printf(lend);
    }

    /****************************************
    ** print key info
    */

    if(kop) {
      printf(lbegin);
      if(iop) {
        sprintf(tmp, "%i", -2);
        print_field(tmp, 0);
      }
      for(i=1; i<=fields; i++) { 
	strcpy(tmp, "        ");
	if(IS_PRI_KEY(field[i]->flags)) tmp[0] = 'P';
	if(IS_NOT_NULL(field[i]->flags)) tmp[1] = 'N';
	tmp[2] = '\0';
        print_field(tmp, i);
      }
      printf(lend);
    }

    /****************************************
    ** print name of table
    */

    if(aop) {
      printf(lbegin);
      if(iop) {
	sprintf(tmp, "%i", -1);
	print_field(tmp, 0);
      }
      for(i=1; i<=fields; i++) { 
	strcpy(tmp, field[i]->table);
        print_field(tmp, i);
      }
      printf(lend);
    }

    /*************************************************
    ** print answer
    */

    if(uop) {
      underline();
      printf("\n");
    }

    line = 0;
    while( (cursor = msqlFetchRow(result)) != NULL ) {
      printf(lbegin);
      if(iop) {
        sprintf(tmp, "%i", line);
	print_field(tmp, 0);
      }
      for(i=1; i<=fields; i++) print_field(cursor[i-1], i);
      printf(lend);
      line++;
    }
    if(uop) {
      fill('-', width);
      printf("\n");
    }

    msqlFreeResult(result);
  }

  /******************
  ** close the connection
  */

  msqlClose(sock);
  return(0);
}







