/* Copyright (C) 1989,1990,1991,1992 by
	Wilfried Koch, Andreas Lampen, Axel Mahler, Juergen Nickelsen,
	Wolfgang Obst and Ulrich Pralle
 
 This file is part of shapeTools.

 This software is published in the hope that it will be useful, but
 WITHOUT ANY WARRANTY for any part of this software to work correctly
 or as described in the manuals. See the ShapeTools Public License
 for details.

 Permission is granted to use, copy, modify, or distribute any part of
 this software but only under the conditions described in the ShapeTools 
 Public License. A copy of this license is supposed to have been given
 to you along with shapeTools in a file named LICENSE. Among other
 things, this copyright notice and the Public License must be
 preserved on all copies.
 */
/*
 *	Shape/AtFS
 *
 *	afcompar.c -- attribute comparison routines 
 *
 *	Author: Andreas Lampen, TU-Berlin (andy@coma.UUCP)
 *					  (andy@db0tui62.BITNET)
 *
 *	$Header: afcompar.c[1.10] Fri Jan 31 18:05:57 1992 andy@cs.tu-berlin.de accessed $
 *
 *	EXPORT:
 *      af_cmpatime -- compare dates of last access
 *      af_cmpauthor -- compare authors
 *      af_cmpctime -- compare dates of last status change
 *      af_cmpgen -- compare generation numbers
 *      af_cmphost -- compare hosts
 *      af_cmphuman -- compare name, type, gen, rev
 *      af_cmplocker -- compare lockers
 *      af_cmpltime -- compare date of last lock change
 *      af_cmpmode -- compare protections
 *      af_cmpmtime -- compare dates of last modification
 *      af_cmpname -- compare names
 *      af_cmpowner -- compare owners
 *      af_cmprev -- compare revision numbers
 *      af_cmpsize -- compare file sizes
 *      af_cmpstate -- compare states
 *      af_cmpsvtime -- compare dates of saving
 *      af_cmpspath -- compare syspaths
 *      af_cmptype -- compare types
 *      af_cmpversion -- compare version numbers (gen.rev)
 *
 *      af_cmpfuncts -- list of names of all compare functions
 *      af_attrlist -- list of attribute names
 *      af_udaname -- global variable holding the name of the
 *                    user defined attribute used for sorting
 *      af_cmpuda -- compare user defined attributes
 */

#include <stdio.h>

#include "afsys.h"
#include "atfs.h"

/*================================================================
 *      af_cmpatime -- compare dates of last access
 *
 *================================================================*/

EXPORT int af_cmpatime (key1, key2)
     Af_key *key1, *key2;
{
  (void) afAccessAso (key1, AF_ATTRS);
  (void) afAccessAso (key2, AF_ATTRS);
  return (VATTR(key1).af_atime - VATTR(key2).af_atime);
}

/*================================================================
 *      af_cmpauthor -- compare authors
 *
 *================================================================*/

EXPORT af_cmpauthor (key1, key2)
     Af_key *key1, *key2;
{
  register int res;

  (void) afAccessAso (key1, AF_ATTRS);
  (void) afAccessAso (key2, AF_ATTRS);
  if (!(res = strcmp (NOTNIL(VATTR(key1).af_auname),
		      NOTNIL(VATTR(key2).af_auname))))
    return (strcmp (NOTNIL(VATTR(key1).af_audomain),
		    NOTNIL(VATTR(key2).af_audomain)));
  else
    return (res);
}

/*================================================================
 *      af_cmpctime -- compare dates of last status change
 *
 *================================================================*/

EXPORT af_cmpctime (key1, key2)
     Af_key *key1, *key2;
{
  (void) afAccessAso (key1, AF_ATTRS);
  (void) afAccessAso (key2, AF_ATTRS);
  return (VATTR(key1).af_ctime - VATTR(key2).af_ctime);
}

/*================================================================
 *      af_cmpgen -- compare generation numbers
 *
 *================================================================*/

EXPORT af_cmpgen (key1, key2)
     Af_key *key1, *key2;
{
  (void) afAccessAso (key1, AF_ATTRS);
  (void) afAccessAso (key2, AF_ATTRS);
  return (VATTR(key1).af_gen - VATTR(key2).af_gen);
}

/*================================================================
 *      af_cmphost -- compare hostnames
 *
 *================================================================*/

EXPORT af_cmphost (key1, key2)
     Af_key *key1, *key2;
{
  (void) afAccessAso (key1, AF_ATTRS);
  (void) afAccessAso (key2, AF_ATTRS);
  return (strcmp (CATTR(key1).af_host, CATTR(key2).af_host));
}

/*================================================================
 *      af_cmphuman -- compare name, type, gen, rev
 *
 *================================================================*/

EXPORT af_cmphuman (key1, key2)
     Af_key *key1, *key2;
{
  register int diff;

  (void) afAccessAso (key1, AF_ATTRS);
  (void) afAccessAso (key2, AF_ATTRS);
  if (!(diff = strcmp (VATTR(key1).af_name, VATTR(key2).af_name)) &&
      !(diff = strcmp (NOTNIL(VATTR(key1).af_type),
		       NOTNIL(VATTR(key2).af_type))) &&
      !(diff = VATTR(key1).af_gen - VATTR(key2).af_gen) &&
      !(diff = VATTR(key1).af_rev - VATTR(key2).af_rev))
    {
      return (0);
    }
  return (diff);
}

/*================================================================
 *      af_cmplocker -- compare lockers
 *
 *================================================================*/

EXPORT af_cmplocker (key1, key2)
     Af_key *key1, *key2;
{
  register int res;

  (void) afAccessAso (key1, AF_ATTRS);
  (void) afAccessAso (key2, AF_ATTRS);
  if (!(res = strcmp (NOTNIL(VATTR(key1).af_lckname),
		      NOTNIL(VATTR(key2).af_lckname))))
    return (strcmp (NOTNIL(VATTR(key1).af_lckdomain),
		    NOTNIL(VATTR(key2).af_lckdomain)));
  else
    return (res);
}

/*================================================================
 *      af_cmpltime -- compare dates of last lock change
 *
 *================================================================*/

EXPORT af_cmpltime (key1, key2)
     Af_key *key1, *key2;
{
  (void) afAccessAso (key1, AF_ATTRS);
  (void) afAccessAso (key2, AF_ATTRS);
  return (VATTR(key1).af_ltime - VATTR(key2).af_ltime);
}

/*================================================================
 *      af_cmpmode -- compare protections
 *
 *================================================================*/

EXPORT af_cmpmode (key1, key2)
     Af_key *key1, *key2;
{
  (void) afAccessAso (key1, AF_ATTRS);
  (void) afAccessAso (key2, AF_ATTRS);
  return (VATTR(key1).af_mode - VATTR(key2).af_mode);
}

/*================================================================
 *      af_cmpmtime -- compare dates of last modification
 *
 *================================================================*/

EXPORT af_cmpmtime (key1, key2)
     Af_key *key1, *key2;
{
  (void) afAccessAso (key1, AF_ATTRS);
  (void) afAccessAso (key2, AF_ATTRS);
  return (VATTR(key1).af_mtime - VATTR(key2).af_mtime);
}

/*================================================================
 *      af_cmpname -- compare names
 *
 *================================================================*/

EXPORT af_cmpname (key1, key2)
     Af_key *key1, *key2;
{
  (void) afAccessAso (key1, AF_ATTRS);
  (void) afAccessAso (key2, AF_ATTRS);
  return (strcmp (VATTR(key1).af_name, VATTR(key2).af_name));
}

/*================================================================
 *      af_cmpowner -- compare owners
 *
 *================================================================*/

EXPORT af_cmpowner (key1, key2)
     Af_key *key1, *key2;
{
  register int res;

  (void) afAccessAso (key1, AF_ATTRS);
  (void) afAccessAso (key2, AF_ATTRS);
  if (!(res = strcmp (CATTR(key1).af_ownname, CATTR(key2).af_ownname)))
    return (strcmp (CATTR(key1).af_owndomain, CATTR(key2).af_owndomain));
  else
    return (res);
}

/*================================================================
 *      af_cmprev -- compare revision numbers
 *
 *================================================================*/

EXPORT af_cmprev (key1, key2)
     Af_key *key1, *key2;
{
  (void) afAccessAso (key1, AF_ATTRS);
  (void) afAccessAso (key2, AF_ATTRS);
  return (VATTR(key1).af_rev - VATTR(key2).af_rev);
}

/*================================================================
 *      af_cmpsize -- compare file size
 *
 *================================================================*/

EXPORT af_cmpsize (key1, key2)
     Af_key *key1, *key2;
{
  (void) afAccessAso (key1, AF_ATTRS);
  (void) afAccessAso (key2, AF_ATTRS);
  return (VATTR(key1).af_fsize - VATTR(key2).af_fsize);
}

/*================================================================
 *      af_cmpstate -- compare states
 *
 *================================================================*/

EXPORT af_cmpstate (key1, key2)
     Af_key *key1, *key2;
{
  (void) afAccessAso (key1, AF_ATTRS);
  (void) afAccessAso (key2, AF_ATTRS);
  return (VATTR(key1).af_state - VATTR(key2).af_state);
}

/*================================================================
 *      af_cmpsvtime -- compare dates of saving
 *
 *================================================================*/

EXPORT af_cmpsvtime (key1, key2)
     Af_key *key1, *key2;
{
  (void) afAccessAso (key1, AF_ATTRS);
  (void) afAccessAso (key2, AF_ATTRS);
  return (VATTR(key1).af_stime - VATTR(key2).af_stime);
}

/*================================================================
 *      af_cmpspath -- compare syspaths
 *
 *================================================================*/

EXPORT af_cmpspath (key1, key2)
     Af_key *key1, *key2;
{
  (void) afAccessAso (key1, AF_ATTRS);
  (void) afAccessAso (key2, AF_ATTRS);
  return (strcmp (CATTR(key1).af_syspath, CATTR(key2).af_syspath));
}

/*================================================================
 *      af_cmptype -- compare types
 *
 *================================================================*/

EXPORT af_cmptype (key1, key2)
     Af_key *key1, *key2;
{
  (void) afAccessAso (key1, AF_ATTRS);
  (void) afAccessAso (key2, AF_ATTRS);
  return (strcmp (NOTNIL(VATTR(key1).af_type),
		  NOTNIL(VATTR(key2).af_type)));
}

/*================================================================
 *      af_cmpversion -- compare version numbers (gen.rev)
 *
 *================================================================*/

EXPORT af_cmpversion (key1, key2)
     Af_key *key1, *key2;
{
  (void) afAccessAso (key1, AF_ATTRS);
  (void) afAccessAso (key2, AF_ATTRS);
  if (VATTR(key1).af_gen == VATTR(key2).af_gen)
    return (VATTR(key1).af_rev - VATTR(key2).af_rev);
  else
    return (VATTR(key1).af_gen - VATTR(key2).af_gen);
}

/*===============================================================
 *      af_cmpfuncts -- array of comparison functions
 *
 *===============================================================*/

struct cmpfun { int (*func) (); };

EXPORT struct cmpfun af_cmpfuncts[] = { af_cmpatime,
					  af_cmpauthor,
					  af_cmpctime,
					  af_cmpgen,
					  af_cmphost,
					  af_cmphuman,
					  af_cmplocker,
					  af_cmpltime,
					  af_cmpmode,
					  af_cmpmtime,
					  af_cmpname,
					  af_cmpowner,
					  af_cmprev,
					  af_cmpsize,
					  af_cmpstate,
					  af_cmpsvtime,
					  af_cmpspath,
					  af_cmptype,
					  af_cmpversion,
					};


/*===============================================================
 *      af_attrlist -- list of attribute names
 *                     (in alphabetical order)
 *
 *===============================================================*/

EXPORT char *af_attrlist[] = { AF_ATTATIME,
				 AF_ATTAUTHOR,
				 AF_ATTCTIME,
				 AF_ATTGEN,
				 AF_ATTHOST,
				 AF_ATTHUMAN,
				 AF_ATTLOCKER,
				 AF_ATTLTIME,
				 AF_ATTMODE,
				 AF_ATTMTIME,
				 AF_ATTNAME,
				 AF_ATTOWNER,
				 AF_ATTREV,
				 AF_ATTSIZE,
				 AF_ATTSTATE,
				 AF_ATTSTIME,
				 AF_ATTSPATH,
				 AF_ATTTYPE,
				 AF_ATTVERSION,
				};

/*================================================================
 *      af_cmpuda -- compare user defined attributes
 *
 *================================================================*/

EXPORT char af_udaname[AF_UDANAMLEN];

EXPORT af_cmpuda (key1, key2)
     Af_key *key1, *key2;
{
  register char *uda1, *uda2;
  char          *vallist1, *vallist2;

  (void) afAccessAso (key1, AF_ATTRS);
  (void) afAccessAso (key2, AF_ATTRS);

  uda1 = afLookupUda (key1, af_udaname);
  uda2 = afLookupUda (key2, af_udaname);

  if (uda1 == (char *)0)
    {
      if (uda2 == (char *)0)
	return (0); /* equal */
      else
	return (1); /* key1 is "greater" than key2 */
    }
  else
    {
      if (uda2 == (char *)0)
	return (-1); /* key2 is "greater" than key1 */
    }

  /* see if ther are values */
  vallist1 = index (uda1, AF_UDANAMDEL);
  vallist2 = index (uda2, AF_UDANAMDEL);

  if (vallist1 == (char *)0)
    {
      if (vallist2 == (char *)0)
	return (0); /* equal */
      else
	return (1); /* key1 is "greater" than key2 */
    }
  else
    {
      if (vallist2 == (char *)0)
	return (-1); /* key2 is "greater" than key1 */
    }

  /* compare values of user defined attributes */
  /* this is a simple textual comparison up to now */
  return (strcmp (vallist1, vallist2));
}

