/*LINTLIBRARY*/
/* Copyright (C) 1989,1990,1991,1992 by
	Wilfried Koch, Andreas Lampen, Axel Mahler, Juergen Nickelsen,
	Wolfgang Obst and Ulrich Pralle
 
 This file is part of shapeTools.

 This software is published in the hope that it will be useful, but
 WITHOUT ANY WARRANTY for any part of this software to work correctly
 or as described in the manuals. See the ShapeTools Public License
 for details.

 Permission is granted to use, copy, modify, or distribute any part of
 this software but only under the conditions described in the ShapeTools 
 Public License. A copy of this license is supposed to have been given
 to you along with shapeTools in a file named LICENSE. Among other
 things, this copyright notice and the Public License must be
 preserved on all copies.
 */
/*
 *	Shape/AtFS
 *
 *	afnames.c -- deal with UNIX-Filesystem names
 *
 *	Author: Andreas Lampen, TU-Berlin (andy@coma.UUCP)
 *					  (andy@db0tui62.BITNET)
 *
 *	$Header: afnames.c[1.7] Fri Jan 31 18:05:14 1992 andy@cs.tu-berlin.de accessed $
 *
 *	EXPORT:
 *      af_uniqpath -- build unified pathname
 *	af_setarchpath -- name directory where archives shall be stored
 *      af_ardir -- realize symbolic links for AtFS subdirectory
 *      af_isarchive -- test if a given file is an archive
 *      af_isbpfile -- see if a given file belongs to derived object cache
 *	af_garname -- build name for archive file
 *	af_gbpname -- build name for binary pool db-file
 *	af_gtmpname -- build name for tmp file
 *	af_gbusname -- build name of busy version
 *	af_afpath -- build af-syspath from UNIX-filename
 *	af_afname -- build af-filename from UNIX-filename
 *	af_aftype -- build af-filetype from UNIX-filename
 *	af_unixname -- build UNIX-filename from af-filename/type
 *      af_bpfilename -- return filename for binary pool file
 *      af_rbphashname -- get unique filename for file in binary pool
 */

#include <stdio.h>

#include "afsys.h"
#include "atfs.h"
#include "afarchive.h"

#include <sys/file.h>

char  *malloc();
Uid_t geteuid();

/*================================================================
 *	af_uniqpath -- build unified pathname
 *
 *================================================================*/

EXPORT char *af_uniqpath (path)
     char *path;
{
  static   char uniqpath[MAXPATHLEN], tmppath[MAXPATHLEN];
  register char *p, *u;
  char          *getwd ();

  if ((path == (char *)0) || (path[0] == '\0') || ((path[0] == '.' && path[1] == '\0')))
    return (getwd (uniqpath));

  /* build absolute pathname if only a relative one is given */
  if (path[0] != '/')
    {
      if (!strcmp (path, ".."))
	{
	  (void) getwd (uniqpath);
	  if ((p = rindex (uniqpath, '/')) == uniqpath)
	    uniqpath[1] = '\0';
	  else
	    *p = '\0';
	  return (uniqpath);
	}
      (void) getwd (tmppath);
      (void) strcat (tmppath, "/");
      (void) strcat (tmppath, path);
      p = tmppath;
    }
  else
    p = path;

  /* eliminate things like "/usr/./bin" and "/usr/../usr/bin" */
  u = uniqpath;
  *u = '/';
  while (*p)
    {
      if ((p[0] == '/') && (p[1] == '.'))
	{
	  if ((p[2] == '/') || (p[2] == '\0'))
	    {
	      p = &p[2];
	      continue;
	    }
	  else
	    if ((p[2] == '.') && ((p[3] == '/') || (p[3] == '\0')))
	      {
		if (u != uniqpath)
		  do { *u = '\0'; u--; } while (*u != '/');
		p = &p[3];
		continue;
	      }
	}
      u++;
      *u = p[1];
      p++;
    }
  
  /* cut slash if present at the end */
  u--;
  if ((u != uniqpath) && (*u == '/'))
    *u = '\0';
  else
    u[1] = '\0';

  return (uniqpath);
}

static char *archpath = (char *)0;

/*================================================================
 *	af_setarchpath
 *
 *================================================================*/

EXPORT char *af_setarchpath (pathname)
     char *pathname;
{
  register char *oldpath = archpath;

  if (pathname == (char *)0)
    archpath = (char *)0;
  else
    archpath = af_entersym (af_uniqpath (pathname));
  return (oldpath);
}

/*========================================================================
 * af_ardir () -- realize symbolic links for AtFS subdirectory
 *
 *========================================================================*/

static struct { char *dirname, *atfsname; } arDirList[AF_MAXLISTS];
static int arDirCount = 0;

EXPORT char *af_ardir (pathname)
     char *pathname;
{
  register int  i = 0, len;
  char          tmpname[MAXPATHLEN];
  struct stat   ibuf;
  register FILE *infile;

  while (arDirList[i].dirname != pathname)
    {
      if (i == arDirCount)
	{
	  arDirList[i].dirname = pathname;

	  ibuf.st_mode = 0;
	  (void) sprintf (tmpname, "%s/%s\0", pathname, AF_SUBDIR);
	  if (stat (tmpname, &ibuf) == ERROR)
	    {
	      /* AtFS subdirectory does not exist. Try old name */
	      char oldTmpName[MAXPATHLEN];

	      (void) sprintf (oldTmpName, "%s/%s\0", pathname, AF_OLDSUBDIR);
	      if (stat (oldTmpName, &ibuf) != ERROR)
		strcpy (tmpname, oldTmpName);
	    }

	  if (ibuf.st_mode)
	    {
	      if ((ibuf.st_mode & S_IFMT) == S_IFREG)
		{
		  /* AF_SUBDIR is a regular file -- maybe it contains
		   * a reference to another AtFS directory.
		   * -> open file and read contents
		   */
		  if ((infile = fopen (tmpname, "r")) != (FILE *)0)
		    {
		      (void) fgets (tmpname, MAXPATHLEN, infile);
		      (void) fclose (infile);
		    }
		  /* remove trailing newline */
		  len = strlen(tmpname);
		  if (tmpname[len-1] == '\n')
		    tmpname[len-1] = '\0';
		  arDirList[i].atfsname = af_entersym (af_uniqpath (tmpname));
		  arDirCount++;
		  break;
		}
	    }
	  arDirList[i].atfsname = af_entersym (tmpname);
	  arDirCount++;
	  break;
	}
      else
	i++;
    }
  return (arDirList[i].atfsname);
}

/*====================================================================
 * af_isarchive -- test if a given file is an archive
 *
 *====================================================================*/

EXPORT af_isarchive (name)
     char *name;
{
  register char ext;

  if (!name || strncmp (name, AF_ATFSFILEID, AF_IDSTRLEN))
    return (FALSE);

  ext = name[strlen (name) - sizeof (char)];
  if ((ext != AF_ARCHEXT) && (ext != AF_DATAEXT))
    return (FALSE);

  return (TRUE);
}

/*==========================================================================
 *	af_isbpfile
 *
 *==========================================================================*/

EXPORT af_isbpfile (name)
     char *name;
{
  return (!strncmp (name, AF_BPFILEID, AF_IDSTRLEN));
}


/*================================================================
 *	af_garname
 *
 *================================================================*/

EXPORT char *af_garname (pathname, name, type)
     char *pathname;
     char *name, *type;
{
  register char *dirname;
  char          arname[MAXPATHLEN];
#if (MAXNAMLEN < 128)
  register int namlen;
#endif

  /* see if there is an explicit pathname where archives shall be stored */
  if (archpath)
    dirname = archpath;
  else
    dirname = af_ardir (pathname);

#if (MAXNAMLEN < 128)
  if ((namlen = strlen (type ? type : "")) == 0)
    namlen = -1;
  if ((namlen = namlen + strlen (name)) > MAXNAMLEN-4)
    name += namlen - (MAXNAMLEN-4);
#endif

  if ((type != (char *)0) && (type[0] != '\0'))
    (void) sprintf (arname, "%s/%s%s.%s%c\0", dirname,
	     AF_ATFSFILEID, NOTNIL(name), NOTNIL(type), AF_ARCHEXT);
  else
    (void) sprintf (arname, "%s/%s%s%c\0", dirname,
	     AF_ATFSFILEID, NOTNIL(name), AF_ARCHEXT);

  return (af_entersym (arname));
} /* af_garname */ 

/*================================================================
 *	af_gbpname
 *
 *================================================================*/

EXPORT char *af_gbpname (pathname)
     char *pathname;
{
  register char *dirname;
  char          bpname[MAXPATHLEN];

  /* see if there is an explicit pathname where archives shall be stored */
  if (archpath)
    dirname = archpath;
  else
    dirname = af_ardir (pathname);

/* superflous ?!?
#ifdef STATISTICS
  _regfileaccess (dirname);
#endif
  if (access (dirname, R_OK))
    return ((char *)0);
*/

  (void) sprintf (bpname, "%s/%s\0", dirname, AF_BPOOLNAME);

  return (af_entersym (bpname));
} /* af_garname */ 


/*================================================================
 *	af_gtmpname
 *
 *================================================================*/

static int count=0;

EXPORT char *af_gtmpname (pathname, filename)
     /*ARGSUSED*/
     char *pathname; /* unused up to now */
     char *filename; /* unused too */
{
  char tmpname[MAXPATHLEN];
  
  (void) sprintf (tmpname, "%s/atfs%d%d\0", AF_TMPDIR, getpid(), count++);
  return (af_entersym (tmpname));
} /* af_gtmpname */


/*================================================================
 *	af_gbusname
 *
 *================================================================*/

EXPORT char *af_gbusname (pathname, name, type)
     char *pathname;
     char *name, *type;
{
  char busyname[MAXPATHLEN];
   
  (void) sprintf (busyname, "%s/%s", pathname, name);
  if ((type != (char *)0) && (type[0] != '\0'))
      {
	(void) strcat (busyname, ".");
	(void) strcat (busyname, type);
      }
  return (af_entersym (busyname));
} /* af_gbusname */ 


/*================================================================
 *	af_afpath
 *
 *================================================================*/

EXPORT char *af_afpath (unixname)
     char *unixname;
{
  register char *nameptr;
  static   char afpath[MAXPATHLEN];

  if (unixname)
    (void) strcpy (afpath, unixname);
  else
    {
      afpath[0] = '\0';
      return (afpath);
    }

  /* cut name */
  if ((nameptr = rindex (afpath, '/')) != (char *)0)
    nameptr[0] = '\0';
  else
    {
      afpath[0] = '\0';
      return (afpath);
    }

  /* cut AtFS subdirectory name if present */
  if ((nameptr = rindex (afpath, '/')) != (char *)0)
    if (!strcmp (AF_SUBDIR, nameptr+1) || !strcmp (AF_OLDSUBDIR, nameptr+1))
      nameptr[0] = '\0';
  else
    if (!strcmp (AF_SUBDIR, afpath) || !strcmp (AF_OLDSUBDIR, afpath))
      afpath[0] = '\0';

  return (afpath);
}

/*================================================================
 *	af_afname
 *
 *================================================================*/

EXPORT char *af_afname (unixname)
     char *unixname;
{
  register char *typeptr, *nameptr;
  char          *afReadName();
  static   char afname[MAXPATHLEN];
  register int  len;

  if (!unixname)
    {
      afname[0] = '\0';
      return (afname);
    }

  /* set nameptr to beginning of name */
  if ((nameptr = rindex (unixname, '/')) == (char *)0)
    nameptr = unixname;
  else
    nameptr++;

  if (af_isarchive (nameptr))
    {
      (void) strcpy (afname, nameptr + strlen (AF_ATFSFILEID));
      len = strlen (afname) - 1;
      afname[len] = '\0';
#if (MAXNAMLEN < 128)
      /* On System V machines, the name might be incomplete */
      if (len == (MAXNAMLEN-3)) /* take a look at the archive */
	{
	  (void) strcpy (afname, afReadName (unixname));
	}
#endif
    }
  else
    (void) strcpy (afname, nameptr);

  /* special handling for "." and ".." */
  if (!strcmp (afname, ".") || !strcmp (afname, ".."))
    return (afname);

  /* if a UNIX type-extension is given -- cut it, except the dot is */
  /*                                      at position 0 (e.g. .cshrc) */
  if ((typeptr = rindex (afname, '.')) != (char *)0)
    if ((typeptr != afname) && typeptr[1])
      typeptr[0] = '\0';

  return (afname);
}

/*================================================================
 *	af_aftype
 *
 *================================================================*/

EXPORT char *af_aftype (unixname)
     char *unixname;
{
  register char *typeptr, *nameptr;
  static   char aftype[MAXTYPLEN];
  register bool isarch = FALSE;

  if (!unixname)
    {
      aftype[0] = '\0';
      return (aftype);
    }

  /* set nameptr to beginning of name */
  if ((nameptr = rindex (unixname, '/')) == (char *)0)
    nameptr = unixname;
  else
    nameptr++;

  if (af_isarchive (nameptr))
    {
      nameptr += strlen (AF_ATFSFILEID);
      isarch = TRUE;
    }

  /* if there is no UNIX type-extension */
  if ((typeptr = rindex (nameptr, '.')) == (char *)0)
    aftype[0] = '\0';
  else
    {
      /* if the found dot indicates a "hidden file" (eg. .cshrc) */
      if (typeptr == nameptr)
	aftype[0] = '\0';
      else
	{
	  (void) strcpy (aftype, typeptr + sizeof(char));
	  /* if the named file is an archive, cut the name-extension */
	  if (isarch)
	    aftype [strlen (aftype) - sizeof (char)] = '\0';
	}
    }
  return (aftype);
}


/*================================================================
 *	af_unixname
 *
 *================================================================*/

EXPORT char *af_unixname (path, name, type)
     char *path, *name, *type;
{
  static char unixname[MAXPATHLEN];

  if ((path == (char *)0) || (path[0] == '\0'))
    (void) strcpy (unixname, NOTNIL(name));
  else
    (void) sprintf (unixname, "%s/%s\0", path, name);

  if ((type != (char *)0) && (type[0] != '\0'))
    {
      (void) strcat (unixname, ".");
      (void) strcat (unixname, type);
    }
  return (unixname);
}

/*================================================================
 *	af_bpfilename
 *
 *================================================================*/

EXPORT char *af_bpfilename (pathname, name)
     char *pathname, *name;
{
  static char bpname[MAXPATHLEN];

  (void) sprintf (bpname, "%s/%s\0", af_ardir (pathname), name);
  return (bpname);
}

/*================================================================
 * af_rbphashname -- get unique filename for file in binary pool
 *
 *================================================================*/

EXPORT char *af_rbphashname (name, type, gen, rev, list, count)
     char *name, *type;
     int  gen, rev;
     /*ARGSUSED*/
     Af_revlist *list; /* unused up to now */
     int count;
{
  char hashname[MAXNAMLEN+1];

#if (MAXNAMLEN < 128)
  char namefrag[MAXNAMLEN-10];
  int namelen = strlen (name);
  
  if (namelen > MAXNAMLEN-11)
    {
      (void) strcpy (namefrag, &name[namelen-(MAXNAMLEN-11)]);
      (void) sprintf (hashname, "%s%s[%d.%d]%d\0", AF_BPFILEID, 
		      namefrag, gen, rev, count);
    }
  else
    (void) sprintf (hashname, "%s%s[%d.%d]%d\0", AF_BPFILEID, 
		    name, gen, rev, count);
#else
  (void) sprintf (hashname, "%s%s.%s[%d.%d]%d\0", AF_BPFILEID, 
		  name, type, gen, rev, count);
#endif

  return (af_entersym (hashname));
}
