/* Copyright (C) 1989,1990,1991,1992 by
	Wilfried Koch, Andreas Lampen, Axel Mahler, Juergen Nickelsen,
	Wolfgang Obst and Ulrich Pralle
 
 This file is part of shapeTools.

 This software is published in the hope that it will be useful, but
 WITHOUT ANY WARRANTY for any part of this software to work correctly
 or as described in the manuals. See the ShapeTools Public License
 for details.

 Permission is granted to use, copy, modify, or distribute any part of
 this software but only under the conditions described in the ShapeTools 
 Public License. A copy of this license is supposed to have been given
 to you along with shapeTools in a file named LICENSE. Among other
 things, this copyright notice and the Public License must be
 preserved on all copies.
 */
#ifndef lint
static char *AtFSid = "$Header: retrv.c[3.24] Fri Feb  7 20:24:57 1992 axel@cs.tu-berlin.de accessed $";
#ifdef CFFLGS
static char *ConfFlg = CFFLGS;
	/* should be defined from within Makefile */
#endif
#endif

/*
 *  retrv [-fmqtx] [-V version] [-a attrs] [-d date] [-g generation]  
 *          [-n author] [-p pname] [-s state] [-lock] [-dest path] fn [fn1 ...]
 *
 *  Retrieve a previously saved version of a file. We orient us pretty much
 *  towards the check-out operation of RCS. Retrieve determines the
 *  archive name to be searched from the given file names. Unless otherwise
 *  specified by the project context (-p), the archive is expected to
 *  reside in the AtFS subdirectory. The retrieved version will be created
 *  in the current directory. Retrieve tries to be careful if an
 *  attempt is made to overwrite an existing busy-version: unless -f
 *  is specified, retrv will ask the caller for permission.
 *  If no busy version exists, one is created with the modes of the
 *  formerly saved version. If one exists, it's modes are kept unless
 *  -m is given.
 *  There's a number of ways to specify which version should be retrieved.
 *  With -V an explicit version can be selected. Another argument to -V
 *  could be a symbolic name (hopefully unique). Alternatively, versions
 *  can be selected by supplying certain attribute values to retrv, such as
 *  the name of the author, the version state, a generation number or a 
 *  set of user defined attributes, possibly describing a variant. In case
 *  that more than one version fits the desired attributes, the newest
 *  of them is selected, unless -x (exact!) is specified. -V implies -x.
 */

#include <stdio.h>
#include <atfs.h>
#include <atfsapp.h>
#include <ParseArgs.h>
#include "retrv.h"

/* #include "project.h" now in afsapp.h */

char *progname;

struct Transaction ThisTransaction;

/* forward decls for option handlers */
extern handle_R_switch ();
extern handle_f_switch ();
extern handle_l_switch ();
extern handle_fix_switch ();
extern handle_last_switch ();
extern handle_m_switch ();
extern handle_q_switch ();
extern handle_t_switch ();
extern handle_x_switch ();
extern handle_V_opt ();
extern handle_a_opt ();
extern handle_d_opt ();
extern handle_dest_opt ();
extern handle_g_opt ();
extern handle_i_opt ();
extern handle_n_opt ();
extern handle_p_opt ();
extern handle_s_opt ();
extern usage ();

unsigned int options;
int StdinFlag = 0;		/* -stdin option */
int NoExpandFlag = 0;		/* -noxp option */
Af_key *NewLock = NULL;
char intent_fname[MAXPATHLEN+1];

static OptDesc retrvargs[] = {
  { "version", PSWITCH, handle_R_switch, NULL, NULL },
  { "f", PSWITCH, handle_f_switch, NULL, NULL },
  { "l", PSWITCH, handle_l_switch, NULL, NULL },
  { "lock", PSWITCH, handle_l_switch, NULL, NULL },
  { "fix", PSWITCH, handle_fix_switch, NULL, NULL },
  { "q", PSWITCH, handle_q_switch, NULL, NULL },
  { "V", PARG, handle_V_opt, NULL, NULL }, 
  { "a", PARG, handle_a_opt, NULL, NULL }, 
  { "d", PARG, handle_d_opt, NULL, NULL }, 
  { "date", PARG, handle_d_opt, NULL, NULL }, 
  { "dest", PARG, handle_dest_opt, NULL, NULL },
  { "g", PARG, handle_g_opt, NULL, NULL }, 
  { "i", PARG, handle_i_opt, NULL, NULL },
  { "s", PARG, handle_s_opt, NULL, NULL }, 
  { "h", PSWITCH, usage, NULL, NULL },
  { "stdin", PSWITCH|PSET, NULL, &StdinFlag, NULL},
  { "xpoff", PSWITCH|PHIDDEN|PSET, NULL, &NoExpandFlag, NULL},
/*  { "?", PARG, usage, NULL, NULL }, */
  { (char *)  NULL, NULL, NULL, NULL, NULL }
};
static OptDesc vcatargs[] = {
  { "version", PSWITCH, handle_R_switch, NULL, NULL },
  { "q", PSWITCH, handle_q_switch, NULL, NULL },
  { "t", PSWITCH, handle_t_switch, NULL, NULL },
  { "V", PARG, handle_V_opt, NULL, NULL }, 
  { "a", PARG, handle_a_opt, NULL, NULL }, 
  { "d", PARG, handle_d_opt, NULL, NULL }, 
  { "date", PARG, handle_d_opt, NULL, NULL }, 
  { "g", PARG, handle_g_opt, NULL, NULL }, 
  { "last", PSWITCH, handle_last_switch, NULL, NULL },
  { "n", PARG, handle_n_opt, NULL, NULL }, 
  { "p", PARG, handle_p_opt, NULL, NULL }, 
  { "s", PARG, handle_s_opt, NULL, NULL }, 
  { "h", PSWITCH, usage, NULL, NULL },
  { "xpoff", PSWITCH|PHIDDEN|PSET, NULL, &NoExpandFlag, NULL},
/*  { "?", PARG, usage, NULL, NULL }, */
  { (char *)  NULL, NULL, NULL, NULL, NULL }
};

static int nfnms;

static struct Vdesc dversion;

char *dp = NULL; /* nasty sideeffect here - see 'handle_dest_opt' */

main (ac, av) int ac; char **av; {
  int i;
  int nac, bound_vid = 0, rc = 0;
  unsigned int options_bak = 0;
  char messg[MSGLEN], *es, *getenv(), **nav, *retrversion(), vstring[16], *cp;
  Project pdesc;
  struct Vdesc *vdescr = &dversion, alt_dversion;

  progname = (cp = rindex (av[0], '/')) ? ++cp : av[0];
  /* make prog-name available to entire program */

  if (ac < 2) {
    pa_ShortUsage (progname, vcatargs, "files ...");
  }

  if (!strcmp (progname, TONAME)) {
    options |= TYPEOUT | XACT;
    if (ParseArgs (ac, av, &nac, &nav, vcatargs)) {
      pa_ShortUsage (progname, vcatargs, "files...");
    }
  }
  else {
    if (ParseArgs (ac, av, &nac, &nav, retrvargs)) {
      pa_ShortUsage (progname, retrvargs, "files...");
    }
  }

  if (!(options & (TYPEOUT | COPY | LOCKIT))) options |= COPY;
    
  if (!(options & ATTRDEF)) {
    es = getenv (RTRATTR);
    if ((es) && (es[0] != '\0')) {
      options |= ATTRDEF;
      (void)strcpy (dversion.v_attrf, es);
    }
  }

  if ((options & PROJCSET) && fail(GetProject (dversion.v_pname, &pdesc))) {
    (void)sprintf (messg, "%s %s", EINVALPROJ, dversion.v_pname);
    logerr (messg);
    exit (1);
  }

  CatchSigs ();
  nfnms = nac;

  if ((options & LOCKIT) && !(options & QUIETPLEASE))
    if (checkForAtfsDirs (nac, nav))
      exit (1);
      
  ThisTransaction.tr_rc = 0;
  for (i = 0, vdescr = &dversion; i < nfnms; i++) {
    if (!setjmp (ThisTransaction.tr_env)) {
      if (bound_vid) {
	bound_vid = FALSE;
	options = options_bak; /* restore original selection options */
	vdescr = &dversion;
      }
      ThisTransaction.tr_seqno = i;
      if (BoundVersion (nav[i], ThisTransaction.tr_fname, vstring)) {
	alt_dversion.v_vno = mkvno (vstring);
	options_bak = options; /* save original selection options */
	options |= (VSPECSET | XACT);
	options &= ~(ATTRDEF | GENSET | AUNSET | STATSET | DATESET);
	vdescr = &alt_dversion;
	bound_vid = TRUE;
	if (options & FIX) {
	  Af_key tmpkey;
	  if (af_getkey (af_afpath (ThisTransaction.tr_fname),
			 af_afname (ThisTransaction.tr_fname),
			 af_aftype (ThisTransaction.tr_fname),
			 gen(alt_dversion.v_vno),
			 AF_LASTVERS,
			 &tmpkey) < 0) {
	    (void) sprintf (messg, "Can't find fixpoint for %s.",
			    nav[i]);
	    logerr (messg);
	    af_perror ("");
	    abort_this (FALSE);
	  }
	  if (vdescr->v_vno != at_rvno (&tmpkey)) {
	    (void) sprintf (messg, 
	      "WARNING: fixpoint for %s is different from baseline %s.", 
			    ThisTransaction.tr_fname, nav[i]);
	    logmsg (messg);
	    vdescr->v_vno = at_rvno (&tmpkey);
	  }
	  af_dropkey (&tmpkey);
	}
      }
      else {
	(void)strcpy (ThisTransaction.tr_fname, nav[i]);
      }
      ThisTransaction.tr_done = FALSE;
      NewLock = NULL;
      RetrieveAFile (ThisTransaction.tr_fname, vdescr, &pdesc, dp);
    }
    else { /* ThisTransaction was aborted */
      rc += ThisTransaction.tr_rc;
    }
  }
  if (nfnms > 1) logdiag ("done.");
  return (rc);
}

Sfunc_t interrupt_action () { /* is executed by appropriate signal handler */
  char messg[MSGLEN];

  disableSig (SIGINT);
  if (NewLock && !(ThisTransaction.tr_done)) {
    /* Try to unlock an eventually placed new lock */
    (void)vc_unlock (NewLock);
  }

  if ((nfnms - ThisTransaction.tr_seqno) > 1) { 
    (void)sprintf (messg, "\ncompletely stop retrieving (%d files pending) ?", 
	     nfnms - ThisTransaction.tr_seqno);
    if (ask_confirm (messg, "no")) {
      if (ThisTransaction.tr_done) {
	(void)sprintf (messg, "\ntoo late, %s already restored", 
		 ThisTransaction.tr_fname);
	logdiag (messg);
	return; /* continue where we've been interrupted */
      }

      (void)sprintf (messg, NORESTORE, ThisTransaction.tr_fname);
      logdiag (messg);
      enableSig ();
      CatchSigs (); /* see comment in save.c */
      af_cleanup ();
      longjmp (ThisTransaction.tr_env, 1);
    }
    else {
      (void)sprintf (messg, NORESTORE, ThisTransaction.tr_fname);
      logmsg (messg);
      exit (1);
    }
  }
  else {
    (void)sprintf (messg, "\n%s not restored", ThisTransaction.tr_fname);
    logdiag (messg);
    exit (1);
  }
}

logmsg (msg) char *msg; {
  if (!(options & QUIETPLEASE)) {
    fprintf (stdout, "%s\n", msg);
  }
}

logdiag (msg) char *msg; {
  if (!(options & QUIETPLEASE)) {
    (void) fflush(stdout);
    fprintf (stderr, "%s\n", msg);
  }
}

logwarn (msg) char *msg; {
  (void) fflush(stdout);
  if (!(options & QUIETPLEASE)) {
    (void) fprintf (stderr, "%s\n", msg);
  }
}

logerr (msg) char *msg; {
  (void) fflush(stdout);
  fprintf (stderr, "%s: %s\n", progname, msg);
}

/*ARGSUSED*/
handle_R_switch (o, a) char *o, *a; {
  char *retrversion(), *at_version(), *af_version();
  
  (void) printf ("This is %s version %s.\n", progname, retrversion ());
  (void) printf ("AtFS toolkit lib version %s.\n", at_version());
  (void) printf ("AtFS version %s.\n", af_version());
  exit (0);
}

/*ARGSUSED*/
handle_f_switch (o, a) char *o, *a; {
  options |= FORCE;
  return 0;
}

/*ARGSUSED*/
handle_l_switch (o, a) char *o, *a; {
  char messg[MSGLEN];

  if (options & TYPEOUT) {
    logerr ("Locking makes no sense in TYPEOUT mode of operation - ignored.");
    return 0;
  }
  if (options & COPY) {
    (void)sprintf (messg, "No checkout (with lock) to distant directory %s.", dp);
    logerr (messg);
    exit (1);
  }
  if (options & FIX) {
    logerr ("Use either \"-fix\" or \"-lock\".");
    exit (1);
  }
  options |= LOCKIT;
  return 0;
}

/*ARGSUSED*/
handle_fix_switch (o, a) char *o, *a; {
  char messg[MSGLEN];

  if (options & TYPEOUT) {
    logerr ("Startfix makes no sense in TYPEOUT mode of operation - ignored.");
    return 0;
  }
  if (options & COPY) {
    (void)sprintf (messg, "No checkout (with lock) to distant directory %s.", dp);
    logerr (messg);
    exit (1);
  }
  if (options & LOCKIT) {
    logerr ("Use either \"-fix\" or \"-lock\".");
    exit (1);
  }
  options |= (FIX | LOCKIT);
  return 0;
}

/*ARGSUSED*/
handle_m_switch (o, a) char *o, *a; {
  options |= KEEPMODE;
  return 0;
}

/*ARGSUSED*/
handle_q_switch (o, a) char *o, *a; {
  options |= QUIETPLEASE;
  return 0;
}

/*ARGSUSED*/
handle_t_switch (o, a) char *o, *a; {
  if (options & LOCKIT) {
    logerr ("Locking makes no sense in TYPEOUT mode of operation - ignored.");
    options &= ~LOCKIT;
  }
  if (options & COPY) {
    logerr ("TYPEOUT mode of operation overrides COPY mode.");
    options &= ~COPY;
    dp = NULL;
  }
  options |= TYPEOUT;
  return 0;
}

/*ARGSUSED*/
handle_x_switch (o, a) char *o, *a; {
  options |= XACT;
  return 0;
}

/*ARGSUSED*/
handle_last_switch (o, a) char *o, *a; {
  if (!strcmp (progname, TONAME)) {
    options &= ~XACT;
    return 0;
  }
  return 0;
}

/*ARGSUSED*/
handle_V_opt (o, a) char *o, *a; {
  dversion.v_vno = mkvno (a);
  (void)strcpy (dversion.v_spec, a);
  options |= VSPECSET;
  options |= XACT;
  return 0;
}

/*ARGSUSED*/
handle_a_opt (o, a) char *o, *a; {
  options |= ATTRDEF;
  (void)strcpy (dversion.v_attrf, a);
  return 0;
}

/*ARGSUSED*/
handle_d_opt (o, a) char *o, *a; {
  char messg[MSGLEN];
  time_t at_mktime();

  if ((options & STATSET) && (dversion.v_state == AF_BUSY)) {
    logmsg (WDOVRBS);
    options &= ~STATSET;
  }
  if (dversion.v_time = at_mktime (a)) {
    options |= DATESET;
    return 0;
  }
  else {
    (void)sprintf (messg, "invalid date specification: %s.", a);
    logerr (messg);
    tusage ();
    return 1;
  }
}

/*ARGSUSED*/
handle_dest_opt (o, a) char *o, *a; {
  char messg[MSGLEN], *lc;
  static char dpath[MSGLEN];
  struct stat statbuf;

  if (!a) return 1;
  if (options & LOCKIT) {
    (void)sprintf (messg, "No checkout (with lock) to distant directory %s.", 
		   a);
    logerr (messg);
    exit (1);
  }
  if (options & TYPEOUT) {
    logerr ("Already TYPEOUT mode of operation selected.");
    return 0;
  }
  if (options & COPY) {
    (void)sprintf (messg, "Destination path already set to %s. %s ignored.",
		   dpath, a);
    logerr (messg);
    return 0;
  }
  options |= COPY;
  (void)strcpy (dpath, a);

  lc = &dpath[strlen(dpath) - 1];
  while (*lc == '/') *lc-- = '\0';

  if (stat (dpath, &statbuf) < 0) {
    (void)sprintf (messg, "Destination path %s does not exist.", dpath);
    logerr (messg);
    exit (1);
  }
  if (!(statbuf.st_mode & S_IFDIR)) {
    (void)sprintf (messg, "Destination %s is not a directory.", dpath);
    logerr (messg);
    exit (1);
  }    
  dp = dpath;
  return 0;
}
  
/*ARGSUSED*/
handle_g_opt (o, a) char *o, *a; {
  char messg[MSGLEN];

  if (dversion.v_genno = mkgenno (a)) {
    options |= GENSET;
    return 0;
  }
  else {
    (void)sprintf (messg, "'%s' is not a legal generation number.", a);
    logerr (messg);
    return 1;
  }
}

/*ARGSUSED*/
handle_i_opt (o, a) char *o, *a; {
  char messg[MSGLEN];

  if (a && *a) {
    options |= INTENTSET;
    (void)strcpy (intent_fname, a);
    if (strcmp (intent_fname, "-") == NULL) {
      StdinFlag = 1;
      options &= ~INTENTSET;
    }
    return 0;
  }
  return 1;
}

/*ARGSUSED*/
handle_n_opt (o, a) char *o, *a; {
  char *cp;

  options |= AUNSET;
  if (cp=index (a, '@')) {
    *cp = '\0';
    (void)strcpy (dversion.v_auhost, ++cp);
  }
  (void)strcpy (dversion.v_aunam, a);
  return 0;
}

/*ARGSUSED*/
handle_p_opt (o, a) char *o, *a; {
  options |= PROJCSET;
  (void)strcpy (dversion.v_pname, a);
  return 0;
}

/*ARGSUSED*/
handle_s_opt (o, a) char *o, *a; {
  char messg[MSGLEN];

  if (!fail((dversion.v_state = mkstate (a)))) {
    if ((options & DATESET) && (dversion.v_state == AF_BUSY)) {
      logmsg (WDOVRBS);
    }
    else {
      options |= STATSET;
    }
  }
  else {
    (void)sprintf (messg, "unrecognized version state: %s.\n", a);
    logerr (messg);
    helpstates ();
    exit (1);
  }
  return 0;
}

tusage () {
  fputs ("specify time as yy/mm/dd[/hh:mm].\n", stderr);
}

helpstates () {
  fputs ("The following states are recognized:\n", stderr);
  fputs ("\tbusy, save, proposed, published, accessed, frozen\n", stderr);
}

usage () {
  if (strcmp (progname, "vcat"))
    pa_ShortUsage (progname, retrvargs, "files ...");
  else 
    pa_ShortUsage (progname, vcatargs, "files ...");
  exit (0);
}
