/***************************************************************************
 * This program reads a TECO macro contained in file CLPARS.TEC.
 * It writes CLPARS.H,  a C include file that is included by ZPRSCL.C.
 ***************************************************************************/

#include <stdlib.h>		/* define prototype for exit() */
#include <stdio.h>		/* define prototypes for fopen(), etc. */
#include <string.h>		/* define prototype for strcpy */

#ifndef FALSE
#define FALSE (1==0)
#endif
#ifndef TRUE
#define TRUE (1==1)
#endif

typedef char BOOLEAN;

#ifndef EXIT_FAILURE
#define EXIT_FAILURE 2
#endif
#define ILINE_SIZE 132	/* maximum input file (CLPARS.TEC) line length */
#define OLINE_SIZE 75	/* maximum output file (CLPARS.H) line length */



static void
open_clpars_tec(FILE **ifile)
{
	*ifile = fopen("clpars.tec","r");
	if (*ifile == NULL)
		{
		puts("genclp: Unable to open file CLPARS.TEC for reading");
		perror("genclp");
		exit(EXIT_FAILURE);
		}
}



static void
open_clpars_h(FILE **ofile)
{
	*ofile = fopen("clpars.h","w");
	if (*ofile == NULL)
		{
		puts("genclp: Unable to open file CLPARS.C for writing");
		perror("genclp");
		exit(EXIT_FAILURE);
		}
}



static void
write_line(char *obuf, FILE *ofile)
{
	if (fputs(obuf, ofile) == EOF)
		{
		puts("genclp: Unable to write to file CLPARS.C");
		perror("genclp");
		exit(EXIT_FAILURE);
		}
}



static void
write_header(FILE *ofile)
{
	write_line("/*\n", ofile);
	write_line(" * This file was created by the GENCLP program.\n",ofile);
	write_line(" * It should not be edited.  To make changes,\n", ofile);
	write_line(" * change the CLPARS.TES file,  then squeeze\n", ofile);
	write_line(" * CLPARS.TES to produce the CLPARS.TEC file,\n", ofile);
	write_line(" * then run GENCLP to produce this file.\n", ofile);
	write_line(" */\n", ofile);
}




static void
close_clpars_tec(FILE *ifile)
{
	if (fclose(ifile) != 0)
		{
		puts("genclp: Unable to close input file CLPARS.C");
		perror("genclp");
		exit(EXIT_FAILURE);
		}
}



static void
close_clpars_h(FILE *ofile)
{
	if (fclose(ofile) != 0)
		{
		puts("genclp: Unable to close output file CLPARS.C");
		perror("genclp");
		exit(EXIT_FAILURE);
		}
}



static void
cnvrt(	FILE *ifile,
	FILE *ofile,
	BOOLEAN ANSI_style)
{
	char *iptr;
	char *optr;
	int total_chars = 0;
	int total_lines = 0;
	char iline[ILINE_SIZE];
	char oline[OLINE_SIZE];

	oline[0] = '\"';
	optr = &oline[1];
	while (fgets(iline, ILINE_SIZE, ifile) != NULL)
		{
		iptr = iline;
		while (*iptr != '\n')
			{
			switch (*iptr) {
			case '"':  *optr++ = '\\'; *optr++ = '"'; break;
			case '\\': *optr++ = '\\'; *optr++ = '\\'; break;
			case '\n': *optr++ = '\\'; *optr++ = 'n'; break;
			case '\r': *optr++ = '\\'; *optr++ = 'r'; break;
			default:
			    if (*iptr >= ' ')
				*optr++ = *iptr;
			    else
				optr += sprintf(optr,
					(ANSI_style) ? "\\%o\"\"" : "\\%03o",
					*iptr);
			}
			iptr++;
			total_chars += 1;
			if (optr > &oline[OLINE_SIZE-10])
				{
				*optr++ = '"';
				if (!ANSI_style)
					*optr++ = ',';
				*optr++ = '\n';
				*optr++ = '\0';
				write_line(oline, ofile);
				total_lines += 1;
				optr = &oline[1];
				}
			}
		}
	*optr++ = '"';
	*optr++ = '\n';
	*optr   = '\0';
	write_line(oline, ofile);
	total_lines += 1;
	write_line("};\n", ofile);
	sprintf(oline, "#define CLPARS_LEN %d\n", total_chars);
	write_line(oline, ofile);
	if (!ANSI_style)
		{
		sprintf(oline, "#define CLPARS_LINES %-3d\n", total_lines);
		write_line(oline, ofile);
		}
}



main(void)
{
	FILE *ifile, *ofile;

	open_clpars_h(&ofile);

	/*
	 * write ANSI_CLPARS
	 */

	open_clpars_tec(&ifile);
	write_header(ofile);

	write_line("#if USE_ANSI_CLPARS\n", ofile);
	write_line("char clpars[] = {\n", ofile);

	cnvrt(ifile, ofile, TRUE);
	close_clpars_tec(ifile);

	/*
	 * write non-ANSI_CLPARS
	 */
	open_clpars_tec(&ifile);

	write_line("#else\n", ofile);
	write_line("char *clpars[] = {\n", ofile);

	cnvrt(ifile, ofile, FALSE);
	write_line("#endif\n", ofile);
	close_clpars_tec(ifile);

	close_clpars_h(ofile);
}
