/* ************************************************************************
*  File: guild.c                                        Part of CircleMUD *
*  Usage: Source file for class-specific code                             *
*                                                                         *
*  All rights reserved.  See license.doc for complete information.        *
*                                                                         *
*  Copyright (C) 1993, 94 by the Trustees of the Johns Hopkins University *
*  CircleMUD is based on DikuMUD, Copyright (C) 1990, 1991.               *
************************************************************************ */

#include "conf.h"
#include "sysdep.h"

#include "structs.h"
#include "comm.h"
#include "db.h"
#include "utils.h"
#include "spells.h"
#include "interpreter.h"
#include "handler.h"
#include "race.h"
#include "guild.h"
#include "screen.h"

/* External variables */
extern struct guild_data *guild_table;
extern struct spell_info_type spell_info[];

/* External functions */
long asciiflag_conv(char *flag);

int guild_info[MAX_GUILD_GUARDS + 1][3];

char *pc_guildrank_types[] = {
	"Member",
	"Officer",
	"Elder",
	"Leader",
	"Patron"
};

void set_guildguard(int guild_nr, int room_vnum, char dir, int slot)
{
	if (slot >= MAX_GUILD_GUARDS) {
		fprintf(stderr, "GuildGuards: out of slots... extend MAX_GUILD_GUARDS\n");
		return;
	}
	/* first set next one to end */
	guild_info[slot + 1][0] = -1;
	guild_info[slot + 1][1] = -1;
	guild_info[slot + 1][2] = -1;

	guild_info[slot][0] = guild_nr;
	guild_info[slot][1] = room_vnum;

	switch (dir) {
	case 'N':
	case 'n':
		guild_info[slot][2] = SCMD_NORTH;
		break;
	case 'S':
	case 's':
		guild_info[slot][2] = SCMD_SOUTH;
		break;
	case 'E':
	case 'e':
		guild_info[slot][2] = SCMD_EAST;
		break;
	case 'W':
	case 'w':
		guild_info[slot][2] = SCMD_WEST;
		break;
	case 'U':
	case 'u':
		guild_info[slot][2] = SCMD_UP;
		break;
	case 'D':
	case 'd':
		guild_info[slot][2] = SCMD_DOWN;
		break;
	default:
		fprintf(stderr, "Error: invalid direction %c in guildguard %d\n", dir, guild_nr);
		exit(1);
		break;
	}
}

void set_guildprereq(int guild_nr, char type, int number)
{
	switch (type) {
	case 'S':
	case 's':
		guild_table[guild_nr].minstr = MIN(20, number);
		break;
	case 'D':
	case 'd':
		guild_table[guild_nr].mindex = MIN(20, number);
		break;
	case 'C':
	case 'c':
		guild_table[guild_nr].mincon = MIN(20, number);
		break;
	case 'I':
	case 'i':
		guild_table[guild_nr].minintel = MIN(20, number);
		break;
	case 'W':
	case 'w':
		guild_table[guild_nr].minwis = MIN(20, number);
		break;
	case 'H':
	case 'h':
		guild_table[guild_nr].mincha = MIN(20, number);
		break;
	case 'L':
	case 'l':
		guild_table[guild_nr].minlevel = MIN(LVL_FULLIMMORT, number);
		break;
	default:
		fprintf(stderr, "GUILD %d: Invalid Minimum type %c\n", guild_nr, type);
		return;
	}
}

void load_guild(FILE *gld_file, int total_guild)
{
  static int guild_nr = 0;
  static int guild_info_num = 0;
  char line[256], letter, letter2;
  int t[3];
 
  if (guild_nr >= NUM_CLASSES) {
	fprintf(stderr, "NUM_CLASSES is not big enough... raise that number.\n");
	return;
  }
  guild_table[guild_nr].fullname = fread_string(gld_file, buf);
  guild_table[guild_nr].shortname = fread_string(gld_file, buf);
  guild_table[guild_nr].abbrev = fread_string(gld_file, buf);
  if (!get_line(gld_file, line)) {
	fprintf(stderr, "Format error in guild %d: Missing available races.\n", guild_nr);
	exit(1);
  }
  guild_table[guild_nr].races = asciiflag_conv(line);
  if (!get_line(gld_file, line)) {
	fprintf(stderr, "Format error in guild %d: Missing current xp.\n", guild_nr);
	exit(1);
  }
  guild_table[guild_nr].currentxp = atol(line);
  if (!get_line(gld_file, line)) {
	fprintf(stderr, "Format error in guild %d: Missing current skill.\n", guild_nr);
	exit(1);
  }
  guild_table[guild_nr].currentskill = atoi(line);

  sprintf(buf, "Format error in guild %d (expecting G/S/M/$)", guild_nr);
  for (;;) {
	if (!get_line(gld_file, line)) {
	   fprintf(stderr, "%s\n", buf);
	   exit(1);
	}
	switch (*line) {
	case 'G':
	case 'g':
		/* G roomvnum dir */
		sscanf(line, " %c %d %c", &letter, t, &letter2);
		set_guildguard(guild_nr, *t, letter2, guild_info_num);
		guild_info_num++;
		break;
	case 'S':
	case 's':
		/* S skillnum level */
		sscanf(line, " %c %d %d", &letter, t, t + 1);
		spell_level(*t, guild_nr, MIN(MAX_PRAC_LEVEL, *(t + 1)));
		break;
        case 'M':
        case 'm':
		/* M s|d|c|i|w|h|l number */
		sscanf(line, " %c %c %d", &letter, &letter2, t);
		set_guildprereq(guild_nr, letter2, *t);
		break;
	case '$':
		top_of_guild_table = guild_nr++;
		return;
		break;
	default:
		fprintf(stderr, "%s\n", buf);
		exit(1);
		break;
	}
  }
}

void save_guild(int guild_nr) {
    FILE *gF;
    struct guild_data *thisguild = &guild_table[guild_nr];
    int i;
    char c;

    /* get name of file */
    sprintf(buf2, "%s/%d.gld", GLD_PREFIX, guild_nr);
    /* open file */
    if (!(gF = fopen(buf2, "w+"))) {
	mudlog("SYSERR: GUILD: Cannot open guild file!", BRF, LVL_GRGOD, TRUE);
	return;
    }

    sprintbitascii(thisguild->races, buf);
    /* print strings */
    if (fprintf(gF, 
	    "%s~\n"
	    "%s~\n"
	    "%s~\n"
	    "%s\n"
            "%ld\n"
            "%d\n",
	    thisguild->fullname,
	    thisguild->shortname,
	    thisguild->abbrev,
	    buf,
	    thisguild->currentxp,
	    thisguild->currentskill) < 0) {
	mudlog("SYSERR: OLC: Cannot write gld file!\r\n", BRF, LVL_GRGOD, TRUE);
	fclose(gF);
	return;
    }

    /* print minimums */
    if (thisguild->minstr > 0) {
	fprintf(gF, "M s %d\n", thisguild->minstr);
    }
    if (thisguild->mindex > 0) {
	fprintf(gF, "M d %d\n", thisguild->mindex);
    }
    if (thisguild->mincon > 0) {
	fprintf(gF, "M c %d\n", thisguild->mincon);
    }
    if (thisguild->minintel > 0) {
	fprintf(gF, "M i %d\n", thisguild->minintel);
    }
    if (thisguild->minwis > 0) {
	fprintf(gF, "M w %d\n", thisguild->minwis);
    }
    if (thisguild->mincha > 0) {
	fprintf(gF, "M h %d\n", thisguild->mincha);
    }
    if (thisguild->minlevel > 0) {
	fprintf(gF, "M l %d\n", thisguild->minlevel);
    }

    /* guildgard, do a reverse set_guildguard */
    /*		G <vnum> <direction>		*/
    for (i = 0; (guild_info[i][0] != -1) && (i < MAX_GUILD_GUARDS) ; i++) {
	if (guild_info[i][0] == guild_nr) {
	   switch (guild_info[i][2]) {
		case SCMD_NORTH:
			c = 'N';
			break;
		case SCMD_SOUTH:
			c = 'S';
			break;
		case SCMD_EAST:
			c = 'E';
			break;
		case SCMD_WEST:
			c = 'W';
			break;
		case SCMD_UP:
			c = 'U';
			break;
		case SCMD_DOWN:
			c = 'D';
			break;
		default:
			mudlog("SYSERR: GUILD: Bad guard info", BRF, LVL_GRGOD, TRUE);		     
			continue;
			break;
	   }
	   fprintf(gF, "G %d %c\n", guild_info[i][1], c);
	}
    }

    /* skills, scan through spelllevel		*/
    /*          S <skillnum> <level>		*/
    for (i = 0; i <= TOP_SPELL_DEFINE; i++) {
	if (spell_info[i].min_level[guild_nr] > 0) {
	   fprintf(gF, "S %d %d\n", i, spell_info[i].min_level[guild_nr]);
	}
    }

    /* put a $ at the end */
    fprintf(gF, "$\n");
    /* close the file */
    fclose(gF);
    return;
}
   

void show_guilds(struct char_data *ch, int guild_number)
{
  int i = guild_number;

  if (guild_number == -1) {
	i = 0;
	guild_number = top_of_guild_table;
  }

  sprintf(buf, "Guilds:        Total Number = %d\r\n", top_of_guild_table);

  if (i <= top_of_guild_table && i >= 0) {
   for (; i <= guild_number; i++) {
	sprintf(buf, "%s  %sName: %s%-13s%s  Abbrev: %s%s%s     Minlevel: %s%3d%s   XP: %s%ld\r\n",
		buf, CCGRN(ch, C_SPR),
		BCCYN(ch, C_SPR), guild_table[i].fullname, CCGRN(ch, C_SPR),
		BCCYN(ch, C_SPR), guild_table[i].abbrev, CCGRN(ch, C_SPR),
		BCCYN(ch, C_SPR), guild_table[i].minlevel, CCGRN(ch, C_SPR),
		BCCYN(ch, C_SPR), guild_table[i].currentxp);
	sprintf(buf, "%s   %sMin: Str: %s%2d%s  Dex: %s%2d%s  Con: %s%2d%s  Int: %s%2d%s  Wis:  %s%2d%s Char: %s%-2d\r\n",
		buf, CCGRN(ch, C_SPR),
		BCYEL(ch, C_SPR), guild_table[i].minstr, CCGRN(ch, C_SPR),
		BCYEL(ch, C_SPR), guild_table[i].mindex, CCGRN(ch, C_SPR),
		BCYEL(ch, C_SPR), guild_table[i].mincon, CCGRN(ch, C_SPR),
		BCYEL(ch, C_SPR), guild_table[i].minintel, CCGRN(ch, C_SPR),
		BCYEL(ch, C_SPR), guild_table[i].minwis, CCGRN(ch, C_SPR),
		BCYEL(ch, C_SPR), guild_table[i].mincha);
        sprintbit(guild_table[i].races, pc_race_shorttypes, buf2);
	sprintf(buf, "%s %sRaces: %s%s\r\n", buf, CCGRN(ch, C_SPR),
		CCCYN(ch, C_SPR), buf2);
   }
  } else {
	strcat(buf, "Invalid guild number.\r\n");
  }
  page_string(ch->desc, buf, 1);
}

