/****************************************************************
*  program: civmover
*
*  Distribute populations of civilians 
*
*  for every sector with more than a given number of civ's or uw's:
*  uses a modified pathfinder algorithm to seek out target sectors
*     that can receive 150 civ's or 150 uw's
*  output move commands put in "civmover.out"
*
*  current options: 
*     -c <civ threshold>       set civ threshold to <>; default 900
*
*     -u <uw threshold>        set uw threshold to <>; default 900
*
*     -i <filename>            specify input file; default "dump"
*
*     -o <filename>            specify output file; default "civmover.out"
*
*     -v                       verbose output
****************************************************************/

#define OPTIONS "vc:u:i:o:"
#define CIVDEFAULT 900		/* if pop exceeds this, move some out */
#define UWDEFAULT 900
#define MAXPATH 64
#define CIVNUM 200		/* number of civs to move out */
#define UWNUM 120
#define KEEPMOB 60			/* maintain this much mobility */
#define NEEDMOB 126		/* need more than this in donor sector */

#include <stdio.h>
#include <string.h>
#include "sve.h"

extern struct maptype Map[XWIDTH][YHEIGHT];
extern struct Secdata *Sector;
extern char Direction[6];

int gotciv[XWIDTH][YHEIGHT], gotuw[XWIDTH][YHEIGHT];

main(argc, argv)
int argc;
char **argv;
{
  float mobcost;
  int verbose = 0;
  char *needpath;
  int i,j;
  int current;			
  int sectorcount;
  int civthresh, uwthresh;
  int findciv;			/* 1 if civmoving; 0 if uwmoving */
  int x,y;
  int target_x,target_y;
  char infile[15];
  char outfile[15];
  int opt;
  extern int optind;
  extern char *optarg;
  FILE *fp, *fopen();

  needpath = (char *) malloc (MAXPATH +1);
  strcpy(infile, "dump");	/* default input file is "dump" */
  strcpy(outfile, "civmover.out"); /* default output file */
  civthresh = CIVDEFAULT;
  uwthresh = UWDEFAULT;

  while ((opt = getopt(argc, argv, OPTIONS)) != EOF)
    {
      switch(opt) {
      case 'c':
	civthresh = atoi(optarg);
	break;
      case 'u':
	uwthresh = atoi(optarg);
	break;
      case 'i':
	strcpy(infile, optarg);
	break;
      case 'o':
	strcpy(outfile, optarg);
	break;
      case 'v':
	verbose = 1;
	break;
      case '?':
      default:
	printf("Usage: civmover -c <civ> -u <uw> -i <infile> -o <outfile> \n");
	exit(0);
      }
    }
  /* Initialize both the map and the Sector structures.
   */

  if (verbose)
    printf("This is civmover version 2.1; Sean Eddy, June 1990\n");

  sectorcount = getdump(infile);
  if (verbose)
    printf("Getdump has processed %d sectors.\n",sectorcount);

  weemapmaker(sectorcount);
  if (verbose)
    printf("Weemapmaker has arrayed map data for those sectors.\n");
  
  for (i=0; i<XWIDTH; i++)
    for (j=0; j<YHEIGHT; j++)
	gotciv[i][j] = gotuw[i][j] = 0;

  fp = fopen(outfile, "w");

				/* for every sector: check uw and civ excess */
  for (current = 0; current < sectorcount; current++)
    {
      if (verbose)
	printf("Looking now at sector %d: %d, %d \n",current,
	       Sector[current].Census.X, Sector[current].Census.Y );

      if ((Sector[current].Census.Civ > civthresh) &&
	  (Sector[current].Census.Occupied != '*') &&
	  (Sector[current].Census.Mob > NEEDMOB))
	{
	  if (verbose)
	    printf("\tToo many civs. Move %d \n", CIVNUM);

	  x = Sector[current].Census.X;
	  y = Sector[current].Census.Y;
	  recoordinate(&x,&y);
	  findciv = 1;

	  strcpy(needpath, (needfinder(x,y,&target_x,&target_y,findciv)));

	  if (strcmp(needpath, "NULL") == 0)
	    continue;

	  mobcalc(x,y,needpath, &mobcost);

	  if (verbose)
	    {
	      printf("Path: %sh\t",needpath);
	      printf("Will cost: %.0f \t", (CIVNUM * mobcost));
	      printf("Has: %d\n", Sector[current].Census.Mob);
	    }
	  
	  if ((CIVNUM * mobcost) > (Sector[current].Census.Mob - KEEPMOB))
	    {
	      if (verbose)
		printf("Insufficient mobility to move people.\n");
	      continue;
	    }

	  gotciv[target_x][target_y] += CIVNUM;

	  fprintf(fp, "move c %d,%d %d %sh \n", Sector[current].Census.X,
		  Sector[current].Census.Y, CIVNUM, needpath);

	  continue;
	}

      if ((Sector[current].Commod.Comm[0] > uwthresh) &&
	  (Sector[current].Census.Occupied != '*') &&
	  (Sector[current].Census.Mob > NEEDMOB))
	{
	  x = Sector[current].Census.X;
	  y = Sector[current].Census.Y;
	  recoordinate(&x,&y);
	  findciv = 0;

	  strcpy(needpath, (needfinder(x,y, &target_x, &target_y, findciv)));

	  if (strcmp(needpath, "NULL") == 0)
	    continue;

	  mobcalc(x, y, needpath, &mobcost);

	  if ((UWNUM * 2 * mobcost) > (Sector[current].Census.Mob - KEEPMOB))
	    continue;

	  gotuw[target_x][target_y] += UWNUM;

	  fprintf(fp, "move uw %d,%d %d %sh \n",Sector[current].Census.X, 
		  Sector[current].Census.Y, UWNUM, needpath);

	  continue;
	}
    }
  fclose(fp);
}

