;; -*-Emacs-Lisp-*-
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; File:         emp-dev.el
;; RCS:          $Header: emp-dev.el,v 1.3 90/12/07 18:31:26 darrylo Exp $
;; Description:  Routines to develop user-specified sectors in a country
;; Author:       Darryl Okahata
;; Created:      Mon Nov 26 18:46:30 1990
;; Modified:     Fri Dec 14 19:39:01 1990 (Darryl Okahata) darrylo@hpsrdmo
;; Language:     Emacs-Lisp
;; Package:      N/A
;; Status: GEET General Release 2d Patch 0
;;
;; TABLE OF CONTENTS
;;   empire-toggle-dev-sector-display -- Toggles map display of current or desired designations
;;   empire-view-developing-sectors -- Displays a buffer with info on sector development plans
;;   empire-develop-sectors -- Redistribute civilians to specified sectors.  This function works
;;   empire-insert-sector-to-develop -- Prompt for a designation, and insert the designation and the current
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; emp-dev.el -- Routines to develop user-specified sectors in a country
;; Copyright (C) 1990 Darryl Okahata (darrylo%hpnmd@hpcea.hp.com)
;; 
;; This program is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 1, or (at your option)
;; any later version.
;; 
;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.
;; 
;; You should have received a copy of the GNU General Public License
;; along with this program; if not, write to the Free Software
;; Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
;; 
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; Note that "(provide 'emp-dev)" is done at the END of this file.
(require 'cl)
(require 'emp-db)
(require 'emp-sector)


(defvar empire-developing-sectors nil
  "List of sectors to develop.")
(put 'empire-developing-sectors 'empire-system t)

(defvar empire-development-buffer "*Development*"
  "Temp buffer used to hold the output of `empire-develop-sectors'.")

(defvar empire-developing-buffer "*Developing*"
  "Temp work buffer used to display a list of developing sectors.")

(defvar empire-designations-buffer "*Desired_Empire_Designations*"
  "Temp buffer used to hold the contents of `empire-designations-file'.")

(defvar empire-dev-current nil
  "A temp variable that holds the list of developing sectors, as they are
processed.  This variable is used only at runtime.")

(defvar empire-development-mode nil
  "*The minor mode var for empire-development-mode.  If non-nil, the
map buffer will display the desired designations of sectors, not what
each sector really is.")

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defconst empire-developing-sectors-db-max 3
  "The number of elements in an entry in the sector development database.")

(defconst empire-developing-pos
  '(
    (location	.		0)	;; (cons x y)
    (des	.		1)
    (plan	.		2)
   )
  "A list, similar to `empire-pos', that describes each entry in the sector
development database.")

(defmacro emp-dev-item-offset (name)
  (let (offset)
    (if (not (symbolp name))
	(error (format "`emp-dev-item': name `%s' must be a symbol." name)))
    (if (not (setq offset (cdr (assq name empire-developing-pos))))
	(error (format "`emp-dev-item': unknown name `%s'" name)))
    offset
    ))

(defmacro emp-dev-recall (item offset)
  (`
   (aref (, item) (, offset))
   )
  )

(defmacro emp-dev-store (item offset value)
  (`
   (aset (, item) (, offset) (, value))
   )
  )

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun empire-development-mode ()
  ""
  (let ()
    (or (assq 'empire-development-mode minor-mode-alist)
	(setq minor-mode-alist (cons '(empire-development-mode " Development")
				     minor-mode-alist)))
    )
  )

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun delequal (lst item)
  "From list LST, delete ITEM, and return the resulting list.  This is a
destructive operation."
  (let (previous result)
    (setq result lst)
    (catch 'done
      (while lst
	(if (equal item (car lst))
	  (progn
	    (if previous
	      (progn
		(setcdr previous (cdr lst))
	      )
	      (progn
		(setq result (cdr lst))
	      )
	    )
	    (throw 'done nil)
	  )
	)
	(setq previous lst
	      lst (cdr lst)
	)
      )
    )
    result
  )
)


(defun insert-empire-designation (x y des)
  ""
  (let (current (loc (cons x y)) entry)
    (if empire-developing-sectors
	;; how atout
	;; (setq current (assoc (cons x y) empire-developing-sectors))
	(setq current (catch 'found
			(progn
			  (dolist (entry empire-developing-sectors)
			    (if (equal (emp-dev-recall entry
						       (emp-dev-item-offset
							location))
				       loc)
				(throw 'found entry))
			    )
			  nil
			  ))
	      )
      )
    (if current
	(progn
	  (emp-dev-store current (emp-dev-item-offset des) des)
	  )
      (progn
	(setq current (make-vector (length empire-developing-pos) nil))
	(emp-dev-store current (emp-dev-item-offset location)
		       (cons x y))
	(emp-dev-store current (emp-dev-item-offset des) des)
	(setq empire-developing-sectors (cons current
					      empire-developing-sectors))
	)
      )
    )
  )


(defun empire-erase-dev-sector (entry)
  ""
  (let (loc x y des)
    (setq loc	(emp-dev-recall entry (emp-dev-item-offset location))
	  x	(car loc)
	  y	(cdr loc)
	  des	(recall x y (position-of des))
	  )
    (map-des x y des)
  )
)


(defun empire-display-dev-sector (entry)
  ""
  (let (loc x y des)
    (setq loc	(emp-dev-recall entry (emp-dev-item-offset location))
	  x	(car loc)
	  y	(cdr loc)
	  des	(emp-dev-recall entry (emp-dev-item-offset des))
	  )
    (map-des x y des)
  )
)


(defun empire-toggle-dev-sector-display ()
  "Toggles map display of current or desired designations"
  (interactive)
  (let ()
    (if empire-development-mode
      (progn
	;;
	;; Display current designations
	;;
	(setq empire-development-mode nil)
	(mapcar 'empire-erase-dev-sector empire-developing-sectors)
      )
      (progn
	;;
	;; Display desired designations
	;;
	(setq empire-development-mode t)
	(mapcar 'empire-display-dev-sector empire-developing-sectors)
      )
    )
  )
)
(put 'empire-toggle-dev-sector-display 'empire t)


(defun empire-display-dev-sectors-if-necessary ()
  ""
  (let ()
    (if empire-development-mode
	(mapcar 'empire-display-dev-sector empire-developing-sectors))
  )
)


(defun empire-print-developing-header ()
  "Print header."
  (let ()
    (princ "    x,y      New des    Old des         Plan\n")
    (princ "   -----    ---------  ---------       ------\n")
  )
)


(defun empire-display-developing-sector (sector)
  "Print a line describing the sector SECTOR."
  (let (loc x y des plan cur-des cur-eff)
    (setq loc		(emp-dev-recall sector (emp-dev-item-offset location))
	  x		(car loc)
	  y		(cdr loc)
	  des		(emp-dev-recall sector (emp-dev-item-offset des))
	  plan		(emp-dev-recall sector (emp-dev-item-offset plan))
	  cur-des	(recall-macro x y (position-of des))
	  cur-eff	(recall-macro x y (position-of eff))
	  )
    (princ (format "%5s,%-5d\t%s\t%s (%3s%%)\t%s\n"
		   x y des
		   (or cur-des "?")
		   (or cur-eff "?")
		   (or plan "None")))
  )
)


(defun empire-view-developing-sectors ()
  "Displays a buffer with info on sector development plans"
  (interactive)
  (let ()
    (empire-save-window-excursion
     (with-output-to-temp-buffer empire-developing-buffer
       (princ "\t*** Sectors being developed ***\n\n")
       (empire-print-developing-header)
       (mapcar 'empire-display-developing-sector empire-developing-sectors)
       )
     (save-excursion
       (set-buffer empire-developing-buffer)
       (empire-data-mode nil nil)
       (local-set-key " " 'empire-toggle-dev-sector-display)
       (goto-line 5)	;; first data line
       )
     )
    (pop-to-buffer empire-developing-buffer)
    )
  )
(put 'empire-view-developing-sectors 'empire t)


(defun empire-develop-a-sector (data)
  "Develop a sector, whose development information is given by DATA.
This function should be called from `empire-develop-sectors', as that
function defines some variables used by this function.

This function returns either `nil' or DATA.  If it returns DATA, the
sector given by data has been completely developed, and should be
removed from the development database."
  (let (des-wanted location x y (delete nil))
    (setq des-wanted	(emp-dev-recall data
					(emp-dev-item-offset des))
	  location	(emp-dev-recall data
					(emp-dev-item-offset location))
	  x		(car location)
	  y		(cdr location)
	  )
    (empire-flash-coordinates "Developing" x y)
    (redistribute-civ-internal x y)
    (check-and-fix-sector-distribution x y)
    (if (not (string= (recall x y des) des-wanted))
      (progn
	;;
	;; The sector does not have the desired designation
	;;
	(if (and (string= (recall-macro x y (position-of 'sdes)) "_")
		 (> (recall x y civ) 10)	;; should be a variable
						;; (FIX ME!)
	    )
	  (progn
	    (make-automated-command (format "des %s,%s %s"
					    x y des-wanted))
	  )
	)
      )
      (progn
	;;
	;; The sector does have the desired designation.
	;; Now check to see if it's reached 100% eff.
	;;
	(if (eq (recall-macro x y (position-of eff)) 100)
	  (progn
	    ;; It has -- flag it for deletion
	    (setq delete data)
	  )
	)
      )
    )
    delete
  )
)

(defun empire-develop-sectors ()
  "Redistribute civilians to specified sectors.  This function works
just like `redistribute-civ', except that the sector designations and
x,y coordinates of the sectors to develop are stored in the variable,
`empire-developing-sectors', which is set by the function
`empire-insert-sector-to-develop'.

If the data sector designation does not match `DES', if the sector
has not already been redesignated, and if the number of civilians in the
sector are greater than 10, the sector will be redesignated to match
`DES' (but only after the civilians have been moved into the sector --
this is useful if all undeveloped sectors are designated as roads)."
  (interactive)
  (require 'emp-sector)
  (let ((des (position-of des))
	(civ (position-of civ))
	(mob (position-of mob))
	(work (position-of work))
	delta
	source
	dest-des
	dest-is-roadp
	mob-needed
	ideal
	path
	bodies-to-move
	xs ys
	(uncivilizable nil)
	(empire-use-inefficient-highways nil)
	completed-sectors
	(header-output nil)
	)
    (save-window-excursion
      (save-excursion
	;;
	;; Here, we don't want the empire flows buffer displayed at the end
	;; of the analysis.
	;;
	(empire-flows)		; do first to get 'distribution attribute set
	)
      )
    (with-output-to-temp-buffer empire-development-buffer
      (princ (format "\t*** Sector Development as of %s ***\n"
		     (current-time-string)))
      (princ (format "At least %s civs will be moved, with a maximum of %s.\n"
		     empire-min-civ-move
		     empire-max-civ-redistribute))
      (princ "See the documentation for `redistribute-civ' for details of the format.\n\n")

      (princ "\t\t   x,y       (D Eff Mob Bodies_Now->Bodies_ideal  Food)\n")
      (princ "\t\t   ---       -----------------------------------------\n")
      (empire-data-mode-buffer t t empire-development-buffer)
      (empire-switch-to-buffer-not-map empire-development-buffer)
      (setq completed-sectors (mapcar 'empire-develop-a-sector
				      empire-developing-sectors))
      (dolist (item completed-sectors)
	(if item
	  (progn
	    (setq empire-developing-sectors (delequal empire-developing-sectors
						      item))
	    (if (not header-output)
	      (progn
		(princ "\n\n\t*** Completed Sectors ***\n\n")
		(empire-print-developing-header)
		(setq header-output t)
	      )
	    )
	    (empire-display-developing-sector item)
	    )
	  )
	)
      )
    (message "Sector development done.")
    )
  )
(put 'empire-develop-sectors 'empire t)


(defun empire-insert-sector-to-develop (x y des)
  "Prompt for a designation, and insert the designation and the current
map sector X,Y location into the empire sector development database, which
is used to develop sectors (see `empire-develop-sectors' for details).
Note that the sectors being developed will not be displayed in the map
buffer, unless the `empire-development-mode' minor mode is enabled, via
\\[empire-toggle-dev-sector-display], or by setting the
`empire-development-mode' variable to non-nil."
  (interactive (let ( (sects (get-map-sector)) current-des)
		 (if (setq current-des (recall (car sects) (cdr sects)
					       (position-of des)))
		     (progn
		       (if (not (assoc current-des sector-type-strings))
			   (error "What kind of sector is `%s'?" current-des))
		       (if (eq (sector-info current-des 'move-cost) 0)
			   (error "You can't develop this sector (`%s')!"
				  current-des))
		       )
		   (progn
		     (error "You cannot develop unknown sectors.")
		     )
		   )
		 (list (car sects) (cdr sects)
		       (empire-prompt-read-designation
			"Develop into des? "))
		 )
	       )
  (let ()
    (insert-empire-designation x y des)
    (if empire-development-mode
	(map-des x y des))
    )
  )
(put 'empire-insert-sector-to-develop 'empire t)

(provide 'emp-dev)
