;;;;;;;;;;;;;;;;;;;;;;;;;;; -*- Mode: Emacs-Lisp -*- ;;;;;;;;;;;;;;;;;;;;;;;;;;
;; emp-modes.el -- Modes and keyboard commands for Gnu Emacs Empire Tool (GEET)
;; 
;; Copyright (c) 1990,1991 Lynn Randolph Slater, Jr
;; 
;; Author          : Lynn Slater  (lrs@indetech.com)
;; Created On      : Fri Oct 26 13:50:14 1990
;; Last Modified By: Lynn Slater x2048
;; Last Modified On: Thu Feb 14 19:16:41 1991
;; Update Count    : 148
;; Status          : GEET General Release 2d Patch 0
;; 
;; PURPOSE
;; 	Provides empire shell, its mode, and interaction command support
;; Also provides map mode and empire-display-mode.
;;
;; Ideally, this file should only define mode maps and the TAB commands and
;; the command definitions should be elsewhere. This will make this file
;; easier to custom modify.
;; HISTORY
;; 9-Feb-1991		Lynn Slater x2048	
;;    Last Modified: Fri Feb  8 18:10:08 1991 #144 (Lynn Slater x2048)
;;    added a mpde map for surveys
;; 5-Feb-1991		Lynn Slater x2048	
;;    Last Modified: Tue Feb  5 09:18:55 1991 #138 (Lynn Slater x2048)
;;    added empire-telegram-map
;; 31-Jan-1991		Lynn Slater x2048	
;;    Last Modified: Tue Jan 29 10:21:53 1991 #128 (Lynn Slater x2048)
;;    split out into emp-cmds, emp-mouse, and emp-buffers
;; 23-Jan-1991		Lynn Slater x2048	
;;    Last Modified: Mon Jan 21 20:28:50 1991 #121 (Lynn Slater x2048)
;;    added ship and plane maps
;; 7-Jan-1991		Lynn Slater x2048	
;;    Last Modified: Sat Jan  5 16:02:03 1991 #112 (Lynn Slater x2048)
;;    removed read-prompt-xy
;; 5-Jan-1991		Lynn Slater x2048	
;;    Last Modified: Sat Jan  5 15:13:49 1991 #106 (Lynn Slater x2048)
;;    added select-this-sector, selected-sector-hooks, and select-a-sector
;;    added ship support
;; 10-Dec-1990		Lynn Slater x2048	
;;    Last Modified: Fri Dec  7 13:17:06 1990 #102 (Lynn Slater x2048)
;;    added v, V, !, and = to fancy map per darryl emp-dev.el
;; 3-Dec-1990		Lynn Slater x2048	
;;    Last Modified: Sun Dec  2 18:06:13 1990 #100 (Lynn Slater x2048)
;;    added a to fancy map -- calcs air defenders
;; 2-Dec-1990		Lynn Slater x2048	
;;    Last Modified: Sun Dec  2 18:05:43 1990 #99 (Lynn Slater x2048)
;;    coastwatch now parses
;; 25-Nov-1990		Lynn Slater x2048	
;;    Last Modified: Sun Nov 25 12:57:34 1990 #80 (Lynn Slater x2048)
;;    added darryls changes: develop command and quite commands
;; 14-Nov-1990		Lynn Slater	
;;    Last Modified: Tue Nov 13 16:54:50 1990 #74 (Lynn Slater)
;;    changed "mov " to "move" in issued command.
;; 13-Nov-1990		Lynn Slater	
;;    Last Modified: Sun Nov 11 16:07:06 1990 #73 (Lynn Slater)
;;    map mode sets the mark
;; 8-Nov-1990		Lynn Slater	
;;    Last Modified: Wed Nov  7 12:10:26 1990 #62 (Lynn Slater)
;;    added empire-des-sector
;;    bound simulate-sector to the fancy map
;; 5-Nov-1990		Lynn Slater	
;;    Last Modified: Mon Nov  5 15:37:14 1990 #52 (Lynn Slater)
;;    added return cap to visible-insert
;;    renamed some interactive fcns, documented them better
;;    flagged some as interactive
;; 5-Nov-1990		Lynn Slater	
;;    Last Modified: Mon Nov  5 15:36:32 1990 #50 (Lynn Slater)
;;    added ping to extract map
;; 2-Nov-1990		Lynn Slater	
;;    Last Modified: Thu Nov  1 13:46:34 1990 #35 (Lynn Slater)
;;    took out empire-general-local-mousemap
;;    empire-path-to-mouse-move-point no longer changes windows
;;    have "other" map from the shell
;;    Added darryl's changes
;; 30-Oct-1990		Lynn Slater	
;;    Last Modified: Tue Oct 30 09:48:29 1990 #23 (Lynn Slater)
;;    sector conditions are asked for only if prefix arg is given
;; 29-Oct-1990		Lynn Slater	
;;    Last Modified: Mon Oct 29 08:56:40 1990 #14 (Lynn Slater)
;;    moved changed switch-to-empire-buffer... and switch-to-map-buffer...
;;      to emp-db as they are macros and need to be present early
;; TABLE OF CONTENTS
;;   empire-census-sector -- Do a census on the current sector.
;;   empire-commodity-sector -- List the commodities in the current sector.
;;   empire-Qdump-sectors -- Do a QUICK dump with no rules.
;;   empire-dump-sector -- Do a dump on the current sector.
;;   empire-level-sector -- Execute a `level' command on the current sector.
;;   empire-resource-sector -- Execute a `resource' command on the current sector.
;;   empire-production-sector -- Execute a `production' command on the current sector.
;;   empire-map-default -- Do a map on the current bounds.
;;   empire-map-bounds -- Do a map on the given bounds.
;;   empire-map-bounds-quietly -- Do a quiet map of sectors in the box defined by (X1,Y1) and (X2,Y2).
;;   empire-map-quietly -- Do a quiet map of SECTORS, which is a string describing the sectors
;;   empire-path-sector -- Execute the `path' command.  The x,y location of the current
;;   empire-distribute-sector -- Execute the `distribute' command.  The x,y location of the current
;;   empire-des-sector -- Execute the `des' command.  The x,y location of the current
;;   empire-prompting-deliver -- Execute the `deliver' command.  The user is prompted for the
;;   empire-prompting-explore -- Prompt the user for some people to explore, and insert the command
;;   empire-prompting-move -- Prompt the user for some goods to move, and insert the command into the
;;   empire-prompting-test -- Prompt the user for some goods to move, and insert the command into the
;;   empire-prompting-survey -- Prompt the user for some goods to survey, insert, and execute the
;;   empire-prompting-threshold -- For the current map sector, send the `threshold' command to the empire
;;   empire-prompting-enlist -- For the current map sector, enlist AMOUNT military.
;;   empire-prompting-demobilize -- For the current map sector, demobilize AMOUNT military.  ACTIVE-RESERVES
;;   empire-ship-command -- Execute the `ship' command on SHIPS.
;;   empire-cargo-command -- Execute the `cargo' command on SHIPS.
;;   empire-census-quietly -- Do a quiet census of SECTORS, which is a string describing the sectors
;;   empire-radar-quietly -- Do a quiet radar of SECTORS, which is a string describing the sectors
;;   empire-dump-quietly -- Do a quiet dump of SECTORS, which is a string describing the sectors
;;   empire-shell-mode -- Major mode for interacting with an inferior shell running empire.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; The contents of this file ARE copyrighted but permission to use, modify,
;; and distribute this code is granted as described in the file
;; emp-install.el which should have been distributed with this file. These
;; terms constitute what the Free Software Foundation calls a COPYLEFT.
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(provide 'emp-modes)
(require 'emp-shell)
(require 'emp-anal)
(require 'emp-auto)
(require 'emp-plane)
(require 'emp-sail)
(require 'emp-help)
(require 'emp-buffers)
(require 'emp-mouse)
(require 'emp-cmds)

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; User variables -- do not edit here, use empire-edit-options
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defvar empire-shell-special-key "\C-c"
  "*This is the key under which the empire shell special commands are placed.")

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; Define some autoloads.  These are placed here to make it easy for the user
;; to see what functions are autoloaded.
;;
(autoload 'dynamic-mobility-show-used-left "emp-sector" "\
Place the map buffer into dynamic-mobility-show-mode and place the
display into the mode where the amount of mobility used/left is
displayed.
"
	  t)

(autoload 'dynamic-mobility-show-amount-can-move "emp-sector" "\
Place the map buffer into dynamic-mobility-show-mode and place the
display into the mode where the amount of mobility used/left is
displayed.
"
	  t)

(autoload 'dynamic-mobility-show-mode "emp-sector" "\
Enter dynamic-mobility-show-mode.  This mode saves and replaces the
existing mode in the map buffer.  In this mode, you are prompted for an
item, an amount to move from the current map sector, and the starting
mobility (the default is the amount of mobility left in the sector), and
the mobility used/left is displayed as you move the cursor over the map.
Pressing the space bar toggles between displaying mobility used/left and
displaying the amount that can be moved given the available mobility.

If you want to see how much of an item can be moved given a particular
amount of mobility, enter that amount when prompted for the mobility.

The full dynamic-mobility-show-mode key bindings are:
\\{dyna-mob-keymap}
"
	  t)

(autoload 'empire-display-plague-chance "emp-sector" "\
Display, in the minibuffer, the estimated chance of plague for the
sector at X,Y.  If this function is run interactively, the chance for
the current map sector is displayed.
"
	  t)

(autoload 'empire-simulate-sector "emp-sector" "\
Simulate the sector at X,Y for up to MAX-ITER iterations, displaying
the simulated values of each iteration in a temp buffer.  If called
interactively, the current map sector is simulated, and the current
value of `empire-simulation-max-iterations' determines the maximum
length of the simulation.  If the sector starves, the simulation will
terminate."
	  t)

(autoload 'empire-calc-plague-chance "emp-sector")

(autoload 'empire-insert-sector-to-develop "emp-dev" "\
Prompt for a designation, and insert the designation and the current
map sector X,Y location into the empire designations buffer, which is
created if it does not already exist."
	  t)

(autoload 'empire-view-developing-sectors "emp-dev" "\
Not yet documented"
	  t)

(autoload 'empire-develop-sectors "emp-dev" "\
Not yet documented"
	  t)

(autoload 'empire-toggle-dev-sector-display "emp-dev" "\
Not yet documented"
	  t)


;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Game commands
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(empire-define-information-command
 "Do a census. If given a prefix arg (\C-u), highlight returned sectors on
the map."
 empire-census-sectors
 "Census sectors? " "#"
 "census" 'empire-read-census t
 )

(defun empire-census-sector ()
  "Do a census on the current sector."
  (interactive)
  (empire-census-sectors (get-map-sector-string) "")
  )

(empire-define-information-command
 "Do a coastwatch."
 empire-coastwatch-sectors
 "Coastwatch from sectors? " nil
 "coastwatch" 'empire-read-coastwatch
 )

(empire-define-information-command
 "Show commodities in sectors."
 empire-commodity-sectors
 "Display commodities in sectors? " nil
 "commodity" nil
 )

(defun empire-commodity-sector ()
  "List the commodities in the current sector."
  (interactive)
  (empire-commodity-sectors (get-map-sector-string) "")
  )

(defun empire-Qdump-sectors (sects condition)
  "Do a QUICK dump with no rules."
  (interactive (list
      	  (empire-read-sector-or-realm "Sectors to quickly dump? "
      				       nil
      				       )
      	  (empire-prompt-read-condition)
      	  ))
  (map-execute-empire-command (format "dump %s %s"
      				sects condition)
      			'empire-quick-read-dump))
(register-empire-command "qdump" 'empire-Qdump-sectors)

(empire-define-information-command
 "Do a dump."
 empire-dump-sectors
 "Sectors to dump? " nil
 "dump" 'empire-read-dump
 )

(defun empire-dump-sector ()
  "Do a dump on the current sector."
  (interactive)
  (empire-dump-sectors (get-map-sector-string) "")
  )

(empire-define-information-command
 "Execute a `level' command."
 empire-level-sectors
 "Get level on sectors? " nil
 "level" nil
 )

(defun empire-level-sector ()
  "Execute a `level' command on the current sector."
  (interactive)
  (empire-level-sectors (get-map-sector-string) "")
  )

(empire-define-information-command
 "Execute a `resource' command."
 empire-resource-sectors
 "Get resource on sectors? " nil
 "resource" 'empire-read-resource
 )

(defun empire-resource-sector ()
  "Execute a `resource' command on the current sector."
  (interactive)
  (empire-resource-sectors (get-map-sector-string) "")
  )

(empire-define-information-command
 "Execute a `production' command."
 empire-production-sectors
 "Get production on sectors? " nil
 "production" 'empire-read-production
 )

(defun empire-production-sector ()
  "Execute a `production' command on the current sector."
  (interactive)
  (empire-production-sectors (get-map-sector-string) "")
  )

(empire-define-information-command
 "Do radars"
 empire-radar-sectors
 "Radar from sectors? " nil
 "radar" 'empire-read-radars
 )

(empire-define-information-command
 "Do version."
 empire-version-command
 nil nil
 "version" 'empire-read-version
 )

(defun empire-map-default ()
  "Do a map on the current bounds."
  (interactive)
  (empire-map-bounds empire-x-start empire-x-stop
      	       empire-y-start empire-y-stop)
  )
(register-empire-command "map" 'empire-map-default)


(defun empire-map-bounds (x1 x2 y1 y2)
  "Do a map on the given bounds."
  (interactive "nStart X:\nnStop X: \nnStart Y: \nnStop Y: ")
  (map-execute-empire-command (format "map %s:%s,%s:%s" x1 x2 y1 y2)
      			'(lambda () (empire-read-map x1)))
  )

(defun empire-map-bounds-quietly (x1 y1 x2 y2 &optional no-redraw)
  "Do a quiet map of sectors in the box defined by (X1,Y1) and (X2,Y2).
If the optional NO-REDRAW is non-nil, the map buffer will not be redrawn
after the map is read."
  (interactive "nStart X:\nnStop X: \nnStart Y: \nnStop Y: ")
  (let (tmp)
    (if (> x1 x2)
	(progn
	  (setq tmp x1)
	  (setq x1 x2)
	  (setq x2 tmp)
	  )
      )
    (if (> y1 y2)
	(progn
	  (setq tmp y1)
	  (setq y1 y2)
	  (setq y2 tmp)
	  )
      )
    (empire-cmd-other-buf-and-exec (format "map %s:%s,%s:%s" x1 x2 y1 y2)
				   nil)
    (save-excursion
      (set-buffer empire-temp-buffer)
      (empire-read-map x1)
      )
    (if (not no-redraw)
	(map-empire)
      )
    )
  )

(defun empire-map-quietly (sectors)
  "Do a quiet map of SECTORS, which is a string describing the sectors
to map."
  (interactive (list
		(empire-read-sector-or-realm "Sectors to map? ")
		))
  (let ()
    (empire-cmd-other-buf-and-exec (format "map %s" sectors)
				   '(empire-read-map map-empire) nil t)
    )
  )
(register-empire-command "Map sectors quietly" 'empire-map-quietly)

(defun empire-path-sector ()
  "Execute the `path' command.  The x,y location of the current
map sector is used."
  (interactive)
  (send-empire-command (format "path %s" (get-map-sector-string)))
  )
(register-empire-command "path"		'empire-path-sector)

(defun empire-distribute-sector ()
  "Execute the `distribute' command.  The x,y location of the current
map sector is used."
  (interactive)
  (send-empire-command (format "distribute %s" (get-map-sector-string)))
  )
(register-empire-command "distribute"	'empire-distribute-sector)

(defun empire-des-sector (new-des)
  "Execute the `des' command.  The x,y location of the current
map sector is used."
  (interactive "sNew Des: ")
  (send-empire-command (format "des %s %s" (get-map-sector-string) new-des))
  (if (or (not (recall (map-x) (map-y) (position-of 'eff)))
	  (eql (recall (map-x) (map-y) (position-of 'eff)) 0))
      (let ((empire-map-des-interactively t))
	(record-des (map-x) (map-y) new-des t t)))
  )
(register-empire-command "designate"	'empire-des-sector)

(defun empire-prompting-deliver (sectors item dir threshold)
  "Execute the `deliver' command.  The user is prompted for the
sector/realm, item, direction, and threshold (if any), if necessary."
  (interactive (let (sectors item dir threshold)
		 (setq sectors (get-map-sector-string))
		 (setq item (prompt-for-item empire-basic-commodity-list))
		 (setq dir (empire-prompt-direction
			    (format "Deliver %s in direction? " item)
			    nil nil t))
		 (setq threshold (empire-prompt-threshold
				  "Threshold (press [Return] for none)? "))
		 (list sectors item dir threshold)
		 ))
  (let ()
    (send-empire-command (format "deliver %s %s %s %s"
				 item sectors dir threshold))
    )
  )
(register-empire-command "deliver" 'empire-prompting-deliver)


(defun empire-prompting-explore ()
  "Prompt the user for some people to explore, and insert the command
into the empire buffer.  The starting sector is updated in terms of
the people who are exploring (but nothing else, like mobility, is
updated).  Note that the final (destination) sector must be manually
updated."
  (interactive)
  (let (sect x y info item amount left)
    (setq sects (get-map-sector))
    (setq x (car sects))
    (setq y (cdr sects))
    (move-to-empire-shell-prompt);; Can we execute a command now?
    (setq info (empire-prompting-read-items x y "explore"
					    empire-people-commodity-list))
    (setq item (nth 0 info))
    (setq amount (nth 1 info))
    (send-empire-command (format "exp %s %s,%s %s " item x y amount))
    ;;
    ;; Update the database
    ;;
    (decriment-sector x y  (dynamic-position-of (read item)) amount)
    )
  )
(register-empire-command "explore"		'empire-prompting-explore)


(defun empire-prompting-move ()
  "Prompt the user for some goods to move, and insert the command into the
empire buffer.  The starting sector is updated in terms of the commodity
moved (but nothing else, like mobility, is updated).  Note that the final
 (destination) sector must be manually updated."
  (interactive)
  (let (sects x y info item amount original-amount amount-left)
    (setq sects (get-map-sector))
    (setq x (car sects))
    (setq y (cdr sects))
    (move-to-empire-shell-prompt);; Can we execute a command now?
    (setq info (empire-prompting-read-items x y "move"
					    empire-complete-commodity-list))
    (setq item (nth 0 info))
    (setq amount (nth 1 info))
    (send-empire-command (format "move %s %s,%s %s " item x y amount))
    ;;
    ;; Update the database
    ;;
    (decriment-sector x y  (dynamic-position-of (read item)) amount)
    )
  )
(register-empire-command "move"		'empire-prompting-move)


(defun empire-prompting-test ()
  "Prompt the user for some goods to move, and insert the command into the
empire buffer.  This is just like `empire-prompting-move', except that the
`test' command in used instead of `move'."
  (interactive)
  (let (sects x y info item amount)
    (setq sects (get-map-sector))
    (setq x (car sects))
    (setq y (cdr sects))
    (move-to-empire-shell-prompt);; Can we execute a command now?
    (setq info (empire-prompting-read-items x y "move (test only)"
					    empire-complete-commodity-list))
    (setq item (nth 0 info))
    (setq amount (nth 1 info))
    (send-empire-command (format "test %s %s %s "
				 item (get-map-sector-string) amount))
    )
  )
(register-empire-command "test"		'empire-prompting-test)

(defun empire-prompting-survey ()
  "Prompt the user for some goods to survey, insert, and execute the
command into the empire buffer."
  (interactive)
  (let (info item sectors condition command)
    (move-to-empire-shell-prompt);; Can we execute a command now?
    (setq item (prompt-for-item empire-survey-commodity-list))
    (setq sectors (empire-read-sector-or-realm "Sectors to survey? "))
    (setq condition (empire-prompt-read-condition))
    (setq command (format "survey %s %s %s" item sectors condition))
    (send-empire-command-and-parse-reply command 'make-survey-readable)
    )
  )
(register-empire-command "survey" 'empire-prompting-survey)


(defun empire-prompting-threshold (item amount)
  "For the current map sector, send the `threshold' command to the empire
client."
  (interactive (progn
		 ;; Can we execute a command now?
		 (move-to-empire-shell-prompt)
		 (list
		  (prompt-for-item empire-basic-commodity-list)
		  (empire-read-number-from-minibuffer
		   (format "Set threshold to? ") nil)
		  )
		 ))
  (if (not (interactive-p))
      (move-to-empire-shell-prompt)	; in case this is non-interactively
					; executed.
    )
  (send-empire-command (format "threshold %s %s %s"
			       item (get-map-sector-string) amount))
  )
(register-empire-command "threshold"	'empire-prompting-threshold)

(defun empire-prompting-enlist (amount)
  "For the current map sector, enlist AMOUNT military."
  (interactive (let (left)
		 (setq left (amount-in-map-sector 'mil))
		 (list
		  (empire-read-number-from-minibuffer
		   (format "Amount to enlist (current = %s) [0]? " left)
		   nil 0)
		  )))
  (let ()
    (send-empire-command (format "enlist %s %s"
				 (get-map-sector-string) amount))
    )
  )
(register-empire-command "enlist"		'empire-prompting-enlist)


(defun empire-prompting-demobilize (amount active-reserves)
  "For the current map sector, demobilize AMOUNT military.  ACTIVE-RESERVES
is a string containing \"yes\" or \"no\", depending on whether or not the
military should be converted into active reserves.
"
  (interactive (let (left)
		 (setq left (amount-in-map-sector 'mil))
		 (list
		  (empire-read-number-from-minibuffer
		   (format "Amount to demobilize (current = %s) [0]? " left)
		   nil 0)
		  (empire-prompt-yes-or-no
		   "New civilians on active reserve (y/n) [y]? " nil "y")
		  )))
  (let ()
    (send-empire-command (format "demobilize %s %s %s"
				 (get-map-sector-string) amount
				 active-reserves))
    )
  )
(register-empire-command "demobilize"	'empire-prompting-demobilize)


;;
;; This will have to be enhanced when ship info is kept.
;;
(defun empire-ship-command (ships condition)
  "Execute the `ship' command on SHIPS."
  (interactive (progn
      	   ;; Can we execute a command now?
      	   (move-to-empire-shell-prompt)
      	   (list
      	    (empire-prompt-read-ship)
      	    (empire-prompt-read-condition)
      	    )
      	   ))
  (let ()
    (switch-to-empire-buffer-if-necessary
     (if (not (interactive-p))
         (move-to-empire-shell-prompt) ; in case this is non-interactively
      				; executed.
       )
     (send-empire-command-and-parse-reply (format "ship %s %s" ships condition)
      				    'empire-read-ships)
     )
    )
  )
(register-empire-command "ship"		'empire-ship-command)

;;
;; This will have to be enhanced when ship info is kept.
;;
(defun empire-cargo-command (ships condition)
  "Execute the `cargo' command on SHIPS."
  (interactive (progn
      	   ;; Can we execute a command now?
      	   (move-to-empire-shell-prompt)
      	   (list
      	    (empire-prompt-read-ship)
      	    (empire-prompt-read-condition)
      	    )
      	   ))
  (let ()
    (switch-to-empire-buffer-if-necessary
     (if (not (interactive-p))
         (move-to-empire-shell-prompt) ; in case this is non-interactively
      				; executed.
       )
     (send-empire-command-and-parse-reply (format "cargo %s %s" ships condition)
      				    'empire-read-cargos)
     )
    )
  )
(register-empire-command "cargo"		'empire-cargo-command)

(defun empire-census-quietly (sectors)
  "Do a quiet census of SECTORS, which is a string describing the sectors
to census."
  (interactive (list
		(empire-read-sector-or-realm "Sectors to census? ")
		))
  (let ()
    (empire-cmd-other-buf-and-exec (format "census %s" sectors)
				   'empire-read-census nil t)
    )
  )
(register-empire-command "Census sectors quietly" 'empire-census-quietly)


(defun empire-radar-quietly (sectors)
  "Do a quiet radar of SECTORS, which is a string describing the sectors
to radar."
  (interactive (list
		(empire-read-sector-or-realm "Sectors to radar? ")
		))
  (let ()
    (empire-cmd-other-buf-and-exec (format "radar %s" sectors)
				   'empire-read-radar nil t)
    )
  )
(register-empire-command "Radar sectors quietly" 'empire-radar-quietly)

(defun empire-dump-quietly (sectors)
  "Do a quiet dump of SECTORS, which is a string describing the sectors
to dump."
  (interactive (list
		(empire-read-sector-or-realm "Sectors to dump? ")
		))
  (let ()
    (empire-cmd-other-buf-and-exec (format "dump %s" sectors)
				   'empire-read-dump nil t)
    )
  )
(register-empire-command "Dump sectors quietly" 'empire-dump-quietly)


(progn
  ;; Define a few special commands that are just packaged fcns
  (register-empire-command 
   "Check empire"		'(check-empire))
  (register-empire-command 
   "Check planes"		'(check-empire-planes))
  (register-empire-command
   "Refresh empire"		'refresh-empire)
  (register-empire-command
   "Adjust empire"		'adjust-empire)
  )

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Special modes
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun empire-shell-mode ()
  "Major mode for interacting with an inferior shell running empire.
Basically, play empire in the shell and run special commands. Commands are
sent to empire by the following techniques:
  Return anywhere in the last line sends line as input.
  Return in other line copies rest of line to end and sends it.

Operation is the same as shell mode except that the prompt pattern that
Emacs looks for is changed to 'Command :' and there are special commands
under Control C. These special commands will read empire output, map the
world, show positions on the map, redistribute food, mil, or commodities,
check your world, and do lots of other special things. Use the emacs help
facility for details on each option.

General help is available via using  \\[describe-function] on
empire-new-world. You may want to print the buffer that creates and this
entire buffer and use them as reference guides. 

			       * Commands *
The following commands provide special GEET functionality.
More organized command help is available through \\[help-for-empire-special-map].

In the explination below, you will see some rather long key sequences.
These will be much easier if you understand that there are three second
level command maps and that no command is more than two 'menus' deep.

The levels of menus are as follows:
  All special empire commands start with C-c. Under this,
    Redistribution  commands start with C-x.
    Data extraction commands start with C-d
    Tool customization commands start with C-c
    Most shell commands (that used to be under C-c) are under 
      C-s Note that some of these (such as
      next and previous can be useful in empire.

All menus will provide you with help if you type ?

The full special empire bindings are as follows:
\\{empire-shell-mode-map}
"
  (interactive)
  (let (process old-filter)
    ;;(kill-all-local-variables)
    (setq major-mode 'empire-shell-mode)
    (setq mode-name "Empire Shell")
    (setq mode-line-process '(": %s"))
    (use-local-map empire-shell-mode-map)
    (make-local-variable 'shell-prompt-pattern)
    (setq shell-prompt-pattern "^\\[[0-9]+:\\([0-9]+\\)\\] Command : ")
    (make-local-variable 'shell-directory-stack)
    (setq shell-directory-stack nil)
    (make-local-variable 'last-input-start)
    (setq last-input-start (or last-input-start (make-marker)))
    (make-local-variable 'last-input-end)
    (setq last-input-end (or last-input-end (make-marker)))
    ;; empire special stuff
    (setq empire-shell-buffer (buffer-name (current-buffer)))
    (if empire-shell-local-mousemap
	(progn
	  (make-local-variable 'current-global-mousemap)
	  (setq current-global-mousemap empire-current-global-mousemap)
	  (use-local-mousemap empire-shell-local-mousemap)))
    (setq process (get-buffer-process empire-shell-buffer))
    (if (not process)
	(error "There is no process associated with the empire shell buffer."))
    ;;
    ;; Here, we try to be a good citizen.  If a process filter already exists,
    ;; we save it and install a new one (which will call the old one).  If
    ;; one does not exist, we install a different process filter.
    ;;
    (if (setq old-filter (process-filter process))
	(progn
	  ;; A process filter already exists
	  (if (and (not (equal old-filter 'empire-output-filter))
		   (not (equal old-filter 'empire-output-postfilter)))
	      (progn
		;; Install a new one that calls the old one ONLY if the one
		;; that is installed is not one that we installed.
		(setq empire-old-process-filter old-filter)
		(set-process-filter process 'empire-output-postfilter)
		)
	    )
	  )
      (progn
	;; A process filter does not exist.  Install one.
	(set-process-filter process 'empire-output-filter)
	)
      )
    )
  )

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Redistribution map
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defvar empire-shell-redistribute-map nil
  "Keymap used in shell or empire shell mode with redistribution commands.")

(let ((c-mp (make-sparse-keymap)))
  (define-key c-mp "?"	  'help-for-empire-redistribute-map)
  (define-key c-mp "\C-h" 'help-for-empire-redistribute-map)
  (define-key c-mp help-character 'help-for-empire-redistribute-map)
  (define-key c-mp "m"		'redistribute-mil)
  (define-key c-mp "f"		'redistribute-food)
  (define-key c-mp "p"		'redistribute-population)
  (define-key c-mp "c"		'redistribute-civ)
  (setq empire-shell-redistribute-map c-mp))

(make-help-screen help-for-empire-redistribute-map
		  "c:civ m:mil p:population f:food ?"
		  "You have discovered the GEET redistribution commands
   From here, you can use the following options:

c	Redistribute civs from overfull sectors into connected underfull ones
	  The functions typically named by empire-ideal-civ-fcn control
          based in part on empire-sector-civ-threshold  
m	Redistribute military using levels given by empire-ideal-mil-fcn
p	Redistribute excess population to highways for max pop growth
	  Excess is any sector so full babies will not be born.
f	Even out food on highways to highway min and leave levels
	  This is good to pump max food to all warehouses/dist pts


Use \\[help-for-empire-redistribute-map] for help on redistribution.
Use \\[help-for-empire-extract-map] for help on data extraction.
Please use \\[describe-key] to find out more about any of the other keys."
		  empire-shell-redistribute-map)

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Customization commands
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defvar empire-shell-customization-map nil
  "Keymap used in shell or empire shell mode to allow user to customize tool.")

(let ((c-mp (make-sparse-keymap)))
  (define-key c-mp "?"	  'help-for-empire-customization-map)
  (define-key c-mp "\C-h" 'help-for-empire-customization-map)
  (define-key c-mp help-character 'help-for-empire-customization-map)
  (define-key c-mp "c"  	'empire-edit-check-hooks)
  (define-key c-mp "d"  	'empire-edit-dump-hooks)
  (define-key c-mp "p"  	'empire-edit-plane-check-hooks)
  (define-key c-mp "a"  	'empire-edit-adjust-hooks)
  (define-key c-mp "v"		'empire-edit-options)
  (define-key c-mp "b"	        'empire-bounds)
  (define-key c-mp "i"	        'empire-set-ideal-civ-function)
  (define-key c-mp "i"	        'empire-set-ideal-mil-function)
  (setq empire-shell-customization-map c-mp))

(make-help-screen help-for-empire-customization-map
		  "c:Checks d:Dumps a:Adjusts p:PLanes v:Vars b:Bounds i:Ideal Civ m:Ideal Mil ?"
		  "You have discovered how to customize GEET.
   From here, you can use the following options:

c	display/edit Check hooks. T toggles
d	display/edit Dump hooks.  T toggles
a	display/edit Adjust hooks. T toggles
c	display/edit Plane check hooks. T toggles
v	display/edit empire user variables. S sets new values
b	set effective map bounds and speed up the game
i	set the ideal-civ function, show the choices
m	set the ideal-mil function, show the choices

Please use \\[describe-key] to find out more about any of these keys.

Try these as alot of the game doc is in the explinations of the hooks or the
user variables."
		  empire-shell-customization-map)

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Extraction
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defvar empire-shell-extract-map nil
  "Keymap used in shell or empire shell mode with data extraction commands.")

(let ((c-mp (make-sparse-keymap)))
  (define-key c-mp "?"	  'help-for-empire-extract-map)
  (define-key c-mp "\C-h" 'help-for-empire-extract-map)
  (define-key c-mp help-character 'help-for-empire-extract-map)

  (define-key c-mp "e"		'empire-read-data) ;; extract everything
  (define-key c-mp "\C-e"	'read-refresh-adjust-empire) ;; EVERYTHING
  (define-key c-mp "a"		'adjust-empire) 
  (define-key c-mp "r"		'refresh-empire) 
  (define-key c-mp "\C-r"	'refresh-adjust-empire) 
  (define-key c-mp "P"		'empire-read-planes)
  (define-key c-mp "p"		'empire-read-production)
  (define-key c-mp "d"		'empire-read-dump)
  (define-key c-mp "q"		'empire-quick-read-dump)
  (define-key c-mp "f"		'empire-read-flight)
  (define-key c-mp "x"		'empire-read-explores)
  (define-key c-mp "g"		'empire-read-pings)
  (define-key c-mp ")"		'empire-read-radars)
  (define-key c-mp "l"		'empire-read-looks)
  (define-key c-mp "m"		'empire-read-map)
  (define-key c-mp "\C-m"	'empire-read-mines)
  (define-key c-mp "c"		'empire-read-census)
  (define-key c-mp "C"		'empire-read-cargos)
  (define-key c-mp "n"		'empire-read-navs)
  (define-key c-mp "N"		'empire-read-nation-report)
  (define-key c-mp "s"		'empire-read-spyreports)
  (define-key c-mp "S"		'empire-read-ships)
  (define-key c-mp "v"		'empire-read-version)

  (setq empire-shell-extract-map c-mp))

(make-help-screen help-for-empire-extract-map
		  "r C-r a e C-e p P d q c C S f x g ) l n N m C-m s v ?"
		  "You have discovered the GEET data extraction commands.
Almost all of them work from where you are to the end of the empire buffer.

	   -*- From here, you can use the following options: -*-
r	Refresh empire via asking the client new questions
C-r	Refresh and then run adjustments based on new data
a	Adjust your empire via various automatic analysis
e	Extract everything from here to the end of the buffer. Good for things
	  not automatically extracted such as the results of manual commands.
	  No analysis/adjustment is done unless a dump is found.
C-e	Do EVERYTHING. Extract all data to end of buffer, do a refresh, do an
	adjust, save.  Very long dings when done
p	Extract the next Production report
P	Extract the next Plane report. If universal arg, resets planes first.
d	Extract the next Dump command and check the levels and commodities.
q	Quickly extract the next dump command and do no checking
c	Extract the next Census data
C	Extract the next ship cargo data
S	Extract the next ship report data
f	Extract all Flight data to end of the file
x	Extract all eXplore data to the end of the file
g	Extract pinGGGGs to the end of the buffer
)	Extract all Radar data to the end of the file
l	Extract all data from Looks to the end of the file
n	Extract all data from navs to the end of the file
N	Extract the next nation data
m	Extract next Map data 
C-M	Extract all deployed and exploded Mine data to the of the file
s	Extract all spy data to the end of the file
v	Extract the next version information

Please use \\[describe-key] to find out more about any of these keys.

Other significant but not bound calls are
\\[empire-summarize-resources] -> Summarizes resources from explores
\\[empire-get-new-world] -> talks with client to set up a new universe

"
		  empire-shell-extract-map)

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Shell subprocess controll
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defvar empire-shell-shell-map nil)

(let ((c-mp (make-sparse-keymap)))
  ;; shell mode essentials
  (define-key c-mp "\C-d" 'shell-send-eof)
  (define-key c-mp "\C-u" 'kill-shell-input)
  (define-key c-mp "\C-c" 'interrupt-shell-subjob)
  (define-key c-mp "\C-z" 'stop-shell-subjob)
  (define-key c-mp "\C-\\" 'quit-shell-subjob)
  (define-key c-mp "\C-o" 'kill-output-from-shell)
  (define-key c-mp "\C-r" 'show-output-from-shell)
  (define-key c-mp "\C-y" 'copy-last-shell-input)
  ;; Diff from 18.52 -- lrs
  (define-key c-mp "\C-k" 'kill-all-output-from-shell)
  (define-key c-mp "\C-n" 'shell-next-command)
  (define-key c-mp "\C-p" 'shell-prev-command)
    
  ;; help
  (define-key c-mp "?"	  'help-for-empire-special-shell-map)
  (define-key c-mp "\C-h" 'help-for-empire-special-shell-map)
  (define-key c-mp help-character 'help-for-empire-special-shell-map)

  (setq empire-shell-shell-map c-mp)
  )

(make-help-screen help-for-empire-special-shell-map
		  "C-d C-c C-u C-z C-\\ C-o C-r C-y C-k C-n C-p ?"
		  "You have discovered the GEET shell commands.
These are the same as in a normal shell but are just moved 1 level deeper. 

   From here, you can use the following options:
C-d	Interrupt the shell (Send a control d)
C-c	Kill the sell job (Send a control c)
C-u	Kill shell input -- zap all since the last shell output
C-\\	Quit shell job
C-o	Kill shell output
C-r	Show shell output
C-y	Copy last shell input to current input
C-k	Kill all shell output made so far
C-n	Goto the next shell command
C-p	Goto the previous shell command

Please use \\[describe-key] to find out more about any of these keys."
		  empire-shell-shell-map)

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Sector manipulation
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defvar empire-sector-map nil "Sector commands")

(let ((c-mp (make-sparse-keymap)))
  (define-key c-mp "d" 'empire-des-in-empire-map)
  (define-key c-mp "u" 'empire-record-as-untaken)
  (define-key c-mp "e" 'empire-edit-sector)
  (define-key c-mp "f" 'forget-sector)

  (define-key c-mp "v" 'empire-view-developing-sectors)
  (define-key c-mp "V" 'empire-toggle-dev-sector-display)
  (define-key c-mp "!" 'empire-develop-sectors)
  (define-key c-mp "i" 'empire-insert-sector-to-develop)

  ;; help
  (define-key c-mp "?"	  'help-for-empire-sector-map)
  (define-key c-mp "\C-h" 'help-for-empire-sector-map)
  (define-key c-mp help-character 'help-for-empire-sector-map)

  (setq empire-sector-map c-mp))

(make-help-screen help-for-empire-sector-map
		  "d u e f  v V ! i"
		  "You have discovered the GEET sector commands
   From here, you can use the following options:
d	Change the tool's memory of the designation of the sector
	  (does not send change to server)
u	Make the sector not be reported as taken by the lost sector check
e	Edit the stored info for a given sector
f	Forget everything about this sector, as it you had never seen it

v	Display sector development plans
V	Toggles map display of current or desired designations
!	Try to develop sectors according to their plans
i	Define a development plan for a sector

Please use \\[describe-key] to find out more about any of these keys.
"
		  empire-sector-map)



;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Telegram handling
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defvar empire-telegram-map nil "Sector commands")

(let ((c-mp (make-sparse-keymap)))
  (define-key c-mp "r" 'reduce-production-reports)
  (define-key c-mp "p" 'reduce-production-reports)
  (define-key c-mp "\C-r" 'really-reduce-production-reports)
  (define-key c-mp "\C-p" 'really-reduce-production-reports)
  (define-key c-mp "t" 'take-out)

  ;; help
  (define-key c-mp "?"	  'help-for-empire-telegram-map)
  (define-key c-mp "\C-h" 'help-for-empire-telegram-map)
  (define-key c-mp help-character 'help-for-empire-telegram-map)

  (setq empire-telegram-map c-mp))

(make-help-screen help-for-empire-telegram-map
		  "r:reduce t:takeout"
		  "You have discovered the GEET telegram commands
   From here, you can use the following options:
r,p	Reduce the verbage from a production report
\C-r	Really reduce the verbage from a production report, even plague and
	  backlog reports.
t	Take-out (eliminate) any further instances of this particular line

Please use \\[describe-key] to find out more about any of these keys.
"
		  empire-telegram-map)

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Fancy Commands
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defvar empire-fancy-map nil "Strange commands")

(let ((c-mp (make-sparse-keymap)))
  (define-key c-mp "w"	'empire-get-new-world)
  (define-key c-mp "b"	'empire-break-new-world)
  (define-key c-mp "d"	'empire-des-new-world)
  (define-key c-mp "x"	'empire-explode-from-sector)
  (define-key c-mp "\C-x" 'empire-explode-all-sectors)
  (define-key c-mp "n"	'nova-from-sector)
  (define-key c-mp "\C-n" 'super-nova-from-sector)
  (define-key c-mp "e" 'nova-from-explored-sectors)

  (define-key c-mp "a" 'calc-air-defenders)

  (define-key c-mp "u" 'unload-dist-node)
  (define-key c-mp "+" 'empire-find-non-road-dist)


  (define-key c-mp "s" 'empire-simulate-sector)

  (define-key c-mp "v" 'empire-view-developing-sectors)
  (define-key c-mp "V" 'empire-toggle-dev-sector-display)
  (define-key c-mp "!" 'empire-develop-sectors)
  (define-key c-mp "=" empire-sector-map)

  ;; help
  (define-key c-mp "?"	  'help-for-empire-fancy-map)
  (define-key c-mp "\C-h" 'help-for-empire-fancy-map)
  (define-key c-mp help-character 'help-for-empire-fancy-map)

  (setq empire-shell-fancy-map c-mp))

(make-help-screen help-for-empire-fancy-map
		  "x C-x n C-n d w b s a e u + v V ! ="
		  "You have discovered the GEET fancy commands
   From here, you can use the following options:
x	Explode 1 sector in any direction from a given sector if the
	  adjacient sectors are wilderness and not ours.
C-x	Try to explode 1 sector from each sector
n	Nova: Use all the civs and all the mob from a given sector to take over
          all known reachable wilderness.
C-n	Super-Nova: Same as above but whenever there seeme to be no more
          wilderness, do a map to make sure we have not found new wilderness.
e	Nova-explored: Nova from all explored sectors, good for early game

d	redesignate all low eff roads to something better. Good for newly
          grabbed land.

w	Talk to the client and get the info needed to set up a NEW WORLD.
b	Talk to the client and manage the world from a sanctuary state.
	  rename country, BREAK, map, and supernova
s	Simulate the sector for awhile

a	calculate AIR defense for a given sector 

e	Edit the stored info for a given sector
u	Try to UNLOAD sectors from a distribution node
+	Find dist paths that are not on roads and bridges (legthy)

v	Display sector development plans
V	Toggles map display of current or desired designations
!	Try to develop sectors according to their plans
=	Do a sector designation/development plan command

Please use \\[describe-key] to find out more about any of these keys.
"
		  empire-shell-fancy-map)

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Map for user driving of surveys
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defvar empire-survey-map nil
  "Keymap used from empire shell OR the map with survey manipulation commands.")

(let ((c-mp (make-sparse-keymap)))
  (define-key c-mp "?"	  'help-for-empire-survey-map)
  (define-key c-mp "\C-h" 'help-for-empire-survey-map)
  (define-key c-mp help-character 'help-for-empire-survey-map)
  
  (define-key c-mp "a"		'empire-calc-survey-all)
  (define-key c-mp "r"		'empire-reset-survey-all)
  (define-key c-mp "c"		'empire-set-survey-commodity)
  (define-key c-mp "s"		'empire-set-survey-scale)
  (define-key c-mp "t"		'empire-toggle-survey)
						   
  (setq empire-survey-map c-mp))

(make-help-screen help-for-empire-survey-map
		  "a:All r:Reset c:Commodity s:Scale t:Toggle"
		  "You have discovered the empire tool survey commands
   From here, you can use the following options:

a	Survey all the known world
r	Reset to show the designations and not the survey data
c	Set the commodity which is surveyed
s	Set the Scale of the survey
t	Toggle map so that new sectors are shown in survey mode or not

Use \\[help-for-empire-survey-map] for help on redistribution.
Use \\[help-for-empire-extract-map] for help on data extraction.
Please use \\[describe-key] to find out more about any of the other keys."
		  empire-survey-map)


(autoload 'empire-calc-survey-all "emp-survey" "" t)
(autoload 'empire-reset-survey-all "emp-survey" "" t)
(autoload 'empire-set-survey-commodity "emp-survey" "" t)
(autoload 'empire-set-survey-scale "emp-survey" "" t)
(autoload 'empire-toggle-survey "emp-survey" "" t)

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; All empire commands from \C-c in a shell
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defvar empire-shell-special-map nil "The C-c map from a empire shell")

(let ((c-mp (make-sparse-keymap)))

  ;; shell mode essentials
  (define-key c-mp "\C-s" empire-shell-shell-map)
  (define-key c-mp "\C-n" 'shell-next-command)
  (define-key c-mp "\C-p" 'shell-prev-command)
  
  (define-key c-mp "c"
    'map-execute-prompted-command)
  (define-key c-mp "\C-c"
    'map-execute-prompted-command)
  (define-key c-mp "\t"
    'map-execute-prompted-command)

  (define-key c-mp "\C-x" empire-shell-extract-map)
  (define-key c-mp "x" empire-shell-extract-map)

  (define-key c-mp "\C-d" empire-shell-redistribute-map)
  (define-key c-mp "d" empire-shell-redistribute-map)

  (define-key c-mp "u" empire-shell-customization-map)

  (define-key c-mp "\C-o" empire-shell-fancy-map)
  (define-key c-mp "o" empire-shell-fancy-map)


  (define-key c-mp "\C-m" 'map-empire)
  (define-key c-mp "m" 'empire-switch-to-map)

  (define-key c-mp "a" 'map-air-umbrella)

  (define-key c-mp "\C-k" 'check-empire)
  (define-key c-mp "k" 'check-empire)

  (define-key c-mp "\C-f" 'empire-flows)
  (define-key c-mp "f" 'empire-flow)


  (define-key c-mp "i" 'empire-insert-map-sector)
  (define-key c-mp "\C-i" 'empire-insert-map-sector)

  (define-key c-mp "w" 'empire-show-previous-where)
  (define-key c-mp "\C-w" 'empire-show-previous-where)

  (define-key c-mp "p" 'empire-path-to-map)

  (define-key c-mp "P" empire-plane-commands-map)
  (define-key c-mp "S" empire-ship-commands-map)

  (define-key c-mp "=" empire-sector-map)
  (define-key c-mp "t" empire-telegram-map)
  (define-key c-mp "\C-u" empire-survey-map)

  (define-key c-mp "s" 'save-empire)
  (define-key c-mp "\C-r" 'restore-empire)
  (define-key c-mp "r" 'restore-empire)


  ;; help
  (define-key c-mp "?"	  'help-for-empire-special-map)
  (define-key c-mp "\C-h" 'help-for-empire-special-map)
  (define-key c-mp help-character 'help-for-empire-special-map)

  (setq empire-shell-special-map c-mp)
  )

(make-help-screen help-for-empire-special-map
		  "u C-u d x c TAB C-s  m C-m a k f C-f  i w p = P S s r ?"
		  "You have discovered the GEET shell special commands
   From here, you can use the following options:

u	Invoke one of the USER customization options
C-u	Invoke one fo the su
d, C-d	Invoke one of the REDISTRIBUTION options
x, C-x	Invoke one of the EXTRACTION options
o, C-o	Invoke an OTHER strange command
=	Invoke a sector designation/edit command
t	Invoke a telegram handling command

TAB,
c,C-c	Run an interactive command with the client

C-s	SHELL commands. Type \\[empire-shell-shell-map] ? for details.

m	Move to the empire map. With prefix arg, goto last named sector
C-m	Map your empire
a	Map the Air umbrella previously calculated. With arg, calcs umbrella
k, C-k	Check empire 
f	Do commodity flow analysis for 1 sector
C-f	Do commodity flow analysis for whole country

i, C-i	Insert the coordinates of the map's current sector
w, C-w	Show current or last sector reference on the map
p	Insert low cost path from last sector to current sector in the map
	  If arg given, prompt for sector to which to calc path
P	Do a plane command
S	Do a ship command

s	Save empire sector data
r, C-r	Restore empire sector data

Please use \\[describe-key] to find out more about any of these keys.
	   \\[describe-mouse-bindings] will explain the mousemap.

\\[empire-new-world] initializes emacs for a new empire.
"
		  empire-shell-special-map)


;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; The main shell map
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defvar empire-shell-mode-map nil)

(if nil ;;empire-shell-mode-map
    nil
  (setq empire-shell-mode-map (copy-keymap shell-mode-map))

  (define-key empire-shell-mode-map "\t"
    'map-execute-prompted-command)

  (define-key empire-shell-mode-map empire-shell-special-key empire-shell-special-map)
  )

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Map mode
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defvar empire-map-mode-map nil "")

(let ((c-mp (make-keymap)))
  (suppress-keymap c-mp t)

  (define-key c-mp "j" 'move-in-empire-map-j)
  (define-key c-mp "g" 'move-in-empire-map-g)
  (define-key c-mp "y" 'move-in-empire-map-y)
  (define-key c-mp "u" 'move-in-empire-map-u)
  (define-key c-mp "n" 'move-in-empire-map-n)
  (define-key c-mp "b" 'move-in-empire-map-b)

  ;;
  (define-key c-mp "J" 'move-in-empire-map-j-and-insert)
  (define-key c-mp "G" 'move-in-empire-map-g-and-insert)
  (define-key c-mp "Y" 'move-in-empire-map-y-and-insert)
  (define-key c-mp "U" 'move-in-empire-map-u-and-insert)
  (define-key c-mp "N" 'move-in-empire-map-n-and-insert)
  (define-key c-mp "B" 'move-in-empire-map-b-and-insert)
  (define-key c-mp "H" 'map-insert-h)

  (define-key c-mp "i" 'select-this-sector)
  (define-key c-mp "p" 'empire-move-to-map)
  (define-key c-mp "s" 'move-to-in-empire-map)
  (define-key c-mp "e" 'empire-switch-to-shell)
  (define-key c-mp ">" 'scroll-left-half-page)
  (define-key c-mp "<" 'scroll-right-half-page)

  (define-key c-mp "C" 'empire-census-sector)
  (define-key c-mp "D" 'empire-distribute-sector)
  (define-key c-mp "d" 'empire-des-sector)
  (define-key c-mp "E" 'empire-prompting-explore)
  (define-key c-mp "L" 'empire-level-sector)
  (define-key c-mp "M" 'empire-prompting-move)
  (define-key c-mp "O" 'empire-commodity-sector)
  (define-key c-mp "P" 'empire-path-sector)
  (define-key c-mp "R" 'empire-resource-sector)
  (define-key c-mp "T" 'empire-prompting-test)

  (define-key c-mp "." 'display-map-sector-info)

  (define-key c-mp "\C-b" 'move-in-empire-map-g)
  (define-key c-mp "\C-f" 'move-in-empire-map-j)
  (define-key c-mp "\C-p" 'move-in-empire-map-u)
  (define-key c-mp "\C-n" 'move-in-empire-map-b)

  (define-key c-mp "=" empire-sector-map)

  ;;
  ;; If the help-char is not C-h, assume that C-h is bound to
  ;; `backward-delete-char' or `backward-delete-char-untabify'.  In this case,
  ;; Remap C-h so that it won't delete any characters from the map buffer.
  ;;
  (if (not (equal help-char ?\C-h))
    (define-key c-mp "\C-h" 'map-unmove)
  )
  ;; Insure that DEL doesn't delete any characters from the map buffer.
  (define-key c-mp "\C-?" 'map-unmove)

  (define-key c-mp "\C-M" 'map-execute-command)
  (define-key c-mp "\t"
    'map-execute-prompted-command)
  (define-key c-mp "\C-c\t"
    'map-execute-prompted-command)
  (define-key c-mp "\C-cc"
    'map-execute-prompted-command)
  (define-key c-mp "\C-c\C-c"
    'map-execute-prompted-command)

  ;; shell mode compatibility -- these bindings whould be the same as under
  ;; \C-c in the shell except that some are missing
  (define-key c-mp "\C-c\C-d" empire-shell-redistribute-map)
  (define-key c-mp "\C-cd" empire-shell-redistribute-map)

  (define-key c-mp "\C-cu" empire-shell-customization-map)

  (define-key c-mp "\C-c\C-o" empire-shell-fancy-map)
  (define-key c-mp "\C-co" empire-shell-fancy-map)

  (define-key c-mp "\C-c\C-m" 'map-empire) 
  (define-key c-mp "\C-cp" 'empire-move-to-map)
  (define-key c-mp "\C-cP" empire-plane-commands-map)
  (define-key c-mp "\C-cS" empire-ship-commands-map)
  (define-key c-mp "\C-c\C-u" empire-survey-map)


  (define-key c-mp "\C-c\C-i" 'select-this-sector)
  (define-key c-mp "\C-c\C-f" 'empire-flows)
  (define-key c-mp "\C-cf" 'empire-flow)
  (define-key c-mp "\C-ca" 'map-air-umbrella)
  (define-key c-mp "\C-c\C-k" 'check-empire)
  (define-key c-mp "\C-ck" 'check-empire)
  (define-key c-mp "\C-cs" 'save-empire)
  (define-key c-mp "\C-cr" 'restore-empire)
  (define-key c-mp "\C-ci" 'select-this-sector)
  (define-key c-mp "\C-ce" 'empire-switch-to-shell)
  (define-key c-mp "\C-c>" 'scroll-left-half-page)
  (define-key c-mp "\C-c<" 'scroll-right-half-page)

  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
  ;;
  ;; Note that these functions are autoloaded.
  ;;
  (define-key c-mp "Q" 'dynamic-mobility-show-mode)
  (define-key c-mp "@" 'dynamic-mobility-show-used-left)
  (define-key c-mp "#" 'dynamic-mobility-show-amount-can-move)
  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

  (define-key c-mp "\C-c?" 'help-for-empire-map-mode)
  (define-key c-mp "?" 'help-for-empire-map-mode)
  (define-key c-mp " " 'empire-toggle-global-desc-fcn)

  (setq empire-map-mode-map c-mp)
  )
  

(make-help-screen help-for-empire-map-mode
		  "jgyunb JGYUNB H . i C D d L P R M E T p s = e Q@# C-cc < > ?"
		  "You have discovered the empire map mode special commands
   From here, you can use the following options:

<	Scroll left.
>	Scroll right.

jgyubn	Moves point the same as empire moves it. Shows sector data.
JGYUBN	Moves point the same as empire moves it, and the corresponding
	  lowercase letter is inserted into the empire shell.
H	Insert `h' into the empire shell.
\\[map-unmove]
	\"Undo\" the last inserted move.
.	Redisplay the current sector info in the minibuffer.
	If given universal arg(\C-u), toggles to another display mode
[Return] Execute the command in the empire shell (acts just as if you pressed
	`[Return]' in the shell buffer).

i	Insert the location of the current sector into the empire shell.
C	Do a `census' on the current sector and update the database.
D	Do a `distribute' on the current sector.
d	do a `des' on the current sector
L	Do a `level' on the current sector.
P	Do a `path' on the current sector.
R	Do a `resource' on the current sector and display the information.
M E T	Interactive `move', `explore', and `test', respectively.  Each command
	will prompt for a commodity and an amount, and will insert into the
	shell buffer a command in the form, `COM T X,Y A ', where `COM' is the
	command, `T' is the commodity to move or explore with, `X,Y' is the
	coordinate of the current sector, and `A' is the amount.

p	Insert in shell path from last sector referenced in shell to
	  current sector.
s	Jump to a given sector.
=	Designate the sector fancy command map -- there are subcommands
	  under here
e	Show the 'e'mpire shell, select it.

Q	Enter move and show mobility mode, \C-g will exit the mode
@	Enter move and show mode showing amount of mob used/left
#	Enter move and show mobility mode, showing movable amount

<>	Scroll window in indicated direction

C-c	most commands under here are the same as in the shell mode.

Please use \\[describe-key] to find out more about any of these keys.
	   \\[describe-mouse-bindings] will explain the mousemap.
"
		  empire-map-mode-map)

(defun empire-map-mode () "Major mode for displaying empire maps.  Shows
cen, production, and commodity data on each selected sector. Allows
'testing' of paths as jgyubn keys move as they do in empire. Permit auto
route calculation.

\\[help-for-empire-map-mode] gives better help.
\\[describe-mouse-bindings] describes the mouse.

Special commands:
\\{empire-map-mode-map}
"
  (kill-all-local-variables)
  (buffer-flush-undo (current-buffer))
  (use-local-map empire-map-mode-map)
  (setq mode-name "Empire Map")
  (setq major-mode 'empire-map-mode)
  ;; turn off line truncation for worlds > screen size
  (setq truncate-lines t)
  ;; Prepare to use overlay arrow
  (setq overlay-arrow-string "    =>")
  (or overlay-arrow-position
      (setq overlay-arrow-position (make-marker)))
  (if (not (mark-marker)) (set-mark (point)))
  (if empire-map-local-mousemap
      (progn
	(make-local-variable 'current-global-mousemap)
	(setq current-global-mousemap empire-current-global-mousemap)
	(use-local-mousemap empire-map-local-mousemap)))
  )
;; Empire-map mode is suitable only for specially formatted data.
(put 'empire-map-mode 'mode-class 'special)

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Empire data Mode
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defvar empire-data-special-mode-map nil "") ; the C-c map

(let ((c-mp (make-sparse-keymap)))

  (define-key c-mp "." 'empire-display-this-sector)
  (define-key c-mp "x" 'empire-send-current-command)
  (define-key c-mp "\C-x" 'empire-send-current-command)

  (define-key c-mp "w" 'empire-show-previous-where)
  (define-key c-mp "\C-w" 'empire-show-previous-where)

  (define-key c-mp "\C-m" 'map-empire)
  (define-key c-mp "m" 'empire-switch-to-map)

  (define-key c-mp "\C-m" 'map-empire)
  (define-key c-mp "m" 'empire-switch-to-map)

  (define-key c-mp "\C-ce" 'empire-switch-to-shell)
  (define-key c-mp "e" 'empire-switch-to-shell)

  (define-key c-mp "\t"
    'map-execute-prompted-command)
  (define-key c-mp "c"
    'map-execute-prompted-command)
  (define-key c-mp "\C-c"
    'map-execute-prompted-command)

  (define-key c-mp "?" 'help-for-empire-data-special-mode)

  (setq empire-data-special-mode-map c-mp)
  )

(make-help-screen help-for-empire-data-special-mode
		  ". TAB m w x e"
		  "You have discovered the empire data mode special commands
   From here, you can use the following options:

.	Show the current sector in the map, display its info
TAB,
c,C-c	Run an interactive command
m	Move to the empire map. With prefix arg, goto last named sector
w, C-w	Show current or last sector reference on the map
x	send the current line to empire as a command

Please use \\[describe-key] to find out more about any of these keys.
	   \\[describe-mouse-bindings] will explain the mousemap.
"
		  empire-data-special-mode-map)

(defvar empire-data-mode-map nil "") ; the map in data buffers
(let ((c-mp (make-keymap)))
  (suppress-keymap c-mp t)

  ;;
  ;; If the help-char is not C-h, assume that C-h is bound to
  ;; `backward-delete-char' or `backward-delete-char-untabify'.  In this case,
  ;; Remap C-h so that it won't delete any characters.
  ;;
  (if (not (equal help-char ?\C-h))
    (define-key c-mp "\C-h" 'undefined)
  )
  ;; Insure that DEL doesn't delete any characters.
  (define-key c-mp "\C-?" 'undefined)
  (define-key c-mp "\C-m" 'undefined)

  (define-key c-mp "\t"
    'map-execute-prompted-command)

  (define-key c-mp empire-shell-special-key
    empire-data-special-mode-map)

  
  (setq empire-data-mode-map c-mp)
  )

(defun empire-data-mode (editablep executablep)
  "Major mode for displaying empire analysis results.
Is mostly text mode so user can edit commands but has a set of special keys.

\\[help-for-empire-data-special-mode] gives help for the special stuff.
\\[describe-mouse-bindings] describes the mouse.

Special commands:
\\{empire-data-mode-map}
"
  (text-mode)
  (use-local-map empire-data-mode-map)
  (setq mode-name "Empire Data")
  (setq major-mode 'empire-data-mode)
  (set-window-hscroll (selected-window) 0)
  (make-local-variable 'empire-editablep)
  (setq empire-editablep editablep)
  (make-local-variable 'empire-executablep)
  (setq empire-editablep executablep)
  (if (not editablep)
      (setq buffer-read-only t))
  
  ;; Only set mouse if we are capable
  (if empire-display-local-mousemap
      (progn
	(make-local-variable 'current-global-mousemap)
	(setq current-global-mousemap empire-current-global-mousemap)
	(use-local-mousemap empire-display-local-mousemap)))
  )

;; Empire-map mode is suitable only for specially formatted data.
(put 'empire-data-mode 'mode-class 'special)

(defun empire-data-mode-buffer (editablep executablep &optional buffer)
  (if (not buffer)
      (empire-data-mode editablep executablep)
    (let ((cb (current-buffer)))
      (set-buffer buffer)
      (empire-data-mode editablep executablep)
      (set-buffer cb))))


;; import-to-sector
