#include <stdio.h>
#include "sect.h"
#include "util.h"
#include "empire.h"

/*
 * This program generates dist commands for setting up warehouse
 * distributions in empire
 * The program takes it's input in the form of a dump file
 * (default is .dump) 
 * Dist paths are set up for every sector in the dump file
 * for a minimum cost/shortest path to a warehouse.
 * the warehouse is automatically chosen
 * 
 * NOTE: paths must be no more than 10 characters, so autodist may
 * not be able to find paths for all sectors
 *
 */

/*
   This version of autodist was written by C. Chase based on inspiration
   (and an earlier version) by Cowboy Bob
*/

FILE *outfile, *fopen();

#define MAX_PATH_LENGTH 80

struct tpathstr {
  char path[MAX_PATH_LENGTH];
  int cost;
  int length;
  sctlst_t  *sct;
} *t_vec; /* an array of the paths to move everywhere */

typedef struct warehouse {
  int x;
  int y;
  sctlst_t * p;
} warehouse;
#define MAXWARES 20

int maxpath = 10;
int maxcost = 10000; /* useful for avoiding dist paths to mountains */


#define FALSE 0
#define TRUE 1


/* country is the complete list of the whole country */
sctlst_t  *country, *read_dump();
int c_size; /* number of sectors in the country */


main(argc,argv)
int argc;
char **argv;
{
  char output_file[80];
  char dump_file[80];
  sctlst_t * p;
  int i;
  warehouse wlist[MAXWARES];
  int warehs; /* number of warehouses in dump file */
  int w;

  strcpy(output_file,"-");
  strcpy(dump_file,".dump");

  /* read in args */
  while(argc > 1) {
    if (argv[1][0] == '-') {
      switch(argv[1][1]) {
      case 'o': /* output file arg */
        argc--; argv++;
        strcpy(output_file,argv[1]);
        break;
      case 'd': /* dump file arg */
        argc--; argv++;
        strcpy(dump_file,argv[1]);
        break;
      default:
        fprintf(stderr,
                "USAGE: autodist -w x,y [-o outfile ] [-d dumpfile]\n");
        exit(1);
      }
      argc--;
      argv++;
    }
  }
  
  if (0 == strcmp(output_file,"-"))
    outfile = stdout;
  else
    if (NULL == (outfile = fopen(output_file,"w"))) {
      fprintf(stderr,"cannot open %s for output\n",output_file);
      exit(1);
    }

  country = read_dump(dump_file);
  init_country();

  /* search through the sector list to find all warehouses */
  warehs = 0;
  for (p = country; p; p = p->next) 
    if (p->data->sct_terr == 99) {
      wlist[warehs].x = p->data->sct_x;
      wlist[warehs].y = p->data->sct_y;
      wlist[warehs].p = p;
      warehs++;
    }


  /* initialize the tree */
  for (i = 0; i < c_size; i++) {
    t_vec[i].cost = -1;
    t_vec[i].length = -1;
    *(t_vec[i].path) = '\0';
  }

  /* construct the dist paths with make_tree */
  for (w = 0; w < warehs; w++) {
    make_tree(wlist[w].p);
  }
  
  /* print out all the dist paths */
  for (i = 0; i < c_size; i++)
    if (strcmp(t_vec[i].sct->data->dist_path, t_vec[i].path) &&
        strlen(t_vec[i].path) > 0)
      fprintf(outfile, "dist %d,%d %sh\n",
              t_vec[i].sct->data->sct_x,
              t_vec[i].sct->data->sct_y,
              t_vec[i].path);
  
  fclose(outfile);
}


init_country()
{
  int i;
  sctlst_t  *p;

  /* count the number of sectors */
  c_size = 0;
  for (p = country; p; p = p->next) 
    p->spec.d = c_size++;
  t_vec = NEWN(struct tpathstr, c_size);
  
  for (i = 0,p = country; i < c_size; i++, p = p->next)
    t_vec[i].sct = p;
    
#ifdef DEBUG
  fprintf(stderr,"%d sectors read from file\n",c_size);
#endif
}


/* set the t_vec array to hold paths to the root node from all sectors */
make_tree(root)
sctlst_t  *root; /* a point on the tree */
{
  int root_id;
  int s_id, d_id;
  int i;
  int thislength;
  int thiscost;
  sctlst_t  *todo; /* the list of sectors to visit */
  sctlst_t  *tail; /* the last sector in the todo list */
  sctlst_t  *tmp; /* used to clear the aux pointer in todo list */
  static char pathdir[7]="nbgyuj";
  char incpath[2]; /* incremental path */
  struct sctstr *sct;

  root_id = root->spec.d;

  sct = root->data;
  t_vec[root_id].cost = 0;
  t_vec[root_id].length = 0;
  *(t_vec[root_id].path) = '\0';

  /* make the root node the only thing on the tree */
  root->aux = NULL;
  todo = root;
  tail = root;
  
  /* build the tree */
  for (; todo != NULL; tmp = todo, todo = todo->aux, tmp->aux = NULL) {
    s_id = todo->spec.d;
    thislength = t_vec[s_id].length + 1;
    if (thislength <= maxpath) {
      for (i = 0; i < 6; i++) {
        if (todo->neighbor[i] != NULL) {
          d_id = todo->neighbor[i]->spec.d;
          sct = todo->neighbor[i]->data;
          thiscost = t_vec[s_id].cost + COST(sct);
          if (thiscost < maxcost) {
            if (t_vec[d_id].cost < 0) {
              /* add it to the tree and put it on the todo list */
              t_vec[d_id].cost = thiscost;
              t_vec[d_id].length = thislength;
              incpath[0] = pathdir[i];
              incpath[1] = '\0';
              strcpy(t_vec[d_id].path,incpath);
              strcat(t_vec[d_id].path,t_vec[s_id].path);
              /* add it to todo list if it's not already there */
              if (!(todo->neighbor[i]->aux)) {
                tail->aux = todo->neighbor[i];
                todo->neighbor[i]->aux = NULL;
                tail = todo->neighbor[i];
              }
            }
            else if (t_vec[d_id].cost > thiscost ||
                     (t_vec[d_id].cost == thiscost && 
                      t_vec[d_id].length > thislength)) {
              /* we've found a better path, make it so */
              t_vec[d_id].cost = thiscost;
              t_vec[d_id].length = thislength;
              incpath[0] = pathdir[i];
              incpath[1] = '\0';
              strcpy(t_vec[d_id].path,incpath);
              strcat(t_vec[d_id].path,t_vec[s_id].path);
              /* add it to todo list if it's not already there */
              if (!(todo->neighbor[i]->aux)) {
                tail->aux = todo->neighbor[i];
                todo->neighbor[i]->aux = NULL;
                tail = todo->neighbor[i];
              }
            }
          }
        }
      }
    }
  }
}

        
      
      

  
