/****************************************************************************
 * fillstruct.c
 * Author Julie Roskies
 * Copyright 1989, Pittsburgh Supercomputing Center, Carnegie Mellon University
 *
 * Permission use, copy, and modify this software and its documentation
 * without fee for personal use or use within your organization is hereby
 * granted, provided that the above copyright notice is preserved in all
 * copies and that that copyright and this permission notice appear in
 * supporting documentation.  Permission to redistribute this software to
 * other organizations or individuals is not granted;  that must be
 * negotiated with the PSC.  Neither the PSC nor Carnegie Mellon
 * University make any representations about the suitability of this
 * software for any purpose.  It is provided "as is" without express or
 * implied warranty.
 *****************************************************************************/

#include <stdio.h>	 
#include <string.h>
#include <ctype.h>
#include "struct_defs.h"
#include "fill_defs.h"
#include "err_defs.h"


/* Fillstruct fills up a model with information from MOVIEBYU data arrays */
Model *fillstruct (movie_data)
FILE *movie_data;
{
        Model *thismodel; /* thismodel is the model we will be filling */
	
	int part_num;	/* number of parts in model */
	long vertex_num; 	/* number of vertices in model */
	long facet_num;	/* number of facets in model */
 	long connect_num; /* number of entries in connectivity array */

	ger_debug("fillstruct:");

	/* Reads in Control variables from MOVIEBYU */
	fscanf (movie_data, "%d %ld %ld %ld", &part_num, &vertex_num, 
	 				      &facet_num, &connect_num);
	/* Set thismodel to an empty model */
	thismodel = create_model();

	/* Fill in part section of thismodel */
	fill_part(movie_data, thismodel, part_num);

	/* Fill in vertex section of thismodel */
	fill_vertex(movie_data, thismodel, vertex_num);

	/* Fill in facet section of thismodel */
	fill_facet(movie_data, thismodel, facet_num);

	/* Find dimensions of model, using bound_box */
	fill_boundary(thismodel, vertex_num);

	/* Return pointer to thismodel */
	return thismodel;
}

/* Fill in part section of model (reading from parts array of MOVIEBYU data */

void fill_part (movie_data, thismodel, part_num)
FILE *movie_data;
Model *thismodel;
int part_num;
{       
	int i;
	long min, max;
	Part *thispart;

	ger_debug("fill_part:");

	/* Memory allocation for the part_list */
	if ( !(thismodel->part_list= (Part *)malloc(part_num * sizeof(Part)) ) )
		ger_fatal (
		  "fill_part: unable to allocate %d bytes",
		  part_num * sizeof(Part) );
        
	/* Initializes thispart pointer to point to first part */
	thispart = thismodel->part_list;
	
	/* Fills up part_list, part by part */
	for (i = 0; i<part_num; i++) {
		fscanf (movie_data, "%ld %ld", &min, &max);
		thispart->min = min;
		thispart->max = max;
		thispart->color = (Color *)NULL;
		thispart++;
		++thismodel->part_count;
		 }
}	


/* 
Fill in vertex section of model (reading from coordinate array 
of MOVIEBYU data. 
*/

void fill_vertex (movie_data, thismodel, vertex_num)
FILE *movie_data;
Model *thismodel;
long vertex_num;
{
	int i;
	float x, y, z;
        Vertex *thisvertex;

	ger_debug("fill_vertex:");

	/* Memory allocation for the vertex_list */
	if ( !(thismodel->vertex_list= 
			(Vertex *)malloc(vertex_num * sizeof(Vertex)) ) )
		ger_fatal (
		  "fill_vertex: unable to allocate %d bytes",
		  vertex_num * sizeof(Vertex) );
	
	/* Initializes thisvertex pointer to point to first vertex */
	thisvertex = thismodel->vertex_list;

	/* Fills up vertex_list, vertex by vertex */
	for (i = 0; i<vertex_num; i++) {
		fscanf (movie_data, "%f %f %f", &x, &y, &z);
		thisvertex->x = x;
		thisvertex->y = y;
		thisvertex->z = z;
		thisvertex->color = (Color *)NULL;
		thisvertex->normal = (Vector *)NULL;
		thisvertex++;
		++thismodel->vertex_count;
		 }
}

/* 
Fill in facet section of model (reading from connectivity array
of MOVIEBYU data.
*/

void fill_facet (movie_data, thismodel, facet_num)
FILE *movie_data;
Model *thismodel;
long facet_num;
{
	int i;
	Vnode *thisnode;
	Vertex_ID thisvertex;
	Facet *thisfacet;
	char ok;	/* for error testing */
	ger_debug("fill_facet:");

	/* Memory allocation for the facet_list */
	if ( !(thismodel->facet_list= 
			(Facet *)malloc(facet_num * sizeof(Facet)) ) )
		ger_fatal (
	  	  "fill_facet: unable to allocate %d bytes",
		  facet_num * sizeof(Facet) );

	/* Initialize thisfacet pointer to point to first facet */
	thisfacet = thismodel->facet_list;

	/* Fills up facet_list, facet by facet */
	for (i = 0; i<facet_num; i++) {
		thisnode = thisfacet;

		while ( (ok = fscanf(movie_data, "%ld", &thisvertex)) != EOF
				&& thisvertex>0 )

			/* repeats until last vertex of facet is 
			   encountered, denoted by negative number */ 
			{
			thisnode->myvertex = thisvertex - 1;
			/* use thisvertex - 1 because in MOVIEBYU vertices
			are numbered starting at 1, not 0. */
			thisnode = thisnode->next = create_vnode();
			}
			
		/* Test to see if loop was exited by an error or because
		last vertex for node was reached */
                if (ok != 1)
			ger_fatal(
			 "fill_facet: error in reading data on facet %d of %d",
                         i-1, facet_num);

		/* Add last node of facet, making myvertex positive */
		thisnode->myvertex = -thisvertex - 1;
		
		thisfacet++; 	/* ready to input next facet */
		++thismodel->facet_count;
		}	
}


void fill_boundary (thismodel, vertex_num)
Model *thismodel;
long vertex_num;
{
	int i;
	float x_min, x_max, y_min, y_max, z_min, z_max;
        Vertex *thisvertex;

	ger_debug("fill_boundary:");

	/* Memory allocation for the bound_box */
	if ( !(thismodel->bound_box= 
			(Boundary *)malloc(sizeof(Boundary)) ) )
		ger_fatal (
		  "fill_boundary: unable to allocate %d bytes",
		  sizeof(Boundary) );
	
	/* Initializes thisvertex pointer to point to first vertex */
	thisvertex = thismodel->vertex_list;

	/* Initializes boundaries to coordinates of first vertex */
	x_min = x_max = thisvertex->x;
	y_min = y_max = thisvertex->y;
	z_min = z_max = thisvertex->z;
                                              
	/* Finds boundaries of model by comparing coordinates of each vertex */
	/* Starts with 2nd vertex, since boundaries were initialized to first */
	for (i = 1; i<vertex_num; i++) {
		thisvertex++;
		if (thisvertex->x > x_max)  
			x_max= thisvertex->x;
			else if (thisvertex->x < x_min)
				x_min= thisvertex->x;
		if (thisvertex->y > y_max)  
			y_max= thisvertex->y;
			else if (thisvertex->y < y_min)
				y_min= thisvertex->y;
		if (thisvertex->z > z_max)  
			z_max= thisvertex->z;
			else if (thisvertex->z < z_min)
				z_min= thisvertex->z;
		 }
	/* Set the bounds in the bound_box */
	thismodel->bound_box->x_min= x_min;
	thismodel->bound_box->x_max= x_max;
	thismodel->bound_box->y_min= y_min;
	thismodel->bound_box->y_max= y_max;
	thismodel->bound_box->z_min= z_min;
	thismodel->bound_box->z_max= z_max;
}
