/*
 * voxels.c
 *
 * Copyright (C) 1989, Craig E. Kolb
 *
 * This software may be freely copied, modified, and redistributed,
 * provided that this copyright notice is preserved on all copies.
 *
 * There is no warranty or other guarantee of fitness for this software,
 * it is provided solely .  Bug reports or fixes may be sent
 * to the author, who may or may not act on them as he desires.
 *
 * You may not include this software in a program or other software product
 * without supplying the source, or without informing the end-user that the
 * source is available for no extra charge.
 *
 * If you modify this software, you should include a notice giving the
 * name of the person performing the modification, the date of modification,
 * and the reason for such modification.
 *
 * $Id: voxels.c,v 1.4 1992/10/23 20:57:07 earhart Exp $
 *
 * $Log: voxels.c,v $
 * Revision 1.4  1992/10/23  20:57:07  earhart
 * Chris Nuuja's fix to add a default gridsize iff undefined.
 *
 * Revision 1.3  1991/12/17  20:44:22  nuuja
 * got rid of stupid debug message to stderr.  Improved error messages.
 *
 * Revision 1.2  1991/12/17  00:02:57  nuuja
 * Added Free_World routine to free up all available memory
 *
 * Revision 1.1  91/01/04  18:35:21  welling
 * Initial revision
 * 
 * Revision 3.0.1.3  90/04/04  19:05:23  craig
 * patch5: Changes to reflect that free() may be destructive.
 * 
 * Revision 3.0.1.2  90/02/12  13:19:50  craig
 * patch4: Added reporting of total number of primitives.
 * 
 * Revision 3.0.1.1  89/12/06  16:33:29  craig
 * patch2: Added calls to new error/warning routines.
 * 
 * Revision 3.0  89/10/27  02:06:09  craig
 * Baseline for first official release.
 * 
 */
#include <math.h>
#include <stdio.h>
#include "constants.h"
#include "typedefs.h"
#include "funcdefs.h"
extern ray_Object *MemRecord;
extern ray_ObjList *CurObj;
extern ray_SurfaceList *Surfaces;	/* Linked list of defined surfaces */

/*
 * Process World object, converting to a Grid or List.
 */
SetupWorld()
{
	extern ray_Object *World;
	extern int WorldXSize, WorldYSize, WorldZSize;

	if (World->type == GRID)
		list2grid(World, WorldXSize, WorldYSize, WorldZSize);
	else
		{
		MemRecord = (ray_Object *)malloc(sizeof(ray_Object));
		MemRecord->data = World->data;
		make_list(World);  
		}
}

/*
 * Add object to grid's unbounded list.
 */
make_unbounded(obj, grid)
ray_Object *obj;
ray_Grid *grid;
{
	ray_ObjList *tmp;

	tmp = (ray_ObjList *)Malloc(sizeof(ray_ObjList));

	tmp->data = obj;
	tmp->next = grid->unbounded;
	grid->unbounded = tmp;
}

/*
 * Place an object in a grid.
 */
engrid(obj, grid)
ray_Object *obj;
ray_Grid *grid;
{
	int x, y, z, low[3], high[3];
	ray_ObjList *ltmp;

	/*
	 * This routine should *never* be passed an unbounded object, but...
	 */
	if (pos2grid(grid, obj->bounds[LOW], low) == 0 ||
	    pos2grid(grid, obj->bounds[HIGH], high) == 0 ||
	    obj->bounds[LOW][X] > obj->bounds[HIGH][X]) {
		/*
		 * Object is partially on wholly outside of
		 * grid -- this should never happen, but just
		 * in case...
		 */
		make_unbounded(obj, grid);
		fprintf(stderr,"Strange, engrid got an unbounded object...\n");
		return;
	    }

	/*
	 * For each voxel that intersects the object's bounding
	 * box, add pointer to this object to voxel's linked list.
 	 */
	for (x = low[X]; x <= high[X]; x++) {
		for (y = low[Y]; y <= high[Y]; y++) {
			for (z = low[Z]; z <= high[Z]; z++) {
				ltmp = (ray_ObjList *)share_malloc(sizeof(ray_ObjList));
				ltmp->data = obj;
				ltmp->next = grid->cells[x][y][z];
				grid->cells[x][y][z] = ltmp;
			}
		}
	}
}

/*
 * Convert 3D point to index into grid's voxels.
 */
pos2grid(grid, pos, index)
ray_Grid *grid;
double pos[3];
int index[3];
{
	index[X] = (int)(x2voxel(grid, pos[0]));
	index[Y] = (int)(y2voxel(grid, pos[1]));
	index[Z] = (int)(z2voxel(grid, pos[2]));

	if (index[X] == grid->xsize)
		index[X] = grid->xsize -1;
	if (index[Y] == grid->ysize)
		index[Y] = grid->ysize -1;
	if (index[Z] == grid->zsize)
		index[Z] = grid->zsize -1;

	if (index[X] < 0 || index[X] >= grid->xsize ||
	    index[Y] < 0 || index[Y] >= grid->ysize ||
	    index[Z] < 0 || index[Z] >= grid->zsize)
		return 0;
	return 1;
}

/*
 * Convert a linked list of objects to a Grid.
 */
list2grid(obj, xsize, ysize, zsize)
ray_Object *obj;
int xsize, ysize, zsize;
{
	ray_Grid *grid;
	ray_Object *otmp;
	ray_ObjList *ltmp, *nltmp;
	int x, y, i;
	extern ray_ObjList *find_bounds();

	grid = (ray_Grid *)Malloc(sizeof(ray_Grid));

	/*
	 * Find bounding box of bounded objects and get list of
	 * unbounded objects.
	 */
	grid->unbounded = find_bounds((ray_ObjList **)&obj->data, obj->bounds);

	grid->xsize = xsize; grid->ysize = ysize; grid->zsize = zsize;

	for (i = 0; i < 3; i++) {
		obj->bounds[LOW][i] -= 2. * EPSILON;
		obj->bounds[HIGH][i] += 2. * EPSILON;
		grid->bounds[LOW][i] = obj->bounds[LOW][i];
		grid->bounds[HIGH][i] = obj->bounds[HIGH][i];
	}
	grid->voxsize[X] = (grid->bounds[HIGH][X]-grid->bounds[LOW][X])/xsize;
	grid->voxsize[Y] = (grid->bounds[HIGH][Y]-grid->bounds[LOW][Y])/ysize;
	grid->voxsize[Z] = (grid->bounds[HIGH][Z]-grid->bounds[LOW][Z])/zsize;

	/*
	 * Allocate voxels.
	 */
	grid->cells = (ray_ObjList ****)share_malloc(xsize * sizeof(ray_ObjList ***));
	for (x = 0; x < xsize; x++) {
		grid->cells[x] = (ray_ObjList ***)share_malloc(ysize*sizeof(ray_ObjList **));
		for (y = 0; y < ysize; y++)
			grid->cells[x][y] = (ray_ObjList **)share_calloc((unsigned)zsize,
							sizeof(ray_ObjList *));
	}

	/*
	 * obj->data now holds a linked list of bounded objects.
	 */
	for(ltmp = (ray_ObjList *)obj->data; ltmp != (ray_ObjList *)0; ltmp = nltmp) {
		otmp = ltmp->data;
		engrid(otmp, grid);
		nltmp = ltmp->next;
		/* 
		  Don't free world list.  We need this list of all objects
		  So we can free it all.  We can't walk the Grid to free due
		  to multiple instantiation (don't want to free the same
		  pointer twice)
		*/
		/* free((char *)ltmp); */
	}
	/* MemRecord is what we will walk when we free */
	fprintf(stderr,"Setting up MemRecord!\n");
	MemRecord = (ray_Object *)malloc(sizeof(ray_Object));
	MemRecord->data = obj->data;
	obj->type = GRID;
	obj->data = (char *)grid;
}

void free_object(obj)
ray_Object *obj;
{
       switch(obj->type)
	  {
	  case BOX: 
	    free((char *)((ray_Primitive *)obj->data)->objpnt.p_box);
	    break;
	  case SPHERE: 
	    free((char *)((ray_Primitive *)obj->data)->objpnt.p_sphere);
	    break;
	  case CYL: 
	    free((char *)((ray_Primitive *)obj->data)->objpnt.p_cylinder);
	    break;
	  case TRIANGLE:
	    free((char *)((ray_Primitive *)obj->data)->objpnt.p_triangle);
	    break;
	  case PHONGTRI:
	    free((char *)((ray_Primitive *)obj->data)->objpnt.p_triangle->vnorm);
	    free((char *)((ray_Primitive *)obj->data)->objpnt.p_triangle->b);
	    free((char *)((ray_Primitive *)obj->data)->objpnt.p_triangle);
	    break;
	  case POLY:
	    free((char *)((ray_Primitive *)obj->data)->objpnt.p_poly->points);
	    free((char *)((ray_Primitive *)obj->data)->objpnt.p_poly);
	    break;
          default:
	    fprintf(stderr,"Error in free_object: unknown object type %d!\n",
		     obj->type);
	    return;
	  }
	if(obj->trans)
		free((char *)(obj->trans));
	free((char *)obj->data);
	free((char *)obj);
}

int Free_World()
{
   ray_ObjList *ltmp,*nltmp;
   ray_Object *otmp;
   ray_SurfaceList *stmp,*nlsurf;
   ray_Grid *grid;
   int xsize,ysize,zsize,x,y,z;
   extern ray_Object *World, *new_object();

   if (World == (ray_Object *)0) 
      {
      fprintf(stderr,"Warning: Free_World call ignored: no world to free\n");
      return(0);
      }
   if (MemRecord == (ray_Object *)0) 
      {
      fprintf(stderr,"Warning: Free_World called before ren_setup. Ignoring\n");
      return(0);
      }

   /*   Free list of all primitives */
   for(ltmp = (ray_ObjList *)MemRecord->data; ltmp != (ray_ObjList *)0; ltmp = nltmp)
     {
	otmp = ltmp->data;
	free_object(otmp);
	nltmp = ltmp->next;
	free((char *)ltmp);
     }
   free((char *)MemRecord);
   MemRecord = (ray_Object *)0;

   
   /* Free all defined surfaces */
   for(stmp = Surfaces; stmp != (ray_SurfaceList *)0 ; stmp = nlsurf)
     {
     if (stmp->surf)
	{
	if (stmp->surf->name)
           free((char *)stmp->surf->name);
        free((char *)stmp->surf);
	}
     nlsurf = stmp->next;
     free((char *)stmp);
     } 
    Surfaces= (ray_SurfaceList *)0;
     
   /*   
	Free grid structure if worls was a grid.  Ignore if world is a list
        Warning.  We currently don't free the grid's unbounded list.  This
        assumes all object are in the grid.  Why wouldn't they be?
   */
   if (World->type == GRID)
      {
      grid = (ray_Grid *) World->data;
      if (grid->unbounded)
	   fprintf(stderr,"Warning,there is an unbounded list! Not freeing it\n");
      xsize= grid->xsize;
      ysize= grid->ysize;
      zsize= grid->zsize;
      for (x = 0; x < xsize; x++) 
         {
         for (y = 0; y < ysize; y++)
	    {
	    for (z = 0; z < zsize; z++)
	       {
	       for( ltmp = grid->cells[x][y][z]; ltmp; ltmp = nltmp )
	          {
	          nltmp = ltmp->next;
	          free((char *)ltmp);
	          }
	       }
	    free((char *)grid->cells[x][y]);
	    }
         free((char *)grid->cells[x]);
         }
      free((char *)grid->cells);
      free((char *)grid);
      }

   /* free object */
   if (World->trans)
   	free((char *)World->trans);
   free((char *)World);
   World = new_object("world", LIST, (char *)NULL, (ray_Trans *)NULL);
   World->trans = NULL;
   /* Do we need to take care of CurObj? */
   return(1);
}

#ifdef MULTIMAX

char *
share_calloc(num, siz)
int num;
unsigned int siz;
{
	char *res;

	res = share_malloc(num*siz);
	bzero(res, num*siz);
	return res;
}
#endif
