/*ScianSockets.c
  Eric Pepke
  April 25, 1990
  Stuff for socket interface in Scian

  Taken over by John R. Murray
  June 29, 1992
*/

#include "Scian.h"
#include "ScianTypes.h"
#include "ScianSockets.h"
#include "ScianArrays.h"
#include "ScianLists.h"
#include "ScianControls.h"
#include "ScianButtons.h"
#include "ScianNetDaemon.h"
#include "ScianWindows.h"
#include "ScianObjWindows.h"
#include "ScianVisWindows.h"
#include "ScianTextBoxes.h"
#include "ScianDialogs.h"
#include "ScianIDs.h"
#include "ScianScripts.h"
#include "ScianErrors.h"
#include "ScianColors.h"
#include "ScianDialogs.h"
#include "ScianEvents.h"
#include "ScianTimers.h"
#include "ScianStyle.h"
#include "ScianDatasets.h"
#include "ScianDraw.h"
#include "ScianIcons.h"
#include "ScianObjFunctions.h"
#include "ScianGarbageMan.h"
#include "ScianNetObjects.h"
#include "ScianHelp.h"

/* temporary? */
#include <errno.h>

/*Line for mkmk*/
/*$cc -I/usr/include/bsd ScianSockets.c -c $(CFLAGS)
*/

/*Public objects*/
ObjPtr advertiseableClass = NULLOBJ;
ObjPtr allAdvertised = NULLOBJ;	/*All objects advertised from
					  this process (list)*/

/* this should probably move to Scian.h and ScianMain.c */
int libRunning = 0;

#ifdef SOCKETS

#define SHORT_TIMEOUT	5.0		/*Short timeout on socket*/
#define LONG_TIMEOUT	15.0		/*Long timeout*/	

/*Kinds of connection*/
#define CK_REMOTE_DAEMON	1	/*Connection to a remote network daemon*/
#define CK_LOCAL_DAEMON		2	/*Connection to a local daemon*/
#define CK_LISTENING_SCIAN	3	/*Listening connection on a local SciAn*/
#define CK_REMOTE_SCIAN		4	/*Connection to a remote SciAn (not implemented)*/
#define CK_LOCAL_SCIAN		5	/*Connection from a local SciAn (not implemented)*/

/*Values for connection state*/
#define CS_NAME_ONLY		0	/*Just have the name of target, nothing else*/
#define CS_WAIT_FOR_VERSION	1	/*Waiting for a version reply*/
#define CS_MORIBUND		2	/*Moribund connection, ready to die*/
#define CS_LIMBO		3	/*Connection in limbo, doing nothing*/
#define CS_IDLING		4	/*Idling a connection.*/
#define CS_FILL_CMD		5	/*Fill a command*/
#define CS_WAIT_FOR_CMD_END	6	/*Wait for the end of a command*/
#define CS_READ_PROC_NAME	7	/*Read a process name*/
#define CS_NOT_LISTENING	8	/*Accepting socket, not listening yet*/
#define CS_LISTENING		9	/*Accepting socket, listening*/
#define CS_WAIT_FOR_PASSWORD	10	/*Of a recieving socket, wait for a password*/
#define CS_READ_OBJECT		11	/*Read an Object */
#define CS_READ_NUMBER_ONE	12	/*Read a number*/
#define CS_READ_NUMBER_TWO	13	/*Read the second number*/
#define CS_READ_NUMBER_THREE	14	/*Read the third number*/

/*Local variables*/
static char *stateName[] = 
	{
	    "name only",
	    "waiting for version",
	    "moribund",
	    "limbo",
	    "idle connection",
	    "fill cmd",
	    "wait for cmd end",
	    "read proc name",
	    "not listening",
	    "listening",
	    "wait for password",
	    "read object",
	    "read number one",
	    "read number two",
	    "read number three"
	};
static char *lastComputer = 0;		/*Address of last computer to be dialed*/
static int lastSocketTried = ND_BASE_SOCKET_NUMBER;
					/*Last process # tried*/
static ObjPtr connectionClass;		/*Class of connections*/
static ObjPtr processClass;		/*Class of processes*/
static ObjPtr allConnections;		/*List of all the connections*/
static ObjPtr processIcon;			/*Icon that represents the process*/
static ObjPtr publicationConnection;	/*Connection for publications*/

/*Local prototypes*/
#ifdef PROTO
static ObjPtr NewConnection(char *, int);
static void SetConnectionState(ObjPtr connection, int state);
static int GetConnectionState(ObjPtr connection);
static void DeleteProcess(ObjPtr connection, int procNum);
static ObjPtr NewProcess(ObjPtr connection, char *procName, int procSocket);
static int SocketPort(void);
#else
static ObjPtr NewConnection();
static void SetConnectionState();
static int GetConnectionState();
static void DeleteProcess();
static ObjPtr NewProcess();
static int SocketPort();
#endif

/*Local methods*/
static ObjPtr ConnectToComputerAgain();
static ObjPtr ContinueWaiting();
static ObjPtr ShowConnectionControls();
static ObjPtr ConnectToProcess();

int writen(fd, cptr, nbytes)
int fd;
char *cptr;
long nbytes;
{
    int nleft, nwritten;

    nleft = nbytes;
    while (nleft > 0)
    {
	nwritten = write(fd, cptr, nleft);
	if (nwritten < 0)
	{
	    if (errno != EWOULDBLOCK && errno != EAGAIN)
	    {
		perror("writen");
		return;
	    }
	}
	else
	{
	    nleft -= nwritten;
	    cptr += nwritten;
	}
    }
}

int readn(sock, cptr, num)
int sock;
char *cptr;
long num;
{
    double lastData;
    long oldnum;
    int olderrno = 0;
    int notTimeOut = 1;

    oldnum = num;
    lastData = WallClock();
    while(num > 0 && (notTimeOut = lastData + JOHNSTIMEOUT > WallClock()))
    {
	long tmp;

	tmp = read(sock, cptr, num);
	if (tmp > 0)
	{
	    num -= tmp;
	    cptr += tmp;
	    lastData = WallClock();
	}
	if (tmp < 0 && olderrno != errno)
	{
	    if (errno != EWOULDBLOCK && errno != EAGAIN)
	    {
fprintf(stderr, "HEY! read returned %d\n", tmp);
		perror("readn");
	    }
	    olderrno = errno;
	}
    }
    if (!notTimeOut)
	fprintf(stderr, "readn timed out\n");

    return 0;
}

#ifdef PROTO
static int SocketPort(void)
#else
static int SocketPort()
#endif
/*Return a socket port which may or may not work*/
{
    return lastSocketTried++;
}

static ObjPtr ConnectToComputerAgain(dummy, whichButton)
ObjPtr dummy;
int whichButton;
/*Asks for a connect again depending on whichButton*/
{
    if (whichButton == 1) DoConnectToComputer();
    return ObjTrue;
}

static ObjPtr ContinueWaiting(connection, whichButton)
ObjPtr connection;
int whichButton;
/*Has a connection continue waiting or not*/
{
    if (whichButton == 1)
    {
	/*Continue trying*/

	/*Set the state*/
	SetVar(connection, CONNECTIONSTATE, 
		GetVar(connection, OLDCONNSTATE));

	/*Set a timeout*/
	SetTimeout(connection, TIMEOUT, WallClock() + LONG_TIMEOUT);
    }
    else
    {
	/*Set the state*/
	SetConnectionState(connection, CS_MORIBUND);
    }
    return ObjTrue;
}

#ifdef PROTO
static void SetConnectionState(ObjPtr connection, int state)
#else
static void SetConnectionState(connection, state)
ObjPtr connection;
int state;
#endif
/*Changes the state of connection to connection*/
{
    ObjPtr oldState;

    /*Get the old state*/
    oldState = GetVar(connection, CONNECTIONSTATE);

    /*If they're not the same, save the old one*/
    if (!oldState || (GetInt(oldState) != state))
    {
	SetVar(connection, OLDCONNSTATE, oldState);
#ifdef DEBUG
	printf("Scian (%ld) connection %ld transition to %s\n",
	    getpid(), connection, stateName[state]);
#endif
    }

    /*Only some states have timeouts, disable them now*/
    DoNotDisturb(connection, TIMEOUT);

    SetVar(connection, CONNECTIONSTATE, NewInt(state));

    /*Determine actions on the transition, such as doing a timeout*/
    switch (state)
    {
	case CS_NAME_ONLY:
	    /*Set a timeout*/
	    SetTimeout(connection, TIMEOUT, WallClock() + SHORT_TIMEOUT);
	    break;
	case CS_READ_NUMBER_ONE:
	    /*Must null the socket num first*/
	    SetVar(connection, READNUMBERONE, NULLOBJ);
	    break;
	case CS_READ_NUMBER_TWO:
	    /*Must null the socket num first*/
	    SetVar(connection, READNUMBERTWO, NULLOBJ);
	    break;
	case CS_READ_NUMBER_THREE:
	    /*Must null the socket num first*/
	    SetVar(connection, READNUMBERTHREE, NULLOBJ);
	    break;
	case CS_READ_PROC_NAME:
	    /*Must null the proc name first*/
	    SetVar(connection, CMDPROCNAME, NULLOBJ);
	    break;
	case CS_IDLING:
	    /*Null the command name*/
	    SetVar(connection, CURCMD, NULLOBJ);
	    break;
	case CS_WAIT_FOR_VERSION:
	    /*Null the command name*/
	    SetVar(connection, CURCMD, NULLOBJ);
	    SetTimeout(connection, TIMEOUT, WallClock() + SHORT_TIMEOUT);
	    break;
	case CS_WAIT_FOR_PASSWORD:
	    /*Null the command name*/
	    SetTimeout(connection, TIMEOUT, WallClock() + SHORT_TIMEOUT);
	    SetVar(connection, CURCMD, NULLOBJ);
	    break;
    }
}

#ifdef PROTO
static int GetConnectionState(ObjPtr connection)
#else
static int GetConnectionState(connection)
ObjPtr connection;
#endif
/*Returns the state of the connection*/
{
    int state;
    ObjPtr var;
    var = GetIntVar("GetConnectionState", connection, CONNECTIONSTATE);
    if (var)
    {
	state = GetInt(var);
    }
    else
    {
	state = CS_LIMBO;
    }
    return state;
}

static ObjPtr TimeoutConnection(connection, elapsedTime)
ObjPtr connection;
double elapsedTime;
/*Connection timeout, performing an action based on the state*/
{
#ifdef INTERACTIVE
    WinInfoPtr alertWindow;
    int state;
    ObjPtr var;
    char alertMessage[400];

    var = GetIntVar("TimeoutConnection", connection, CONNECTIONSTATE);
    state = GetInt(var);

    switch (state)
    {
	case CS_NAME_ONLY:
	    var = GetIntVar("TimeoutConnection", connection, CONNECTIONKIND);
	    sprintf(alertMessage, "The SciAn %s did not respond during the time out period.  You can cancel \
the connection or keep waiting.", GetInt(var) == CK_REMOTE_SCIAN ? "process" : "daemon");
	    alertWindow = AlertUser(UISEVEREALERT, (WinInfoPtr) connection, 
		alertMessage, ContinueWaiting, 
		2, "Cancel", "Keep Waiting");
		    if (alertWindow) SetVar((ObjPtr) alertWindow, HELPSTRING,
    NewString("Something responded to the socket connection on the remote host, \
but it did not identify itself as a SciAn process within the timeout period.  It \
is possible but unlikely that some other process is running in the daemon's place \
It is more likely that the process is just busy or slow, or there is a temporary \
slowdown in the network."));
	    SetConnectionState(connection, CS_LIMBO);
	    break;
	case CS_WAIT_FOR_VERSION:
	    var = GetIntVar("TimeoutConnection", connection, CONNECTIONKIND);
	    sprintf(alertMessage, "The SciAn %s did not respond during the time out period.  You can cancel \
the connection or keep waiting.", GetInt(var) == CK_REMOTE_SCIAN ? "process" : "daemon");
	    alertWindow = AlertUser(UISEVEREALERT, (WinInfoPtr) connection, 
		alertMessage, ContinueWaiting, 
		2, "Cancel", "Keep Waiting");
		    if (alertWindow) SetVar((ObjPtr) alertWindow, HELPSTRING,
    NewString("Something responded to the socket connection on the remote host, \
but it did not identify itself as a SciAn process within the timeout period.  It \
is possible but unlikely that some other process is running in the daemon's place \
It is more likely that the process is just busy or slow, or there is a temporary \
slowdown in the network."));
	    SetConnectionState(connection, CS_LIMBO);
	    break;
	case CS_WAIT_FOR_PASSWORD:
	    sprintf(alertMessage, "An incoming connection has not validated itself as a SciAn process during the time out period.  You can cancel \
the connection or keep waiting.");
	    alertWindow = AlertUser(UISEVEREALERT, (WinInfoPtr) connection, 
		alertMessage, ContinueWaiting, 
		2, "Cancel", "Keep Waiting");
		    if (alertWindow) SetVar((ObjPtr) alertWindow, HELPSTRING,
    NewString("Something attempted to connect to this process, but, \
but it did not identify itself as a SciAn process within the timeout period.  It \
is possible but unlikely that some other process is running in the daemon's place \
It is more likely that the process is just busy or slow, or there is a temporary \
slowdown in the network."));
	    SetConnectionState(connection, CS_LIMBO);
	    break;
	default:
	    fprintf(stderr, "timed out in funny state!\n");
    }

    return ObjTrue;
#endif
}

static void HostNameErrorAlert()
/*Creates a host name error*/
{
    WinInfoPtr alertWindow;
    alertWindow = AlertUser(UIERRORALERT, (WinInfoPtr) 0, 
	"A host by that name or address could not be found.", ConnectToComputerAgain, 
	2, "Cancel", "Re-enter");
    if (alertWindow) SetVar((ObjPtr) alertWindow, HELPSTRING,
	NewString("A valid host could not be found for that name or address.  \
This may be it is not in the name server or the name server is not available.  You can check the \
address to make sure that it is valid and try again."));
}

#ifdef PROTO
static void NoDaemonAlert(ObjPtr connection)
#else
static void NoDaemonAlert(connection)
ObjPtr connection;
#endif
/*Creates a "no daemon" error*/
{
    WinInfoPtr alertWindow;
    ObjPtr var;
    char *message = "";

    var = GetIntVar("NoDaemonAlert", connection, CONNECTIONKIND);
    if (var)
    {
	switch(GetInt(var))
	{
	    case CK_REMOTE_DAEMON:
		message = "There is no SciAn network daemon running on that host.";
		break;
	    case CK_LOCAL_DAEMON:
		message = "There is no SciAn network daemon running on your local computer.";
		break;
	}
    }

    alertWindow = AlertUser(UISEVEREALERT, (WinInfoPtr) 0, 
	message, ConnectToComputerAgain, 
	2, "Cancel", "Re-enter");
    if (alertWindow) SetVar((ObjPtr) alertWindow, HELPSTRING,
	NewString("In order for you to be able to connect \
to SciAn processes running on a computer, the SciAn network daemon must be running \
on the computer.  The SciAn network daemon can by started by entering 'snd &' on \
the command line of the host computer."));
}

static ObjPtr KillConnection(connection)
ObjPtr connection;
/*Kills a connection*/
{
    DeleteFromList(allConnections, connection);
    return ObjTrue;
}

void FindCommand(connection)
ObjPtr connection;
/*Finds a command corresponding to CURCMD in connection and changes the
connection's state appropriately to deal with it.*/
{
    ObjPtr command;
    command = GetVar(connection, CURCMD);
    if (command)
    {
	char *cmd;
	cmd = GetString(command);
	if (strlen(cmd) == 4)
	{
	    long cmdNum;

	    cmdNum = CMD(cmd);

	    if (cmdNum == CMD(ND_INFO_LIVE_PROCESS) ||
		cmdNum == CMD(ND_INFO_DEAD_PROCESS))
	    {
		/*Live or dead process*/
		SetConnectionState(connection, CS_READ_NUMBER_ONE);
	    }
	    else if (cmdNum == CMD(ND_CLOSE))
	    {
		SetConnectionState(connection, CS_MORIBUND);
	    }
	    /* This may be temporary. provides an exit for comp process */
	    else if (libRunning && cmdNum == CMD(ND_EXIT))
	    {
		SetConnectionState(connection, CS_MORIBUND);
		fprintf(stderr, "Got exit command, turning off library\n");
		libRunning = 0;
	    }
	    else if (cmdNum == CMD(ND_LIST_OBJECTS))
	    {
		ThingListPtr runner;

		runner = LISTOF(allAdvertised);
		while (runner)
		{
		    sprintf (tempStr, "%s %ld\n", ND_REG_OBJECT, GetInt(GetIntVar("FindCommand", runner -> thing, NETWORKID)));
		    SendSocketCommandLater(connection,tempStr);
		    SendObjectLater(connection, runner -> thing);
		    runner = runner -> next;
		}
		SetConnectionState(connection, CS_WAIT_FOR_CMD_END);
	    }
	    else if (cmdNum == CMD(ND_REG_OBJECT))
	    {
		SetConnectionState(connection, CS_READ_NUMBER_ONE);
	    }
	    else if (cmdNum == CMD(NO_MESG_OBJECT))
	    {
		SetConnectionState(connection, CS_READ_OBJECT);
	    }
	    else if (cmdNum == CMD(NO_MESG_SEND_OBJECT))
	    {
		SetConnectionState(connection, CS_READ_NUMBER_ONE);
	    }
	    else if (cmdNum == CMD(NO_MESG_GETVAR))
	    {
		SetConnectionState(connection, CS_READ_NUMBER_ONE);
	    }
	    else if (cmdNum == CMD(NO_MESG_GOTVAR))
	    {
		SetConnectionState(connection, CS_READ_NUMBER_ONE);
	    }
	    else if (cmdNum == CMD(NO_MESG_UPDATEVAR))
	    {
		SetConnectionState(connection, CS_READ_NUMBER_ONE);
	    }
	    else if (cmdNum == CMD(NO_MESG_MACHINE_IRIS4D))
	    {
#if MACHINE == IRIS4D
fprintf(stderr, "fast socket\n");
#endif
		SetConnectionState(connection, CS_WAIT_FOR_CMD_END);
	    }
	    else if (cmdNum == CMD(NO_MESG_MACHINE_RS6000))
	    {
#if MACHINE == RS6000
#endif
		SetConnectionState(connection, CS_WAIT_FOR_CMD_END);
	    }
	    else if (cmdNum == CMD(NO_MESG_MACHINE_UNKNOWN))
	    {
		SetConnectionState(connection, CS_WAIT_FOR_CMD_END);
	    }
	    else
	    {
		sprintf(tempStr, "Bad socket command: %s", cmd);
		ReportError("FindCommand", tempStr);
		SetConnectionState(connection, CS_WAIT_FOR_CMD_END);
	    }
	}
	else
	{
	    SetConnectionState(connection, CS_WAIT_FOR_CMD_END);
	    ReportError("FindCommand", "Socket command wrong size");
	}
    }
    else
    {
	SetConnectionState(connection, CS_WAIT_FOR_CMD_END);
	ReportError("FindCommand", "No command specified");
    }
}

#ifdef PROTO
Bool ThisWouldBeAReallyBadThing(NameTyp var)
#else
Bool ThisWouldBeAReallyBadThing(var)
NameTyp var;
#endif
{
    switch(var)
    {
	case SELECTED:
	    return true;
	case NETWORKID:
	    return true;
	case OWNERCONNECTION:
	    return true;
	default:
	    return false;
    }
}

#ifdef PROTO
void SendGetVarMessage(ObjPtr obj, NameTyp var)
#else
void SendGetVarMessage(obj, var)
ObjPtr obj;
NameTyp var;
#endif
{
    ObjPtr connection;
    int netid;

    /* object had better be a Remote object */
    connection = GetVar(obj, OWNERCONNECTION);
    netid = GetInt(GetVar(obj, NETWORKID));
    if (connection && netid)
    {
	sprintf(tempStr, "%s %d %d\n", NO_MESG_GETVAR, netid, var);
	SendSocketCommandLater(connection, tempStr);
    }
    else
    {
	ReportError("SendGetVarMessage", "object has no OWNERCONNECTION or NETWORKID");
    }
}

#ifdef PROTO
void SendVarUpdateNotice(ObjPtr obj, NameTyp var)
#else
void SendVarUpdateNotice(obj, var)
ObjPtr obj;
NameTyp var;
#endif
{
    ThingListPtr runner;
    int netid;

    netid = GetInt(GetVar(obj, NETWORKID));
    if (!netid)
    {
fprintf(stderr, "SendVarUpdateNotice: obj has no NETWORKID\n");
	return;
    }
    runner = LISTOF(allConnections);
    while (runner)
    {
	/* any non-daemon connection */
	if (CK_LOCAL_SCIAN == GetInt(GetIntVar("AdvertiseObject", runner -> thing, CONNECTIONKIND)))
	{
	    sprintf(tempStr, "%s %ld %ld\n", NO_MESG_UPDATEVAR, netid, var);
	    SendSocketCommandLater(runner -> thing, tempStr);
	}
	runner = runner -> next;
    }
}

ObjPtr AdvertiseObject(object)
ObjPtr object;
/*Advertises an object*/
{
    ObjPtr networkID;
    int netid;
    ThingListPtr runner; /* runs down connection list */

    if (GetPredicate(object, ADVERTISED))
    {
	/* already advertised */
	return ObjFalse;
    }

    SetVar (object, ADVERTISED, ObjTrue);

/*
    if (!(networkID = GetVar(object, NETWORKID)))
    {
	networkID = NewInt(GetNetworkID());
	SetVar(object, NETWORKID, networkID);
    }
    netid = GetInt(GetIntVar("AdvertiseObject", object, NETWORKID));
*/
    netid = PublishObject(object);

    PrefixList (allAdvertised, object);

    runner = LISTOF(allConnections);
    while (runner)
    {
	/* any non-daemon connection */
	if (CK_LOCAL_SCIAN == GetInt(GetIntVar("AdvertiseObject", runner -> thing, CONNECTIONKIND)))
	{
	    sprintf(tempStr, "%s %ld\n", ND_REG_OBJECT, netid);
	    SendSocketCommandLater(runner -> thing, tempStr);
	    SendObjectLater(runner -> thing, object);
	}
	runner = runner -> next;
    }
    
    return ObjTrue;
}

ObjPtr UnadvertiseObject(object)
ObjPtr object;
/*Unadvertises an object*/
{
    ThingListPtr runner;
    if (!GetPredicate(object, ADVERTISED))
    {
	/* never advertised, or already unadvertised */
	return ObjFalse;
    }

    SetVar (object, ADVERTISED, ObjFalse);

    DeleteFromList(allAdvertised, object);

    runner = LISTOF(allConnections);
    while (runner)
    {
	if (CK_LOCAL_SCIAN == GetInt(GetIntVar("AdvertiseObject", runner -> thing, CONNECTIONKIND)))
	{
	    sprintf(tempStr, "%s %ld %ld\n", ND_UNREG_OBJECT, GetIntVar("AdvertiseObject", runner -> thing, NETWORKID));
	    SendSocketCommandLater(runner -> thing, tempStr);
	}
	runner = runner -> next;
    }

    return ObjTrue;
}

#ifdef PROTO
static ObjPtr NewProcess(ObjPtr connection, char *procName, int procSocket)
#else
static ObjPtr NewProcess(connection, procName, procSocket)
ObjPtr connection;
char *procName;
int procSocket;
#endif
/*Registers a process with the given name and num from connection*/
{
    WinInfoPtr dialogExists;
    ObjPtr allProcesses;
    ObjPtr process;

    /*Make a new process*/
    process = NewObject(processClass, 0);
    SetVar(process, NAME, NewString(procName));
    SetVar(process, SOCKETID, NewInt(procSocket));
    SetVar(process, OWNERCONNECTION, connection);
    SetVar(process, ALLOBJECTS, NewList());
    SetVar(process, ALLADVERTISED, NewList());

    /*Add to the list of all processes*/
    allProcesses = GetListVar("NewProcess", connection, ALLPROCESSES);
    if (allProcesses)
    {
	PrefixList(allProcesses, process);
    }

    dialogExists = DialogExists((WinInfoPtr) connection, NewString("Controls"));
    if (dialogExists)
    {
	ObjPtr icon, corral;

	/*Make an icon and put it in the corral*/
	corral = GetObjectVar("NewProcess", (ObjPtr) dialogExists, CORRAL);
	if (!corral) return NULLOBJ;

	icon = NewObject(processIcon, 0L);
	SetVar(icon, ICONLOC, NULLOBJ);

	SetVar(icon, NAME, NewString(procName));
	SetVar(icon, REPOBJ, process);
	SetVar(icon, CORRAL, corral);
	DropIconInCorral(corral, icon);
    }

    return process;
}

#ifdef PROTO
ObjPtr FindRemoteObject(ObjPtr process, int netid)
#else
ObjPtr FindRemoteObject(process, netid)
ObjPtr process;
int netid;
#endif
{
    ObjPtr allObjects;
    ThingListPtr runner;

    allObjects = GetVar(process, ALLOBJECTS);
    if (!allObjects || !IsList(allObjects))
    {
	allObjects = NewList();
	SetVar (process, ALLOBJECTS, allObjects);
    }

    runner = LISTOF(allObjects);
    while(runner)
    {
	if (netid == GetInt(GetVar(runner -> thing, NETWORKID)))
	{
	    break;
	}
	runner = runner -> next;
    }

    if (runner)
    {
	return runner -> thing;
    }
    else
    {
	return NULLOBJ;
    }
}

#ifdef PROTO
static void DeleteProcess(ObjPtr connection, int socketNum)
#else
static void DeleteProcess(connection, socketNum)
ObjPtr connection;
int socketNum;
#endif
/*Deletes a process with the given name and num from connection*/
{
    WinInfoPtr dialogExists;
    ObjPtr allProcesses;
    ObjPtr process = NULLOBJ;
    ThingListPtr runner;
    /*Find the process*/
    allProcesses = GetListVar("DeleteProcess", connection, ALLPROCESSES);
    if (allProcesses)
    {
	runner = LISTOF(allProcesses);
	while (runner)
	{
	    ObjPtr var;
	    var = GetVar(runner -> thing, SOCKETID);
	    if (var && socketNum == GetInt(var))
	    {
		process = runner -> thing;
	    }
	    runner = runner -> next;
	}
    }

    if (process)
    {
	DeleteFromList(allProcesses, process);

        dialogExists = DialogExists((WinInfoPtr) connection, NewString("Controls"));
	if (dialogExists)
	{
	    ObjPtr contents, corral, delIcon = NULLOBJ;

	    /*Delete the icon from the corral*/
	    corral = GetObjectVar("DeleteProcess", (ObjPtr) dialogExists, CORRAL);
	    if (!corral) return;

	    contents = GetListVar("DeleteProcess", corral, CONTENTS);
	    if (!contents) return;
	    runner = LISTOF(contents);
	    while (runner)
	    {
		if (GetVar(runner -> thing, REPOBJ) == process)
		{
		    delIcon = runner -> thing;
		}
		runner = runner -> next;
	    }
	    DeleteFromList(contents, delIcon);
	    ImInvalid(corral);
	}
    }
}

#ifdef PROTO
void UnRemoteObject(ObjPtr obj)
#else
void UnRemoteObject(obj)
ObjPtr obj;
#endif
{
    ObjPtr connection;
    if (!IsRemote(obj))
    {
	ReportError("UnRemoteObject", "not remote");
	return;
    }

    connection = GetVar(obj, OWNERCONNECTION);
    if (obj)
    {
	ObjPtr allObjects;

	allObjects = GetVar(connection, ALLOBJECTS);
	if (allObjects)
	{
	    DeleteFromList(allObjects, obj);
	    obj -> flags = obj -> flags & ~ISREMOTE;
fprintf(stderr, "unremoted object\n");
	}
    }
    else
    {
	ReportError("UnRemoteObject", "no OWNERCONNECTION");
	return;
    }
}

static ObjPtr NewRemoteObject(filledObj)
ObjPtr filledObj;
/*Method to register a new remote object and add its icon to the process window*/
{
    WinInfoPtr dialogExists;
    ObjPtr connection, process;

    connection = GetVar(filledObj, OWNERCONNECTION);
    if (!connection)
    {
	ReportError("NewRemoteObject", "Object has no connection");
	return ObjFalse;
    }

    process = GetVar(connection, OWNERPROCESS);
    if (!process)
    {
	ReportError("NewRemoteObject", "John lied");
	return ObjFalse;
    }

    dialogExists = DialogExists((WinInfoPtr) process, NewString("Controls"));
    if (dialogExists)
    {
	ObjPtr icon, corral, name;

	/*Make an icon and put it in the corral*/
	corral = GetObjectVar("IdleSockets", (ObjPtr) dialogExists, CORRAL);
	MakeVar(filledObj, DEFAULTICON);
	icon = GetVar(filledObj, DEFAULTICON);
	if (icon)
	{
	    icon = NewObject(icon, 0L);
	}
	else
	{
	    icon = NewIcon(ICONQUESTION, 0, 0, "?");
	}
        SetVar(icon, ICONLOC, NULLOBJ);
	MakeVar(filledObj, NAME);
	name = GetVar(filledObj, NAME);
        SetVar(icon, NAME, name);
	SetVar(icon, FORMAT, GetVar(process, NAME)); 
        SetVar(icon, REPOBJ, filledObj);
        SetVar(icon, CORRAL, corral);
        DropIconInCorral(corral, icon);
    }
    return ObjTrue;
}

#ifdef PROTO
void IdleConnection(ObjPtr connection)
#else
void IdleConnection(connection)
ObjPtr connection;
#endif
/*Idles a connection*/
{
    int state;

    state = GetConnectionState(connection);

    switch (state)
    {
	case CS_NOT_LISTENING:		/*Accepting socket, not listening*/
	    {
		struct sockaddr_in templateAddr;
		int template;
		int proto;
		int port;

		/*Initialize internet socket template to internet, any machine*/
		port = SocketPort();
		templateAddr . sin_port = port;
		templateAddr . sin_family = PF_INET;
		templateAddr . sin_addr.s_addr = INADDR_ANY;

		proto = getprotobyname("tcp") -> p_proto;
		template = socket(PF_INET, SOCK_STREAM, proto);
		if (template >= 0)
		{
		    int status;

		    status = bind(template, &templateAddr, sizeof(templateAddr));
		    if (status >= 0)
		    {
			ObjPtr localConnection;
			/*Name was bound OK*/

			/*Set socket to be non blocking*/
			fcntl(template, F_SETFL, 
				fcntl(template, F_GETFL) | FNDELAY);

			/*Listen for a maximum of 5 connections*/
			listen(template, 5);
			((ConnectionPtr) connection) -> template = template;
			SetVar(connection, SOCKETID, NewInt(port));

			/*Send out a newp, now that we know who we are*/
			localConnection = NewLocalDaemonConnection();
			sprintf(tempStr, "%s %d %d %s_(%d)\n", 
			    ND_REG_PROCESS, getpid(), port, whatsMyName, getpid());
			SendSocketCommandLater(localConnection, tempStr);

			SetConnectionState(connection, CS_LISTENING);
		    }
		    else
		    {
			/*Connection failed, try again later.*/
		    }
		}
		else
		{
		    ReportError("IdleSockets", "socket call failed!");
		}
	    }
	    break;
	case CS_LISTENING:		/*Accepting socket, listening*/
	    {
		int sock, template;

		template = ((ConnectionPtr) connection) -> template;
		if (template >= 0)
		{
		    sock = accept(template, (struct sockaddr *) 0, (int *) 0);
		    if (sock != -1)
		    {
			ObjPtr newConnection;

			/*Wow, a connection*/ printf("Wow, a connection\n");
			newConnection = NewLocalScianConnection(sock);
		    }
		    else
		    {
			/*No connection yet*/
			break;
		    }
		}
		else
		{
		    ReportError("IdleSocket", "Bad listener template");
		}
	    }
	    break;
	case CS_NAME_ONLY:		/*We've got the name only*/
	    {
		struct hostent *host;
		ObjPtr name;
		char *s;
		struct sockaddr_in inSocketAddr;
		WinInfoPtr alertWindow;

		name = GetStringVar("IdleConnection", connection, NAME);
		if (!name)
		{
		    break;
		}

		/*Try to get the host information based on the address or name*/
		host = gethostbyname(GetString(name));

		if (host)
		{
		    ObjPtr var;
		    int k, inSocket, port;
		    struct protoent *proto;

		    /*Now try to connect*/
		    for (k = 0; k < sizeof(inSocketAddr); ++k)
		    {
			*(k + (char *) &inSocketAddr) = 0;
		    }
		    for (k = 0; k < host -> h_length; ++k)
		    {
			*(k + (char *) &inSocketAddr . sin_addr) =
			    *(k + (char *) host -> h_addr);
        	    }
		    var = GetVar(connection, SOCKETID);
		    if (var)
		    {
			port = GetInt(var);
		    }
		    else
		    {
			port = ND_BASE_SOCKET_NUMBER;
		    }
		    inSocketAddr . sin_port = htons((u_short) port);
		    inSocketAddr . sin_family = PF_INET;	/*Internet protocol*/

		    proto = getprotobyname("tcp");

		    inSocket = socket(PF_INET, SOCK_STREAM, proto -> p_proto);
		    if (inSocket >= 0)
		    {
			int status;

		 	status = connect(inSocket, &inSocketAddr, sizeof(inSocketAddr));
			if (status >= 0)
			{
			    /*Set socket to be non blocking*/
			    fcntl(inSocket, F_SETFL, 
				fcntl(inSocket, F_GETFL) | FNDELAY);

			    /*Send a validator*/
			    sprintf(tempStr, "%s\n", ND_PASSWORD);
			    writen(inSocket, tempStr, strlen(tempStr));

			    /*Put the fd into the connection*/
			    ((ConnectionPtr) connection) -> sock = inSocket;

			    /*Set the state*/
			    SetConnectionState(connection, CS_WAIT_FOR_VERSION);
			}
			else
			{
			    NoDaemonAlert(connection);
			    SetConnectionState(connection, CS_MORIBUND);
			}
		    }
		    else
		    {
			ReportError("IdleSocket", "Problem making socket.");
			SetConnectionState(connection, CS_MORIBUND);
		    }
		}
		else
		{
		    /*Error with the host name.*/
		    switch (h_errno)
		    {
			case HOST_NOT_FOUND:		/*No host by that name*/
			default:			/*Any other fatal error*/
			    HostNameErrorAlert();
			    SetConnectionState(connection, CS_MORIBUND);
			    break;
			case TRY_AGAIN:			/*Temporary error*/
			    break;
		    }
		}
	    }
	    break;
	case CS_WAIT_FOR_VERSION:			/*Waiting for a VERS to come back*/
	    {
		int sock;
		int nChars;
		ObjPtr command;
		ObjPtr curCmd;

		curCmd = GetVar(connection, CURCMD);
		if (!curCmd) curCmd = NewString(""); 

		sock = ((ConnectionPtr) connection) -> sock;
		if (sock >= 0)
		{
		    Bool anything;
		    char inChar;
		    anything = false;

		    /*See if there are characters waiting*/
		    while ((1 == read(sock, &inChar, 1)) &&
			   inChar &&
			   !isspace(inChar))
		    {
			char cmd[2];
			/*There's a command of some sort*/
			cmd[0] = inChar;
			cmd[1] = 0;

			curCmd = ConcatStrings(curCmd, NewString(cmd));
			anything = true;
		    }
		    if (anything)
		    {
			SetVar(connection, CURCMD, curCmd);
			if (inChar == '\n' || inChar == 0)
			{
			    ((ConnectionPtr) connection) -> gotCmdEnd = true;
			}
			if (inChar == 0 || isspace(inChar))
			{
			    char *cmd;

			    /*The command is done.  Find it*/
			    cmd = GetString(curCmd);
			    if (strlen(cmd) == 4 && CMD(cmd) == CMD(ND_INFO_VERSION))
			    {
				ObjPtr var;
				/*Got a version.  Either register our process
				  or ask for updates*/

				var = GetIntVar("IdleSocket", connection, CONNECTIONKIND);
				if (var)
				{
				    switch (GetInt(var))
				    {
					case CK_REMOTE_DAEMON:
					    sprintf(tempStr, "%s\n", ND_UPDATE_PROCESSES);
					    SendSocketCommandLater(connection, tempStr);
					    ShowConnectionControls(connection, NULLOBJ, GetString(GetVar(connection, NAME)));
					    break;
					case CK_LOCAL_DAEMON:
					    {
						/*Make a local listener*/
						NewListenerConnection();
					    }
					    break;
					case CK_REMOTE_SCIAN:
#ifdef DEBUG
					    printf("Gonna make a haha\n");
					    SendSocketCommandLater(connection, "Haha\n");
#endif

					    /* say what kind of machine I am */
#if MACHINE == IRIS4D
					    sprintf(tempStr, "%s\n", NO_MESG_MACHINE_IRIS4D);
#else
#if MACHINE == RS6000
					    sprintf(tempStr, "%s\n", NO_MESG_MACHINE_RS6000);
#else
					    sprintf(tempStr, "%s\n", NO_MESG_MACHINE_UNKNOWN);
#endif
#endif
					    SendSocketCommandLater(connection, tempStr);
					    sprintf(tempStr, "%s\n", ND_LIST_OBJECTS);
					    SendSocketCommandLater(connection, tempStr);
					    break;
				    }
				}
				SetConnectionState(connection, CS_WAIT_FOR_CMD_END);
			    }
			    else
			    {
				/*DIKEO make user alert*/
				ReportError("IdleSockets", "Funny return");
				SetConnectionState(connection, CS_MORIBUND);
				++TrashDayFlag;
			    }
			}
		    }
		}
		else
		{
		    ReportError("IdleSockets", "No socket file descriptor");
		    SetConnectionState(connection, CS_MORIBUND);
		    ++TrashDayFlag;
		}
	    }
	    break;
	case CS_WAIT_FOR_PASSWORD:			/*Waiting for a RUN? to come back*/
	    {
		int sock;
		int nChars;
		ObjPtr command;
		ObjPtr curCmd;

		curCmd = GetVar(connection, CURCMD);
		if (!curCmd) curCmd = NewString(""); 

		sock = ((ConnectionPtr) connection) -> sock;
		if (sock >= 0)
		{
		    Bool anything;
		    char inChar;
		    anything = false;

		    /*See if there are characters waiting*/
		    while ((1 == read(sock, &inChar, 1)) &&
			   inChar &&
			   !isspace(inChar))
		    {
			char cmd[2];
			/*There's a command of some sort*/
			cmd[0] = inChar;
			cmd[1] = 0;

			curCmd = ConcatStrings(curCmd, NewString(cmd));
			anything = true;
		    }
		    if (anything)
		    {
			SetVar(connection, CURCMD, curCmd);
			if (inChar == '\n' || inChar == 0)
			{
			    /* ignore it */
			    ((ConnectionPtr) connection) -> gotCmdEnd = true;
			}
			if (inChar == 0 || isspace(inChar))
			{
			    char *cmd;

			    /*The command is done.  Find it*/
			    cmd = GetString(curCmd);
			    if (strlen(cmd) == 4 && CMD(cmd) == CMD(ND_PASSWORD))
			    {
				char *s;
				ObjPtr var;
				/*Got a password.  Return a version.*/

				sprintf(tempStr, "%s Scian %s\n", ND_INFO_VERSION, SCIANVERSION);
				s = tempStr + 5;
				while (*s)
				{
				    if (*s == ' ') *s = '_';
				    ++s;
				}
				SendSocketCommandLater(connection, tempStr);
				/* say what kind of machine I am */
#if MACHINE == IRIS4D
				sprintf(tempStr, "%s\n", NO_MESG_MACHINE_IRIS4D);
#else
#if MACHINE == RS6000
				sprintf(tempStr, "%s\n", NO_MESG_MACHINE_RS6000);
#else
				sprintf(tempStr, "%s\n", NO_MESG_MACHINE_UNKNOWN);
#endif
#endif
				SendSocketCommandLater(connection, tempStr);
				SetConnectionState(connection, CS_WAIT_FOR_CMD_END);
			    }
			    else
			    {
				/*DIKEO make user alert*/
				ReportError("IdleSockets", "Funny return");
				SetConnectionState(connection, CS_MORIBUND);
			    }
			}
		    }
		}
		else
		{
		    ReportError("IdleSockets", "No socket file");
		    SetConnectionState(connection, CS_MORIBUND);
		}
	    }
	    break;
	case CS_IDLING:			/*At connection idle*/
	    {
		int sock; /* socket file descriptor */
		sock = ((ConnectionPtr) connection) -> sock;
		if (sock >= 0)
		{
		    char inChar;
		    /*See if there's a character waiting*/
		    if ((1 == read(sock, &inChar, 1)) &&
			isalpha(inChar))
		    {
			char cmd[2];
			/*There's a command of some sort*/
			cmd[0] = inChar;
			cmd[1] = 0;

			SetVar(connection, CURCMD, NewString(cmd));
			SetConnectionState(connection, CS_FILL_CMD);
		    }
		    else
		    {
			/*See if there are commands to be sent*/
			ObjPtr cmdQueue;
			cmdQueue = GetVar(connection, OUTPUTQUEUE);
			if (cmdQueue)
			{
			    ThingListPtr runner;
			    runner = LISTOF(cmdQueue);
			    while (runner)
			    {
				if (!IsPublished(runner -> thing))
				{
				    char *str;
				    str = GetString(runner -> thing);
				    if (str)
				    {
#ifdef DEBUG
					fprintf(stderr, "Sending string: %s\n", str);
#endif
	     		        	writen(sock, str, strlen(str));
				    }
				    else
				    {
#ifdef DEBUG
					fprintf(stderr, "null string in OUTPUTQUEUE\n");
#endif
				    }
				}
				else
				{
#ifdef DEBUG
				    fprintf(stderr, "Sending object: 0x%x\n", runner -> thing);
#endif
				    TransmitObject(runner -> thing, connection);
				}
				runner = runner -> next;
			    }
			    SetVar(connection, OUTPUTQUEUE, NewList());
			}
		    }
		}
	    }
	    break;
	case CS_FILL_CMD:			/*Filling command*/
	    {
		int sock;
		int nChars;
		ObjPtr command;
		ObjPtr curCmd;

		curCmd = GetVar(connection, CURCMD);
		if (!curCmd) curCmd = NewString(""); 

		sock = ((ConnectionPtr) connection) -> sock;
		if (sock >= 0)
		{
		    Bool anything;
		    char inChar = 'x';
		    anything = false;

		    /*See if there are characters waiting*/
		    while ((1 == read(sock, &inChar, 1)) &&
			   inChar &&
			   !isspace(inChar))
		    {
			char cmd[2];
			/*There's a command of some sort*/
			cmd[0] = inChar;
			cmd[1] = 0;

			curCmd = ConcatStrings(curCmd, NewString(cmd));
			anything = true;
		    }
		    SetVar(connection, CURCMD, curCmd);
		    if (inChar == '\n' || inChar == 0)
		    {
			((ConnectionPtr) connection) -> gotCmdEnd = true;
		    }
		    if (inChar == 0 || isspace(inChar))
		    {
			/*The command is done.  Find it*/
			FindCommand(connection);
		    }
		}
	    }
	    break;
	case CS_READ_NUMBER_ONE:		/*Read first number*/
	case CS_READ_NUMBER_TWO:		/*Read second number*/
	case CS_READ_NUMBER_THREE:		/*Read third number*/
	    {
		int sock;
		int oldState;

		oldState = state;

		sock = ((ConnectionPtr) connection) -> sock;
		if (sock >= 0)
		{
		    unsigned long num;
		    Bool someDigits = false;
		    char inChar;
		    ObjPtr var;

		    /*Get process num.  If null, it means that one hasn't been
		      set yet, and leading spaces should be tolerated.*/

		    switch (oldState)
		    {
			case CS_READ_NUMBER_ONE:
			    var = GetVar(connection, READNUMBERONE);
			    break;
			case CS_READ_NUMBER_TWO:
			    var = GetVar(connection, READNUMBERTWO);
			    break;
			case CS_READ_NUMBER_THREE:
			    var = GetVar(connection, READNUMBERTHREE);
			    break;
			default:
			    ReportError("IdleConnection", "Internal state error");
		    }
		    if (var)
		    {
			num = GetInt(var);
			someDigits = true;
		    }
		    else
		    {
			num = 0;
		    }

		    /*See if there's a character waiting*/
		    while (1 == read(sock, &inChar, 1))
		    {
			if (isspace(inChar) && (inChar != '\n') && !someDigits)
			{
			    
			    /*Just skip the blanks*/
			}
			else if (inChar >= '0' && inChar <= '9')
			{
			    someDigits = true;
			    num *= 10;
			    num += inChar - '0';
			}
			else
			{
			    /*This is the end of the number*/
			    ObjPtr cmdObj;

			    if (inChar == '\n' || inChar == '\0')
			    {
				((ConnectionPtr) connection) -> gotCmdEnd = true;
			    }

			    cmdObj = GetStringVar("IdleConnection", connection, CURCMD);
			    if (cmdObj)
			    {
				char *cmd;

				cmd = GetString(cmdObj);
				if (CMD(cmd) == CMD(ND_INFO_LIVE_PROCESS))
				{
				    SetVar(connection, CMDPROCSOCKET, NewInt(num));
				    SetConnectionState(connection, CS_READ_PROC_NAME);
				}
				else if (CMD(cmd) == CMD(ND_INFO_DEAD_PROCESS))
				{
				    SetVar(connection, CMDPROCSOCKET, NewInt(num));
				    SetConnectionState(connection, CS_READ_PROC_NAME);
				}
				else if (CMD(cmd) == CMD(NO_MESG_SEND_OBJECT))
				{
				    ObjPtr sendObject;
				    sendObject = FindPublishedObject(num);
				    if (sendObject)
				    {
					SendObjectLater(connection, sendObject);
				    }
				    SetConnectionState(connection, CS_WAIT_FOR_CMD_END);
				}
				else if (CMD(cmd) == CMD(ND_REG_OBJECT))
				{
				    ObjPtr process;

				    process = GetVar(connection, OWNERPROCESS);
				    if (process && num)
				    {
					SetVar(process, TARGETOBJECT, NULLOBJ);
					SetVar(process, TARGETVAR, NULLOBJ);
				    }
				    SetConnectionState(connection, CS_WAIT_FOR_CMD_END);
				}
				else if (CMD(cmd) == CMD(NO_MESG_GETVAR))
				{
				    /* HAK! don't know how to deal with this yet!*/
				    switch (oldState)
				    {
					case CS_READ_NUMBER_ONE:
					    SetConnectionState(connection, CS_READ_NUMBER_TWO);
					    break;
					case CS_READ_NUMBER_TWO:
					    {
						ObjPtr obj, val, valNetidObj;
						int netid, valNetid;

						netid = GetInt(GetIntVar("IdleConnection", connection, READNUMBERONE));
						obj = FindPublishedObject(netid);
						if (obj)
						{
						    val = GetVar(obj, num);
						    if (val)
						    {
							valNetid = PublishObject(val);
						    }
						    else
						    {
							valNetid = 0;
						    }
						    sprintf(tempStr, "%s %d %d %d\n", NO_MESG_GOTVAR, netid, num, valNetid);
						    SendSocketCommandLater(connection, tempStr);
						    SendObjectLater(connection, val);
						}
						SetConnectionState(connection, CS_WAIT_FOR_CMD_END);
						break;
					    }
					default:
					    ReportError("IdleConnection", "Internal State Error");
					    break;
				    }
				}
				else if (CMD(cmd) == CMD(NO_MESG_GOTVAR))
				{
				    /* HAK! have to set the objs var! */
				    switch (oldState)
				    {
					case CS_READ_NUMBER_ONE:
					    SetConnectionState(connection, CS_READ_NUMBER_TWO);
					    break;
					case CS_READ_NUMBER_TWO:
					    SetConnectionState(connection, CS_READ_NUMBER_THREE);
					    break;
					case CS_READ_NUMBER_THREE:
#if 0
					    /* for now, automatically request it */
					    if (num) /* num = 0 means it was a null object */
					    {
						sprintf(tempStr, "%s %d\n", NO_MESG_SEND_OBJECT, num);
						SendSocketCommandLater(connection, tempStr);
					    }
#else
					    {
					    ObjPtr process;

					    process = GetVar(connection, OWNERPROCESS);
					    if (process)
					    {
						ObjPtr target;
						/* these are always integers, I hope */
						

						target = FindRemoteObject(process, GetInt(GetVar(connection, READNUMBERONE)));
						SetVar(connection, TARGETOBJECT, target);
						SetVar(connection, TARGETVAR, GetVar(connection, READNUMBERTWO));
					    }
					    }
#endif
					    SetConnectionState(connection, CS_WAIT_FOR_CMD_END);
					    break;
					default:
					    ReportError("IdleConnection", "Internal State Error");
					    break;
				    }
				}
				else if (CMD(cmd) == CMD(NO_MESG_UPDATEVAR))
				{
				    switch(oldState)
				    {
					case CS_READ_NUMBER_ONE:
					    SetConnectionState(connection, CS_READ_NUMBER_TWO);
					    break;
					case CS_READ_NUMBER_TWO:
					    {
						ObjPtr process;
						process = GetVar(connection, OWNERPROCESS);
						if (process)
						{
						    ObjPtr object;
						    object = FindRemoteObject(process, GetInt(GetVar(connection, READNUMBERONE)));
						    if (object)
						    {
							UnRemoteObject(object);
							SetVar(object, num, NetworkUpdated);
						    }
						    if (num == DATA)
						    {
							DatasetChanged(object);
						    }

						}
					    }
					    SetConnectionState(connection, CS_WAIT_FOR_CMD_END);
					    break;
				    }
				}
				else
				{
				    ReportError("IdleConnection", "Funny command in CS_READ_NUMBER_num");
				    SetConnectionState(connection, CS_WAIT_FOR_CMD_END);
				}
			    }
			    break;
			}
		    }
		    if (someDigits)
		    {
			switch (oldState)
			{
			    case CS_READ_NUMBER_ONE:
				var = SetVar(connection, READNUMBERONE, NewInt(num));
				break;
			    case CS_READ_NUMBER_TWO:
				var = SetVar(connection, READNUMBERTWO, NewInt(num));
				break;
			    case CS_READ_NUMBER_THREE:
				var = SetVar(connection, READNUMBERTHREE, NewInt(num));
				break;
			    default:
				ReportError("IdleConnection", "Internal state error");
			}
		    }
		}
	    }
	    break;
	case CS_READ_PROC_NAME:			/*Read the proc name*/
	    {
		int sock;

		sock = ((ConnectionPtr) connection) -> sock;
		if (sock)
		{
		    Bool someAlpha = false;		/*True iff there's some alpha*/
		    char inChar;
		    Bool escaped;
		    ObjPtr var;

		    var = GetVar(connection, CMDPROCNAME);
		    if (var)
		    {
			someAlpha = true;
		    }

		    escaped = GetPredicate(connection, ESCAPED);

		    /*See if there's a character waiting*/
		    while ((1 == read(sock, &inChar, 1)))
		    {
			if (!escaped && !someAlpha && (inChar != '\n') && isspace(inChar))
			{
			    /*Just skip the blanks*/
			}
			else if (!escaped && (inChar == 0 || inChar == '\n'))
			{
			    ObjPtr cmdObj;
			 
			    /*This is the end of the name*/
			    cmdObj = GetStringVar("IdleConnection", connection, CURCMD);
			    if (cmdObj)
			    {
				char *cmd;
				cmd = GetString(cmdObj);
				if (CMD(cmd) == CMD(ND_INFO_LIVE_PROCESS))
				{
				    NewProcess(connection, GetString(var), GetInt(GetVar(connection, CMDPROCSOCKET)));
				}
				else if (CMD(cmd) == CMD(ND_INFO_DEAD_PROCESS))
				{
				    DeleteProcess(connection, GetInt(GetVar(connection, CMDPROCSOCKET)));
				}
			    }
			    SetConnectionState(connection, CS_IDLING);
			    break;
			}
			else if (!escaped && (inChar == '\\'))
			{
			    SetVar(connection, ESCAPED, ObjTrue);
			    escaped = true;
			}
			else
			{
			    char s[2];

			    if (escaped)
			    {
				SetVar(connection, ESCAPED, ObjFalse);
			    }
			    else if (inChar == '_')
			    {
				inChar = ' ';
			    }

			    s[0] = inChar;
			    s[1] = 0;

			    if (var)
			    {
				var = ConcatStrings(var, NewString(s));
			    }
			    else
			    {
				var = NewString(s);
				someAlpha = true;
			    }
			}
		    }
		    if (someAlpha)
		    {
			SetVar(connection, CMDPROCNAME, var);
		    }
		}
	    }
	    break;
	case CS_READ_OBJECT:
	    {
		ObjPtr filledObj, process;
		filledObj = ReceiveObject(connection); /* currently, wait until obj is filled */
#ifdef DEBUG
if (filledObj)
   fprintf(stderr, "well, it returned something\n");
else
   fprintf(stderr, "it didn't return anything\n");
#endif
		if (filledObj && (process = GetVar (connection, OWNERPROCESS)))
		{
		    ObjPtr allObjects, target;
		    int varnum;
		    allObjects = GetVar(process, ALLOBJECTS);
		    if (!allObjects)
		    {
			allObjects = NewList();
		    }
		    PrefixList(allObjects, filledObj);

		    SetVar(process, ALLOBJECTS, allObjects);

		    target = GetVar(connection, TARGETOBJECT);
		    if (!target)
		    {
			/* must be a top-level advertised object */
			ObjPtr allAdvertised;

			allAdvertised = GetVar(process, ALLADVERTISED);
			if (!allAdvertised)
			{
			    allAdvertised = NewList();
			}
			PrefixList(allAdvertised, filledObj);
			SetVar(process, ALLADVERTISED, allAdvertised);
			SetMethod(filledObj, NEWREMOTEOBJ, NewRemoteObject);
			DeferMessage(filledObj, NEWREMOTEOBJ);
		    }
		    else
		    {
			varnum = GetInt(GetIntVar("IdleConnection", connection,
					TARGETVAR));
			if (varnum)
			{
			    SetVar(target, varnum, filledObj);
/* HAK this might ought to be done more intelligently... */
#if 0
			    if (varnum == DATA)
			    {
				DatasetChanged(target);
			    }
#endif
			}
			else
			{
			    ReportError("IdleConnection","bad variable number");
			}
			SetVar(connection, TARGETOBJECT, NULLOBJ);
		    }
		}
		else
		{
#ifdef DEBUG
fprintf(stderr, "I guess the connection didn't have an owner process\n");
#endif
		}

		SetConnectionState(connection, CS_WAIT_FOR_CMD_END);
	    }
	    break;
	case CS_WAIT_FOR_CMD_END:		/*Wait for the end of the command*/
	    {
		int sock;

		sock = ((ConnectionPtr) connection) -> sock;
		if (((ConnectionPtr) connection) -> gotCmdEnd)
		{
		    SetVar(connection, CURCMD, NULLOBJ);
		    SetConnectionState(connection, CS_IDLING);
		}
		else if (sock)
		{
		    char inChar = 'x';
		    /*See if there's a character waiting*/
		    while ((1 == read(sock, &inChar, 1)))
		    {
			if (inChar == '\n' || inChar == 0)
			{
			    SetVar(connection, CURCMD, NULLOBJ);
			    SetConnectionState(connection, CS_IDLING);
			    break;
			}
		    }
		}
		((ConnectionPtr) connection) -> gotCmdEnd = false;
	    }
	    break;
	case CS_MORIBUND:			/*Moribund.  Kill connection*/
	    KillConnection(connection);
	    break;
	case CS_LIMBO:				/*Process is in limbo.  Do nothing and let idle expire.*/
	    break;
    }
}

/* returns true IFF all connections are in idle state, and have no queued
 * output waiting. If flag == true, prints the reason it's returning false
 */
#ifdef PROTO
Bool AllConnectionsIdle(Bool flag)
#else
Bool AllConnectionsIdle(flag)
Bool flag;
#endif
{
    ThingListPtr runner;
    int state;
    ObjPtr cmdQ;

    runner = LISTOF(allConnections);
    while (runner)
    {
	state = GetConnectionState(runner -> thing);
	if (state != CS_IDLING && state != CS_MORIBUND && state != CS_LIMBO && state != CS_LISTENING)
	{
	    if (flag)
		fprintf(stderr, "found connection in state %s\n", stateName[state]);
	    return false;
	}
	else if (state == CS_IDLING || state == CS_LIMBO)
	{
	    cmdQ = GetVar(runner -> thing, OUTPUTQUEUE);
	    if (cmdQ)
	    {
		/* if there's something waiting in the queue, it's not idle */
		if (LISTOF(cmdQ))
		{
		    if (flag)
			fprintf(stderr, "found pending output in socket in state %s\n", stateName[state]);
		    return false;
		}
	    }
	}
	runner = runner -> next;
    }
    return true;
}

#ifdef PROTO
void IdleAllConnections(void)
#else
void IdleAllConnections()
#endif
/*Idles all the connections*/
{
    ThingListPtr runner;


    runner = LISTOF(allConnections);
    while (runner)
    {
	IdleConnection(runner -> thing);
	runner = runner -> next;
    }
}

#ifdef PROTO
static ObjPtr NewConnection(char *machineName, int connectionKind)
#else
static ObjPtr NewConnection(machineName, connectionKind)
char *machineName;
Bool connectionKind;
#endif
/*Makes a new connection to machine machineName.  connectionKind
  is the kind of connection:
	CK_REMOTE_DAEMON	Connection to a remote network daemon
	CK_LOCAL_DAEMON		Connection to a local daemon
	CK_LISTENING_SCIAN	Listening connection on a local SciAn
	CK_REMOTE_SCIAN		Connection to a remote SciAn (not implemented)
	CK_LOCAL_SCIAN		Connection from a local SciAn (not implemented)

  Does not do kind-specific initialization for the connection, such
  as socket ID, and leaves the connection in limbo.
*/
{
    char *s;
    ObjPtr connection;

    /*Stuff this in lastComputer*/
    if (lastComputer)
    {
	free(lastComputer);
    }
    lastComputer = malloc(strlen(machineName) + 1);
    strcpy(lastComputer, machineName);

    /*Create the new connection*/
    connection = NewObject(connectionClass, sizeof(Connection) - sizeof(Thing));
    SetVar(connection, CONNECTIONKIND, NewInt(connectionKind));
    SETOBJTYPE(connection -> flags, CONNECTION);
    ((ConnectionPtr) connection) -> sock = -1;
    ((ConnectionPtr) connection) -> gotCmdEnd = false;
    ((ConnectionPtr) connection) -> template = -1;

    PrefixList(allConnections, connection);

    SetVar(connection, NAME, NewString(machineName));
    SetVar(connection, OUTPUTQUEUE, NewList());
    SetVar(connection, ALLPROCESSES, NewList());

    SetVar(connection, CONNECTIONSTATE, NewInt(CS_LIMBO));

    return connection;
}

#ifdef PROTO
ObjPtr NewLocalDaemonConnection(void)
#else
ObjPtr NewLocalDaemonConnection()
#endif
/*Make a new connection to the local daemon*/
{
    char hostName[200];
    ThingListPtr runner;
    ObjPtr retVal;

    /*See if there's connection to the daemon already*/
    runner = LISTOF(allConnections);
    while (runner)
    {
	ObjPtr var;
	var = GetIntVar("NewLocalDaemonConnection", runner -> thing, CONNECTIONKIND);
	if (var && GetInt(var) == CK_LOCAL_DAEMON)
	{
	    return runner -> thing;
	}
	runner = runner -> next;
    }

    /*There must not be.*/
    gethostname(hostName, 199);
    hostName[199] = 0;
    retVal = NewConnection(hostName, CK_LOCAL_DAEMON);
    SetConnectionState(retVal, CS_NAME_ONLY);
    return retVal;
}

#ifdef PROTO
ObjPtr NewRemoteDaemonConnection(char *hostName)
#else
ObjPtr NewRemoteDaemonConnection(hostName)
char *hostName;
#endif
/*Make a new connection to the daemon on hostName*/
{
    ObjPtr retVal;

    retVal = NewConnection(hostName, CK_REMOTE_DAEMON);
    SetVar(retVal, SOCKETID, NewInt(ND_BASE_SOCKET_NUMBER));
    SetConnectionState(retVal, CS_NAME_ONLY);
    return retVal;
}

#ifdef PROTO
ObjPtr NewRemoteScianConnection(char *hostName, int port, ObjPtr process)
#else
ObjPtr NewRemoteScianConnection(hostName, port, process)
char *hostName;
int port
ObjPtr process;
#endif
/*Make a new connection to the SciAn on hostName on socket port*/
{
    ObjPtr retVal;

    retVal = NewConnection(hostName, CK_REMOTE_SCIAN);
    SetVar(retVal, SOCKETID, NewInt(port));
    SetConnectionState(retVal, CS_NAME_ONLY);
    SetVar(retVal, OWNERPROCESS, process); /* JRM connection needs assoc. proc. */
    return retVal;
}


#ifdef PROTO
static ObjPtr NewListenerConnection(void)
#else
static ObjPtr NewListenerConnection();
#endif
{
    ObjPtr retVal;
    ThingListPtr runner;

    /*See if there's connection to the daemon already*/
    runner = LISTOF(allConnections);
    while (runner)
    {
	ObjPtr var;
	var = GetIntVar("NewListenerConnection", runner -> thing, CONNECTIONKIND);
	if (var && GetInt(var) == CK_LISTENING_SCIAN)
	{
	    return runner -> thing;
	}
	runner = runner -> next;
    }

    retVal = NewConnection("", CK_LISTENING_SCIAN);

    SetConnectionState(retVal, CS_NOT_LISTENING);
    return retVal;
}

#ifdef PROTO
static ObjPtr NewLocalScianConnection(int file)
#else
static ObjPtr NewLocalScianConnection(file)
int file;
#endif
/*Opens a new local Scian connection using file as descriptor*/
{
    ObjPtr retVal = NULLOBJ;

    if (file >= 0)
    {
	retVal = NewConnection("", CK_LOCAL_SCIAN);
	((ConnectionPtr) retVal) -> sock = file;
	SetConnectionState(retVal, CS_WAIT_FOR_PASSWORD);
    }
    else
    {
	ReportError("NewLocalScianConnection", "bad file descriptor");
	retVal = NULLOBJ;
    }
    return retVal;
}

static ObjPtr ConnectToProcess(process)
ObjPtr process;
/*Method to connect to a process*/
{
    ThingListPtr runner;
    ObjPtr ownerConnection;
    ObjPtr connection = NULLOBJ;
    ObjPtr var;
    char *name;
    int port;

    /*First see if there is already a process in connection*/
    runner = LISTOF(allConnections);
    while (runner)
    {
	if (GetVar(runner -> thing, OWNERPROCESS) == process)
	{
	    connection = runner -> thing;
	    break;
	}
	runner = runner -> next;
    }

    if (!connection)
    {
	/*Have to make the connection.  Get machine name from owner connection*/
	ownerConnection = GetVar(process, OWNERCONNECTION);
	if (!ownerConnection)
	{
	    return ObjFalse;
	}

	var = GetStringVar("ConnectToProcess", ownerConnection, NAME);
	if (!var)
	{
	    return ObjFalse;
	}
	name = GetString(var);

	var = GetIntVar("ConnectToProcess", process, SOCKETID);
	if (!var)
	{
	    return ObjFalse;
	}
	port = GetInt(var);

	connection = NewRemoteScianConnection(name, port, process);
	SetConnectionState(connection, CS_NAME_ONLY);
    }
    NewControlWindow(process);

    return ObjTrue;
}

#ifdef PROTO
void DoEnablePublication(void)
#else
void DoEnablePublication()
#endif
/*Enables publication*/
{
    NewLocalDaemonConnection();
}

static ObjPtr ReplyConnection(dummy, replyText)
ObjPtr dummy;
char *replyText;
/*Makes a new remote daemon connection to machine replyText*/
{
    NewRemoteDaemonConnection(replyText);
}

void DoConnectToComputer(void)
/*Initiates a connection to another computer by asking the user for an
  address and then trying to connect.*/
{
    WinInfoPtr askWindow;
    char hostName[200];

    gethostname(hostName, 199);
    hostName[199] = 0;
    askWindow = AskUser((WinInfoPtr) NULLOBJ, "Enter the name or the IP address of the computer to connect:",
		 (FuncTyp) ReplyConnection, 
		lastComputer ? lastComputer : hostName);
    if (askWindow) SetVar((ObjPtr) askWindow, HELPSTRING,
	NewString("In order to connect to a computer, you must first enter \
its internet name (e.g. foo.bar.moo.cow.edu) or its internet address (e.g. \
128.182.2.255).  Enter the name or address here and press OK to connect to \
the computer.  Press Cancel to cancel the connection."));
}

#ifdef PROTO
void CloseConnection(ObjPtr connection)
#else
void CloseConnection(connection)
ObjPtr connection;
#endif
{
    if (OBJTYPE(connection -> flags) == CONNECTION)
    {
	ObjPtr var;
	int connectionKind;
	int sock;

	sock = (((ConnectionPtr) connection) -> sock);
	var = GetIntVar("CleanupConnection", connection, CONNECTIONKIND);

	connectionKind = GetInt(var);

	if (connectionKind == CK_LOCAL_DAEMON)
	{
	    if (sock >= 0)
	    {
	            /*I am not here*/
	            sprintf(tempStr, "%s %d\n", ND_UNREG_PROCESS, getpid());
		    writen(sock, tempStr, strlen(tempStr));
		    sleep(1);
	    }
	}

	/*Say bye*/
	if (sock >= 0 && (connectionKind == CK_LOCAL_DAEMON ||
			   connectionKind == CK_REMOTE_DAEMON))
	{
	    /*Send a close*/
	    sprintf(tempStr, "%s\n", ND_CLOSE);
	    writen(sock, tempStr, strlen(tempStr));
	    sleep(1);
	}

	if (sock >= 0)
	{
	    close(sock);
	}
	else if (((ConnectionPtr) connection) -> template >= 0)
	{
	    close(((ConnectionPtr) connection) -> template);
	}
	((ConnectionPtr) connection) -> sock = -1;
	((ConnectionPtr) connection) -> template = -1;
    }
}

ObjPtr CleanupConnection(connection)
ObjPtr connection;
/*Clean up a connection*/
{
    if (OBJTYPE(connection -> flags) == CONNECTION)
    {
	CloseConnection(connection);
    }

    return ObjTrue;
}

#ifdef PROTO
void SendSocketCommandLater(ObjPtr connection, char *command)
#else
void SendSocketCommandLater(connection, command)
ObjPtr connection;
char *command;
#endif
/*Sends command to connection later*/
{
    ObjPtr cmdQueue;

    cmdQueue = GetVar(connection, OUTPUTQUEUE);
    if (!cmdQueue)
    {
	cmdQueue = NewList();
    }

    PostfixList(cmdQueue, NewString(command));
    SetVar(connection, OUTPUTQUEUE, cmdQueue);
}

void SendObjectLater(connection, obj)
ObjPtr connection, obj;
{
    ObjPtr cmdQueue;

    cmdQueue = GetVar(connection, OUTPUTQUEUE);
    if (!cmdQueue)
    {
	cmdQueue = NewList();
    }

    PostfixList(cmdQueue, obj);
    SetVar(connection, OUTPUTQUEUE, cmdQueue);
}

static ObjPtr ShowProcessControls(process, windowName)
ObjPtr process;
char *windowName;
/*Makes a new control window to control a process */
{
    WinInfoPtr controlWindow;
    ObjPtr var;
    ObjPtr panel;
    ObjPtr corral;
    ObjPtr contents;
    WinInfoPtr dialogExists;

    dialogExists = DialogExists((WinInfoPtr) process, NewString("Controls"));
    controlWindow = GetDialog((WinInfoPtr) process, NewString("Controls"), windowName, 
	PXWINWIDTH, PXWINHEIGHT, SCRWIDTH, SCRHEIGHT, WINDBUF + WINFIXEDSIZE + WINRGB);
    
    if (!dialogExists)
    {
	ObjPtr connection, textBox, checkBox, icon, palette, name, button, sw;
	int l, r, b, t, bw;
	int left, right, bottom, top;
	char *s;
	ObjPtr allObjects;
	ThingListPtr runner;

	SetVar((ObjPtr) controlWindow, REPOBJ, process);

	/*Set help string*/
	SetVar((ObjPtr) controlWindow, HELPSTRING, NewString("This window \
shows status and running processes for a remote network connection."));

	SelWindow(controlWindow);
	GetWindowBounds(&l, &r, &b, &t);

	/*Add in a panel*/
	panel = NewPanel(greyPanelClass, 0, r - l, 0, t - b);
	if (!panel)
	{
	    return ObjFalse;
	}
	contents = GetVar((ObjPtr) controlWindow, CONTENTS);
	PrefixList(contents, panel);
	SetVar(panel, PARENT, (ObjPtr) controlWindow);
	SetVar(panel, STICKINESS, NewInt(STICKYLEFT + STICKYRIGHT +
				     STICKYBOTTOM + STICKYTOP));
	contents = GetVar(panel, CONTENTS);

	/*Make an icon corral*/
	corral = NewIconCorral(NULLOBJ, l + MINORBORDER, r - MINORBORDER, b + 3 * MINORBORDER + 2 * BUTTONHEIGHT, t - MINORBORDER, BARRIGHT + BARBOTTOM);
	SetVar(corral, STICKINESS, NewInt(STICKYLEFT + STICKYRIGHT +
				     STICKYBOTTOM + STICKYTOP));
	SetVar(corral, TOPDOWN, ObjTrue);
	SetVar((ObjPtr) controlWindow, CORRAL, corral);
	SetVar(corral, NAME, NewString("Process Objects Corral"));
	SetVar(corral, HELPSTRING,
	NewString("This corral contains icons for all the objects available through \
a process connection.  You can visualize, show the controls of, or modify the datasets by selecting \
some of them and pressing the buttons at the bottom of the window.  You can delete \
datasets by choosing Delete from the Object menu.  You can also open new datasets by dragging icons from \
the file window into this corral."));
	PrefixList(contents, corral);
	SetVar(corral, PARENT, panel);

	/*Drop the objects in it*/
	allObjects = GetVar(process, ALLADVERTISED);
	if (!allObjects)
	{
	    allObjects = NewList();
	    SetVar(process, ALLADVERTISED, allObjects);
	}

	runner = LISTOF(allObjects);
	while (runner)
	{
	    icon = GetVar(runner -> thing, DEFAULTICON);
	    if (icon)
	    {
		icon = NewObject(icon, 0L);
	    }
	    else
	    {
		icon = NewIcon(0, 0,ICONQUESTION, "?");
	    }
	    SetVar(icon, REPOBJ, runner -> thing);
	    SetVar(icon, ICONLOC, NULLOBJ);
	    SetVar(icon, CORRAL, corral);
	    DropIconInCorral(corral, icon);
	    runner = runner -> next;
	}

	l += MINORBORDER;
	r -= MINORBORDER;
	b += 2 * MINORBORDER + BUTTONHEIGHT;
	t = b + BUTTONHEIGHT;
	bw = (r - l - MINORBORDER) / 2;

	/*Make a visualize button*/
	button = NewFunctionButton(controlWindow,
		    l, l + bw,
		    b, b + BUTTONHEIGHT, OF_VISUALIZE); 
	if (button)
	{
	    SetVar(button, PARENT, panel);
	    SetVar(button, STICKINESS, NewInt(STICKYBOTTOM + STICKYLEFT + FLOATINGRIGHT));
	    PrefixList(contents, button);
	}
	    
	/*Make a visualize as... button*/
	button = NewFunctionButton(controlWindow,
		    r - bw, r,
		    b, b + BUTTONHEIGHT, OF_VISUALIZE_AS); 
	if (button)
	{
	    SetVar(button, PARENT, panel);
	    SetVar(button, STICKINESS, NewInt(STICKYBOTTOM + FLOATINGLEFT + STICKYRIGHT));
	    PrefixList(contents, button);
	}

	t = b - MINORBORDER;
	b = t - BUTTONHEIGHT;
	/*Make a show info button*/
	button = NewFunctionButton(controlWindow,
		    l, l + bw, 
		    b, b + BUTTONHEIGHT, OF_SHOW_CONTROLS); 
	if (button)
	{
	    SetVar(button, PARENT, panel);
	    SetVar(button, STICKINESS, NewInt(STICKYBOTTOM + STICKYLEFT + FLOATINGRIGHT));
	    PrefixList(contents, button);
	}

	/*Make a modify button*/
	button = NewFunctionButton(controlWindow, 
		    r - bw, r,
		    b, b + BUTTONHEIGHT, OF_MODIFY);
	if (button)
	{
	    SetVar(button, PARENT, panel);
	    SetVar(button, STICKINESS, NewInt(STICKYBOTTOM + FLOATINGLEFT + STICKYRIGHT));
	    PrefixList(contents, button);
	}
    }
    return ObjTrue;
}

static ObjPtr ShowConnectionControls(connection, ownerWindow, windowName)
ObjPtr connection;
WinInfoPtr ownerWindow;
char *windowName;
/*Makes a new control window to control a connection*/
{
    WinInfoPtr controlWindow;
    ObjPtr var;
    ObjPtr panel;
    ObjPtr corral;
    ObjPtr contents;
    WinInfoPtr dialogExists;
    char hostName[200];

    dialogExists = DialogExists((WinInfoPtr) connection, NewString("Controls"));
    controlWindow = GetDialog((WinInfoPtr) connection, NewString("Controls"), windowName, 
	SCWINWIDTH, SCWINHEIGHT, SCRWIDTH, SCRHEIGHT, WINDBUF + WINFIXEDSIZE + WINRGB);
    
    if (!dialogExists)
    {
	ObjPtr textBox, checkBox, icon, palette, name, button, sw;
	int l, r, b, t;
	int left, right, bottom, top;
	char *s;
	SetVar((ObjPtr) controlWindow, REPOBJ, connection);

	/*Set help string*/
	SetVar((ObjPtr) controlWindow, HELPSTRING, NewString("This window \
shows status and running processes for a remote network connection."));

	SelWindow(controlWindow);
	GetWindowBounds(&l, &r, &b, &t);

	/*Add in a panel*/
	panel = NewPanel(greyPanelClass, 0, r - l, 0, t - b);
	if (!panel)
	{
	    return ObjFalse;
	}
	contents = GetVar((ObjPtr) controlWindow, CONTENTS);
	PrefixList(contents, panel);
	SetVar(panel, PARENT, (ObjPtr) controlWindow);
	SetVar(panel, STICKINESS, NewInt(STICKYLEFT + STICKYRIGHT +
				     STICKYBOTTOM + STICKYTOP));

	contents = GetVar(panel, CONTENTS);
	
	/*Add in the icons*/

	left = l + MAJORBORDER;
	right = l + MAJORBORDER + SCGRAPHICWIDTH;
	top = t - MAJORBORDER;
	bottom = top - ICONSIZE;

	gethostname(hostName, 199);
	hostName[199] = 0;
	icon = NewIcon(left + ICONSIZE / 2, top - ICONSIZE / 2, ICONWORKSTATION,
		"SciAn");
	PrefixList(contents, icon);
	SetVar(icon, PARENT, (ObjPtr) controlWindow);
	SetVar(icon, STICKINESS, NewInt(STICKYLEFT + STICKYTOP));
	SetVar(icon, HELPSTRING, NewString("This icon represents the workstation running \
SciAn that you are using right now.  It's part of a graphic that shows that the window \
is a network connection window.  Other than that, it doesn't really do anything."));

	icon = NewIcon(right - ICONSIZE / 2, top - ICONSIZE / 2, ICONCOMPUTER,
		"Host");
	PrefixList(contents, icon);
	SetVar(icon, PARENT, (ObjPtr) controlWindow);
	SetVar(icon, STICKINESS, NewInt(STICKYLEFT + STICKYTOP));
	SetVar(icon, HELPSTRING, NewString("This icon represents the remote computer \
that you are connected to.  It's part of a graphic that shows that the window \
is a network connection window.  Other than that, it doesn't really do anything."));

	/*Put a switch between the icons*/
	sw = NewFlowLine(left + ICONSIZE + MINORBORDER, right - ICONSIZE - MINORBORDER, bottom, top, "Graphic Flow Line");
	PrefixList(contents, sw);
	SetVar(sw, PARENT, (ObjPtr) controlWindow);
	SetVar(sw, STICKINESS, NewInt(STICKYLEFT + STICKYTOP));

	/*Add the info description*/
	left = l + 2 * MAJORBORDER + SCGRAPHICWIDTH;
	right = r - MINORBORDER;
	top = t - MINORBORDER;
	bottom = top - SCTEXTHEIGHT;

	s = tempStr;
	sprintf(s, "Network Connection\nHost: ");
	while (*s) ++s;

	var = GetVar(connection, NAME);
	if (var)
	{
	    sprintf(s, "%s\n", GetString(var));
	}
	else
	{
	    sprintf(s, "Unknown\n");
	}
	while (*s) ++s;

	/*Create the text box*/
	textBox = NewTextBox(left, right, 
			 bottom, top,
			 0, "Info Text", tempStr);
	PrefixList(contents, textBox);
	SetVar(textBox, PARENT, panel);
	SetVar(textBox, TEXTFONT, NewString(SCFONT));
	SetVar(textBox, TEXTSIZE, NewInt(SCFONTSIZE));
	SetVar(textBox, STICKINESS, NewInt(STICKYLEFT + STICKYRIGHT + STICKYTOP));

	/*Make an icon corral*/
	top = bottom - MAJORBORDER;
	left = MINORBORDER;
	right = r - MINORBORDER;
	bottom = b + 2 * MINORBORDER + BUTTONHEIGHT;
	corral = NewIconCorral(NULLOBJ, left, right, bottom, top, BARRIGHT + BARBOTTOM);
	SetVar(corral, STICKINESS, NewInt(STICKYLEFT + STICKYRIGHT +
				     STICKYBOTTOM + STICKYTOP));
	SetVar(corral, TOPDOWN, ObjTrue);
	SetVar((ObjPtr) controlWindow, CORRAL, corral);
	SetVar(corral, NAME, NewString("Processes Corral"));
	SetVar(corral, HELPSTRING,
	    NewString("This corral contains icons for all the processes running on the \
host machine.  You can show the controls of a process by selecting it and pressing \
the Show Controls button."));
	PrefixList(contents, corral);
	SetVar(corral, PARENT, panel);

	bottom = b + MINORBORDER;
	top = bottom + BUTTONHEIGHT;
	right = (l + r) / 2;

	/*Make a show controls button*/
	button = NewFunctionButton(controlWindow,
		left, right, bottom, top, OF_CONNECT_TO_PROCESS); 
	if (button)
	{
	    SetVar(button, PARENT, panel);
	    SetVar(button, STICKINESS, NewInt(STICKYBOTTOM + STICKYLEFT + FLOATINGRIGHT));
	    PrefixList(contents, button);
	}
    }

    return (ObjPtr) controlWindow;
}
#endif

/* any vars of object that are gotten in here **MUST** have been set locally! */
void WaitForNetObjectVar(object, varNode)
ObjPtr object;
VarsPtr varNode;
{
#ifdef SOCKETS
    ObjPtr connection, process, allObjects;
    ThingListPtr thing;
    ObjPtr oldThing;
    double time;

    connection = GetVar(object, OWNERCONNECTION);

    if (!connection)
    {
	ReportError("WaitForNetObjectVar", "network object has no OWNERCONNECTION");
#ifdef DEBUG
	if (IsRemote(object))
	{
	    fprintf(stderr, "Yeah, it's remote\n");
	}
	else
	{
	    fprintf(stderr, "Hey! it's not really remote!\n");
	}
#endif
    }
    else
    {
#if 0
	/* this is a REALLY terrible hack. Relies on internal stuff in IdleConnection */
	/* Also relies on ReceiveObject keeping the CPU until it gets the whole object. */

	process = GetVar(connection, OWNERPROCESS);
	allObjects = GetVar(process, ALLOBJECTS);
	thing = LISTOF(allObjects);

	if (thing)
	    oldThing = thing -> thing;
	else
	    oldThing = (ObjPtr) 0;

	time = WallClock();
	while (WallClock() < time + 60.0 && GetConnectionState(connection) != CS_READ_OBJECT)
	{
	    IdleAllConnections();
	}

	/* Also relies on the new object being at the top of the list */
	IdleConnection(connection);

	process = GetVar(connection, OWNERPROCESS);
	allObjects = GetVar(process, ALLOBJECTS);
	thing = LISTOF(allObjects);
	if (thing -> thing != oldThing)
	{
	    varNode -> value = thing -> thing;
	    varNode -> chCount = ++anotherDamnReferenceCount;
	    /* Another hack */
	    DeleteFromList(allObjects, thing -> thing);
	}
	else
	{
#ifdef DEBUG
	    fprintf(stderr, "Doesn't look like we got remote var %s\n",
			GetInternalString(varNode -> name));
#endif
	}
#else
	if (!varNode)
	{
	    ReportError("WaitForNetObjectVar", "waiting for nothing!");
	    return;
	}

	if (!IsNetworkWaiting(varNode -> value))
	{
	    varNode -> value = NetworkWaiting;
	    SendGetVarMessage(object, varNode -> name);
	}

	time = WallClock();
	while (WallClock() < time + 60.0 && IsNetworkWaiting(varNode -> value))
	{
	    IdleAllConnections();
	}
#endif
    }
#endif
}

void InitSockets(void)
/*Initializes the socket system*/
{
#ifdef SOCKETS
    advertiseableClass = NewObject(NULLOBJ, 0L);
    SetMethod(advertiseableClass, ADVERTISE, AdvertiseObject);
    SetMethod(advertiseableClass, UNADVERTISE, UnadvertiseObject);
    allAdvertised = NewList();
    AddToReferenceList(advertiseableClass);
    AddToReferenceList(allAdvertised);

    connectionClass = NewObject(NULLOBJ, 0L);
    AddToReferenceList(connectionClass);
    SetMethod(connectionClass, TIMEOUT, TimeoutConnection);
    SetMethod(connectionClass, CLEANUP, CleanupConnection);
    SetMethod(connectionClass, NEWCTLWINDOW, ShowConnectionControls);

    /*Make class for processes*/
    processClass = NewObject(NULLOBJ, 0L);
    AddToReferenceList(processClass);
    SetMethod(processClass, NEWCTLWINDOW, ShowProcessControls);
    SetMethod(processClass, CONNECTTOPROCESS, ConnectToProcess);

    /*Make icons for process*/
    processIcon = NewIcon(10, 10, ICONPROCESS, "Process");
    AddToReferenceList(processIcon);

    allConnections = NewList();
    AddToReferenceList(allConnections);
#endif
}

void KillSockets(void)
/*Kills the socket system*/
{
#ifdef SOCKETS

    RemoveFromReferenceList(allConnections);
    RemoveFromReferenceList(processIcon);
    RemoveFromReferenceList(processClass);
    RemoveFromReferenceList(connectionClass);
    if (lastComputer)
    {
	free(lastComputer);
	lastComputer = 0;
    }
    RemoveFromReferenceList(allAdvertised);
    RemoveFromReferenceList(advertiseableClass);
#endif

}
