/* Name: echoascii - Echo binary format to ascii filter */

static char SccsId[] = "@(#)echoascii.c	1.10";

/* Synopsis:	echoascii [-ncvit] imagefile [asciifile]
 *
 * Description:	Echoascii reads a echo binary format file from the standard
 *		input and writes an ascii translation to the standard output.
 *		The output may be of two forms (see options below):
 *
 *		theta y [radius VOID]
 *		theta y radius
 *
 *		Where 'theta' is a integer value in microradian units
 *		(0...2pi); 'y' is the displacement along the cylindrical
 *		axis in microns; and 'r' is the radius of the point in microns.
 *		Output is sorted by increasing theta and then increasing y.
 *
 *		The data file may contain void values where either no
 *		data was recorded or the image has been trimmed.
 *		This program may be directed with the -v switch to
 *		indicate these points by outputting the radius as the
 *		word 'VOID'. Otherwise these points are silently skipped.
 *
 *		A line giving the name and some stats is output to stderr.
 *
 *		Options:
 *
 *		-n suppress documentation.
 *		-c use cartesian coordinates.
 *		-v use the string "VOID" to show void data points.
 *		-i data input to the image, format must be identical.
 *		-t outputs theta in radians, y and radius in tenths of mm.
 *
 *		When using the -i switch the format must exactly match the
 *		cylindrical data format used to extract the same data points.
 *		Because the echo system requires that the radii fall on exact
 *		y and theta coordinates any conversion from cartesian will
 *		probably require interpolation to find r at the fixed theta.
 *		In this mode the values of theta and y must be present but are
 *		ignored.
 *
 *		Compiler: cc -o echoascii echoascii.c cyfile.c -lm
 *
 *		int	32 bit signed
 *		short	int 16 bit signed
 *		float	32 bit floating point
 *
 *		Otherwise assumes Unix and the Standard I/O Library.
 *		Backward byte processors (Intel, DEC, etc) will need to swap
 *		bytes in the header and data portions. Floats and dates are
 *		probably not portable and should be ignored.
 *
 * Programmers: If you make use of this code or an adaptation of it you will
 *		have some assurance of compatibility with future images.
 *		Further, it is very wise to use the header variables
 *		ltincr, lgincr, and rshift as they vary across different
 *		configurations of the digitizer.
 *
 *		NOT portable to byte-swapped processors.
 */

#include <stdio.h>
#include <fcntl.h>
#include <math.h>
#include "cyfile.h"

#define real float
extern int errno;
extern void perror();

char nodoc_switch = 0;		/* flag, -n switch */
char cartesian_switch = 0;	/* flag, -c switch */
char void_switch = 0;		/* flag, -v switch */
char input_switch = 0;		/* flag, -i switch */
char tenths_switch = 0;		/* flag, -t switch */

char *usage = "usage: echoascii [-ncvit] imagefile [asciifile]";


#if 1

main(argc, argv)

int argc;
char *argv[];
{
	int fd;				/* image file descriptor */
	GSPEC *gs;			/* file header buffer base pointer */
	int lt, lg;			/* data array subscripts */
	long int theta, y, radius;	/* coordinates, cylindrical */
	long int x, z;			/* coordinates, cartesian */
	float sin_theta, cos_theta;	/* time savers */
	char *image_pathname = 0;	/* pathname of target image */

	/* interpret command line switches */
	if (argc < 2) {
		puts(usage);
		exit(-1);
	}
	while (--argc) {
		if (*argv[argc] == '-') {
			while (*++argv[1] != 0) {
				switch (*argv[1]) {
				default:
					puts(usage);
					exit(-1);
				case 'n':
					nodoc_switch = 1;
					break;
				case 'c':
					cartesian_switch = 1;
					break;
				case 'v':
					void_switch = 1;
					break;
				case 'i':
					input_switch = 1;
					break;
				case 't':
					tenths_switch = 1;
					break;
				}
			}
		} else {
			if (image_pathname != 0) {
				puts(usage);
				exit(-1);
			} else {
				image_pathname = argv[argc];
			}
		}
	}

	/* read the image file */
	if ((fd = open(image_pathname, O_RDWR)) == -1) {
		perror("echoascii");
		exit(-1);
	}
	if ((gs = cyread(NULL, fd)) == NULL) {
		exit(-1);
	}

	/* display image name for verification */
	if (!nodoc_switch) {
		fprintf(stderr, "image: %s ", gs->name);
		fprintf(stderr, "hdrsiz=%d ltincr=%d\n",
			sizeof(GSPEC), gs->ltincr);
		fprintf(stderr, "ltrange=%d:%d lgrange=%d:%d\n",
		    gs->ltmin, gs->ltmax, gs->lgmin, gs->lgmax);
	}

	if (!input_switch) {
		/* output as ascii mode */
		if (!cartesian_switch) {
			/* output each point as an ascii cylindrical point */
			theta = gs->lgmin * gs->lgincr;
			for (lg = gs->lgmin; lg <= gs->lgmax; ++lg) {
				y = gs->ltmin * gs->ltincr;
				for (lt = gs->ltmin; lt <= gs->ltmax; ++lt) {
					radius = GETR(gs, lt, lg);
					output(gs, theta, y, radius);
					y += gs->ltincr;
				}
				theta += gs->lgincr;
			}
		} else {
			/* output each point as an ascii cartesian point */
			theta = gs->lgmin * gs->lgincr;
			for (lg = gs->lgmin; lg <= gs->lgmax; ++lg) {
				y = gs->ltmin * gs->ltincr;
				cos_theta = cos(URTOR(-theta));
				sin_theta = sin(URTOR(theta));
				for (lt = gs->ltmin; lt <= gs->ltmax; ++lt) {
					radius = GETR(gs, lt, lg);
					if (radius != VOID) {
						x = sin_theta * radius;
						z = cos_theta * radius;
					} else {
						x = VOID;
						z = VOID;
					}
					output(gs, x, y, z);
					y += gs->ltincr;
				}
				theta += gs->lgincr;
			}
		}
	} else {
		/* input from ascii mode */
		for (lg = gs->lgmin; lg <= gs->lgmax; ++lg) {
			for (lt = gs->ltmin; lt <= gs->ltmax; ++lt) {
				if (scanf("%*d%*d%d", &radius) == 1) {
					PUTR(gs, lt, lg, radius);
				} else {
					PUTR(gs, lt, lg, VOID);
				}
			}
		}
		cywrite(gs, fd);
	}
	cyfree(gs);
	close(fd);
	exit(0);
}



output(gs, a, b, c)

GSPEC *gs;
long int a, b, c;
{
	if (!void_switch && (a == VOID || b == VOID || c == VOID)) {
		return;
	}

	if (tenths_switch) {
		if (cartesian_switch) {
			/* output x as tenths */
			if (a != VOID) {
				printf("%ld", a / 100);
			} else {
				printf("VOID");
			}
		} else {
			/* output theta as radians */
			if (a != VOID) {
				printf("%.3f", (float)a * 1e-6);
			} else {
				printf("VOID");
			}
		}
	} else {
		if (cartesian_switch) {
			/* output x as microns */
			if (a != VOID) {
				printf("%ld", a);
			} else {
				printf("VOID");
			}
		} else {
			/* output theta as microradians */
			if (a != VOID) {
				printf("%ld", a);
			} else {
				printf("VOID");
			}
		}
	}
	/* output y */
	if (b != VOID) {
		if (tenths_switch) {
			b = b / 100;	/* convert to tenths */
		}
		printf(" %ld", b);
	} else {
		printf(" VOID");
	}
	/* output radius or z */
	if (c != VOID) {
		if (tenths_switch) {
			c = c / 100;	/* convert to tenths */
		}
		printf(" %ld", c);
	} else {
		printf(" VOID");
	}
	putchar('\n');
}

#else

#include <ctype.h>

char exp_switch = 0;		/* flag, -e switch */

/* char *usage = "usage: echoascii [-ncvite] imagefile [asciifile]"; */

main(argc, argv)

int argc;
char *argv[];
{
	int fd;							/* image file descriptor */
	GSPEC *gs;						/* file header buffer base pointer */
	char *format = "%*d%*d%s";		/* control string for scanf */
	char r_alpha[32];				/* r from asciifile (+? "VOID"s) */
	int lt, lg, cnt;				/* data array subscripts */
	long int theta, radius; 		/* coordinates, cylindrical */
	long int x, y, z;				/* coordinates, cartesian */
	float sin_theta, cos_theta;		/* time savers */
	char *image_pathname = 0;		/* pathname of target image */

	/* interpret command line switches */
	if (argc < 2) {
		puts(usage);
		exit(-1);
	}
	while (--argc > 0) {
		if (*argv[argc] == '-') {
			while (*++argv[argc] != 0) {
				switch (*argv[argc]) {
				default:
					puts(usage);
					exit(-1);
				case 'n':
					nodoc_switch = 1;
					break;
				case 'c':
					cartesian_switch = 1;
					break;
				case 'v':
					void_switch = 1;
					break;
				case 'i':
					input_switch = 1;
					break;
				case 't':
					tenths_switch = 1;
					break;
				case 'e':
					exp_switch = 1;
					break;
				}
			}
		} else {
			if (image_pathname == 0) {
				image_pathname = argv[argc];
			} else {
				if (strcmp(argv[argc], "echoascii") != 0) {
					puts(usage);
					exit(-1);
				}
			}
		}
	}

	/* read the image file */
	if ((fd = open(image_pathname, O_RDWR)) == -1) {
		perror("echoascii");
		exit(-1);
	} else {
		;
	}
	if ((gs = cyread(NULL, fd)) == NULL) {
		exit(-1);
	} else {
		;
	}

	/* display image name for verification */
	if (!nodoc_switch) {
		fprintf(stderr, "image: %s ", gs->name);
		fprintf(stderr, "hdrsiz=%d ltincr=%d\n",
			sizeof(GSPEC), gs->ltincr);
		fprintf(stderr, "ltrange=%d:%d lgrange=%d:%d\n",
		    gs->ltmin, gs->ltmax, gs->lgmin, gs->lgmax);
	} else {
		;
	}

	if (!input_switch) {
		/* output as ascii mode */
		if (!cartesian_switch) {
			/* output each point as an ascii cylindrical point */
			theta = gs->lgmin * gs->lgincr;
			for (lg = gs->lgmin; lg <= gs->lgmax; ++lg) {
				y = gs->ltmin * gs->ltincr;
				for (lt = gs->ltmin; lt <= gs->ltmax; ++lt) {
					radius = GETR(gs, lt, lg);
					output(gs, theta, y, radius);
					y += gs->ltincr;
				}
				theta += gs->lgincr;
			}
		} else {
			/* output each point as an ascii cartesian point */
			theta = gs->lgmin * gs->lgincr;
			cnt = 0;
			for (lg = gs->lgmin; lg <= gs->lgmax; ++lg) {
				y = gs->ltmin * gs->ltincr;
				cos_theta = cos(URTOR(-theta));
				sin_theta = sin(URTOR(theta));
				for (lt = gs->ltmin; lt <= gs->ltmax; ++lt) {
					radius = GETR(gs, lt, lg);
					if (radius != VOID) {
						x = sin_theta * radius;
						z = cos_theta * radius;
					} else {
						x = VOID;
						z = VOID;
					}
					output(gs, ++cnt, x, y, z);
					y += gs->ltincr;
				}
				theta += gs->lgincr;
			}
		}
	} else {
		/* input from ascii mode */
		for (lg = gs->lgmin; lg <= gs->lgmax; ++lg) {
			for (lt = gs->ltmin; lt <= gs->ltmax; ++lt) {
				if (scanf(format, r_alpha) == 1) {
					if (isdigit(*r_alpha)) {
						radius = atoi(r_alpha);
						PUTR(gs, lt, lg, radius);
					} else {
						PUTR(gs, lt, lg, VOID);
					}
				} else {
					puts("format conflict?");
					return(-1);
				}
			}
		}
		cywrite(gs, fd);
	}
	cyfree(gs);
	close(fd);
	exit(0);
}



output(gs, cnt, a, b, c)

GSPEC *gs;
long int a, b, c;
int cnt;
{
	if (!void_switch && (a == VOID || b == VOID || c == VOID)) {
		return;
	}

	if (exp_switch || tenths_switch) {
		if (cartesian_switch) {
			printf("%d", cnt);
			if (a != VOID) {
				if (tenths_switch) {
					/* output x as tenths */
					printf(" %.1d", a / 100);
				} else {
					/* output x to 3 decimals in exponential notation */
					printf(" %.3e", (float)(a / 100));
				}
			} else {
				printf("VOID");
			}
		} else {
			/* output theta as radians */
			if (a != VOID) {
				printf("%.3f", (float)(a * 1e-6));
			} else {
				printf("VOID");
			}
		}
	} else {
		if (cartesian_switch) {
			/* output x as microns */
			if (a != VOID) {
				printf("%ld", a);
			} else {
				printf("VOID");
			}
		} else {
			/* output theta as microradians */
			if (a != VOID) {
				printf("%ld", a);
			} else {
				printf("VOID");
			}
		}
	}
	/* output y */
	if (b != VOID) {
		if (exp_switch || tenths_switch) {
			if (tenths_switch) {
				/* output x as tenths */
				printf(" %.1d", b / 100);
			} else {
				/* output x to 3 decimals in exponential notation */
				printf(" %.3e", (float)(b / 100));
			}
		}
	} else {
		printf(" VOID");
	}
	/* output radius or z */
	if (c != VOID) {
		if (exp_switch || tenths_switch) {
			if (tenths_switch) {
				/* output x as tenths */
				printf(" %.1d", c / 100);
			} else {
				/* output x to 3 decimals in exponential notation */
				printf(" %.3e", (float)(c / 100));
			}
		}
	} else {
		printf(" VOID");
	}
	putchar('\n');
}

#endif
