/* automata.d/src file kb.c */
/* This is the main file dealing with the Knuth-Bendix process.
*/
#include <stdio.h>
#include "defs.h"
#include "list.h"
#include "word.h"
#include "input.h"
#include "lg.h"
#include "kb.h"

extern list rels;
extern lg* archive;
extern lg* to_add;
extern lg * being_refined;
extern boolean print_kb_rules;
extern list * to_be_considered;
extern list tree_list;
extern FILE * wfile;

extern int max_archive_size;
int num_rules = 0; /*The number of current Knuth-Bendix rules*/
int max_num_rules = 0;/*During the Knuth-Bendix process, rules are
sometimes added and sometimes deleted. In this variable we record the
maximum number of rules reached at any point during the process.*/
int archive_size_last_word_diff = 1; /*|archive| is the name of the
structure in which we keep all words so far come across. The |archive|
is only cleared out very occasionally, if at all. This variable
records the size of the archive at the time the last new word difference
was discovered. We use this to decide when the calculation has gone on
long enough since the last word difference was discovered. At the
moment the Knuth-Bendix process is brought to a halt either when a
(finite) confluent system is found (i.e. |to_be_considered| is an
empty list), or else we insist that the |archive| be at least a certain size and
furthermore that it size has doubled since the last word difference.  */

static void mk_rule PARMS((word *, word *));
static void overlap_with_xX PARMS((word* lhs_ptr));
static void rebuild PARMS ((VOID));
static void find_confluent_pairs PARMS((vindex v,word*leftp,word*rightp,int
mode));
static vindex max_suffix PARMS((lg * lgp,word *wp, int tree_type));
static void suffix_erase PARMS((word * wp,vindex vsuf, word * prefix));
static void consider PARMS ((vindex));
static void unmark_in_archive PARMS((vindex v, vindex inv_v));
static void lhs_unmark PARMS((lg * lgp, word * wp, word * invp));
static vindex smallest_con_leaf PARMS((vindex v));
static vindex biggest_con_leaf PARMS((vindex v));
static void remove_bad_lhss PARMS((vindex smallest,vindex biggest));
static vindex just_smaller PARMS ((vindex));
static vindex just_bigger  PARMS ((vindex));
static void rebuild_after PARMS ((VOID));
static void 
run_thru_overlaps PARMS
	((word* first_stringp, vindex overlap, word* rightp,int mode));
#define WEIGHTING 1 
/*We want the rules which have produced new word
differences to be given priority, so that they are used to generate
overlaps with other left hand sides earlier than they might otherwise
have been. If |WEIGHTING| is small, then the
extra credit given for producing new word differences is small.*/



/* This procedure converts into KB rules the relators which are presented to the
program by the user. The program has previously formed these into a
list called |rels|.
*/
void
rels2KB()
{
	word w;
	word triv; /*the empty string*/
	list_traverser trels;
	/*Initialize*/
	list_traverser_init(&trels,&rels);
	word_init(&w);
	word_init(&triv);
	while (list_next(&trels,(dp)&w)) {
		(void)reduction(&w,&w);/*reduce using current KB rules*/
		if (word_sgn(&triv,&w) > 0) /*|w| is bigger than the trivial
					string, i.e. |w| is non-trivial.*/
		mk_rule(&w,&triv); /*make a rule saying that |w| should be
				replaced by the empty string*/
	}
	rebuild(); /*The main thing done here is to compute the failure
		pointers for the KMP algorithm*/
	/*Clean up*/
	word_clear(&triv);
	word_clear(&w);
	list_traverser_clear(&trels);
}


/* Install the rule |lhs->rhs|.
*/
static void
mk_rule(lhs,rhs)
	word *lhs;
	word *rhs;
{
	vindex lhsv = wtree_word_insert(archive,lhs);
	vindex rhsv = wtree_word_insert(archive,rhs);
	vindex t = UNDEFINED;
	word inv_lhs;
	vindex inv_v = UNDEFINED;
	int l = word_length(lhs) - word_length(rhs);
	int n = 0;
	assert(word_sgn(lhs,rhs) == -1); /*the left hand side really is
	greater than the right hand side, with respect to the chosen ordering*/
	assert(rhsv == get_rhs(rhsv)); /* This assert
		statement is saying that there is no immediate known way of
		reducing the right hand side.  */
	num_rules++;
	if (num_rules > max_num_rules)
		max_num_rules = num_rules;
	set_rhs(lhsv,rhsv); /*We record the fact that there is an
		immediate reduction of left hand side to the right hand side.
		We can use this to help during future reductions*/
	set_status_on_tree(lhsv,LHS,TRUE);
	/*We have to keep track of the inverses of left hand sides. This
	is because the KMP structure (failure pointers) is convenient for
	finding each suffix of our present |lhs| which is equal to a prefix of
	some other left hand side, but not for finding each prefix of
	|lhs| which is equal to a suffix of some other left hand side. We
	interchange prefixes and suffixes by using the formal inverses of
	the strings.
	*/
	word_init(&inv_lhs);
	word_inv(lhs,&inv_lhs);
	inv_v = wtree_word_insert(archive,&inv_lhs);
	set_status_on_tree(inv_v,INV_LHS,TRUE);
	/*|to_add| is a KMP tree consisting
		of the strings and rules found since the last
	call of |rebuild()|. This store of words and rules is incorporated
	as soon as convenient into one of the KMP trees of |tree_list|. In
		the mean time, information in |to_add| will also be used to make
		reductions. */
	t = wtree_word_insert(to_add,lhs);
	set_status_on_tree(t,LHS,TRUE);
	/* Recall that |archive| does not contain failure pointer information.
		Getting hold of a suffix of |lhs| entails getting
		hold of a prefix of its inverse, and this means we should also
	put the inverse into |to_add|, where failure pointers ARE computed.
	*/
	inv_v = wtree_word_insert(to_add,&inv_lhs);
	set_status_on_tree(inv_v,INV_LHS,TRUE);
	word_clear(&inv_lhs);
	 /*Check all overlaps with relators of the form xX*/
	overlap_with_xX(lhs);
	if (print_kb_rules){
	/*Print out the new rule*/
		fprintf(wfile,"\t# ");
		word_print(wfile,lhs);
		fprintf(wfile,"->");
		word_print(wfile,rhs);
		fprintf(wfile,"\n");
	}
	/*If the lengths of |lhs| and |rhs| differ by at most 2, then
	we compute the word differences which result, denoting  by |n| the
	number of new word differences. (If the difference in lengths
	is greater than 2, this rule will be replaced almost
	immediately by another rule where the lengths differ by less,
	and which immediately implies this one. Therefore we do not compute word
	differences in that case.) We also record the size of |archive| if
	n > 0. This is used to decide when to bring the computation
	to a halt when the group has not been found to be confluent
	(i.e. when we think that all word differences have
		probably now been found).*/
	if (l <= 2  && (n=wdiffs_from_wpair(lhs,rhs)) > 0) {
		archive_size_last_word_diff = get_num_verts(archive);
		set_priority(lhsv,get_priority(lhsv) - n*WEIGHTING);
	}
/*Add this left hand side to the list of left hand sides which need to be
		compared with other left hand sides in the Knuth-Bendix
		process.*/
	(void)list_insert(to_be_considered,(dp)&lhsv);
}


/* We need to
look at the case where alpha beta = Xx and the case where beta gamma = Xx.
We do not need to worry about both of these occurring simultaneously,
because both ways of reducing XxX give the answer X.
The next procedure looks for overlaps of the word pointed to by
|lhs_ptr| with such trivial words of length 2, and takes appropriate
action.
*/

static void
overlap_with_xX(lhs_ptr)
	word *lhs_ptr;
{
	gen x=INVALID_GEN;
	word temp1, temp2;
	word rhs;
	gen X = INVALID_GEN;
	int n;
	assert(lhs_ptr);
	assert(!word_empty(lhs_ptr));
	word_init(&temp1);
	word_init(&temp2);
	word_init(&rhs);
	/*first the case where the trivial relator overlaps on the right*/
	word_cpy(lhs_ptr,&temp1);
	/*copy because later steps change the contents*/
	(void)reduction(lhs_ptr,&rhs); /*first step in first way of reducing*/
	word_cpy(&rhs,&temp2); 
	(void)word_delget_first(&temp1,&x);
	/*first step in second way of reducing*/
	X = inv(x);
	word_put_first(&temp2,X); /*first way*/
	reduction(&temp2,&temp2); /*first way*/
	reduction(&temp1,&temp1); /*second way*/
	 /*Have we found a new rule? If so, install it.*/
	common_prefix_erased(&temp1,&temp2);
	common_suffix_erased(&temp1,&temp2);
	if ((n = word_sgn(&temp1,&temp2)) > 0) /*|temp2| is bigger*/
		mk_rule(&temp2,&temp1);
	else if (n < 0)
		mk_rule(&temp1,&temp2);
/*this completes the case where the trivial relator overlaps on the right*/
	/*now the case where the trivial relator overlaps on the left*/
	word_cpy(lhs_ptr,&temp1); /*later steps change the contents*/
	(void)word_delget_last(&temp1,&x);
	/*first step in first way of reducing*/
	word_cpy(&rhs,&temp2);
	X = inv(x);
	word_put_last(&temp2,X); /*first step in second way of reducing*/
	reduction(&temp1,&temp1); /*first way*/
	reduction(&temp2,&temp2); /*second way*/
	 /*Have we found a new rule? If so, install it.*/
	common_prefix_erased(&temp1,&temp2);
	common_suffix_erased(&temp1,&temp2);
	if ((n = word_sgn(&temp1,&temp2)) > 0) /*|temp2| is bigger*/
		mk_rule(&temp2,&temp1);
	else if (n < 0)
		mk_rule(&temp1,&temp2);
/*this completes the case where the trivial relator overlaps on the left*/
	word_clear(&rhs);
	word_clear(&temp1);
	word_clear(&temp2);
}


/* We start the next procedure with a KMP tree |to_add|. The object is
to incorporate the data into |tree_list|, a list of KMP trees which
contain all the rules we have so far found.
Having amalgamated as necessary with trees in |tree_list|,
we change the name of |to_add|
to |being_refined|. (In the course of this procedure, a new |to_add| is
likely to spring up, so the name has to be changed to avoid
confusion.)
We then compute the KMP structure on the labelled graph pointed to by
|being_refined|. The problem is that if more vertices are added to the labelled
graph, the previously computed KMP structure is no longer correct. So
the KMP structure needs first to be removed
(which is done in |refine_without_looping()|) and then recomputed.
*/
static void
rebuild()
{
	assert(to_add);
	while (non_trivial(to_add)) {
	/*first add on all the graphs in |tree_list| of roughly the same size*/
		boolean absorbing = TRUE;
		while (absorbing) {
			lg *absorb = 0;
			list_traverser tt_l;/*traverser of |tree_list|*/
			absorbing = FALSE;
			list_traverser_init(&tt_l,&tree_list);
			while (list_next(&tt_l,(dp)&absorb)) {
				assert(non_trivial(absorb)); 
			/*as we run through |tree_list|, |absorb| keeps on
					getting smaller*/
			if (2*get_num_verts(to_add) > get_num_verts(absorb))
						break;
				absorb = 0;
			}
			list_traverser_clear(&tt_l);
			if (absorb) {
				(void)list_delete(&tree_list,(dp)&absorb);
				assert(get_type(absorb) == KMP_WTREE);
				kill_and_devour(to_add,absorb);
				absorb = 0;
				absorbing = TRUE;
			}
		} /*All amalgamations have now been completed*/
		if (non_trivial(to_add)) {
			set_type(to_add,KMP_WTREE);
		/* If there have been changes in the structure of |to_add|
this type may have changed, and needs resetting */
			being_refined = to_add;
			to_add_create();
			refine();
			/*check over |being_refined| carefully, to see
			whether we want to keep all the current rules in
			|being_refined|. Prune away unnecessary parts of
			|being_refined|. Failure pointers are recalculated.  */
			if (non_trivial(being_refined))
			(void)list_insert(&tree_list,(dp)&being_refined);
		else { /*|being_refined| is trivial and should be thrown away*/
				lg_clear(being_refined);
				Free_dp((dp)being_refined);
			}
			being_refined = 0;
		}
	}
}


/*
We deal with the left hand sides (or long strings) one by one. These
are extracted from the list |to_be_considered|. If |to_be_considered|
becomes empty, then the (finite) set of Knuth-Bendix rules we have
found is confluent, and this is reported to the user. The procedure
|consider()| will usually add to the list |to_be_considered|, and will
cause a number of other changes to take place.
*/
void
deal_with(which_tree)
	int which_tree;
{
	vindex u;
	assert(which_tree == LHS || which_tree == LS);
		/*left hand sides or long strings*/ 
	if (which_tree == LS) {
		fprintf(wfile,"\t# For now, we are not computing long strings\n");
		return;
	}
	while (list_delget_first(to_be_considered,(dp)&u)) {
	/*|u| represents the left hand side with the lowest priority score*/
#ifdef DEBUG
	{
	word lhs;
	word_init(&lhs);
	vindex2word(u,&lhs);
		set_priority(u,word_length(&lhs)); /*The priority for most
		vertices of a KMP tree is equal to the length of the string
			it represents. However, when |u| represents a left hand
			side of a rule which gives rise to new word differences,
			then the priority is affected by the number of word
			differences found. This is just what we want.
		*/
	word_clear(&lhs);
	}
#endif DEBUG
		consider(u); /*We compare the left hand side corresponding to
			|u| to all left hand sides already considered. This may
		result in new Knuth-Bendix rules being found, and in other
			changes to the structures in the program.
		*/
		assert(to_add); /*|to_add| is the KMP tree which contains all
			the new information found by considering |u|.  */
		if (non_trivial(to_add))
		rebuild(); /*Arrange for the information in |to_add| to be
				inserted into the trees in |tree_list|.*/
		if (list_empty(to_be_considered))
		rebuild_after(); /*Amalgamate all the trees in |tree_list|
					into one tree.*/
		 /*We stop if it seems as though all word differences have
				been found.*/
		if (get_num_verts(archive) > max_archive_size &&
			get_num_verts(archive) > 2*archive_size_last_word_diff)
				break;
	}
	if (list_empty(to_be_considered))
		fprintf(wfile,"\t# Rules are confluent.\n");
	else
	fprintf(wfile,"\t# We don't know if the rules are confluent or not.\n");
}


/* The vertex |u| is a vertex of |archive|, which
represents the left hand side of some Knuth-Bendix rule.
We first apply the procedure |reject(u)|. This decides whether the rule in
question should continue to be a rule. |consider()| only really comes
into effect if this initial test is passed.

We have to find left hand sides of rules which overlap either on the
left or on the right with this rule.  |lhs|
is the string represented by |u|.
*/
#define STANDARD 0
#define INVERSE 1
static void
consider(u)
	vindex u;
{
	if (!reject(u)) { /*We decide whether the Knuth-Bendix rule
			represented by |u| should continue to be a rule.*/
		word lhs;
		word inv_lhs;
		word rhs ;
		word inv_rhs;
		vindex inv_u = UNDEFINED;
		vindex rhsv = get_rhs(u); /*the vertex representing the right
		hand side of the rule, whose left hand side is represented by
			|u|*/
		assert(get_status(u,LHS));
		assert(rhsv != u); /*we do actually have a rule*/
		word_init(&lhs);
		word_init(&rhs);
		word_init(&inv_lhs);
		word_init(&inv_rhs);

	 /*Deal with overlaps on the right of |lhs|.*/
		vindex2word(u,&lhs);
		vindex2word(rhsv,&rhs);
		find_confluent_pairs(u,&lhs,&rhs,STANDARD);

	 /*Deal with overlaps on the left of |lhs|.*/
		word_inv(&lhs,&inv_lhs);
		inv_u=lg_word_find(archive,&inv_lhs);
		assert(inv_u != UNDEFINED);
		word_inv(&rhs,&inv_rhs);
		find_confluent_pairs(inv_u,&inv_lhs,&inv_rhs,INVERSE);
		word_clear(&inv_rhs);
		word_clear(&rhs);
		word_clear(&inv_lhs);
		word_clear(&lhs);
	}
}



/* We are given a vertex |v| in |archive| which represents the left
hand side |leftp|. |leftp| is either a left hand side (in which case
|mode == STANDARD|), or the inverse of a left hand side (in which
case |mode == INVERSE|). If |mode == STANDARD|, we find all suffixes
of |*leftp| equal to prefixes of considered left hand sides. If
|mode == INVERSE|, we find all suffixes of |*leftp| equal to prefixes
of inverses of considered left hand sides. In the second case, the
strings found must then be inverted to get hold of the actual
strings of interest.


*/
static void
find_confluent_pairs(v,leftp,rightp,mode)
	vindex v;
	word * leftp; /* points to word corresponding to v */
	word * rightp;
	int mode; /* equals |STANDARD| or |INVERSE| */
{
	lg * lgp=0;
	list overlaps; /* Various vertices in |archive| will represent overlaps
		during the current procedure. |overlaps| is a list of all
		these vertices. 
	*/
	list_traverser tt_l; /*for traversing |tree_list|*/
	vindex overlap=UNDEFINED; 	
	vindex bigger = (vindex)UNDEFINED; 
	vindex smaller = (vindex)UNDEFINED; 	
	list_init(&overlaps,VINDEX,FIFO); 
	if (mode == STANDARD) 
		set_status_on_tree(v,CONSIDERED,TRUE); 
	/*we do this before actually considering |v| in order to make sure
		 that we try to find overlaps of the string represented by |v|
			 with itself*/ 
	else {
		assert(mode == INVERSE);
		set_status_on_tree(v,INV_CONSIDERED,TRUE); 
	}
	/*The first job is to maintain the linked list of considered left
	hand sides and inverse left hand sides. We find out where |v|
	should be linked in, and then do the linking.*/
	smaller = just_smaller(v); /*If there are no considered left hand
		sides or inverse left hand sides smaller than |v|,
		|just_smaller(v)| will return |v|, and similarly for
		|just_bigger(v)|.*/
	bigger = just_bigger(v);
	set_next(smaller,v); /*Notice that we would get the wrong linking
		if the statement before this comment were interchanged with the
		statement after it and if |smaller == v != bigger|.*/
	set_next(v,bigger);
	/*Look for overlaps in each tree of |tree_list|.*/
	list_traverser_init(&tt_l,&tree_list);
	while (list_next(&tt_l,(dp)&lgp)) {
		int tree_type;
		if (mode == STANDARD) 
			tree_type = CONSIDERED_TREE;
		else 
			tree_type = INV_CONSIDERED_TREE;
		overlap = max_suffix(lgp,leftp,tree_type); /*this gives the
			vertex representing the maximal suffix of |*leftp| which
			fits into the tree |lgp|, starting at the basepoint.*/
/*
|leftp| is a pointer to a string which is
either a left hand side or the inverse of a left hand side, and this
string is constant within the next fragment. 
We are working in one of the KMP trees in |tree_list|, and |overlap|
is a vertex in this tree, which represents a suffix of |leftp|.
We will have to look for all considered left hand sides or inverses of
considered left hand sides respectively, which have a prefix equal to
this suffix. These will be found not in the tree containing |overlap|,
but in |archive|, which is the best place to look because it contains
all the data.
During this fragment, |overlap| represents a shorter and shorter
suffix of |leftp|, until it disappears entirely, when we are done.
*/
		while (is_basepoint(overlap) == FALSE) {
			/*must be a non-trivial overlap*/
			vindex a_overlap = archive_image(overlap);
		/*Both |overlap| and |a_overlap| represent the same string,
			and this string is a suffix of |leftp|.*/
			if (get_status(a_overlap,tree_type)
				&& get_status(a_overlap,OVERLAP) == FALSE) {
				word alpha;
				word_init(&alpha);
				suffix_erase(leftp,overlap,&alpha);
				run_thru_overlaps(&alpha,a_overlap,rightp,mode);
				/* We are looking
		for reductions of expressions of the form alpha beta gamma,
		where alphabeta is a left hand side and beta gamma is a
		left hand side (or both are inverses of left hand sides).
			|run_thru_overlaps()| finds all possible gamma's.*/
				/*Mark this vertex to make sure
				we don't use this overlap again in
				conjunction with this particular left hand side
		(|*leftp|), and put it on the list |overlaps|, so that we
			can later unmark it and use it again for different left
				hand sides.*/
				set_status(a_overlap,OVERLAP,TRUE);
				(void)list_insert(&overlaps,(dp)&a_overlap);
				word_clear(&alpha);
			}
			overlap = get_failure(overlap);
			/*find the next proper suffix within the graph*/
		}
	}
	list_traverser_clear(&tt_l);
	while (list_delget_first(&overlaps,(dp)&overlap))
		set_status(overlap,OVERLAP,FALSE);
				/*wipe the slate clean before the next call of
				|find_confluent_pairs()|*/
	list_clear(&overlaps);
}



/* |max_suffix| returns the address of the vertex
in |*lgp| corresponding to the longest
proper suffix of |*wp| that can be traced out from the basepoint within
the subtree determined by |tree_type|. We use the Knuth Morris Pratt
failure pointers.
*/
static vindex
max_suffix(lgp,wp,tree_type)
	lg * lgp;
	word * wp;
	int tree_type;
	
{

	vindex v;
	word suffix;
	gen g=INVALID_GEN;
	word_init(&suffix);
	word_cpy(wp,&suffix); /*avoiding changing |*wp|*/
	assert(get_type(lgp) == KMP_WTREE);
	(void)word_del_first(&suffix);/*only interested in overlaps which
			are proper suffixes of |*wp|.*/
	v = basept(lgp);
	while (word_get_first(&suffix,&g)) { /* The induction hypothesis is
			as follows. Let T be the subtree of |lgp| corrresponding
			to |tree_type|, that is the subtree of considered left
	hand sides or the subtree of considered inverses of left hand sides.
			We have factorized |*wp| into alpha beta gamma,
			with |suffix| equal to gamma.
			The string beta is a path in T starting at the
			basepoint and ending at |v|.
			The maximal suffix of alpha beta g, which gives a path
			in T starting at the basepoint, is a suffix of beta g.
			The induction is by lexicographical order of the pair,
		consisting of the length of gamma and the length of beta.
		*/
		vindex t = get(v,g); /*try to advance by one*/
		if (t != UNDEFINED && backg(t) == g &&
				get_status(archive_image(t),tree_type)) {
				/*beta g fits into T?*/
			v = t;
			(void)word_del_first(&suffix); 
		/*Induction step: delete |g| from the
			beginning of \gamma and adjoin it to beta.*/
		}
		else if (is_basepoint(v) == FALSE) /*beta is not trivial*/
			v = get_failure(v); /*The failure pointer in the KMP
			structure gives the maximal proper suffix of the
			string representing |v| which fits into the tree. This
				step replaces beta by a proper suffix
				beta' and alpha by alpha', with
				alphabeta = alpha'beta'. gamma is
				unchanged and beta is shortened.*/
		else
			(void)word_del_first(&suffix);/*The maximal suffix of
			alphabeta g which fits in T is trivial. beta is trivial.
			We shorten gamma by one, lengthen alpha by one
			and leave beta trivial.*/
	}
	word_clear(&suffix);
	return v;
}

/* We start with |vsuf| a vertex of a KMP tree representing a suffix of
the word |*wp|.
Without changing the word pointed to by |wp| or the vertex |vsuf|
(arguments to procedures are never changed in C),
delete the suffix and put the answer into |*prefix|, which is assumed
to start as an initialized word.
*/
static void
suffix_erase(wp,vsuf,prefix)
	word * wp;
	vindex vsuf;
	word * prefix;
{
	gen g = INVALID_GEN;
	gen h = INVALID_GEN;
	word_cpy(wp,prefix); 
	while (!is_basepoint(vsuf)) {
		g = backg(vsuf);
		word_delget_last(prefix,&h);
		assert(h == g);
		vsuf = get(vsuf,inv(g));
		assert(vsuf != UNDEFINED);
	}
}




/* 
|v| is a vertex in |archive| which currently represents a left hand
side of a rule. Should it be retained as a rule?
*/
boolean
reject(v)
	vindex v;
{
	boolean ans = FALSE;
	word lhs, rhs, new_lhs, new_rhs;
	int n;
	word_init(&lhs);
	word_init(&rhs);
	word_init(&new_lhs);
	word_init(&new_rhs);

	vindex2word(v,&lhs);
	vindex2word(get_rhs(v),&rhs);
	assert(word_sgn(&rhs,&lhs) == 1);
	if (reduction(&rhs,&new_rhs))
	ans = TRUE; /*Reduction has taken place. This rule is obsolete.*/
	if(reduction2(&lhs,&new_lhs)) { /*|reduction2()| reduces, but carefully
		avoids using the particular rule we are busy checking up on.*/
		ans = TRUE;
	}
	if (common_prefix_erased(&new_lhs,&new_rhs)
			||common_suffix_erased(&new_lhs,&new_rhs))
		ans = TRUE;
	if ((n = word_sgn(&new_lhs,&new_rhs)) == 0) /*The rule under
	consideration can be dropped because it is implied by other rules.*/
		ans = TRUE;

	if (ans == TRUE) {/*we are going to throw away this rule*/
		vindex inv_v = UNDEFINED;
		word inv_lhs;
		list_traverser tt_l;
		lg *lgp = 0;
		word_init(&inv_lhs);
		word_inv(&lhs,&inv_lhs);
		inv_v = lg_word_find(archive,&inv_lhs);
		unmark_in_archive(v,inv_v);
		/*Remove from |archive| all indications that |v|
		represents a considered left hand side and |inv_v| the inverse
		of a considered left hand side.*/
	/*Remove from all other relevant trees all indications that |lhs|
			is a left hand side and |inv_lhs| is the inverse
				of a left hand side.*/
		lhs_unmark(to_add,&lhs,&inv_lhs);
		if (being_refined)
			lhs_unmark(being_refined,&lhs,&inv_lhs);
		list_traverser_init(&tt_l,&tree_list);
		while (list_next(&tt_l,(dp)&lgp))
			lhs_unmark(lgp,&lhs,&inv_lhs);
		list_traverser_clear(&tt_l);
		num_rules--;/*One rule has been thrown away*/
		word_clear(&inv_lhs);
	}
	if (ans == TRUE) {/*We are going to drop this rule and we now
	ensure that any information that ought to be retained is retained,
			though now in a more efficient form.*/
		if (n == 1) { /*This means that |new_rhs| is bigger than
				|new_lhs|.*/
			mk_rule(&new_rhs,&new_lhs);
		}
		else if (n == -1 && ans == TRUE)
			mk_rule(&new_lhs,&new_rhs);
	}
	word_clear(&new_rhs);
	word_clear(&new_lhs);
	word_clear(&rhs);
	word_clear(&lhs);
	return ans;
}

/* Remove from |archive| all the indications that |u| and |inv_u| represent
respectively a considered left hand side and an inverse of a
considered left hand side.
*/
static void
unmark_in_archive(v,inv_v)
	vindex v;
	vindex inv_v;
{
	if (inv_v != UNDEFINED) {
		set_status_on_tree(inv_v,INV_CONSIDERED,FALSE);
		set_status_on_tree(inv_v,INV_LHS,FALSE);
	}
	set_status_on_tree(v,CONSIDERED,FALSE);
	set_status_on_tree(v,LHS,FALSE);
	if (get_status(v,ON_PQ))
		(void)list_delete(to_be_considered,(dp)&v);
}


/* |wp| and |invp| point to words which are formal inverses of each
other. We remove from |lgp| andy indication that these represent
respectively a left hand side or the inverse of a left hand side.
*/
static void
lhs_unmark(lgp,wp,invp)
	lg * lgp;
	word * wp, *invp;
{
	vindex v=UNDEFINED;
	vindex inv_v=UNDEFINED;
	inv_v=lg_word_find(lgp,invp);
	if (inv_v != UNDEFINED) {
		set_status_on_tree(inv_v,INV_LHS,FALSE);
	}
	v=lg_word_find(lgp,wp);
	if (v != UNDEFINED) {
		set_status_on_tree(v,LHS,FALSE);
	}
}



/* This procedure takes the word in pointed to by |in|, and reduces it,
using the information contained in |tree_list|, the tree |to_add|, and the tree
|*being_refined| if it exists.

To understand how this works, one should first understand reduction by
a single KMP tree, that is a tree with failure pointers representing
the longest proper suffixes which fit in the tree. Such a tree also
incorporates knowledge of certain Knuth-Bendix rules. The reader is
advised to go through the code and the description that follows
first making the assumption that only one tree is involved.

Let |K_i| be one of the KMP trees |to_add|, |being_refined| (if it
exists) or one of the KMP trees in |tree_list|. Let |L_i| be the
subtree of |K_i| corresponding to the set of (prefixes of) left hand
sides recorded in |K_i|.

One is given a word called |given|. We suppose inductively that
there is a factorization in the group (not to be
confused with formal factorization, corresponding to concatenation of
strings) of |given| into the concatenation |prefix circ suffix|.
For each |i| as in the preceding paragraph, we suppose that we have a
formal factorization |prefix = alpha_i beta_i g|, where |g| is a
letter.
The induction assumptions are as follows:
 |beta_i| fits in the tree, starting from the basepoint.
 The maximal suffix of |alpha_i beta_i g| which fits in the tree
|L_i| (starting from the basepoint) is some suffix of |beta_i g|.
 |alpha_i beta_i| is irreducible.
|beta_i g| does fit in |L_i| and contains a left hand side |lambda_i| of
a rule in |L_i| as a suffix. In this
case, let |rho_i| be the corresponding right hand side.

The induction is lexicographic on the tuple
||(prefixcirc suffix, suffix, beta_0, dots, beta_{trees-1}),||
where |trees| is the total number of KMP trees being used in the reduction.
We refer to the first entry as the {it concatenation} in the comments
to the code below.
The induction starts with |alpha_i| and |beta_i| trivial and
with |suffix| equal as a word to |given|.

We regard the vertices of the tree as the states of a finite state automaton.
A vertex representing a left hand side is a failure state. (We are using
``failure'' in ``failure state'' and ``failure pointer'' in two entirely
different senses, which are not to be confused.) Also, any vertex
whose failure pointer points to a failure state is a failure state.
In other words, a string is represented by a failure state if and only
if it has a left hand side of a KB rule as a suffix (not necessarily
proper).

When we find a left hand side and replace it by a right hand side,
then, to avoid doing unnecessary work, we should jump back to where we
were when the part of |alphabeta_i| which remains was first read in.
In order to do this easily, we keep a record of the history of the
states arrived at as the original string was read in.

If we have a number of trees, then a similar process works, running
all the trees as finite state automata in parallel.
We are now keeping track of this history simultaneously in a
whole lot of different KMP trees, so history is a 2-dimensional array,
where one dimension allows the tree to vary and one dimension allows
the time to vary. Since |history| has to be declared as a pointer, we
have access to the entries of the array only by doing a little arithmetic.

Returns |TRUE| if |*given| can be reduced, and |FALSE |otherwise.  */ 
boolean
reduction(given,reduced)
	word *given, *reduced;
{
	list_traverser ltrav;
	lg *lgp;
	boolean ans = FALSE;
	int trees = 1; /*initial count includes copy of |to_add| */
	int i = 0;
	int prefix_length = 0;
	vindex * history = 0;
	vindex curr = UNDEFINED;
	word prefix;
	word suffix;
	word lhs, rhs;
	gen g = INVALID_GEN;
	vindex t = UNDEFINED;
	boolean advancing_along_tree = TRUE;
	boolean substitution_found = FALSE;
	assert(given);
	word_init(&suffix);
	word_init(&prefix);
	word_init(&lhs);
	word_init(&rhs);
	word_cpy(given,&suffix);
	list_traverser_init(&ltrav,&tree_list);
	/*count the number of trees in |tree_list|*/
	while (list_next(&ltrav,(dp)&lgp)) {
		assert(get_type(lgp) == KMP_WTREE);
		trees++;
	}
	list_traverser_clear(&ltrav);
	if (being_refined)
		trees++;
	history = vzalloc2(vindex,trees * (1 + word_length(given)));
	/*Make enough room to store all history for each tree, assuming
	that no reduction takes place.*/
	history[0] = to_add->basepoint;
	i = 0;
	list_traverser_init(&ltrav,&tree_list);
	while (list_next(&ltrav,(dp)&lgp))
		history[++i] = lgp->basepoint;
	list_traverser_clear(&ltrav);
	if (being_refined)
		history[++i] = being_refined->basepoint;
	while (word_delget_first(&suffix,&g)) { /*|suffix| decreases.*/
		/* |g| followed by |inv(g)|?. If yes, take action and continue
		with the while loop. */
		gen h;
		if (word_get_last(&prefix,&h) && h == inv(g)) {
			(void)word_del_last(&prefix);
			ans = TRUE;
			continue;
		}
/*Concatenation decreased.  Each beta_i has its final letter (|h|)
		removed. Each alpha_i is unaltered.*/
		word_put_last(&prefix,g);/*Now prefix = alpha_ibeta_i g.*/
		prefix_length = word_length(&prefix);
		substitution_found = FALSE;
		for (i = 0;!substitution_found && i < trees; i++) { /*Go
		through the various KMP trees one by one, trying to find a
			reduction. |i| represents the i-th tree. |trees| is
				the total number of KMP trees being used in the
				induction process.*/
			curr = history[(prefix_length-1)*trees + i];
			/*We hop to the situation recorded in |history|, where
			we are looking in the i-th tree, and all except the
				last lstter |g| of |prefix| has
			just been read in. |curr| now represents |beta_i|. The
			arithmetic is necessary to convert a linear array to a
				2-dimensional array (coordinates |i| and
				|prefix_length-1|).*/
			/* Move |curr|, using the failure pointers, trying to
		apply |g| to |curr|, naming the result |t|. If this is possible,
			set |advancing_along_tree| to |TRUE|.*/
			while(TRUE) {
		/*|curr| represents beta_i.*/
				t = get(curr,g);
		advancing_along_tree = (t != UNDEFINED && backg(t) == g &&
											is_lhs_tree(t));
				if (is_basepoint(curr) && !advancing_along_tree)
					break;
				else if (!advancing_along_tree)
					curr = get_failure(curr);
				else break;
			}/*|curr| represents beta_i.*/

			if (advancing_along_tree)
				curr = history[prefix_length*trees + i] = t;
			/*We do not
					change alpha_i, beta_i and |suffix|.
					|curr| represents beta_i g.*/
			else {
				assert(is_basepoint(curr));
				history[prefix_length*trees + i] = curr;
				/*Make a record for posterity.
			beta_i is trivial. If |suffix| is empty, the induction
			is complete. Otherwise we need to start the main loop
			again, to re-establish the induction hypotheses.*/
			}
			if (is_basepoint(curr)) continue;/*beta_i is trivial*/
			/* If possible, do a substitution.*/
			while (does_fail_to_reducible(curr))
				/*|curr| represents beta_i g,
			which has a left hand side lambda_i as a proper suffix*/
		curr = get_failure(curr); /*concatenation and suffix constant,
			beta_i decreased, alpha_i correspondingly increased.
				|curr| still represents beta_i g.*/
			if (is_reducible(curr)) {
				/*beta_i g is equal to the left hand
				side lambda_i*/
				ans = TRUE;
				substitution_found = TRUE;
			vindex2word(curr,&lhs); /*beta_i g = \lambda_i = lhs*/
				vindex2word(get_rhs(archive_image(curr)),&rhs);
				(void)common_suffix_erased(&prefix,&lhs);
	/*now we have erased from |prefix| the suffix corresponding to |lhs|*/
				assert(word_empty(&lhs));
				word_concat(&rhs,&suffix,&suffix);
		/*Concatenation decreased. */
			}
		}
	}
	word_cpy(&prefix,reduced);
	assert(word_sgn(reduced,given)>=0);
	word_clear(&rhs);
	word_clear(&lhs);
	word_clear(&prefix);
	word_clear(&suffix);
	Free_dp((dp)history);
	history=0;
	assert(ans == TRUE || ans == FALSE);
	return ans;
}


/*
This procedure does as much reduction on |in| as possible, without
using any rule in which |in| is actually equal to the left hand side.
This is useful when checking whether
a KB rule is implied by other shorter rules. We reduce all proper
substrings, and, if such a reduction is possible, we reduce further.
*/
boolean
reduction2(in,out) 
	word * in;
	word * out;
{
	word copy;
	gen g ;
	boolean ans = FALSE;
	word_init(&copy);
	word_cpy(in,&copy);
	(void)word_delget_last(&copy,&g);
		/*Knock off the last letter of |copy|.*/
	if (reduction(&copy,&copy)) {
		ans = TRUE;
		word_put_last(&copy,g);
	}
	else {
		word_put_last(&copy,g); /*Put the last letter back on and
				knock off the first letter*/
		(void)word_delget_first(&copy,&g);
		if (reduction(&copy,&copy))
			ans = TRUE;
		word_put_first(&copy,g); /*Put the first letter back on*/
	}
	if (ans == TRUE) /*we are allowed to reduce further because we are
				no longer operating on the original word*/
		(void)reduction(&copy,&copy);
	word_cpy(&copy,out);
	word_clear(&copy);
	return ans;
}



/* We describe only the case where |mode == STANDARD|.
|rightp| points to the right hand side corresponding to
the left hand side pointed to by |leftp|. We are looking
for reductions of expressions of the form alpha \beta \gamma,
where alpha\beta is a left hand side and beta \gammais a
left hand side.
|overlap| is a vertex of |archive| corresponding to beta
and |alpha| is equal to alpha.
|run_thru_overlaps()| finds all possible
gamma's and for each gamma
carries out two distinct processes of reduction.

The code below also works for inverses of left hand sides, when |mode
== INVERSE|. In that case the connections of alpha, beta and
gamma with the variables in the code are slightly different, and we
run over all possibile alpha's while keeping beta\gammafixed. (See
the comments to the code below.)
*/
static void
run_thru_overlaps(first_stringp,overlap,rightp,mode)
	word * first_stringp; /*points to alpha or to gamma^{-1},
			depending on |mode|.*/
	vindex overlap;/*represents beta or beta^{-1}*/
	word * rightp;
	/*points to overline{\beta \gamma} or to
			overline{\alpha\beta})^{-1}.*/
	int mode;
{
	int n=0;
	vindex s;
	vindex temp;
	gen g;
	vindex smallest;
	vindex biggest;
	int leaf_type,tree_type;
	word abc1, abc2, third_stringp;
	word lhs;
	word_init(&abc1);
	word_init(&abc2);
	word_init(&third_stringp);
	word_init(&lhs);
	/*The next two procedures use the way that the considered
		left hand sides and inverses of considered left hand sides are
		linked together in |archive| following dictionary order (and
		completely ignoring length).*/
	smallest = smallest_con_leaf(overlap);
		/*the smallest considered left hand
		side or inverse left hand side which contains the string
		represented by |overlap| as a prefix (not necessarily a proper
		prefix)*/
	biggest = biggest_con_leaf(overlap); /*the biggest considered left hand
		side or inverse left hand side which contains the string
		represented by |overlap| as a prefix (not necessarily a proper
		prefix).*/
	if (mode == STANDARD) {
		leaf_type = CONSIDERED;
		tree_type = CONSIDERED_TREE;
	}
	else {
		leaf_type = INV_CONSIDERED;
		tree_type = INV_CONSIDERED_TREE;
	}
	s = smallest;
	while(TRUE) {
		/*We now run through all considered left hand sides and
			inverses of considered left hand sides which contain the
		string represented by |overlap| as a prefix. This uses the
		``next'' field associated to a vertex of |archive|. We are
	only interested in the appropriate kind of vertex. Each considered
		left hand side that we come across is tested to see if it
			should remain as the left hand side of a rule.*/
		if (get_status(s,leaf_type) && (mode==INVERSE || !reject(s))) {
			temp = s;
		/*|overlap| represents beta or beta^{-1}, and |s| represents
			beta\gamma or \alpha\beta)^{-1} respectively
			(depending on the value of |mode|.
			Therefore when we follow |s| towards the
			basepoint until we hit |overlap|, we will exactly trace
			out gamma or alpha^{-1} respectively.*/
			while (temp != overlap) {
				assert(get_status(temp,tree_type));
				g = backg(temp);
				assert(g != BACK_FROM_BASE);
				temp = get(temp,inv(g));
				word_put_first(&third_stringp,g);
			}
			/*Now we construct our two different methods of
			reducing \alpha \beta \gamma.*/
			word_concat(rightp,&third_stringp,&abc1);
				/* If |mode == STANDARD|,
					abc1 = \overline{\alpha\beta}\gamma. If
					|mode == INVERSE|, abc1 =
				(\overline{\beta\gamma})^{-1} \alpha^{-1}.*/
			if (mode==INVERSE) {
				word_inv(&abc1,&abc1);
				/*abc1 = \alpha(\overline{\beta\gamma})*/
				vindex2word(s,&lhs);
					/*lhs = (\alpha\beta)^{-1}*/
				word_inv(&lhs,&lhs); /*lhs = \alpha\beta*/
				temp=lg_word_find(archive,&lhs);
				assert(temp != UNDEFINED);
			}
			else
				temp = s; /*|temp| represents betagamma if |mode
				== STANDARD| and |temp| represents alphabeta if
				|mode == INVERSE|.*/ 
			vindex2word(get_rhs(temp),&abc2); 
			if (mode==INVERSE) word_inv(&abc2,&abc2);
			word_concat(first_stringp,&abc2,&abc2);
			if (mode==INVERSE) word_inv(&abc2,&abc2);
			/*Now |abc1| and |abc2| are equal to
				(alphabeta(reduced))gamma and to
			alpha(betagamma(reduced)) in that order if |mode ==
			STANDARD| and in the reverse order if |mode == INVERSE|.
			We now complete the reductions, and make a new KB rule
				if necessary.
			*/
			reduction(&abc1,&abc1);
			reduction(&abc2,&abc2);
			common_prefix_erased(&abc1,&abc2);
			common_suffix_erased(&abc1,&abc2);
			if ((n=word_sgn(&abc1,&abc2)) > 0) /*|abc2| is bigger*/
				mk_rule(&abc2,&abc1);
			else if (n < 0)
				mk_rule(&abc1,&abc2); /*|abc1| is bigger*/
			word_reset(&third_stringp);
				/*empty out |gamma|. This is faster
			than clearing it and re-initializing.*/
		}
		if (s == biggest)
			break; /*We have reached the end of the linked list of
			considered and inverse considered left hand sides.*/
		s = get_next(s); /*We have not yet reached the end of the
		linked list.*/
	}
	remove_bad_lhss(smallest,biggest); /*Some of the vertices in the
		linked list of considered and inverse considered left hand sides
		had no right to be there and are now thrown off. We must do
		this after running through the list once as above, because the
		call to |reject()| means that we are likely to find a number
		of new vertices which need to be thrown off the list. What was
		disastrous in earlier attempts to write this code was the
		attempt to conflate the two processes, doing the removal as we
		came to the vertex |s| in the above loop. This was terribly
		confusing and proved to be impossible to get right.*/
	 /*Clean up*/
	word_clear(&abc1);
	word_clear(&abc2);
	word_clear(&third_stringp);
	word_clear(&lhs);
}


/* Find the vertex representing the smallest string in dictionary order
(ignoring length) which contains the string represented by |v| as a
prefix (not necessarily proper) and which is a considered or inverse
considered left hand side.
*/
static vindex
smallest_con_leaf(v)
	vindex v;
{
	vindex curr = v;
	gen g;
	g = 1; /*Try to advance using as small a value of |g| as
	possible.*/
	while (g <= num_gens ) {
		vindex t=UNDEFINED;
		if (CON_LEAF(curr)) break;
	if ((t=get(curr,g)) != UNDEFINED &&  backg(t) == g && CON_TREE(t)){
			/*Advancing along the (combined) tree of considered and
			inverse considered left hand sides is possible.*/
			curr = t;
			g = 1;
		}
		else g++;
	}
	assert(CON_LEAF(curr));
	return curr;
}


/* Find the vertex representing the biggest string in dictionary order
(ignoring length) which contains the string represented by |v| as a
prefix and which is a considered or inverse considered left hand side.
*/
static vindex
biggest_con_leaf(v)
	vindex v;
{
	vindex curr = v;
	gen g;
	g = num_gens;
	while (1 <= g ) {
		vindex t=UNDEFINED;
		/*Advance along combined tree of considered and inverse
		considered left hand sides, if possible*/
	if ((t=get(curr,g)) != UNDEFINED &&  backg(t) == g && CON_TREE(t)){
					curr = t;
					g = num_gens;
		}
		else g--;
	}
	assert(CON_LEAF(curr));
	return curr;
}



/* 
The considered and inverse considered left hand sides are represented
by vertices in |archive|, arranged in
a linked list, in dictionary order of the strings, ignoring length.
The pointers are manipulated by the procedures |get_next()| and
|set_next()|.
The final vertex in the list points to itself.

We throw away the bad left hand sides, between |smallest| and
|biggest|, except that |smallest| is not thrown away because there might
be a pointer into it which we don't know about.
*/
static void
remove_bad_lhss(smallest,biggest)
	vindex smallest;
	vindex biggest;
{
	vindex prevs;/*pronounced ``previous s''*/
	vindex t;
	prevs = smallest; /*initialize traversal of the list*/
	while (prevs != biggest && get_next(prevs) != prevs){
		/*make sure |prevs| is at least one step away from being
		|biggest|*/
		t = get_next(prevs);/*definitely one along from |prevs|*/ 
		/*|t| may equal |biggest|, but is no further along the list.*/
		if (!CON_LEAF(t)) {
			/*Then we have to remove |t| from the linked list.*/
			 /*suck list in towards |prevs|, deleting as we suck*/
			vindex temp = get_next(t);
			/*|temp| may be beyond |biggest|, but it
						doesn't matter*/
			assert( t != smallest);
			set_next(t,(vindex)UNDEFINED);
			if (t == temp) { /*|t| was at the new end of the list*/
				assert(t == biggest);
				set_next(prevs,prevs);
				/*make |prevs| the end of the list*/
			}
			else set_next(prevs,temp);
		}
		else
			prevs = t; /*one more step along in the traversal of the
				linked list*/
		if (t == biggest)
			break;
	}
}



/* We order words using dictionary order. That is, the length does not
count. This procedure looks for leaves of a certain tree, namely the
combined tree of words which are either considered left hand sides or
considered inverses of left hand sides.
Assuming that |orig| is a leaf in the tree, we find the largest leaf which is
definitely smaller than |orig|. If there is no such leaf, |orig| is
returned.
*/
static vindex
just_smaller(orig)
	vindex		  orig;
{
	gen			 g =  0;
	gen			 i =  0;
	boolean		 turning_found = FALSE;
	vindex		  t = UNDEFINED;
	vindex		  v = orig;
	assert(CON_LEAF(orig));
	while (!turning_found && !is_basepoint(v)) {
	/*|v| is on the direct route \gamma from |orig| to the basepoint.
	|v| is not equal to the answer. Either the answer lies on the direct
	route from |v| to the basepoint, or the direct route \rho from the
	answer to the basepoint meets \gamma between |v| and the
		basepoint.*/
		g = backg(v);
		v = get(v,inv(g)); /*One step towards the basepoint, so now
			|v| might possibly be the answer.*/
		for (i = g - 1; i > 0; i--) { /*And now try to start going
				away from the basepoint again.*/
			t = get(v,i);
			if (t != UNDEFINED && backg(t) == i && CON_TREE(t)) {
				/*We can turn, and \rho passes through |t|.*/
				turning_found = TRUE;
				v = t;/*\rho passes through |v == t|.*/
				break; /*from for loop*/
			}
		}
		if (!turning_found && CON_LEAF(v)) /*|v| is the answer*/
			break; /*from while loop*/
	}
	if (!turning_found) {
		if (is_basepoint(v))
			/*there simply is no smaller leaf*/
			v = orig;
#ifdef DEBUG
		else assert(CON_LEAF(v));
#endif
	}
	else { /*We have turned and rho passes through |v == t|.*/
		while (v == t) {
			/*Try to go further from the basepoint. Since we have
		turned, if we can go further we will always remain smaller
				than |orig|*/
			for (i = num_gens; i > 0; i--) {
				/*the largest possibility
					is what we are looking for*/
				t = get(v,i);
			if (t != UNDEFINED && backg(t) == i && CON_TREE(t)) {
					/*We can go further from the basepoint*/
					v = t;
					break; /*from for loop*/
				}
			}
		/*The only way that |v| can be equal to |t| is as a result
			of the if clause just above.*/
		}
	}
	assert(CON_LEAF(v));
	return v;
}


/* We order words using dictionary order. That is, the length does not
count. This procedure looks for leaves of a certain tree, namely the
tree of words which are either considered left hand sides or
considered inverses of left hand sides.
Assuming that |orig| is a leaf in the tree, we find the smallest leaf which is
definitely larger than |orig|. If there is no such leaf, |orig| is
returned.
*/
static		  vindex
just_bigger(orig)
	vindex		  orig;
{
	gen			 g = 0, i = 0;
	vindex		  t = (vindex)UNDEFINED;
	vindex		  v = orig;
	boolean moved_ahead = FALSE;
	assert(CON_LEAF(v));
	do { /* we must do something to avoid finding |orig| again*/
/* When |v == orig|, |g == 0|. Otherwise |g| is the arrow pointing in
the direction of increasing distance from the basepoint along the
route which we followed
backwards to get to this point. We look forwards to the RIGHT of |g|,
to see if we can move ahead along as small a route as possible.
*/
		assert(CON_TREE(v));
for (i = g + 1/*to the RIGHT of |g|*/; !moved_ahead && i <= num_gens; i++) {
			t = get(v,i);
			if (t != UNDEFINED &&  backg(t) == i && CON_TREE(t)) {
		/*Yes, we can move one step further from the basepoint.*/
				v = t;
				moved_ahead = TRUE;
			}
		}
		if (is_basepoint(v))
			break; /*from do loop*/
		else if (!moved_ahead) {/*There is no way of increasing the
			distance from the basepoint, and we have to retreat.*/
			g = backg(v);
			v = get(v,inv(g));
		}
	} while (!moved_ahead);
	if (moved_ahead) {
		assert(moved_ahead);
		while (!CON_LEAF(v)) {
			t = UNDEFINED;
			for (i = 1; t == UNDEFINED && i <= num_gens; i++) {
				/*Keep as far to the LEFT as possible.*/
				t =  get(v,i);;
			if (t != UNDEFINED &&  backg(t) == i && CON_TREE(t)) {
					v = t;
				}
				else
			/*This makes sure we do not break from the for loop.*/
					t = UNDEFINED;
			}
			assert(CON_TREE(v));
			assert(v == t || CON_LEAF(v));
		}
	}
	if (is_basepoint(v)) /*There is no bigger leaf than |orig|.*/
		v = orig;
	assert(CON_LEAF(v));
	return v;
}



/*
If the word |*wp| cannot be found in the tree |*lgp| it is added in. The
address of the vertex at the end of the path it traces out is returned.
*/
vindex
wtree_word_insert(lgp,wp)
	lg *lgp;
	word *wp;
{
	extern lg * to_add;
	vindex v = basept(lgp);
	mp_fntab * fntab = lgp->fntab;
	vindex t = UNDEFINED;
	vindex last_elt = UNDEFINED;
	word copy;
	gen g ;
	word_init(&copy);
	word_cpy(wp,&copy);
	while(word_get_first(&copy,&g)) {
		t = get(v,g);
		if (t == UNDEFINED)
			break;
		v = t;
		(void)word_del_first(&copy);
	}
	last_elt = v;
	while(word_delget_first(&copy,&g)) {
		t = lg_vertex_create(lgp);
		set(v,g,t);
		set(t,inv(g),v);
		set_back_ptr(t,g);
		if (fntab==KMP)
			set_priority(t,get_priority(v)+1);
		if (to_add == lgp) {
			vindex al = archive_image(v);
			assert(get(al,g) != UNDEFINED);
			set_archive_image(t,get(al,g));
		}
		v = t;
	}
	word_clear(&copy);
	/*you may think it is not necessary to compute the failure
	function if no new vertices have been added. But |comp_failure()|
	also computes whether the vertex fails to a left hand side, and
	that has certainly changed*/
	if (lgp == to_add) {
		bfs_traverser t_to_add;
		boolean start = FALSE;
		assert(fntab==KMP);
		bfs_init(&t_to_add,to_add);
		while (bfs_next(&t_to_add,&t)) {
			if (t == last_elt) {
				start = TRUE;
			}
			if (start)
				comp_failure(t);
		}
		bfs_clear(&t_to_add);
	}
	return v;
}

/* Take all the trees on |tree_list| and amalgamate them with |to_add|.
Then call |rebuild()| to compute the failure pointers in |to_add| and
make |to_add| into the unique tree on |tree_list| (provided that the
rebuilding process does not result in more stuff being discovered
which then has to be added in).
*/
static void
rebuild_after()
{
	lg *lgp = 0;
	assert(to_add);
	while (list_delget_first(&tree_list,(dp)&lgp)) {
		assert(non_trivial(lgp));
		assert(get_type(lgp) == KMP_WTREE);
		kill_and_devour(to_add,lgp);
		lgp = 0;
	}
	rebuild();
	assert(being_refined == 0);
}

