/* lib.d file list.c */
/*
This file contains the definition of all functions dealing with lists.
A list may be of one of four types, an ordered list, a fifo list, a lifo
list or a priority queue. The details of these structures are given in 
the `llist' (for the linked list structure used for ordered, fifo and lifo
 lists) and the `pq' (for the priority queue structure) files. 
A list is manipulated using standard
functions contained in one of the three list function tables (pointers
to which are called |ORDERED|, |FIFO|, |LIFO| and |PQUEUE|)
relevant to that particular list. 
Blank entries appear in each function table for operations which are
particulary inefficient for that type of list.
Here is the format of the table of functions for manipulating lists.
\item{0.}
An insert function to insert a specified item.
\item{1.}
A function to determine whether or not a specified item is in the list.
\item{2.}
A delete function to delete a specified item if it can be found in the list 
\item{3.}
A function to take the first item from a list, without affecting the list.
\item{4.}
A function to take the first item from a list, while also deleting it.
\item{5.}
A function to determine whether or not a list is empty.
\item{6.}
A copy function.
\item{7.}
A create function that returns initialized space for a list..
\item{8.}
A kill function, that deallocates the space assigned to a list.
\item{9.}
A reset function.
\item{10.}
A print function.
\item{11.}
A debugging print function.
\item{12.}
A function that creates initialized space for a list traverser to run through 
the list.
\item{13.}
A function that deallocates the space assigned to a list traverser.
\item{14.}
Is there a next item in a list being traversed?
\item{15.}
A signature function taking two arguments, each the address of a list,
which returns +1, 0 or -1, according to whether the two lists appear
in their desired order, are equal, or appear in order opposite to that
desired.
*/

#include <stdio.h>
#include "defs.h"
#include "list.h"

void 
list_init(lp,datafns,listfns) 
	list *lp; 
	elt_fntab *datafns; 
	list_fntab *listfns; 
{ 
	lp->fntab = listfns; 
	lp->address = (*(listfns->create))(datafns);
	lp->lock=0;
} 

void 
list_clear(lp) 
        list * lp; 
{ 
	assert(lp->lock==0);
	(*(lp->fntab->kill))(lp->address);
	lp->address = 0;
	lp->fntab = 0;
} 

void 
list_reset(lp) 
	list *lp; 
{ 
	assert(lp->lock==0);
	(*(lp->fntab->reset))(lp->address);
} 

void list_traverser_init(ltp,lp) 
        list_traverser * ltp; 
        list * lp; 
{ 
	ltp->address = (*(lp->fntab->trav_create))(lp->address); 
	ltp->fntab = lp->fntab; 
	ltp->lockaddress = &(lp->lock);
	(lp->lock)++;
} 

void
list_traverser_clear(ltp) 
	list_traverser * ltp; 
{ 
	(*(ltp->fntab->trav_kill))(ltp->address);
	(*(ltp->lockaddress))--;
} 

boolean 
list_next_dp(ltp,out) 
        list_traverser * ltp; 
        dp out; 
{ 
	assert((*(ltp->lockaddress))>0);
	return (*(ltp->fntab->next))(ltp->address,out); 
} 

boolean
list_empty(lp)
	list * lp;
{ 
	return (*(lp->fntab->empty))(lp->address);
} 


boolean 
list_find_dp(lp,in,out) 
        list * lp; 
        data_ptr in, out; 
{ 
	return (*(lp->fntab->find))(lp->address,in,out); 
} 

boolean
list_insert_dp(lp,in) 
        list * lp; 
        dp in; 
{ 
	assert(lp->lock==0);
	return (*(lp->fntab->insert))(lp->address, in); 
} 

boolean
list_delete_dp(lp,in) 
        list * lp; 
        dp in; 
{ 
	assert(lp->lock==0);
	return (*(lp->fntab->deletefn))(lp->address,in);
} 

boolean 
list_get_first_dp(lp,out) 
        list * lp; 
        dp out; 
{ 
	return (*(lp->fntab->get_first))(lp->address, out); 
} 

boolean 
list_delget_first_dp(lp,out) 
        list * lp; 
        dp out; 
{ 
	assert(lp->lock==0);
	return (*(lp->fntab->delget_first))(lp->address,out);
} 

void 
list_print(wfile,lp) 
		FILE * wfile;
        list * lp; 
{ 
	(*(lp->fntab->print))(wfile,lp->address);
} 

void 
list_cpy(oldp, newp) 
	list * oldp, * newp; 
{ 
	assert(oldp != newp);
	assert (oldp->fntab == newp->fntab);
	(*(oldp->fntab->copy))(oldp->address,newp->address);
	
} 

int
list_sgn(lp1, lp2)
	list *lp1, *lp2;
{
	int ans;
	assert((lp1->fntab->signature)!=0);
	assert (lp1->fntab == lp2->fntab);
	ans= (*(lp1->fntab->signature))(lp1->address,lp2->address);
	return ans;
}


