/* ================================================================== 
    Clean.c   This file contains the functions that take care of
    cleaning up the data structures. 
   ==================================================================*/

#include "structs.h"
#include "macros.h"

/*-------------------------------------------------------------------------
       Clean_up goes through each data structure list and clears all
   flags and NULLs out some pointers.  In the vertex list,  the active
  flag is set to 0 and the duplicate pointer is set to NULL. In the
  edge list any struture that is marked deleted gets deleted from the 
  edge list, and in the vertex list any face that is marked visible gets
  deleted from the face list.  
  ------------------------------------------------------------------------*/
clean_up()
{

	clean_edges();
	clean_faces();
	clean_vertices();

}

/*------------------------------------------------------------------------
      Clean_edges runs through the edge list and cleans up the structure.
   If there is a face in the edges adjacent face[2] then it will put
   that face in place of the visible face and set adjacent face[2] to 
   NULL.  It will also delete any edge marked deleted.
  -----------------------------------------------------------------------*/
clean_edges()
{
	register struct tedge *e, *tmp_edge;
	
        /* replace the visible face with the new face */
	if ( e = edges )
    	     do {
                if ( e->adjface[2] ) { 
                     if ( e->adjface[0]->visible )
                          e->adjface[0] = e->adjface[2]; 
                     else e->adjface[1] = e->adjface[2];
                     e->adjface[2] = NULL;
                     }
                  e = e->next;
                  } while ( e != edges );

        /* delete any edges marked deleted */
	while ( edges && edges->deleted ) {
                e = edges;
                DEL_QUEUE ( edges, e );
                }
        if ( edges )  {
             e = edges->next;
             do {
                if ( e->deleted ) {
                     tmp_edge = e;
                     e = e->next;
                     DEL_QUEUE( edges, tmp_edge );
                     }
                else e = e->next;
                } while ( e != edges );
        }

}

/*------------------------------------------------------------------------
         Clean_faces runs through the face list and deletes any face
   marked visible.
  -----------------------------------------------------------------------*/
clean_faces()
{
	register struct tface *f, *tmp_face;

        while ( faces && faces->visible ) { 
                f = faces;
                DEL_QUEUE( faces, f );
                }
	if ( faces ) {
             f = faces->next;
             do {
                if ( f->visible ) {
                     tmp_face = f;
                     f = f->next;
                     DEL_QUEUE( faces, tmp_face );
                     }
                else f = f->next;
                } while ( f != faces );
        }

}

/*-------------------------------------------------------------------------
      Clean_vertices runs through the vertex list and deletes the 
   vertices that are marked as used but are not incident to any undeleted
   edges. It will also set the duplicate pointer to NULL and the active
   flag to not active.
  -------------------------------------------------------------------------*/
clean_vertices()
{
	struct tedge *e;
	struct tvertex *v, *tmp_vert;

        /* mark all vertices that are incident */
        /* to an edge as active */
	if ( e = edges )
             do {
	         e->endpts[0]->active = e->endpts[1]->active = ACTIVE;
	         e = e->next;
             } while (e != edges);

        /* delete all vertices that are */
        /* marked but not active */
        if ( v = vertices ) 
             do {
                if ( v->mark && !v->active ) {    
                     tmp_vert = v; 
                     v = v->next;
                     DEL_QUEUE( vertices, tmp_vert )
                     }
                else v = v->next;
             } while ( v != vertices );

        /* reset active flag and duplicate pointer */
	if ( v = vertices )
       	     do {
                v->duplicate = NULL; 
                v->active = !ACTIVE; 
                v = v->next;
                } while ( v != vertices );
   
}
