#define UBUG
/*************************************************************
*  This file is part of the Surface Evolver source code.     *
*  Programmer:  Ken Brakke, brakke@geom.umn.edu              *
*************************************************************/

/*********************************************************************
*
*    file:      simplex.c
*
*    Contents:  Functions calculating energy, volume, and their
*               gradients for the LINEAR simplex model.
*/

#include "include.h"

/**********************************************************************
*
*  function:  calc_simplex_energy()
*
*  purpose:   Calculates energy due to one simplex facet.
*             Uses Gram determinant to find simplex area.
*
*/

void calc_simplex_energy(f_id)
facet_id f_id;
{
   vertex_id *v = get_facet_vertices(f_id);
   REAL side[MAXCOORD][MAXCOORD];
   REAL *x[MAXCOORD];
   REAL matspace[MAXCOORD][MAXCOORD];
   REAL *mat[MAXCOORD];
   REAL energy;
   int k,j;

   x[0] = get_coord(v[0]);
   for ( k = 1 ; k <= web.dimension ; k++ )
     {  x[k] = get_coord(v[k]);
        for ( j = 0 ; j < web.sdim ; j++ )
          side[k-1][j] = x[k][j] - x[0][j];
     }
   for ( k = 0 ; k < web.dimension ; k++ )
     { mat[k] = matspace[k]; /* set up for matrix.c */
       for ( j = 0 ; j <= k ; j++ )
         mat[j][k] = mat[k][j] = dot(side[j],side[k],web.sdim);
     }
   energy = sqrt(determinant(mat,web.dimension))/simplex_factorial;

  web.total_area   += energy;

  /* accumulate 1/n area around each vertex to scale motion */
    for ( k = 0  ; k <= web.dimension ; k++ )
      add_vertex_star(v[k],energy);

  set_facet_area(f_id,energy);
  if ( get_fattr(f_id) & DENSITY )
       energy *= get_facet_density(f_id);
  web.total_energy += energy;
     
}



/**********************************************************************
*
*  function:  calc_simplex_forces()
*
*  purpose:   Calculates vertex forces due to one simplex facet.
*
*/

void calc_simplex_forces(f_id)
facet_id f_id;
{
   vertex_id *v = get_facet_vertices(f_id);
   REAL side[MAXCOORD][MAXCOORD];
   REAL *x[MAXCOORD];
   REAL *f[MAXCOORD];
   REAL matspace[MAXCOORD][MAXCOORD];
   REAL *mat[MAXCOORD];
   REAL force;
   int k,j,i;
   double det;
   double factor;

   x[0] = get_coord(v[0]);
   f[0] = get_force(v[0]);
   for ( k = 1 ; k <= web.dimension ; k++ )
     {  x[k] = get_coord(v[k]);
        f[k] = get_force(v[k]);
        for ( j = 0 ; j < web.sdim ; j++ )
          side[k-1][j] = x[k][j] - x[0][j];
     }
   for ( k = 0 ; k < web.dimension ; k++ )
     { mat[k] = matspace[k]; /* set up for matrix.c */
       for ( j = 0 ; j <= k ; j++ )
         mat[j][k] = mat[k][j] = dot(side[j],side[k],web.sdim);
     }
   det = det_adjoint(mat,web.dimension);  /* leaves mat adjoint transpose */
   if ( det <= 0.0 ) return; /* degenerate triangle */

   if ( get_fattr(f_id) & DENSITY )
       factor = get_facet_density(f_id)/simplex_factorial/sqrt(det);
   else factor = 1.0/simplex_factorial/sqrt(det);

   for ( k = 0 ; k < web.dimension ; k++ )
    for ( i = 0 ; i < web.dimension ; i++ )
    { REAL *ss,*ff,*ff0, fudge =  factor*mat[k][i];
      for ( j=web.sdim, ff0=f[0], ff=f[k+1],ss=side[i] ; j ; j-- )
       { 
         *(ff++) -= force = fudge*(*(ss++));
         *(ff0++) += force;
       }
    }
/* unoptimized
   for ( k = 1 ; k <= web.dimension ; k++ )
    for ( i = 1 ; i <= web.dimension ; i++ )
     for ( j = 0 ; j < web.sdim ; j++ )
       { force = factor*mat[k-1][i-1]*side[i-1][j];
         f[k][j] -= force;
         f[0][j] += force;
       }
*/

  /* accumulate 1/n area around each vertex to scale motion */
    { double energy = sqrt(det)/simplex_factorial;
      for ( k = 0  ; k <= web.dimension ; k++ )
        add_vertex_star(v[k],energy);
    }

}


/**********************************************************************
*
*  function:  calc_simplex_volume()
*
*  purpose:   Calculates volume due to one simplex facet.
*             Symmetric style only.
*
*/

void calc_simplex_volume(f_id)
facet_id f_id;
{
  vertex_id *v = get_facet_vertices(f_id);
  REAL *x[MAXCOORD];
  REAL matspace[MAXCOORD][MAXCOORD];
  REAL *mat[MAXCOORD];
  int k,j;
  body_id b_id0,b_id1;
  REAL vol;
    
  b_id0 = get_facet_body(f_id);
  b_id1 = get_facet_body(facet_inverse(f_id));
  if ( !valid_id(b_id0) && !valid_id(b_id1) ) return;
    
  for ( k = 0 ; k <= web.dimension ; k++ )
     {  x[k] = get_coord(v[k]);
        mat[k] = matspace[k];
        for ( j = 0 ; j < web.sdim ; j++ )
          mat[k][j] = x[k][j];
     }
   vol = determinant(mat,web.sdim);
   vol /= simplex_factorial*web.sdim;

  /* add to body volumes */
  if ( valid_id(b_id0) ) 
    set_body_volume(b_id0,get_body_volume(b_id0) + vol);
  if ( valid_id(b_id1) ) 
    set_body_volume(b_id1,get_body_volume(b_id1) - vol);
}

/******************************************************************
*   
*  Function: simplex_grad_l()
*
*  Purpose: Compute volume gradients for vertices on facets.
*/

void simplex_grad_l()
{
  body_id bi_id;  /* identifier for body i */
  body_id bj_id;  /* identifier for body j */
  vertex_id *v;
  REAL *x[MAXCOORD];
  REAL matspace[MAXCOORD][MAXCOORD];
  REAL *mat[MAXCOORD];
  int k,j,i;
  facet_id f_id;
    
  volgrad *vgptr;

  FOR_ALL_FACETS(f_id)
   { 
     bi_id = get_facet_body(f_id);
     bj_id = get_facet_body(facet_inverse(f_id));
     v = get_facet_vertices(f_id);
     for ( k = 0 ; k <= web.dimension ; k++ )
        {  x[k] = get_coord(v[k]);
           mat[k] = matspace[k];
           for ( j = 0 ; j < web.sdim ; j++ )
                   mat[k][j] = x[k][j];
        }
     det_adjoint(mat,web.sdim); /* mat adjoint transpose */

     if ( valid_id(bi_id) && (get_battr(bi_id) & (PRESSURE|FIXEDVOL)) )
       { 
          for ( k = 0 ; k <= web.dimension ; k++ )
            { vgptr = get_bv_new_vgrad(bi_id,v[k]);
              for ( i = 0 ; i < web.sdim ; i++ )
                    vgptr->grad[i] +=  mat[i][k]/volume_factorial;
            }
       }
     if ( valid_id(bj_id) && (get_battr(bj_id) & (PRESSURE|FIXEDVOL)) )
       { 
          for ( k = 0 ; k <= web.dimension ; k++ )
            { vgptr = get_bv_new_vgrad(bj_id,v[k]);
              for ( i = 0 ; i < web.sdim ; i++ )
                    vgptr->grad[i] -=  mat[i][k]/volume_factorial;
            }
       }
   } /* end facet loop */
}


/* Recursive refining of a simplex divided at the midpoints of its edges. */


struct divedge { int endpt[2];  /* endpoints */
              int divpt;     /* dividing point of edge */
              };

struct simplex { int pt[MAXCOORD+1]; };

void refine_simplex(dim,vlist,elist,slist)
    int dim;    /* dimension of simplex */
    int *vlist;    /* vertices of simplex  */
    struct divedge *elist; /* edges of simplex */
    struct simplex *slist; /* generated simplices */
{ int v1,v2;
  int basept;
  int numedges = (dim*(dim+1))/2;
  int snum = 0;  /* simplex number */
  int pnum;      /* point number in simplex */
  struct divedge *newelist
               = (struct divedge *)mycalloc(numedges,sizeof(struct divedge));
  int i,j;
  int newfaces;
  int edgenum;

  /* pick base point as first point on list  */
  v1 = elist[0].endpt[0]; v2 = elist[0].endpt[1];
  basept = elist[0].divpt;

  /* do simple opposite simplices */
  /* one for each vertex not an endpoint on base edge */
  for ( i = 0 ; i <= dim ; i++ )
    { if ( (vlist[i] == v1) || (vlist[i] == v2) ) continue; /* on base edge */
      for ( j = 0, pnum = 0 ; j < numedges ; j++ )
        if ( (elist[j].endpt[0]==vlist[i]) || (elist[j].endpt[1]==vlist[i]))
          slist[snum].pt[pnum++] = elist[j].divpt;
      slist[snum].pt[pnum++] = basept; /* add common basepoint */
      snum++;
    }
  if ( dim <= 2 ) goto exxit; /* end recursion */

  /* now do compound opposite faces */
  /* first, opposite v1 */
  /* first, get edges after base not including  v2 */
  for ( i = 1, edgenum = 0 ; i < numedges ; i++ )
    if ( (elist[i].endpt[0]!=v1) && (elist[i].endpt[1]!=v1) )
        newelist[edgenum++] = elist[i];
  /* get vertex list in good order with v1 first */
  for ( i = 0 ; i <= dim ; i++ )
    if ( v1 == vlist[i] )
      { vlist[i] = vlist[0]; vlist[0] = v1; break; }
  /* recurse */
  refine_simplex(dim-1,vlist+1,newelist,slist+snum);
  /* add base point */
  newfaces = (1<<(dim-1))-dim;
  for ( i = 0 ; i < newfaces ; i++ )
    slist[snum++].pt[dim] = basept;

  /* second, opposite v2 */
  /* first, get edges after base not including  v2 */
  for ( i = 1, edgenum = 0 ; i < numedges ; i++ )
    if ( (elist[i].endpt[0]!=v2) && (elist[i].endpt[1]!=v2) )
        newelist[edgenum++] = elist[i];
  /* get vertex list in good order with v2 first */
  for ( i = 0 ; i <= dim ; i++ )
    if ( v2 == vlist[i] )
      { vlist[i] = vlist[0]; vlist[0] = v2; break; }
  /* recurse */
  refine_simplex(dim-1,vlist+1,newelist,slist+snum);
  /* add base point */
  newfaces = (1<<(dim-1))-dim;
  for ( i = 0 ; i < newfaces ; i++ )
    slist[snum++].pt[dim] = basept;

exxit:
  free((char*)newelist);
}

static struct simplex *slist;
static struct simplex *slist_edge;
static int dim;
static int count,scount;
static double (*vcoord)[MAXCOORD];

/* temporary explicit refinements */
static struct simplex slist3[8] = { {0,4,5,6}, {1,4,8,7}, {2,5,7,9},
   {3,6,9,8}, {4,5,8,7}, {4,8,5,6}, {6,8,5,9}, {5,8,7,9} };
static struct simplex slist2[4] = { {0,3,4},{1,5,3},{2,4,5},{3,5,4}};
static struct simplex slist1[2] = { {0,2}, {2,1}};

void refine_simplex_init( )
{
  int vlist[MAXCOORD+1];
  int i,j;
  struct divedge *elist;
  int numedges;
  int pnum,snum=0;

  if ( slist && (slist != slist3)  && (slist != slist2) && (slist != slist1))
    free((char*)slist);
  switch ( web.dimension )
    { case 4: slist_edge = slist3; break;
      case 3: slist = slist3; slist_edge = slist2; return; 
      case 2: slist = slist2; slist_edge = slist1; return;
      case 1: slist = slist1; slist_edge = slist1; return;
      default: error("Too high dimension for simplex refinement.\n",
         WARNING);
    }
      
  dim = web.dimension;
  if ((dim < 1) || (dim > MAXCOORD) )
    error("Simplex dimension must be between 1 and MAXCOORD.\n",RECOVERABLE);

  vcoord = (double(*)[MAXCOORD])mycalloc((MAXCOORD+1)*(MAXCOORD+2)/2*MAXCOORD,
              sizeof(double));
  /* set up initial arrays */
  /* outer vertices */
  for ( i = 0 ; i <= dim ; i++ ) 
    { vlist[i] = i;
      if ( i > 0 ) vcoord[i][i-1] = 2.0;
    }
  /* edges */
  numedges = (dim*(dim+1))/2;
  elist = (struct divedge *)mycalloc(numedges,sizeof(struct divedge));
  for ( i = 0, count = 0 ; i < dim ; i++ )
    for ( j = i+1 ; j <= dim ; j++ )
      { int pt,k;
        elist[count].endpt[0] = i;
        elist[count].endpt[1] = j;
        elist[count].divpt = pt = dim+1 + count;
        count++;
        for ( k = 0 ; k < dim ; k++ )
          vcoord[pt][k] = (vcoord[i][k] + vcoord[j][k])/2.0;
      }

  scount = (1 << dim);
  slist = (struct simplex *)mycalloc(scount,sizeof(struct simplex));

  /* do simplices on corners */
  for ( i = 0 ; i <= dim ; i++ )
    { 
      for ( j = 0, pnum = 0 ; j < numedges ; j++ )
        if ( (elist[j].endpt[0]==vlist[i]) || (elist[j].endpt[1]==vlist[i]))
          slist[snum].pt[pnum++] = elist[j].divpt;
      slist[snum].pt[pnum++] = vlist[i]; /* add common basepoint */
      snum++;
    }
  /* do interior blob */
  if ( dim > 1 )
    refine_simplex(dim,vlist,elist,slist+snum);

  check_orientation();
  free((char*)vcoord);
  free((char*)elist);

}

void check_orientation()
{
  int i,j,k;
  double **mat;

  mat = dmatrix(0,dim-1,0,dim-1);
  for ( i = 0 ; i < scount ; i++ )
    { 
      /* load matrix with side vectors of simplex */
      for ( j = 0 ; j < dim ; j++ )
        for ( k = 0 ; k < dim ; k++ )
          mat[j][k] = vcoord[slist[i].pt[j+1]][k] - vcoord[slist[i].pt[0]][k];

      /* test and exchange if necessary */
      if ( determinant(mat,dim) < 0.0 )
        { j = slist[i].pt[0]; slist[i].pt[0] = slist[i].pt[1];
          slist[i].pt[1] = j;
        }
    }
  free_matrix(mat);
}

/* Now come the routines for refining all the simplices.  Idea is to
go through simplices refining and keep subdivided edges in hash table
to prevent duplication */

/* hashtable stuff */
struct entry { vertex_id endpt[2], divpt; };
struct entry *hashtable;
int maxload;  /* maximum entries before expanding */
int hashentries;  /* current number of entries */
int hashmask;     /* for getting applicable bits of hash key */
int tablesize;    /* current table size, power of 2 */
#define hash_1(id)    (((id)*701)>>7)
 /*#define hash_2(id)    (((id)*1003)>>+7) */
#define hash_2(id) 1

void init_hash_table()
{
  hashentries = 0;
  tablesize = 64; hashmask = tablesize-1; maxload = 48;
  while ( maxload < web.skel[EDGE].count )
    { hashmask += tablesize; tablesize *= 2; maxload *= 2; }
  hashtable = (struct entry *)mycalloc(tablesize,sizeof(struct entry));
}

void rehash()
{ struct entry *oldhashtable = hashtable;
  int i,j;
  struct entry *h;
  unsigned int hash1,hash2,hash;
  int recount=0;

  hashtable = (struct entry *)mycalloc(2*tablesize,sizeof(struct entry));

  /* rehash all entries */
  hashmask += tablesize;
  for ( i = 0, h = oldhashtable ; i < tablesize ; i++,h++ )
    {
      if ( !h->divpt ) continue;
      hash1 = hash_1(h->endpt[0]) + hash_1(h->endpt[1]);
      hash2 = hash_2(h->endpt[0]) + hash_2(h->endpt[1]);
      hash = hash1 % hashmask;
      for ( j = 0 ; j < hashentries ; j++ )
        {
          if ( !hashtable[hash].divpt ) break; /* empty spot */
          hash += hash2;
          hash %= hashmask;
        }
if ( hashtable[hash].divpt )
  error("Fatal hash conflict in rehash.\n",UNRECOVERABLE);
      hashtable[hash] = *h;  /* move in entry */
      recount++;
    }
  if ( recount != hashentries )
    { end_hash_table();
      error("Hash table expansion error.\n",RECOVERABLE);
    }
  tablesize <<= 1;  /* double tablesize */
  maxload <<= 1;
  free((char*)oldhashtable);
}

void end_hash_table()
{ 
  free((char*)hashtable);
  hashtable = NULL;
}

vertex_id simplex_edge_divide(v1,v2)
vertex_id v1,v2;
{ vertex_id newv;
  REAL *x1,*x2;
  REAL newx[MAXCOORD];
  unsigned int hash1,hash2,hash;
  int i;
  MAP conmap;

  /* get vertices in canonical order */
  if ( v1 > v2 ) { vertex_id temp = v1; v1 = v2; v2 = temp;}

  /* look up in hash table */
  hash1 = hash_1(v1) + hash_1(v2);
  hash2 = hash_2(v1) + hash_2(v2);
  hash = hash1 % hashmask;
  for ( i = 0 ; i <= hashentries ; i++ )
    { 
      if ( !hashtable[hash].divpt ) break; /* missing */
      if ( (hashtable[hash].endpt[0]==v1) && (hashtable[hash].endpt[1]==v2) )
        return hashtable[hash].divpt; /* found! */
      hash += hash2;
      hash %= hashmask;
    }
if ( hashtable[hash].divpt )
  error("Fatal hash conflict.\n",UNRECOVERABLE);
  /* need to insert new edge and dividing point */
  hashtable[hash].endpt[0] = v1;
  hashtable[hash].endpt[1] = v2;
  x1 = get_coord(v1); x2 = get_coord(v2);
  for ( i = 0 ; i < web.sdim ; i++ )
    newx[i] = (x1[i] + x2[i])/2;
  hashtable[hash].divpt = newv = new_vertex(newx);
  set_attr(newv,get_vattr(v1)&get_vattr(v2));

  conmap = get_v_constraint_map(v1) & get_v_constraint_map(v2);
  set_v_conmap(newv,conmap);

  conmap = get_v_constraint_state(v1) & get_v_constraint_state(v2);
  set_v_constraint_state(newv,conmap);

  if ( get_boundary(v1) == get_boundary(v2) )
    set_boundary(newv,get_boundary(v1));
  hashentries++;

  /* see if hashtable needs expanding */
  if ( hashentries > maxload )
    rehash();

  return newv;
}

void refine_all_simplices()
{
  facet_id f_id;
  edge_id e_id;
  int i,j;
  vertex_id vlist[(MAXCOORD+1)*(MAXCOORD+2)/2]; /* full list of vertices */
  tagtype tag;
  body_id b_id1,b_id2;
  int vcount;
  ATTR attr;
  REAL density;

  /* allocate space for new elements all at once to reduce fragmentation */
  extend(VERTEX,web.skel[EDGE].count);
  extend(FACET,web.skel[FACET].count*((1<<web.dimension)-1));

  init_hash_table();  /* for subdivided edges */

  FOR_ALL_FACETS(f_id)
    {
      vertex_id *v = get_facet_vertices(f_id);

      attr = get_fattr(f_id);
      if ( attr & NEWELEMENT ) continue;

      /* transfer old vertices to start of full list */
      for ( vcount = 0 ; vcount <= web.dimension ; vcount++ )
        vlist[vcount] = v[vcount];

      /* go through edges, subdividing */
      for ( i = 0 ; i < web.dimension ; i++ )
        for ( j = i+1 ; j <= web.dimension ; j++ )
          vlist[vcount++] = simplex_edge_divide(v[i],v[j]);

      /* construct new simplices using pre-computed decomposition */
      /* copy relevant properties of old facet */
      attr |= NEWELEMENT;  /* so don't repeat refinement */
      tag = get_tag(f_id);
      density = get_facet_density(f_id);
      b_id1 = get_facet_body(f_id);
      b_id2 = get_facet_body(inverse_id(f_id));
      for ( i = 0 ; i < (1<<web.dimension) ; i++ )
        { facet_id newf;
          if ( i == 0 ) newf = f_id;   /* re-use old facet */
          else newf = new_facet();
          set_attr(newf,attr);
          set_tag(newf,tag);
          set_original(newf,get_original(f_id));
          set_facet_density(newf,density);
          set_facet_body(newf,b_id1);
          set_facet_body(inverse_id(newf),b_id2);
	  set_facet_density(newf,1.0);
          v = get_facet_vertices(newf);
          for ( j = 0 ; j <= web.dimension ; j++ )
            v[j] = vlist[slist[i].pt[j]];
        }
    }


  FOR_ALL_EDGES(e_id)
    {
      vertex_id *v = get_edge_vertices(e_id);

      attr = get_eattr(e_id);
      if ( attr & NEWELEMENT ) continue;

      /* transfer old vertices to start of full list */
      for ( vcount = 0 ; vcount < web.dimension ; vcount++ )
        vlist[vcount] = v[vcount];

      /* go through edges, subdividing */
      for ( i = 0 ; i < web.dimension-1 ; i++ )
        for ( j = i+1 ; j < web.dimension ; j++ )
          vlist[vcount++] = simplex_edge_divide(v[i],v[j]);

      /* construct new edge simplices using pre-computed decomposition */
      /* copy relevant properties of old facet */
      attr |= NEWELEMENT;  /* so don't repeat refinement */
      for ( i = 0 ; i < (1<<(web.dimension-1)) ; i++ )
        { edge_id newedge;
          if ( i == 0 ) newedge = e_id;   /* re-use old facet */
          else newedge = new_edge(NULLID,NULLID);
          set_attr(newedge,attr);
          set_original(newedge,get_original(e_id));
          set_e_conmap(newedge,get_e_constraint_map(e_id));
          v = get_edge_vertices(newedge);
          for ( j = 0 ; j < web.dimension ; j++ )
            v[j] = vlist[slist_edge[i].pt[j]];
        }
    }

  end_hash_table();

  level++;
}

/******************************************************************
*
*  function: hi_dim_graph()
*
*  purpose:  graph wire frames of hi dimension simplices projected
*            to 3D.
*/

void hi_dim_graph()
{
  struct graphdata gdata[2];
  facet_id f_id;
  int i,j,k;

  gdata[0].color = WHITE;
  FOR_ALL_FACETS(f_id)
    {
      vertex_id *v = get_facet_vertices(f_id);
      if ( breakflag ) break;
      q_id = f_id;
      if ( show_expr.root && !eval(&show_expr,NULL) ) continue;
      gdata[0].id = f_id;
      for ( i = 0 ; i < web.dimension ; i++ )
        { REAL *x = get_coord(v[i]);
          for ( j = 0 ; j < 3 ; j++ )
            gdata[0].x[j] = x[j];
          for ( k = i+1 ; k <= web.dimension ; k++ )
            { x = get_coord(v[k]);
              for ( j = 0 ; j < 3 ; j++ )
                 gdata[1].x[j] = x[j];
              (*graph_edge)(gdata);
            }
        }
    }
}

/******************************************************************
*
*  function: calc_simplex_edge_energy()
*
*  purpose:  find energy contribution of edge integrand
*
*/

void calc_simplex_edge_energy(e_id)
edge_id e_id;
{
  struct constraint *constr;
  int i,j,k,m;
  REAL energy = 0.0;
  REAL side[MAXCOORD][MAXCOORD];
  REAL *sides[MAXCOORD];
  REAL green[MAXCONCOMP];
  MAP conmap;
  REAL midpt[MAXCOORD];  /* evaluation point for integrand */
  int sign;
  REAL kvector[MAXCONCOMP];  /* k-vector representing simplex */
  REAL *x[MAXCOORD];
  vertex_id *v;

  conmap = get_e_constraint_map(e_id);
  if ( get_eattr(e_id) & NEGBOUNDARY ) sign = -1;
  else sign = 1;
  if ( inverted(e_id) ) sign = -sign;

  v = get_edge_vertices(e_id);
  x[0] = get_coord(v[0]);
  for ( k = 1 ; k < web.dimension ; k++ )
     {  x[k] = get_coord(v[k]);
        for ( j = 0 ; j < web.sdim ; j++ )
          side[k-1][j] = x[k][j] - x[0][j];
        sides[k-1] = side[k-1];  /* pointers for matrix routines */
     }
  exterior_product(sides,kvector,web.dimension-1,web.sdim);

  for ( j = 0 ; j < web.concount ; j++,conmap >>= 1 )
   {
    if ( !(conmap & 1) ) continue;
    constr = get_constraint(j);
    if ( !(constr->attr & BDRY_ENERGY) ) continue;
    for ( k = 0 ; k < 1 /* web.gauss1D_order */ ; k++ )
     {
       for ( i = 0 ; i < web.sdim ; i++ )
         { midpt[i] = 0.0;
           for ( m = 0 ; m < web.dimension ; m++ )
             midpt[i] += x[m][i]/web.dimension;
         }
       for ( i = 0 ; i < constr->compcount ; i++ )
            green[i] = eval(constr->envect[i],midpt);  /* constraint energy */
       energy += sign*dot(kvector,green,constr->compcount);
     }

   }

  web.total_energy += energy/simplex_factorial*web.dimension;

}


/******************************************************************
*
*  function: calc_simplex_edge_force()
*
*  purpose:  find force contribution of edge integrand
*
*/

void calc_simplex_edge_force(e_id)
edge_id e_id;
{
  struct constraint *constr;
  int i,j,k,m;
  REAL *hforce,*tforce;
  REAL side[MAXCOORD][MAXCOORD];
  REAL *sides[MAXCOORD];
  REAL green[MAXCONCOMP];
  REAL green_deriv[MAXCONCOMP][MAXCOORD];
  MAP conmap;
  REAL midpt[MAXCOORD];  /* evaluation point for integrand */
  int sign;
  REAL kvector[MAXCONCOMP];  /* k-vector representing simplex */
  REAL *x[MAXCOORD];
  vertex_id *v;
  double fudge = simplex_factorial/web.dimension;
  REAL grad[MAXCOORD];

  conmap = get_e_constraint_map(e_id);
  if ( get_eattr(e_id) & NEGBOUNDARY ) sign = -1;
  else sign = 1;
  if ( inverted(e_id) ) sign = -sign;

  v = get_edge_vertices(e_id);
  x[0] = get_coord(v[0]);
  for ( k = 1 ; k < web.dimension ; k++ )
     {  x[k] = get_coord(v[k]);
        for ( j = 0 ; j < web.sdim ; j++ )
          side[k-1][j] = x[k][j] - x[0][j];
        sides[k-1] = side[k-1];  /* pointers for matrix routines */
     }
  exterior_product(sides,kvector,web.dimension-1,web.sdim);

  tforce = get_force(v[0]);
  for ( j = 0 ; j < web.concount ; j++,conmap >>= 1 )
   {
    if ( !(conmap & 1) ) continue;
    constr = get_constraint(j);
    if ( !(constr->attr & BDRY_ENERGY) ) continue;
    for ( k = 0 ; k < 1 /* web.gauss1D_order */ ; k++ )
     {
       for ( i = 0 ; i < web.sdim ; i++ )
         { midpt[i] = 0.0;
           for ( m = 0 ; m < web.dimension ; m++ )
             midpt[i] += x[m][i]/web.dimension;
         }
       for ( i = 0 ; i < constr->compcount ; i++ )
         eval_all(constr->envect[i],midpt,web.sdim,&green[i],
               green_deriv[i]);  /* constraint value and derivs */
        
       /* part due to motion of midpoint changing integrand */
       for ( m = 0 ; m < web.sdim ; m++ )
         for ( i = 0 , grad[m] = 0.0 ; i < constr->compcount ; i++ )
           grad[m] += kvector[i]*green_deriv[i][m];
       for ( i = 0 ; i <= web.dimension ; i++ )
         { hforce = get_force(v[i]);
           for ( m = 0 ; m < web.sdim ; m++ )
             hforce[m] -= grad[m]/web.dimension/fudge;
         }
       /* part due to changing kvector */
       for ( i = 0 ; i < web.dimension-1 ; i++ ) /* side by side */
       { hforce = get_force(v[i+1]);
         for ( m = 0 ; m < web.sdim ; m++ )
         { double f;
           sides[i] = identity[m];
           exterior_product(sides,kvector,web.dimension-1,web.sdim);
           f = dot(kvector,green,constr->compcount)/fudge;
           hforce[m] -= f;
           tforce[m] += f;
         }
         sides[i] = side[i];
       }        
     }
   }
}

/*********************************************************************
*
*  function: simplex_equiangulate()
*
*  purpose: equiangulation for simplex representation.
*
*/

void simplex_equiangulate()
{
  vertex_id v_id;
  vertex_id face[MAXCOORD+1],*v;
  int dim;
  int i,j,k;
  int count = 0;
  double temperature;
  facet_id f_id;
  edge_id e_id;
   REAL **side,**kvector;
   REAL *x[MAXCOORD];
   int kcomp = binom_coeff(web.sdim,web.dimension);
   vertex_id tempv;

  /* use one old simplex to establish orientation of starter new simplex */
  side = dmatrix(0,web.dimension-1,0,web.sdim-1);
  kvector = dmatrix(0,1,0,kcomp-1);
  FOR_ALL_FACETS(f_id) break; /* get one old simplex */
  v = get_facet_vertices(f_id);
  x[0] = get_coord(v[0]);
  for ( k = 1 ; k <= web.dimension ; k++ )
     {  x[k] = get_coord(v[k]);
        for ( j = 0 ; j < web.sdim ; j++ )
          side[k-1][j] = x[k][j] - x[0][j];
     }
  exterior_product(side,kvector[0],web.dimension,web.sdim);

  /* get initial face */
  face[0] = v[0]; /* get a starter vertex */
  for ( dim = 0 ; dim < web.dimension ; dim++ )
   { v_id = find_other_vertex(face,dim,NULLID);
     if ( !valid_id(v_id) )
       { 
	 sprintf(errmsg,"Cannot find simplex-completing vertex for\n");
	 for ( j = 0 ; j <= dim ; j++ ) 
	   sprintf(errmsg+strlen(errmsg)," %d",ordinal(face[j])+1);
	 error(errmsg,RECOVERABLE);
       }
     face[dim+1] = v_id;
   }
  x[0] = get_coord(face[0]);
  for ( k = 1 ; k <= web.dimension ; k++ )
     {  x[k] = get_coord(face[k]);
        for ( j = 0 ; j < web.sdim ; j++ )
          side[k-1][j] = x[k][j] - x[0][j];
     }
  exterior_product(side,kvector[1],web.dimension,web.sdim);
  if ( dot(kvector[0],kvector[1],kcomp) < 0.0 )
    { /* need to flip orientation */
      v_id = face[0]; face[0] = face[1]; face[1] = v_id;
    }
  free_matrix(side);
  free_matrix(kvector);

  /* get rid of old */
  FOR_ALL_FACETS(f_id)
    free_element(f_id);
  FOR_ALL_EDGES(e_id)
    free_element(e_id);

  /* jiggle to remove degeneracy */
  save_coords();
  temperature = 1e-5;
  FOR_ALL_VERTICES(v_id)
    { REAL *x;
      x = get_coord(v_id);
      for ( i = 0 ; i < web.sdim ; i++ )
        x[i] += gaussian()*temperature*web.max_len*overall_size;
    }

  /* start face stack */
  init_face_stack(web.dimension-1);
  dim = web.dimension-1;
  v_id = face[dim+1];
  for ( i = 0 ; i <= dim+1 ; i++ )
    { if ( i == dim+1 )  /* for proper orientation */ 
        { tempv = face[1]; face[1] = face[0]; face[0] = tempv; }
      face[dim+1] = face[i]; face[i] = v_id; 
      push_face(face);
      face[i] = face[dim+1];
    }
  tempv = face[1]; face[1] = face[0]; face[0] = tempv;  /* fix order */

  /* add to list */
  f_id = new_facet();
  v = get_facet_vertices(f_id);
  for ( i = 0 ; i <= web.dimension ; i++ ) v[i] = face[i];
  count++;

  /* go through and complete each face on stack */
  while ( pop_face(face) )
   { v_id = find_other_vertex(face,dim,face[dim+1]);
     if ( !valid_id(v_id) )
       { 
	 sprintf(errmsg,"Cannot find simplex-completing vertex for\n");
	 for ( j = 0 ; j <= dim ; j++ ) 
	   sprintf(errmsg+strlen(errmsg)," %d",ordinal(face[j])+1);
	 error(errmsg,RECOVERABLE);
       }
     for ( i = 0 ; i <= dim ; i++ )
       { face[dim+1] = face[i];
         face[i] = v_id;
         push_face(face);
         face[i] = face[dim+1];
       }

     /* add to list */
     f_id = new_facet();
     v = get_facet_vertices(f_id);
     for ( i = 0 ; i < web.dimension ; i++ ) v[i] = face[i];
     v[web.dimension] = v_id;
     count++;
   }
 end_face_stack();
 fprintf(outfd,"Count: %d\n",count);

 simplex_delauney_test();

 restore_coords();  /* undo jiggling */
}
     
static vertex_id *face_stack;
static int face_stack_size;
static int face_dim;
static int face_stack_max;

void init_face_stack(dim)
int dim;  /* dimension of faces */
{
  face_stack_max = web.skel[VERTEX].count*2*(dim+2);
  face_stack = (vertex_id *)temp_calloc(face_stack_max,sizeof(vertex_id)); 
  face_stack_size = 0;
  face_dim = dim;
}

void end_face_stack()
{
  temp_free((char *)face_stack);
}

int pop_face(face)
vertex_id *face;  /* to copy vertex list for face into */
{ int i;
  if ( face_stack_size <= 0 ) return 0;
  memcpy((char*)face,(char*)(face_stack + face_stack_size - (face_dim+2)),
     (face_dim+2)*sizeof(vertex_id));
  if ( valid_id(face[face_dim+1]) )
     face_stack_size -= face_dim+2;  /* pop if already done one side */

#ifdef UBUG
fprintf(outfd,"pop %d:",face_stack_size); 
for ( i = 0 ; i <= face_dim+1 ; i++ ) 
  fprintf(outfd," %d",ordinal(face[i])+1);
fprintf(outfd,"\n");
#endif

}

void push_face(face)
vertex_id *face;  /* face to put on list if not already there */
{ int i,j;
  vertex_id v_id,vsort[MAXCOORD+1],*v;
  MAP conmap;
  int swaps;

  /* see if all on same constraint */
  for ( i = 0, conmap = -1 ; i <= face_dim ; i++ )
    conmap &= get_v_constraint_state(face[i]);
  if ( conmap && valid_id(face[face_dim+1]) )  /* outer edge */
    { /* add to face list */
      edge_id e_id = new_edge(NULLID,NULLID);
      v = get_edge_vertices(e_id);
      for ( j = 0 ; j <= face_dim ; j++ )
        v[j] = face[j];
      set_attr(e_id,CONSTRAINT);
      set_attr(e_id,BDRY_ENERGY);  /* maybe not, but won't hurt */
      set_attr(e_id,BDRY_CONTENT);  /* maybe not, but won't hurt */
      set_e_conmap(e_id,conmap);
      return; /* don't add to face list */
    }

  /* sort face vertices for easy comparison */
  /* insertion sort */
  for ( i = 0, swaps = 0 ; i <= face_dim ; i++ )
    { for ( j = i ; j > 0 ; j-- )
        if ( ordinal(vsort[j-1]) < ordinal(face[i]) ) break;
        else { vsort[j] = vsort[j-1]; swaps++; }
      vsort[j] = face[i];
    }
  if ( !(swaps & 1) ) /* force odd swaps */
    { v_id = vsort[0]; vsort[0] = vsort[1]; vsort[1] = v_id; }
  vsort[face_dim+1] = face[face_dim+1];

#ifdef UBUG
fprintf(outfd,"push %d ",face_stack_size); 
for ( i = 0 ; i <= face_dim+1 ; i++ ) fprintf(outfd," %d",ordinal(vsort[i])+1);
fprintf(outfd,"\n");
#endif

  /* see if already have */
  /* crude linear search */
  for ( i = 0 ; i < face_stack_size ; i += face_dim+2 )
    { for ( j = 0 ; j <= face_dim ; j++ )
        if ( !equal_id(face_stack[i+j],vsort[j]) ) break;
      if ( j > face_dim ) /* found */
        break;
    }
  if ( i < face_stack_size ) /* found */
   { 
#ifdef UBUG
fprintf(outfd,"found\n");
#endif
     if ( valid_id(face_stack[i+face_dim+1]) )
       { /* now have both sides, so drop from list */
         face_stack_size -= face_dim+2;
         if ( face_stack_size > i ) /* move top face into empty slot */
           memcpy((char*)(face_stack+i),(char*)(face_stack + face_stack_size),
                       (face_dim+2)*sizeof(vertex_id));
       }
     else /* have just first simplex */
       face_stack[i+face_dim+1] = face[face_dim+1];
   }
  else /* not found, add to stack */
   {
#ifdef UBUG
fprintf(outfd,"not found\n");
#endif
     if ( face_stack_size >= face_stack_max )
       error("Face stack overflow.\n",RECOVERABLE);
     v_id = vsort[0]; vsort[0] = vsort[1]; vsort[1] = v_id; /* even swap */
     memcpy((char*)(face_stack + face_stack_size),(char*)vsort,
                                 (face_dim+2)*sizeof(vertex_id)); 
     face_stack_size += face_dim+2; 
    } 
}

vertex_id void_test(v,dim)
vertex_id *v;
int dim;
{
   REAL *x[MAXCOORD+1];
   REAL ss[MAXCOORD];  /* squares of sides */
   int k,j;
   REAL rr;  /* square radius of void */
   REAL center[MAXCOORD];
   REAL lam[MAXCOORD];
   vertex_id v_id,bad_v = NULLID;
  REAL **mat = dmatrix(0,web.dimension-1,0,web.dimension-1);
  REAL **side = dmatrix(0,web.sdim-1,0,web.sdim-1);

   /* first, calculate center of void */
   x[0] = get_coord(v[0]);
   for ( k = 1 ; k <= dim ; k++ )
     {  x[k] = get_coord(v[k]);
        for ( j = 0 ; j < web.sdim ; j++ )
          side[k-1][j] = x[k][j] - x[0][j];
     }
   for ( k = 0 ; k < dim ; k++ )
     { 
       for ( j = 0 ; j <= k ; j++ )
         mat[j][k] = mat[k][j] = dot(side[j],side[k],web.sdim);
       ss[k] = mat[k][k];
     }
   mat_inv(mat,dim);
   matvec_mul(mat,ss,lam,dim,dim);
   rr = dot(lam,ss,dim)/4;
   vec_mat_mul(lam,side,center,dim,web.sdim);
   for ( k = 0 ; k < web.sdim ; k++ )
     center[k] = x[0][k] + center[k]/2;

   /* now see if any other vertices are in the void */
   FOR_ALL_VERTICES(v_id)
     { REAL *y;
       REAL z[MAXCOORD];

       for ( k = 0 ; k <= dim ; k++ )
         if ( equal_id(v_id,v[k]) ) break;
       if ( k <= dim ) continue;  /* skip vertices in facet */

       y = get_coord(v_id);
       for ( j = 0 ; j < web.sdim ; j++ )
         z[j] = y[j] - center[j];
       if ( dot(z,z,web.sdim) >= rr - 1e-10 ) continue;
#if 1      
fprintf(outfd,"Void violation by %g\n",-dot(z,z,web.sdim) + rr);
#endif
       bad_v = v_id;
       break;
     }
 free_matrix(side);
 free_matrix(mat);
 return bad_v;
}

void simplex_delauney_test()
{
  facet_id f_id;
  
  FOR_ALL_FACETS(f_id)
  {
   vertex_id v_id, *v = get_facet_vertices(f_id);
   v_id = void_test(v,web.dimension);
   if ( valid_id(v_id) )
       fprintf(outfd,"Vertex %d inside void of facet %d\n",ordinal(v_id)+1,
           ordinal(f_id)+1);
  }

}

vertex_id find_other_vertex(v,dim,otherv)
vertex_id *v;  /* vertices of face */
int dim;  /* dimension of face */
vertex_id otherv; /* known vertex for other simplex on face */
{
   REAL *x[MAXCOORD+1];
   int k,j,i;
   REAL RRv;  /* square radius of void */
   REAL minRR;
   REAL rr;  /* square radius of face void */
   REAL center[MAXCOORD];
   REAL lam[MAXCOORD];
   REAL **qform,**temp,**side,**mat;
   REAL ss[MAXCOORD];  /* squares of sides */
   REAL *otherx;
   vertex_id v_id;
#define MAXKEEPERS 15
   struct keeper {
     vertex_id v;  /* completers of face */
     REAL z[MAXCOORD];
     REAL center[MAXCOORD];
     REAL RR;
   } keeper[MAXKEEPERS];
   int  keepers = 0;
   vertex_id best_v;

/*
   qform = dmatrix(0,web.sdim-1,0,web.sdim-1);
   temp = dmatrix(0,web.sdim-1,0,web.sdim-1);
   side = dmatrix(0,dim-1,0,web.sdim-1);
   mat = dmatrix(0,dim-1,0,dim-1);
*/
   qform = dmatrix(0,MAXCOORD,0,MAXCOORD);
   temp  = dmatrix(0,MAXCOORD,0,MAXCOORD);
   side  = dmatrix(0,MAXCOORD,0,MAXCOORD);
   mat   = dmatrix(0,MAXCOORD,0,MAXCOORD);

   /* first, calculate center of void */
   x[0] = get_coord(v[0]);
   for ( k = 1 ; k <= dim ; k++ )
     {  x[k] = get_coord(v[k]);
        for ( j = 0 ; j < web.sdim ; j++ )
          side[k-1][j] = x[k][j] - x[0][j];
     }
   for ( k = 0 ; k < dim ; k++ )
     { 
       for ( j = 0 ; j <= k ; j++ )
         mat[j][k] = mat[k][j] = dot(side[j],side[k],web.sdim);
       ss[k] = mat[k][k];
     }
   mat_inv(mat,dim);
   matvec_mul(mat,ss,lam,dim,dim);
   rr = dot(lam,ss,dim)/4;
   vec_mat_mul(lam,side,center,dim,web.sdim);
   for ( j = 0 ; j < web.sdim ; j++ )
      center[j] /= 2.0;

    /* quadratic form for evaluating b^2 */
   mat_mult(mat,side,temp,dim,dim,web.sdim);
   tr_mat_mul(side,temp,qform,dim,web.sdim,web.sdim);
   for ( j = 0 ; j < web.sdim ; j++ )
     for ( k = 0 ; k < web.sdim ; k++ )
       { qform[j][k] = -qform[j][k];
         if ( j == k ) qform[j][k] += 1.0;
       }        

   /* go thru all other vertices */
   if ( valid_id(otherv) ) 
     { otherx = get_coord(otherv);
       keeper[keepers].v = otherv;
       for ( j = 0 ; j < web.sdim ; j++ )
         keeper[keepers].z[j] = otherx[j] - x[0][j];
       keepers++;
     }
   FOR_ALL_VERTICES(v_id)
    { REAL *y,z[MAXCOORD],aa,bb,b,c,cvec[MAXCOORD],bvec[MAXCOORD];
      REAL vcenter[MAXCOORD];
      REAL zc[MAXCOORD];

      /* skip those in current face */
      for ( j = 0 ; j <= dim ; j++ )
        if ( equal_id(v_id,v[j]) ) break;
      if ( j <= dim ) continue;
      if ( equal_id(v_id,otherv) ) continue;

      y = get_coord(v_id);
      for ( j = 0 ; j < web.sdim ; j++ )
        z[j] = y[j] - x[0][j];  /* relative to v[0] */
      matvec_mul(qform,z,bvec,web.sdim,web.sdim);
      bb = dot(z,bvec,web.sdim);
      if ( fabs(bb) < 1E-14 ) continue;
      for ( j = 0 ; j < web.sdim ; j++ )
        zc[j] = z[j] - center[j];
      aa = dot(zc,zc,web.sdim);
      RRv = rr + (aa - rr)*(aa - rr)/4/bb;
      c = (aa - rr)/2/bb;
      for ( j = 0 ; j < web.sdim ; j++ )
        vcenter[j] = center[j] + c*bvec[j];
#ifdef xxx
{
double qq=dot(vcenter,vcenter,web.sdim);
fprintf(outfd,"RRv: %f   vcenter: %f\n",RRv,(double)qq);
fprintf(outfd,"new v: %f\n",dot(z,z,web.sdim)-2*dot(z,vcenter,web.sdim)+qq);
for ( j = 0 ; j < dim ; j++ )
  fprintf(outfd," %f ",ss[j]-2*dot(side[j],vcenter,web.sdim)+qq);
fprintf(outfd,"\n"); getchar();
}
#endif

      if ( RRv < 0.0 ) continue; /* really infinite */
      /* see if new void contains any previous keepers */
      for ( i = 0 ; i < keepers ; i++ )
        { double dd;
          for ( j = 0 ; j < web.sdim ; j++ )
            cvec[j] = keeper[i].z[j] - vcenter[j];
          dd = dot(cvec,cvec,web.sdim );
          if ( dd < RRv ) goto cont;
        }
      /* see if any previous keepers displaced */
      for ( i = (valid_id(otherv)?1:0) ; i < keepers ; i++ )
        { double dd;
          for ( j = 0 ; j < web.sdim ; j++ )
            cvec[j] = keeper[i].center[j] - z[j];
          dd = dot(cvec,cvec,web.sdim );
          if ( dd < keeper[i].RR ) 
            keeper[i] = keeper[--keepers]; /* pop */
        }
      /* add to keeper list */
      if ( keepers >= MAXKEEPERS )
        { error("Too many keepers.\n",WARNING); continue; }
      keeper[keepers].v = v_id;
      keeper[keepers].RR = RRv;
      for ( j = 0 ; j < web.sdim ; j++ )
        { keeper[keepers].z[j] = z[j];
          keeper[keepers].center[j] = vcenter[j];
        }
      keepers++;

cont: ;
    }
  free_matrix(mat);
  free_matrix(temp);
  free_matrix(qform);
  free_matrix(side);

  /* recheck all voids */
 {
  vertex_id vv[MAXCOORD+1];
  best_v = NULLID;     
  for ( i = 0 ; i <= dim ; i++ ) vv[i] = v[i];
  for ( i =  (valid_id(otherv)?1:0) ; i < keepers ; i++ )
   { vv[dim+1] = keeper[i].v;
     v_id = void_test(vv,dim+1);
     if ( valid_id(v_id) )
       { keeper[i] = keeper[--keepers]; i--; }  /* pop and retest */
   }
  }
  for ( i =  (valid_id(otherv)?1:0), minRR = 1e30 ; i < keepers ; i++ )
   { if ( keeper[i].RR < minRR )
       { minRR = keeper[i].RR;
         best_v = keeper[i].v; 
       } /* keep smallest void */
   }

#ifdef UBUG
fprintf(outfd,"search "); 
for ( i = 0 ; i <= dim ; i++ ) fprintf(outfd," %d",ordinal(v[i])+1);

if ( valid_id(best_v) )
  fprintf(outfd,"  found %d\n",ordinal(best_v)+1);
else fprintf(outfd,"  vertex not found.\n");
#endif

  return best_v;
}


