/* sparse.c */

/* symmetric sparse matrix factoring and system solving */

#include "include.h"

/********************************************************************
*
* function: sp_order()
*
* purpose: find order of elimination of variables
*
* input: N,IA,JA
*
* output: P,IP
*/

void sp_order(N,IA,JA,P,IP)
int N;  /* size of matrix */
int *IA;  /* row starts in JA, size N+1 (last entry beyond end of JA) */
int *JA;  /* column labels for each row, increasing order */
int *P;   /* permutation list; variables in order of elimination */
int *IP;  /* inverse permutation list */
{
  int i,j;
  int minnbr,mini,minj;
  int count = 0;
  int emax = 2*IA[N];  /* oriented edges */
  int *IE = (int*)calloc(N+1,sizeof(int));
  int *JE = (int*)calloc(emax,sizeof(int));
  int *nbrs = (int*)calloc(N,sizeof(int));
  int active_lo,active_hi;
  int temp;

  /*********************************/
  /* construct full adjacency list */
  /*********************************/

  /* count adjacent vertices of each vertex */
  for ( i = 0 ; i < N ; i++ )
    for ( j = IA[i]+1 ; j < IA[i+1] ; j++ )
      { IE[JA[j]]++;
        IE[i]++;
      }

  /* allocate space in JE */
  for ( i = 0, count = 0 ; i < N ; i++ )
    { int temp = IE[i];
      IE[i] = count;
      count += temp;
    }
  IE[N] = count;

  /* fill in adjacency list, keeping counts in nbrs */
  for ( i = 0 ; i < N ; i++ )
    for ( j = IA[i]+1 ; j < IA[i+1] ; j++ )
      { int ii = JA[j];
        JE[IE[i]+nbrs[i]++] = ii;
        JE[IE[ii]+nbrs[ii]++] = i;
      }
     
  /**********************************************
  *  Now eliminate in order so that eliminated  *
  *  active variable introduces fewest new      *
  *  active variables.                          *
  **********************************************/

#define INACTIVE -2
#define ACTIVE   -1
  for ( i = 0, minnbr = N ; i < N ; i++ ) IP[i] = INACTIVE;

  /* starting vertex, with lowest number of nbrs */
  for ( i = 0, minnbr = N ; i < N ; i++ )
    if ( nbrs[i] < minnbr ) { mini = i ; minnbr = nbrs[i]; }
  P[0] = mini;
  active_lo = 0; active_hi = 1; /* activate first variable */

  /* main loop, eliminating active variable with fewest new nbrs */
  for ( i = 0 ; i < N ; i++ )
    {
      /* find active var with fewest inactive nbrs */
      for ( j = active_lo, minnbr = N ; j < active_hi ; j++ )
         if ( -IP[P[j]] < minnbr ) { minj = j; minnbr = -IP[P[j]]; }
      if ( minnbr == N )
        { printf("Variables disconnected.\n"); 
          exit(1);
        }
      temp = P[minj]; P[minj] = P[active_lo]; P[active_lo] = temp;
      IP[P[active_lo]] = active_lo;
      active_lo++;

      /* activate nbrs */
      for ( j = IE[temp] ; j < IE[temp+1] ; j++ )
        if ( IP[JE[j]] == INACTIVE )
          { /* add to active list */
            int k,aj = JE[j];
            IP[aj] = ACTIVE;
            P[active_hi++] = aj;
            /* remove as inactive from its nbrs */
            for ( k = JE[aj] ; k < JE[aj+1] ; k++ )
              nbrs[JE[k]]--;
          }
     }
  free((char*)IE);
  free((char*)JE);
  free((char*)nbrs);
}


/********************************************************************
*
*  function:  sp_fill_setup()
*
*  purpose:   Set up indexing for filled matrix.
*
*  Inputs:  N, IA, JA, P, IP
*
*  Output:  IW
*
*  Return:  entries to be allocated to hold filled matrix
*/

int sp_fill_setup(N,IA,JA,P,IP,IW)
int N;  /* size of matrix */
int *IA;  /* row starts in JA, size N+1 (last entry beyond end of JA) */
int *JA;  /* column labels for each row, increasing order */
int *P;   /* permutation list; variables in order of elimination */
int *IP;  /* inverse permutation list */
int *IW;  /* starts of filled rows, size N+1 */
{
  int i,j;
  int count = 0;
  int final;

  /* find last entries in rows */
  for ( i = 0 ; i < N ; i++ )
    { int pi = IP[i];
      for ( j = IA[i] ; j < IA[i+1] ; j++ )
        { int pj = IP[JA[j]];
          if ( pi <= pj )
            { if ( pj > IW[pi] ) IW[pi] = pj;
            }
          else if ( pi > IW[pj] ) IW[pj] = pi;
        }
    }
        
  /* fill in IW offsets */
  for ( i = 0, final = 0 ; i < N ; i++ )
    { 
      if ( IW[i] > final ) final = IW[i];
      IW[i] = count;
      count += final - i + 1;
    }
  IW[N] = count;

  return count;
}

void sp_factor(N,IA,JA,P,IP,IW,A,F)
int N;  /* size of matrix */
int *IA;  /* row starts in JA, size N+1 (last entry beyond end of JA) */
int *JA;  /* column labels for each row */
int *P;   /* permutation list; variables in order of elimination */
int *IP;  /* inverse permutation list */
int *IW;  /* starts of filled rows, size N+1 */
double *A;  /* sparse matrix entries, as labelled by JA */
double *F;  /* allocated room for filled matrix, returned factor */
{
  int n; /* row index */

  /* copy A over to F */
  memset((char*)F,0,IW[N]*sizeof(double));
  for ( n = 0 ; n < N ; n++ )
    { int j, maxj = IA[n+1];
      int pn = IP[n];
      for ( j = IA[n] ; j < maxj ; j++ )
        { int pj = IP[JA[j]];
          if ( pj > pn )
            F[IW[pn]+pj-pn] = A[j];
          else
            F[IW[pj]+pn-pj] = A[j];
        }
    }

  /* main row loop for factoring */
  for ( n = 0 ; n < N ; n++ )
    { int i,maxi = IW[n+1] -IW[n];
      double piv = F[IW[n]];
      if ( piv == 0.0 )
        { printf("Zero pivot.");
          exit(1);
        }
      for ( i = 1 ; i < maxi ; i++ )
        { double r = F[IW[n]+i]/piv; /* col multiplier */
          int j;
          for ( j = n+1 ; j <= n+i ; j++ )
            F[IW[j] + n + i - j] -= r*F[IW[n] + j - n];
        }
    }
}


void sp_backsub(N,P,IP,IW,F,B,X)
int N;  /* size of matrix */
int *P;   /* permutation list; variables in order of elimination */
int *IP;  /* inverse permutation list */
int *IW;  /* starts of filled rows, size N+1 */
double *F;  /* allocated room for filled matrix, returned factor */
double *B;  /* incoming right side */
double *X;  /* returned solution */
{
  int n; /* row index */
  double *Y = X;  /* synonym */

  /* solve LY = B */
  for ( n = 0 ; n < N ; n++ )
    { int i;
      double sum = B[P[n]];
      int firsti = 0;

      /* find first contribution to sum */
      while ( (n - firsti) >= (IW[firsti+1] - IW[firsti]) ) 
        firsti++;
      
      /* do sum */
      for ( i = firsti ; i < n ; i++ )
        sum -= F[IW[i] + n - i]*Y[P[i]];
      Y[P[n]] = sum/F[IW[n]];
    }
      
  /* solve DUX = Y */
  for ( n = N-1 ; n >= 0 ; n-- )
    { int i, maxi = IW[n+1] - IW[n];
      double piv = F[IW[n]];
      double sum = piv*Y[P[n]];
      for ( i = 1 ; i < maxi ; i++ )
        sum -= F[IW[n]+i]*X[P[n+i]];
      X[P[n]] = sum/piv;
    }
}

#ifdef SPTEST

/* testing */

int IA[7] = { 0, 3, 5, 8, 10, 12, 13};
int JA[19] = { 0,1,2,1,3,2,3,4,3,5,4,5,5};
double A[19] = { 3.0,0.0,0.0,3.0,0.0,3.0,0.0,0.0,3.0,0.0,3.0,0.0,3.0,3.0 };
double B[6] = {3.0,3.0,6.0,6.0,9.0,9.0};
double X[6];
int P[6];
int IP[6];

main()
{
  int fillsize;
  int N = 6;
  int n;
  int *IW;
  double *F;

  IW = calloc(N+1,sizeof(int));
  sp_order(N,IA,JA,P,IP);
  fillsize = sp_fill_setup(N,IA,JA,P,IP,IW);
  printf("fillsize = %d\n",fillsize);
  F = calloc(fillsize,sizeof(double));
  sp_factor(N,IA,JA,P,IP,IW,A,F);
  sp_backsub(N,P,IP,IW,F,B,X);
  for ( n = 0 ; n < N ; n++ )
    printf("%d.   %f\n",n,X[n]);
}
#endif
    
  
