/*************************************************************
*  This file is part of the Surface Evolver source code.     *
*  Programmer:  Ken Brakke, brakke@geom.umn.edu              *
*************************************************************/

/************************************************************************
*
*  File:  stringl.c
*
*  Contents: Functions to calculate area, energy and their gradients
*            according to the LINEAR STRING model.
*/

#include "include.h"

/************************************************************************
*
*  Calculates all forces on control points due to edge and
*  accumulates them at each control point.
*/

void edge_force_l(e_id)
edge_id e_id;
{
  REAL len;
  REAL side[MAXCOORD];
  int i,j;
  vertex_id tv = get_edge_tailv(e_id);
  vertex_id hv = get_edge_headv(e_id);
  REAL density = get_edge_density(e_id);
  REAL forces[2][MAXCOORD];  /* total forces from this facet */
  REAL *forceptr[2];   /* pointers to forces */
  REAL *force[2];  /* vertex forces */
  WRAPTYPE wraps[2];

  memset((char*)forces,0,sizeof(forces));  /* set to 0 */

  get_edge_side(e_id,side);

  /* force due to linear tension */
  if ( web.metric_flag )
    { vertex_id v[2];
      REAL *x[2];
      REAL xx[2][MAXCOORD];
      v[0] = tv; x[0] = get_coord(tv);
      v[1] = hv; x[1] = get_coord(hv);
      if ( web.symmetry_flag )
        { for ( j = 0 ; j < web.sdim ; j++ )
            xx[1][j] = x[1][j] + side[j];  /* unwrap */
          x[1] = xx[1];
        }
      forceptr[0] = forces[0]; forceptr[1] = forces[1];
      simplex_force_metric(v,x,density,forceptr);
      goto dograv;
    }

  len = sqrt(dot(side,side,web.sdim));
  set_edge_length(e_id,len);
  /* add to endpoint stars */
    { REAL *hw,*tw,dw;
      /* following two lines for old area normalization option */
      add_vertex_star(tv,len);
      add_vertex_star(hv,len);

      add_vertex_valence(tv,1);
      add_vertex_valence(hv,1);
    }
  if ( len != 0.0 ) 
    for ( i = 0 ; i < web.sdim ; i++ )
      {
        forces[0][i] += density*side[i]/len;
        forces[1][i] -= density*side[i]/len;
      }


dograv:
  /* calculate gravitational forces */
  if ( web.gravflag )
    { REAL *t,*h;
      REAL density = edge_grav_density(e_id);

      if ( density != 0.0 )
        {
          t = get_coord(tv);
          h = get_coord(hv);
          forces[0][0] += -density*(t[1]*t[1] + t[1]*h[1] + h[1]*h[1])/6;
          forces[1][0] +=  density*(t[1]*t[1] + t[1]*h[1] + h[1]*h[1])/6;
          forces[0][1] += -density*(t[0]-h[0])*(2*t[1] + h[1])/6;
          forces[1][1] += -density*(t[0]-h[0])*(t[1] + 2*h[1])/6;
        }
    }               
  force[0] = get_force(tv);
  force[1] = get_force(hv);
  if ( web.symmetry_flag )
    { REAL wforce[MAXCOORD];  /* unwrapped forces */
      REAL *t = get_coord(tv);

      for ( j = 0 ; j < web.sdim ; j++ )
           force[0][j] += forces[0][j];
      wraps[1] = get_edge_wrap(e_id);
      (*sym_form_pullback)(t,wforce,forces[1],wraps[1]);
      for ( j = 0 ; j < web.sdim ; j++ )
           force[1][j] += wforce[j];
    }
  else
   for ( i = 0 ; i < 2 ; i++ )
     for ( j = 0 ; j < web.sdim ; j++ )
           force[i][j] += forces[i][j];
}


/************************************************************************
*
*  Returns energy due to one edge.
*  Find edge's contribution to total length and facet areas.
*  Areas with respect to origin are calculated 
*  and then oriented contributions added for each facet. 
*
*/

void edge_energy_l(e_id)
edge_id e_id;
{
  REAL side[MAXCOORD];
  REAL wulff[MAXCOORD];   /* energy covector to side vector */
  REAL energy;
  vertex_id v[2];
  int i,j;

  v[0] = get_edge_tailv(e_id);
  v[1] = get_edge_headv(e_id);
  get_edge_side(e_id,side);              

  /* energy due to linear tension */
  if ( web.metric_flag )
    { REAL *x[2];
      REAL xx[2][MAXCOORD];
      for ( i = 0 ; i < 2 ; i++ ) x[i] = get_coord(v[i]);
      if ( web.symmetry_flag )
        { for ( j = 0 ; j < web.sdim ; j++ )
            xx[1][j] = x[0][j] + side[j];  /* unwrap */
          x[1] = xx[1];
        }
      energy = simplex_energy_metric(v,x);
    }
  else
    { /* calculate length energy */
      energy = sqrt(dot(side,side,web.sdim));
      set_edge_length(e_id,energy);
      if ( web.wulff_flag ) 
         { (*get_wulff)(side,wulff);
           energy = dot(side,wulff,web.sdim);
         }
    }
  if ( web.dimension == STRING )
     web.total_area   += energy;   /* don't count triple junction as area */

  /* accumulate 1/2 area around each vertex to scale motion */
/*  
    for ( i = 0 ; i < 2 ; i++ )
      add_vertex_star(v[i],energy);
*/

  energy *= get_edge_density(e_id);

  /* calculate gravitational energy */
  if ( web.gravflag )
    { REAL *t,*h;
      REAL grav_e;
      REAL density = edge_grav_density(e_id);

      if ( density != 0.0 )
        {
          t = get_coord(v[0]);
          h = get_coord(v[1]);
          grav_e = (t[0]-h[0])*(t[1]*t[1] + t[1]*h[1] + h[1]*h[1])/6;
          energy += density*grav_e;
        }
    }               

  web.total_energy += energy;

}

/******************************************************************
*
*  Function: edge_grav_density()
*
*  Purpose: Finds gravitational energy density multiplier across
*           an edge in the string model. Includes grav const.
*/

REAL edge_grav_density(e_id)
edge_id e_id;
{
  facetedge_id fe;
  body_id  b_id;
  REAL density = 0.0;
  facet_id f_id;

  /* if body is on facet agreeing with edge orientation,
     then pressure is positive. */

  generate_edge_fe_init();
  while ( generate_edge_fe(e_id,&fe) )
   {
     f_id = get_fe_facet(fe);
     b_id = get_facet_body(f_id);
     if ( (valid_id(b_id)) && (get_battr(b_id) & DENSITY) )
       density += get_body_density(b_id);
     b_id = get_facet_body(inverse_id(f_id));
     if ( (valid_id(b_id)) && (get_battr(b_id) & DENSITY) )
       density -= get_body_density(b_id);
   }

  return web.grav_const*density;
}
 
/**********************************************************************
*
*  Adds contribution of edge to areas of neighboring facets, via
*  Green's theorem.   x-y plane only.
*
*/

void edge_area_l(fe_id)
facetedge_id fe_id;
{
  REAL *x,side[MAXCOORD];
  REAL area;
  body_id b_id;
  edge_id e_id = get_fe_edge(fe_id);

  if ( web.torus_flag )
    {
      REAL *xt,*xh;
      double v1,v2,v3,v4;
      int wx=0,wy=0; /* wraps */
      WRAPTYPE wrap = get_edge_wrap(e_id);
  
    /* new way, accurate modulo unit cell area */

      xt = get_coord(get_edge_tailv(e_id));    
      xh = get_coord(get_edge_headv(e_id));    
      switch ( wrap & WRAPMASK )
        { 
          case POSWRAP: wx = 1 ; break;
          case NEGWRAP: wx = -1; break;
        }
      switch ( (wrap>>TWRAPBITS) & WRAPMASK )
        { 
          case POSWRAP: wy = 1 ; break;
          case NEGWRAP: wy = -1; break;
        }
    
      v1 = dot(web.inverse_periods[0],xt,2);
      v2 = dot(web.inverse_periods[0],xh,2);
      v3 = dot(web.inverse_periods[1],xt,2);
      v4 = dot(web.inverse_periods[1],xh,2);

      area = ((v1 + v2 + wx)*(v4 - v3 + wy)/2 - wx*v4)*web.torusv;

    } /* end new way */
  else
    { 
      get_edge_side(e_id,side);              
      x = get_coord(get_edge_tailv(e_id));    
    
      /* calculate area above x1 axis */ 
      area = -side[0]*(x[1] + side[1]/2);

#ifdef OLDWAY
      /* accurate only within half a cell area */
printf("fe %lX    raw area: %f      ",fe_id,area);
      if ( web.torus_flag )
        { /* adjust for wrapping */
          int w = get_edge_wrap(e_id);
          for ( i = 0 ; i < 2 ; i++,w>>TWRAPBITS )
            {
              if ( (w&WRAPMASK) == 0 ) continue;
              if ( (w&WRAPMASK) == POSWRAP ) /* can get true position of head x[0] */
                area += (x[0] + side[0])*web.torus_period[i][1];
              else 
                area -= (x[0] + side[0] + web.torus_period[i][0])
                       *web.torus_period[i][1];
            }
        }
printf("adjusted: %f \n",area);
#endif
     }

  /* add to cell areas */
  b_id = get_facet_body(get_fe_facet(fe_id));
  if ( valid_id(b_id ) )
    set_body_volume(b_id,get_body_volume(b_id) + area);
  
  b_id = get_facet_body(facet_inverse(get_fe_facet(fe_id)));
  if ( valid_id(b_id ) )
    set_body_volume(b_id,get_body_volume(b_id) - area);
}

/***********************************************************************
*
*  Function: string_grad_l()
*
*  Purpose:  construct vertex area gradients
*            Note: force is local, so toroidal wrappings don't matter,
*            as long as we get the edges right, which get_edge_side() does.
*
*/

void string_grad_l()
{
  body_id bi_id;  /* identifier for body i */
  facetedge_id fe_id;
  REAL side[MAXCOORD];
  volgrad *hvgptr,*tvgptr;
  vertex_id headv,tailv;
  edge_id e_id;

  FOR_ALL_FACETEDGES(fe_id)
    {
      /* see which side has body, if any */
      bi_id = get_facet_body(get_fe_facet(fe_id));
      if ( !valid_id(bi_id) ) 
        { invert(fe_id);
          bi_id = get_facet_body(get_fe_facet(fe_id));
        }
      if ( !valid_id(bi_id) ) continue;  /* nothing here */
      if ( !(get_battr(bi_id) & FIXEDVOL) ) continue;

      headv = get_fe_headv(fe_id);
      tailv = get_fe_tailv(fe_id);

      /* gradient due to edge */
      e_id = get_fe_edge(fe_id);
      get_edge_side(e_id,side);

      hvgptr = get_bv_new_vgrad(bi_id,headv);
      tvgptr = get_bv_new_vgrad(bi_id,tailv);

      hvgptr->grad[0] +=  side[1]/2;
      hvgptr->grad[1] += -side[0]/2;
      tvgptr->grad[0] +=  side[1]/2;
      tvgptr->grad[1] += -side[0]/2;
   }
}

/*******************************************************************
*
*  Function: string_bdry_grad()
*
*  Purpose: Add cell area gradients due to boundary integrals.
*/

void string_bdry_grad()
{
  vertex_id v_id;
  REAL dummy,partial; /* for eval_all */

  FOR_ALL_VERTICES(v_id)
    {
      volgrad *vgptri;
      struct boundary *bdry;
      ATTR attr = get_vattr(v_id);

      if ( attr & FIXED ) continue;
      if ( !(attr & BOUNDARY) ) continue;
      if ( !(attr & BDRY_CONTENT) ) continue;

      /* assuming boundary vertex is on just one cell */

      vgptri = get_vertex_vgrad(v_id);
      if ( !vgptri ) continue;
      bdry = get_boundary(v_id);
      if ( bdry->pcount != 1 ) return ;

      eval_all(bdry->convect[0],get_param(v_id),1,&dummy,&partial);
      vgptri->grad[0] += partial;  
   }
}

/*******************************************************************
*
*  Function: string_constr_grad()
*
*  Purpose: Add cell area gradients due to constraint integrals.
*/

void string_constr_grad()
{
  vertex_id v_id;

  FOR_ALL_VERTICES(v_id)
    {
      struct volgrad *vgptri;
      struct constraint *constr;
      facetedge_id fe_id;
      ATTR attr = get_vattr(v_id);

      if ( attr & FIXED ) continue;
      if ( !(attr & CONSTRAINT) ) continue;
      if ( !(attr & BDRY_CONTENT) ) continue;
      fe_id = get_vertex_fe(v_id);
      if ( !valid_id(fe_id) ) continue;

      vgptri = get_vertex_vgrad(v_id);
      while ( vgptri )
        {
          int sign=1,i,j;
          MAP conmap = get_v_constraint_map(v_id);

          if ( !equal_id(get_facet_body(get_fe_facet(fe_id)),vgptri->b_id) )
            { fe_id = inverse_id(fe_id);
              if ( !equal_id(get_facet_body(get_fe_facet(fe_id)),vgptri->b_id) )
                { outstring("Can't find vertex body.\n");
                }
              else sign = -1;
            }

          for ( j = 0 ; j < web.concount ; j++, conmap >>= 1 )
            { REAL val,derivs[MAXCOORD];
              if ( !(conmap & 1) ) continue;
              constr = get_constraint(j);
              if ( (constr->attr & BDRY_CONTENT) ) continue;
              eval_all(constr->convect[0],get_coord(v_id),web.sdim,&val,derivs);
              for ( i = 0 ; i < web.sdim ; i++ )
                vgptri->grad[i] += sign*derivs[i];  
            }
          vgptri = vgptri->chain;
        }
   }
}
