(* $Id: Complex.m,v 2.0 89/10/03 21:38:31 mbp Exp $
 *    
 * Complex.m: teach Mathematica a few things about complex expressions
 *)

(***************************************************************************
 *                          Copyright (C) 1990 by                          *
 *        Mark B. Phillips, William M. Goldman, and Robert R. Miner        *
 *                                                                         *
 *  Permission to use, copy, modify, and distribute this software, its     *
 *  documentation, and any images it generates for any purpose and without *
 *  fee is hereby granted, provided that                                   *
 *                                                                         *
 *  (1) the above copyright notice appear in all copies and that both that *
 *      copyright notice and this permission notice appear in supporting   *
 *      documentation, and that the names of Mark B.  Phillips, William M. *
 *      Goldman, Robert R.  Miner, or the University of Maryland not be    *
 *      used in advertising or publicity pertaining to distribution of the *
 *      software without specific, written prior permission.               *
 *                                                                         *
 *  (2) Explicit written credit be given to the authors Mark B. Phillips,  *
 *      William M. Goldman, and Robert R. Miner in any publication which   *
 *      uses part or all of any image produced by this software.           *
 *                                                                         *
 * This software is provided "as is" without express or implied warranty.  *
 ***************************************************************************)

BeginPackage["Complex`"]

DeclareReal::usage = "DeclareReal[x, y, ...] declares x, y, ... to be
real variables or functions which take real numbers to real numbers."

ComplexNormSquared::usage = "ComplexNormSquared[z] returns the square of
the absolute value of the complex number z."

Begin["`Private`"]

protected = Unprotect[Re, Im, Conjugate, Arg]

ComplexNormSquared[z_] := Re[z]^2 + Im[z]^2

RealQ[_] = False

DeclareReal[x_] := (DeclaredRealQ[x] ^= True; Re[x] ^= x; Im[x] ^= 0; Null)
DeclareReal[x_,y__] := (Map[DeclareReal, {x,y}]; Null)

Re[Complex[x_,y_]] := x
Im[Complex[x_,y_]] := y
  
Im[Re[z_]]  := 0
Im[Im[z_]]  := 0
Im[Abs[z_]] := 0
Re[Re[z_]]  := Re[z]
Re[Im[z_]]  := Im[z]
Re[Abs[z_]] := Abs[z]
Re[Pi]	    := Pi
Im[Pi]      := 0
Re[E]	    := E
Im[E]       := 0
  
Re[Literal[Plus[summands__]]] := Apply[Plus,Map[Re,{summands}]]
Im[Literal[Plus[summands__]]] := Apply[Plus,Map[Im,{summands}]]

Re[x_ y_] := Re[x] Re[y] - Im[x] Im[y]
Im[x_ y_] := Re[x] Im[y] + Im[x] Re[y]

Re[1 / x_] := Block[{rex=Re[x]},  rex / (rex^2   + Im[x]^2)]
Im[1 / x_] := Block[{imx=Im[x]}, -imx / (Re[x]^2 + imx^2  )]

Re[ x_?DeclaredRealQ ^ n_Integer] := x^n
Im[ x_?DeclaredRealQ ^ n_Integer] := 0

Re[ x_ ^ n_?EvenQ ] := Re[x^(n/2)]^2 - Im[x^(n/2)]^2

Im[ x_ ^ n_?EvenQ ] := 2 Re[x^(n/2)] Im[x^(n/2)]

Re[ x_ ^ n_Integer ] :=
  Block[{a, b},
    a = Round[n/2];
      b = n-a;
      Re[x^a] Re[x^b] - Im[x^a] Im[x^b]
  ]

Im[ x_ ^ n_Integer ] :=
  Block[{a, b},
    a = Round[n/2];
      b = n-a;
      Re[x^a] Im[x^b] + Im[x^a] Re[x^b]
  ]

Re[E^x_] := Cos[Im[x]] Exp[Re[x]]
Im[E^x_] := Sin[Im[x]] Exp[Re[x]]

Re[x_Integer ^ n_Rational] := 0		/; IntegerQ[2n] && x<0
Im[x_Integer ^ n_Rational] :=
  (-x)^n (-1)^((Numerator[n]-1)/2)	/; IntegerQ[2n] && x<0

Re[x_Integer ^ n_Rational] := x^n	/; OddQ[Denominator[n]] || x>0
Im[x_Integer ^ n_Rational] := 0		/; OddQ[Denominator[n]] || x>0

Re[(-1) ^ n_Rational] := Cos[n Pi]
Im[(-1) ^ n_Rational] := Sin[n Pi]

Re[Sin[x_]] := Sin[Re[x]] Cosh[Im[x]]
Im[Sin[x_]] := Cos[Re[x]] Sinh[Im[x]]
Re[Cos[x_]] :=  Cos[Re[x]] Cosh[Im[x]]
Im[Cos[x_]] := -Sin[Re[x]] Sinh[Im[x]]

Re[Log[r_?Positive]] := Log[r]
Im[Log[r_?Positive]] := 0
Re[Log[r_?Negative]] := Log[-r]  
Im[Log[r_?Negative]] := Pi

Re[Log[z_]] := (1/2) Log[Re[z]^2 + Im[z]^2]
Im[Log[z_]] := Arg[z]

Re[Log[a_ b_]] := Re[Log[a] + Log[b]]
Re[Log[a_ b_]] := Im[Log[a] + Log[b]]
Re[Log[a_ ^ c_]] := Re[c Log[a]]
Im[Log[a_ ^ c_]] := Im[c Log[a]]

Re[Tan[x_]] := Re[Sin[x]/Cos[x]]
Im[Tan[x_]] := Im[Sin[x]/Cos[x]]

Re[Conjugate[z_]] :=  Re[z]
Im[Conjugate[z_]] := -Im[z]

Conjugate[Literal[Plus[summands__]]] :=
  Apply[Plus, Map[Conjugate, {summands}]]
Conjugate[Literal[Times[factors__]]] :=
  Apply[Times, Map[Conjugate, {factors}]]
Conjugate[x_ ^ n_Integer] := Conjugate[x]^n
Conjugate[Conjugate[x_]] := x
Conjugate[x_?DeclaredRealQ] := x  

Arg[z_] := 0 /; Im[z] == 0

Protect[ Release[protected] ]

End[]

EndPackage[]

