/*
 * sliders.c: slider submodule
 */

/***************************************************************************
 *                Copyright (C) 1990 by Mark B. Phillips                   *
 *                                                                         *
 *  Permission to use, copy, modify, and distribute this software, its     *
 *  documentation, and any images it generates for any purpose and without *
 *  fee is hereby granted, provided that                                   *
 *                                                                         *
 *  (1) the above copyright notice appear in all copies and that both      *
 *      that copyright notice and this permission notice appear in         *
 *      supporting documentation, and that the names of Mark B.            *
 *      Phillips, or the University of Maryland not be used in             *
 *      advertising or publicity pertaining to distribution of the         *
 *      software without specific, written prior permission.               *
 *                                                                         *
 *  (2) Explicit written credit be given to the author Mark B. Phillips    *
 *      in any publication which uses part or all of any image produced    *
 *      by this software.                                                  *
 *                                                                         *
 * This software is provided "as is" without express or implied warranty.  *
 ***************************************************************************/

#include <stdio.h>
#include "glg.h"
#include "sliders.h"
#include "internal.h"
#include "device.h"
#include "get.h"

static int (*DrawWindow)() = NULL;

/*-----------------------------------------------------------------------
 * Function:	GlgSliderInit
 * Description:	initialize the slider subpackage
 * Args  IN:	drawwindow: procedure for redrawing entire window.
 *		  Call syntax is just 'drawwindow()'.
 * Returns:	nothing
 * Author:	mbp
 * Date:	Mon Jun  4 14:12:16 1990
 * Notes:	Call this once before processing any gadget events.
 */
int
  GlgSliderInit(drawwindow)
int (*drawwindow)();
{
  DrawWindow = drawwindow;
}

/*-----------------------------------------------------------------------
 * Function:	GlgSliderDraw
 * Description:	Draw a slider
 * Args  IN:	*data: the slider to draw
 * Returns:	nothing
 * Author:	slevy, with modifications by mbp
 * Date:	Sun Jun  3 16:00:33 1990
 */
int
  GlgSliderDraw(data)
register char *data;
{
  register GlgSlider *sl = (GlgSlider*)data;
  register double xs, ys;
  register double xc;
  
  color(sl->bgnd);
  rectf(sl->xcen - sl->xsz, sl->ycen - sl->ysz,
	sl->xcen + sl->xsz, sl->ycen + sl->ysz);
  
  color(sl->inbar);
  xs = .9 * sl->xsz;
  ys = .7 * sl->ysz;
  rectf(sl->xcen - xs, sl->ycen - ys,
	sl->xcen + xs, sl->ycen + ys);
  
  color(sl->barmark);
  xc = (sl->umin == sl->umax) ? .5 :
    (sl->u - sl->umin) / (sl->umax - sl->umin);
  if(xc < 0) xc = 0;
  else if(xc > 1) xc = 1;
  xc = sl->xcen + .9 * sl->xsz * (2*xc - 1);
  ys = .9 * sl->ysz;
  xs = ys * .2;
  rectf(xc - xs, sl->ycen - ys,  xc + xs, sl->ycen + ys);
  
  if (sl->format != NULL) {
    char msg[50];
    
    switch (sl->type) {
    case S_DOUBLE:
      sprintf(msg, sl->format, sl->u);
      break;
    case S_INT:
      sprintf(msg, sl->format, (int)(sl->u+.5));
      break;
    default:
      strcpy(msg, "(???)");
      break;
    }
    GlgWriteString(msg, sl->xcen, sl->ycen - 1.3*sl->ysz, 1, 2, 1,
			sl->bgnd, sl->txtcol);
  }
  
  color(sl->outline);
  rect(sl->xcen - sl->xsz, sl->ycen - sl->ysz,
       sl->xcen + sl->xsz, sl->ycen + sl->ysz);
  
}

/*-----------------------------------------------------------------------
 * Function:	GlgSliderEvent
 * Description:	Let a slider process an event
 * Args  IN:	*data: the slider
 *		event: the event
 * Returns:	1 if the slider acts on the event, 0 if not.
 * Author:	mbp
 * Date:	Mon Jun  4 17:26:53 1990
 */
int
  GlgSliderEvent(data, event)
register char *data;
GlgEvent event;
{
  register GlgSlider *sl = (GlgSlider*)data;
  register double tu;
  long buffers;
  
  if (!GlgEventInRect(event, (sl->xcen)-(sl->xsz), (sl->ycen)-(sl->ysz),
		     2*(sl->xsz), 2*(sl->ysz)))
    return(0);
  
  switch (event.dev) {
  case LEFTMOUSE:
    {
      double xold = event.x;
      int first = 1;
      
      while (getbutton(event.dev)) {
	GlgGetMouseXY(&event.x,&event.y);
	
	if ((first) || (event.x != xold)) {
	  first = 0;
	  xold = event.x;
	  
	  /* Set the slider value */
	  tu = ((event.x - sl->xcen) / (.9 * sl->xsz)) * .5 + .5;
	  if(tu < 0) tu = 0;
	  else if(tu > 1) tu = 1;
	  sl->u = sl->umin + (sl->umax - sl->umin) * tu;
	  
	  /* Redraw the slider */
	  buffers = getbuffer();
	  frontbuffer(TRUE);
	  GlgSliderDraw(sl);
	  if ( ! (buffers & FRNTBUFFER) )
	    frontbuffer(FALSE);
	  
	  /* Call the callback proc, if any */
	  if (sl->proc != NULL) {
	    switch (sl->type) {
	    case S_DOUBLE:
	      (*(sl->proc))(sl->u);
	      break;
	    case S_INT:
	      (*(sl->proc))((int)(sl->u+.5));
	      break;
	    default:
	      break;
	    }
	  }
	}
	
      }
    }
    break;
    
  case MIDDLEMOUSE:
    if (DrawWindow != NULL) {

      double xold = event.x;
      int first = 1;
      while (getbutton(event.dev)) {
	GlgGetMouseXY(&event.x,&event.y);
	if ((first) || (event.x != xold)) {
	  first = 0;
	  xold = event.x;	
	  sl->xcen = event.x;
	  sl->ycen = event.y;
	  (*DrawWindow)();
	}
      }

    }
    break;
    
  case RIGHTMOUSE:
    if (getbutton(RIGHTMOUSE))
      printf("slider is at (%5.3f, %5.3f)\n", sl->xcen,sl->ycen);
    break;
    
  }
  
  return(1);
}

