/* $Id: matrix.c,v 4.0 89/06/06 15:38:49 mbp Exp $
 *
 * matrix.c: matrix package
 *
 * By Michael J. Raue at the University of Maryland, Spring 1988.
 *
 * Minor modifications by Mark Phillips
 */

/************************************************************************
 *      Copyright (C) 1989 by Michael J. Raue, Mark B. Phillips         *
 *			and William M. Goldman                          *
 * 									*
 * Permission to use, copy, modify, and distribute this software and	*
 * its documentation for any purpose and without fee is hereby granted,	*
 * provided that the above copyright notice appear in all copies and	*
 * that both that copyright notice and this permission notice appear in	*
 * supporting documentation, and that the names of Michael J. Raue,	*
 * Mark B. Phillips, William M. Goldman, or the University of Maryland	*
 * not be used in advertising or publicity pertaining to distribution	*
 * of the software without specific, written prior permission.  This	*
 * software is provided "as is" without express or implied warranty.	*
 ************************************************************************/

/* This file is part of a general matrix package which can be used
 * separately from HEISENBERG.  The package consists of this file and
 * the file "matrix.h".  It also uses the HEISENBERG-specific file
 * "matrixentry.h".
 */

/*
   This file contains a package of matrix operations which can be executed
   on matrices of any size and any type.  The size of each matrix must be
   indicated in the parameter list, and the type of matrix entries must be
   defined in a user created file named "matrixentry.h".  Several basic 
   operations for the manipulation of this data type must also be defined
   by the user in this file.  Optionally, there are files included for
   entries of types: integer, real, double precision, and complex.  See
   "matrix.real" for an example of how such a file should be written.

   WARNING: None of the functions included checks that function arguments
            are of the correct matrix dimension.


   The following is a list of functions included and a brief description:

        1. scalar_mat() multiplies a matrix by a scalar of type 
                        matrix_entry.
 
        2. mult_mat() multiplies two matrices together.

        3. add_mat() adds two rxc matrices together.

        4. sub_mat() subtracts two rxc matricestogether().

        5. copy_mat() copys one rxc matrix to another.

        6. print_mat() prints the values of a matrix (for debugging
                       purposes).

        7. exp_mat() multiplies a matrix by itself n times.
 
        8. equal_mat() checks for equality of two rxc matrices.

        9. det() computes the determinant of a nxn matrix (currently,
                 n must be < 4). 

       10. fnorm_mat() computes the Frobenius norm of a matrix or the
                       Euclidean norm of a vector.

       11. transpose_mat() returns the transpose of a matrix.

       12. trace_mat() computes the trace of a matrix.

       13. dist_vect() computes the Euclidean distance between two
                       n-vectors.

       14. dot_vect() computes the dot product of two n-vectors.

*/

#include "matrix.h"                           /* Matrix package definitions. */
#include "matrixentry.h"                      /* Definition of matrix data 
                                                 type and operations. */

typedef Matrix_entry *Matrix;

/* Function declarations: */

    int equal_mat();
    Matrix_entry det(), fnorm_mat(), trace_mat(), dist_vect(),
                 dot_vect();

/* Function:   scalar_mat() multiplies a  matrix by a  scalar.
 * Arguments:  A - input matrix.
               C - resulting matrix.
               k - constant.
               r - # of rows.
               c - # of columns.
 * Returns:    Nothing.
 * Notes:
 */

scalar_mat(C, k, A, r, c)
  Matrix C, A;
  Matrix_entry k;
  int r, c;
{
  int i, j; 

  for (i = 0; i < r; i++)
    for (j = 0; j < c; j++)
      mult( C[ IND(i, j, c) ], k, A[ IND(i, j, c) ] );
}
  


/* Function:	mult_mat() multiplies two matrices.
 * Arguments:	A ,B:	matrices to be multiplied.
 *                      A is of dimension ra x ca.
 *                      B is of dimension rb x cb. 
 *              C:	matrix to hold the result of A x B.
 * Returns:  	Nothing.
 * Notes:       To function correctly, ca must equal rb.
 */

mult_mat(C, A, B, r1, c1, r2, c2)
     Matrix C, A, B;
     int r1, c1, r2, c2;
{
  Matrix_entry temp1, temp2;
  int i, j, k;
  
  for (i = 0; i < r1; i++)
    for (j = 0; j < c2; j++)
      {
        copy( temp1, ZERO);
        for (k = 0; k < c1; k++)
          {
            mult( temp2, A[IND(i,k,c1)], B[IND(k,j,c2)] );
            add( temp1, temp2, temp1);
          }     
        copy( C[IND(i, j, c2)], temp1);
      }
}



/* Function:	add_mat() adds two matrices.
 * Arguments:	A, B:	matrices to be added.
 *              C:   	matrix to hold the result.
 *              r:      # of rows of A and B.
 *              c:      # of columns of A and B.
 * Returns:     Nothing.
 * Notes:       Both A and B must have the same dimension.
 */
 
add_mat(C, A, B, r, c)
  Matrix C, A, B;
  int r, c; 
{
  int i, j;

  for (i = 0; i < r; i++)
    for (j = 0; j < c; j++)
      add( C[IND(i, j, c)], A[IND(i, j, c)], B[IND(i, j, c)] );
}

 

/* Function:	sub_mat() subtracts matrix B from matrix A.
 * Arguments:	A, B: matrices to be subtracted.
 *              C:    to hold the result.
 *              r:    # of rows.
 *              c:    # of columns.
 * Returns:     Nothing.
 * Notes:
 */

sub_mat(C, A, B, r, c)
    Matrix C, A, B;
    int r, c;
{
  int i,j;
               
  for (i = 0; i < r; i++)
    for (j = 0; j < c; j++)
      sub( C[IND(i, j, c)], A[IND(i, j, c)], B[IND(i, j, c)] );
}
  


/* Function:	copy_mat() copies one matrix to another.
 * Arguments:	source:		Matrix to be copied.
 *              destination:	Destination matrix.
                r:  # of rows in the matrices.
                c:  # of columns.
 * Returns:     Nothing.
 * Notes:       Both matrices must be of the same dimension.
 */

copy_mat(destination, source, r, c)
     Matrix source, destination;
     int r, c;
{
  int i, j;

  for( i = 0; i < r; i++) 
  	for( j = 0; j < c; j++)
		copy(destination[IND(i,j,c)], source[IND(i,j,c)]);
}



/* Function:	print_mat() prints a matrix.
 * Arguments:	A:	Matrix to be printed.
 *              r:  # of rows. 
 *              c:  # of columns.
 * Returns:     Nothing.
 * Notes:
 */

print_mat(A, r, c)
     Matrix A;
     int r, c;
{
  int i,j;				/* Index variable. */

  for (i = 0; i < r; i++)
    for (j = 0; j < c; j++)
      write(A[IND(i, j, c)]);
}



/* Function:     exp_mat() exponentiates the matrix A.
 * Arguments:    C - result.              
 *               A - matrix.
 *               exp - exponent.
 *               n - # of rows and columns in A.
 * Returns:      Nothing.
 * Notes:        A must be a square nxn matrix.   
 */  

exp_mat(C, A, exp, n)
    Matrix C, A;
    int exp, n;
{
  int i;

  copy_mat(C, A, n, n);
  for (i = 1; i < exp; i++)
	mult_mat(C, A, C, n, n, n, n);
}



/* Function:     equal_mat() returns TRUE if both matrices are equal.
 * Arguments:    A,B - two matrices. 
                 r - # of rows.
                 c - # of columns.
 * Returns:      TRUE or FALSE.
 * Notes:        A and B must both be rxc.
 */

int equal_mat(A, B, r, c)
  Matrix A, B;
  int r, c;
{
  int i, j, result;

  result = TRUE;
  for (i = 0; i < r; i++)
	for (j = 0; j < c; j++)
	  if (!(equal(A[IND(i, j, c)], B[IND(i, j, c)])))
		  result = FALSE;
  return (result);
}
   
 

/* Function:    det() computes the determinant of a nxn matrix.
 * Arguments:   A - represents matrix.
                n - # of rows and columns in A.
 * Returns:     determinant of type Matrix_entry.
 * Notes:       A must be square, with n < 4.
 */

Matrix_entry det(A,n)
    Matrix A;
    int n;
{
  Matrix_entry t1, t2, t3;  /* temporary variables used in computations. */
  switch (n)
  {
    case 1:
        return(A[0]);
    case 2:
        mult(t1, A[0], A[3]);
        mult(t2, A[1], A[2]);
        sub(t1, t1, t2);
        return(t1);
    case 3:
        mult(t1,A[0],A[4]);
        mult(t1,t1,A[8]);
        mult(t2,A[1],A[5]);
        mult(t2,t2,A[6]);
        add(t3,t1,t2);
        mult(t1,A[2],A[3]);
        mult(t1,t1,A[7]);
        add(t3,t3,t1);
        mult(t1,A[2],A[4]);
        mult(t1,t1,A[6]);
        sub(t3,t3,t1);
        mult(t1,A[1],A[3]);
        mult(t1,t1,A[8]);
        sub(t3,t3,t1);
        mult(t1,A[0],A[5]);
        mult(t1,t1,A[7]);
        sub(t3,t3,t1);
        return(t3);
    default:
        return(ZERO);
  }
}



/* Function:    fnorm_mat() computes the Frobenius norm of a matrix,
                            or the Euclidean norm of a vector.
 * Arguments:   A - represents matrix.
                r - # of rows.
                c - # of columns.
 * Returns:     norm of type Matrix_entry.
 * Notes:
 */

Matrix_entry fnorm_mat(A, r, c)
  Matrix A;
  int r, c;
{
  int i, j;
  Matrix_entry temp1, temp2;       /*temporaries used in computations.*/
  copy(temp2, ZERO);
  for (i = 0; i < r; i++)
    for (j = 0; j < c; j++) 
      {
        mult(temp1, A[IND(i, j, c)], A[IND(i, j, c)]);
        add(temp2, temp1, temp2);
      }
  root(temp2, temp2);
  return( temp2 );
}



/* Function:   transpose_mat() computes the transpose of a nxn matrix.
 * Arguments:  C - transpose of A;
 *             A - input matrix.
               r - # of rows.
               c - # of columns.
 * Returns:    Nothing.
 * Notes:      C will be a cxr matrix.
 */
 
transpose_mat(C, A, r, c)
  Matrix C, A; 
  int r, c;
{ 
  int i, j;

  for (i = 0; i < r; i++)
    for (j = 0; j < c; j++)
      copy(C[IND(i, j, c)], A[IND(j, i, c)]);
}



/* Function:   trace_mat() finds the trace of a nx matrix.
 * Arguments:  A - input matrix.
               n - # of rows and columns in A.
 * Returns:    the trace of A, of type Matrix_entry.
 * Notes:      A must be square.
 */

Matrix_entry trace_mat(A, n)
  Matrix A;
  int n;
{
  Matrix_entry temp;
  int i;

  copy(temp, ZERO);
  for (i = 0; i < n; i++)
    add(temp, temp, A[IND(i, i, n)]);
  return (temp);
}



/* Function:  dist_vector() computes the Euclidean distance between two
                            n-vectors.
 * Arguments: u,v - input vectors.
 *            n - # of components in vectors u and v.
 * Returns:   scalar of type Matrix_entry.
 * Notes:     u and v must both be n-vectors.
 */

Matrix_entry dist_vect(u, v, n)
    Matrix u, v; 
    int n;
{
  int i;
  Matrix_entry t1, t2, t3;           /*temporaries used in computations.*/
  copy(t3, ZERO);
  for (i = 0; i < n; i++) 
    {
      sub(t1, u[i], v[i]);
      mult(t2,t1, t1);
      add(t3, t3, t2);
    }
  root(t3, t3);
  return( t3 );
}



/* Function:  dot_vect() computes the dot product of two n-vectors.
 * Arguments: u,v - input vectors;
              n - size of vectors u and v.
 * Returns:   the dot product of type Matrix_entry.
 * Notes:      u and v must both be n-vectors.
 */

Matrix_entry dot_vect(u, v, n) 
    Matrix u, v;
    int n;
{
  int i;
  Matrix_entry t1, t2;                  /* temporary variables. */

  copy(t2, ZERO);
  for(i = 0; i < n; i++)
    {
      mult(t1, u[i], v[i]);
      add(t2, t2, t1);
    }
  return(t2);
}






