/***************************************************************************
 *                Copyright (C) 1990 by Mark B. Phillips                   *
 *                                                                         *
 *  Permission to use, copy, modify, and distribute this software, its     *
 *  documentation, and any images it generates for any purpose and without *
 *  fee is hereby granted, provided that                                   *
 *                                                                         *
 *  (1) the above copyright notice appear in all copies and that both      *
 *      that copyright notice and this permission notice appear in         *
 *      supporting documentation, and that the names of Mark B.            *
 *      Phillips, or the University of Maryland not be used in             *
 *      advertising or publicity pertaining to distribution of the         *
 *      software without specific, written prior permission.               *
 *                                                                         *
 *  (2) Explicit written credit be given to the author Mark B. Phillips    *
 *      in any publication which uses part or all of any image produced    *
 *      by this software.                                                  *
 *                                                                         *
 * This software is provided "as is" without express or implied warranty.  *
 ***************************************************************************/

#include "gr.h"

#define MIN(x,y) ((x) < (y) ? (x) : (y))
#define ROUNDSHORT(x) ((short)((x)+.5))

int GrNoDepthCue=0;

/* Color Ramp indices: */
Colorindex RampLo[7], RampHi[7];

/* Color Ramp offsets: */
Colorindex RampLoOffset=0, RampHiOffset=0;

/* Z offsets: */
long GrNearZOffset=NEARZOFFSETDEFAULT;
long GrFarZOffset=FARZOFFSETDEFAULT;

Colorindex GrRampLen;

/* Color Ramp boundary values: */
short
  GrRedLo[]	= {255, 0,   0	},  GrRedHi[]	  = {0, 0, 0},
  GrGreenLo[]	= {0,	255, 0	},  GrGreenHi[]	  = {0, 0, 0},
  GrBlueLo[]	= {0,	0,   255},  GrBlueHi[]	  = {0, 0, 0},
  GrYellowLo[]	= {255, 255, 0	},  GrYellowHi[]  = {0, 0, 0},
  GrMagentaLo[] = {255, 0,   255},  GrMagentaHi[] = {0, 0, 0},
  GrCyanLo[]	= {0,	255, 255},  GrCyanHi[]	  = {0, 0, 0},
  GrWhiteLo[]	= {255, 255, 255},  GrWhiteHi[]   = {0, 0, 0};

/*-----------------------------------------------------------------------
 * Function:	GrColor
 * Description:	Set the current color for non-depthcued drawing
 * Args  IN:	n: colormap index
 * Notes:	n should be an actual index into the colormap
 */
GrColor(n)
     int n;
{
  depthcue(0);
  color((Colorindex)n);
}

/*-----------------------------------------------------------------------
 * Function:     GrDepthCueColor
 * Description:  Set the current drawing color for (depthcued) graphics
 * Arguments IN: n: the color to become current.  Should be one of
 *		 GR_RED, GR_GREEN, GR_BLUE, GR_YELLOW, GR_MAGENTA,
 *		 GR_CYAN, or GR_WHITE.
 * Returns:      nothing
 * Notes:	 n is NOT an index into the colormap, as in GrColor.
 */
GrDepthCueColor(n)
     int n;
{
  if (GrNoDepthCue) {
    GrNoDepthCueColor(n);
    return;
  }
  depthcue(1);
  switch (n) {
  case GR_CYAN:
    MyShadeRange(RampLo[GR_CYAN], RampHi[GR_CYAN]);
    break;
  case GR_GREEN:
    MyShadeRange(RampLo[GR_GREEN], RampHi[GR_GREEN]);
    break;
  case GR_RED:
    MyShadeRange(RampLo[GR_RED], RampHi[GR_RED]);
    break;
  case GR_BLUE:
    MyShadeRange(RampLo[GR_BLUE], RampHi[GR_BLUE]);
    break;
  case GR_YELLOW:
    MyShadeRange(RampLo[GR_YELLOW], RampHi[GR_YELLOW]);
    break;
  case GR_MAGENTA:
    MyShadeRange(RampLo[GR_MAGENTA], RampHi[GR_MAGENTA]);
    break;
  case WHITE:
  default:
    MyShadeRange(RampLo[GR_WHITE], RampHi[GR_WHITE]);
    break;
  }
}

/*-----------------------------------------------------------------------
 * Function:     GrNoDepthCueColor
 * Description:  Set the current drawing color for (non-depthcued) graphics
 * Arguments IN: n: the color to become current.  Should be one of
 *		 GR_RED, GR_GREEN, GR_BLUE, GR_YELLOW, GR_MAGENTA,
 *		 GR_CYAN, or GR_WHITE.
 * Returns:      nothing
 * Notes:	 n is NOT an index into the colormap, as in GrColor.
 */
GrNoDepthCueColor(n)
     int n;
{
  switch (n) {
  case GR_CYAN:
    color(CYAN);
    break;
  case GR_GREEN:
    color(GREEN);
    break;
  case GR_RED:
    color(RED);
    break;
  case GR_BLUE:
    color(BLUE);
    break;
  case GR_YELLOW:
    color(YELLOW);
    break;
  case GR_MAGENTA:
    color(MAGENTA);
    break;
  case WHITE:
  default:
    color(WHITE);
    break;
  }
}

MyShadeRange(lo,hi)
  Colorindex lo, hi;
{
  shaderange(lo+RampLoOffset,
	     hi-RampHiOffset,
	     LO_SCREEN_COORD+GrFarZOffset,
	     HI_SCREEN_COORD-GrNearZOffset );
}

LoadRamps()
{
  int bp, maplen;

  bp = getplanes();		/* (number of bitplanes available) */
  /* Note: 2^12 = 4096 = total length of colormap.  We may have less than
     this available for use because of mex and doublebuffering.  Also,
     the first 8 indices are used for the default colormap, so we don't
     want to mess with these.  Therefore, the number of colormap entries
     which we have available is 2^min(12, bp) - 8. */
  maplen = (1 << MIN(12,bp)) - 8;

  /* If we don't have room for each of the 7 ramps to be at least 8 
     entries long, then forget depth cueing completely. */
  if (maplen < 7*8) {
    GrNoDepthCue = 1;
    return;
  }

  ComputeRamps(RampLo, RampHi,
	       (Colorindex)8, (Colorindex)(8 + maplen - 1), 7);

  LoadRamp(RampLo[GR_RED],     RampHi[GR_RED],     GrRedLo,     GrRedHi);
  LoadRamp(RampLo[GR_GREEN],   RampHi[GR_GREEN],   GrGreenLo,   GrGreenHi);
  LoadRamp(RampLo[GR_BLUE],    RampHi[GR_BLUE],    GrBlueLo,    GrBlueHi);
  LoadRamp(RampLo[GR_YELLOW],  RampHi[GR_YELLOW],  GrYellowLo,  GrYellowHi);
  LoadRamp(RampLo[GR_MAGENTA], RampHi[GR_MAGENTA], GrMagentaLo, GrMagentaHi);
  LoadRamp(RampLo[GR_CYAN],    RampHi[GR_CYAN],    GrCyanLo,    GrCyanHi);
  LoadRamp(RampLo[GR_WHITE],   RampHi[GR_WHITE],   GrWhiteLo,   GrWhiteHi);
}

/*-----------------------------------------------------------------------
 * Function:     LoadRamp
 * Description:  load a color ramp (for a single color)
 * Arguments IN: i1: colormap index of low end of ramp
 *               i2: colormap index of high end of ramp
 *               rgb1: color for index i1
 *               rgb2: color for index i2
 * Returns:      nothing
 * Notes:     1. This loads the range [i1,i2] in the colormap with RGB
 *		 values linearly interpolated between rgb1 and
 *		 rgb2.
 *            2. i1 should be less than or equal to i2.  If i1==i2, then
 *		 this location is loaded with the value rgb1.
 */
LoadRamp(i1, i2, rgb1, rgb2)
     Colorindex i1, i2;
     short rgb1[3], rgb2[3];
{
  double r,g,b, r_inc, g_inc, b_inc, ramplen;
  Colorindex i;

  /* Check for degenerate cases */
  if (i1>i2) return;
  if (i1==i2) {
    mapcolor(i1,rgb1[0],rgb1[1],rgb1[2]);
    return;
  }

  /* Determine increments */
  ramplen = i2 - i1 + 1;
  r_inc = ((double)(rgb2[0] - rgb1[0])) / (ramplen - 1);
  g_inc = ((double)(rgb2[1] - rgb1[1])) / (ramplen - 1);
  b_inc = ((double)(rgb2[2] - rgb1[2])) / (ramplen - 1);

  /* Load ramps */
  r = rgb1[0];  g = rgb1[1];  b = rgb1[2];
  for (i=i1; i<=i2; ++i) {
    mapcolor(i, ROUNDSHORT(r), ROUNDSHORT(g), ROUNDSHORT(b));
    r += r_inc;
    g += g_inc;
    b += b_inc;
  }

}

/*-----------------------------------------------------------------------
 * Function:     ComputeRamps
 * Description:  Compute the low and high indices of a sequence of color
 *		   ramps.
 * Arguments IN: i1[]: array in which to store low indices of ramps
 *		 i2[]: array in which to store high indices of ramps
 *		 lo: lowest index to be used by any ramp
 *		 hi: highest index to be used by any ramp
 *		 n: number of ramps to be stored in [lo,hi]
 * Returns:      nothing
 * Notes:     1. The i1 and i2 arrays should be declared to have length
 *		 at least n.
 *	      2. hi - lo should be at least n, so that there is room for
 *		 at least one entry in each ramp.
 *	      3. Ramp lengths will be differ by at most one.
 */
ComputeRamps(i1,i2, lo, hi, n)
Colorindex i1[],i2[], lo, hi;
int n;
{
  double index, ramplen;
  int ramp;

  ramplen = ((double)(hi - lo + 1)) / ((double)n);
  GrRampLen = ramplen;
  index = lo;
  for (ramp=0; ramp<n; ++ramp) {
    i1[ramp] = ROUNDSHORT(index);
    index += ramplen;
    i2[ramp] = ROUNDSHORT(index-1);
  }
}
