/*
 * mathpipe.c: pipe manager for allowing Mathematica to communicate
 * 		via sockets with a program running on a remote host.
 *
 * usage: mathpipe
 *
 * NOTE: this program should really take the server host name and an
 * optional port number as command-line arguments.  But because the
 * Mathematica 'StartProcess' function has no way of passing arguments
 * to the program it starts, we resort to a sleazy environment
 * variable mechanism (see below).
 */

/************************************************************************
 *		Copyright (C) 1989 by Mark B. Phillips                  *
 * 									*
 * Permission to use, copy, modify, and distribute this software and    *
 * its documentation for any purpose and without fee is hereby granted, *
 * provided that the above copyright notice appear in all copies and    *
 * that both that copyright notice and this permission notice appear in *
 * supporting documentation, and that the name of Mark B. Phillips or   *
 * the University of Maryland not be used in advertising or publicity   *
 * pertaining to distribution of the software without specific, written *
 * prior permission.  This software is provided "as is" without express *
 * or implied warranty.                                                 *
 ************************************************************************/

/* This program functions as a client in the server-client model.  It
 * attempts to establish a connection with the 'elbow daemon' (elbowd)
 * on the server host.  If this connection is successful, the elbow
 * daemon starts its designated process, whose stdin/stdout is
 * connected to the socket 'sock' in this program.  Mathematica can
 * then communicate with this process via the MathPipe command.
 *
 * We read the environment variable ELBOWSERVER to determine the name
 * of the server host; this variable may be either an ascii host name
 * or an Internet address of the form "a.b.c.d".  ELBOWSERVER *must*
 * be set.
 *
 * If the environment variable ELBOWPORT is set, we assume it is the
 * port number on which the elbow daemon is listening for connection
 * requests on the server host.  If ELBOWPORT is not set, we use a
 * default port number (constant ELBOWPORT in bothpipe.h).
 */

/* BOTHPIPE_H should be the pathname of "bothpipe.h", in quotes */
#include BOTHPIPE_H
#include "mathlink.h"

int sock;			/* socket for talking to server */
int serverportnumber=ELBOWPORT;	/* server port number */
Sockaddr server;		/* socket address for server */

char *mathpipe();
char *getenv();

main(argc, argv)
     int argc;
     char *argv[];
{
  connect_to_server();
  connect_to_mathematica();
  MathStart();
  quit("mathpipe: exiting because MathStart returned");
}

connect_to_server()
{
  Sockaddr *addr_ptr;
  char *serverhost, *serverportstring;

  /* Get server host name from ELBOWSERVER env var */
  if ((serverhost=getenv("ELBOWSERVER")) == NULL)
    quit("mathpipe: environment variable ELBOWSERVER is not set.");

  /* Maybe get server port number from ELBOWPORT env var */
  if ((serverportstring=getenv("ELBOWPORT")) != NULL)
    serverportnumber = atoi(serverportstring);

  /* Create the socket */
  if ((sock=socket(AF_INET, SOCK_STREAM, 0)) == -1)
    pquit("mathpipe (socket())");

  /* Construct server's address */
  if ((addr_ptr=resolve_host(serverhost)) == NULL)
    quit("mathpipe: can't resolve elbow host address");
  bcopy((char*)addr_ptr, (char*)&server, sizeof(server));
  server.sin_port = serverportnumber;

  /* Connect does the bind() for us */
  if (connect(sock, (char *)&server, sizeof (server)) != 0)
    pquit("mathpipe (connect)");
}

connect_to_mathematica()
{
  MathInit();
  MathExec("MathPipe::usage = \"MathPipe[string] sends string to \
a program running on a remote host, and returns the string returned by \
the program.\"");
  MathInstall(mathpipe, "MathPipe", "char *", "s_String", "char *");
}

/*-----------------------------------------------------------------------
 * Function:	mathpipe
 * Description:	send a string down the pipe (sock) to the program, and return
 *		  the result returned by the program (a string)
 * Args  IN:	s: string to send down pipe
 * Returns:	the string returned by the program
 * Notes:	s should not contain any newlines
 */
char *
  mathpipe(s)
char *s;
{
  static char ibuf[512], obuf[512];
  int len, cc;

  /* Write this line to socket */
  sprintf(obuf, "%s\n", s);
  len = strlen(obuf);
  cc = write(sock, obuf, len);
  if (cc != len) {
    fprintf(stderr,"mathpipe: pipe write error, cc = %1d, len = %1d\n",
	    cc, len);
    return("WRITE_ERROR");
  }
  
  /* Read a line from socket, and return that line */
  if (fgetline(ibuf, 512, sock) == NULL) {
    fprintf(stderr, "mathpipe: pipe read error\n");
    return("READ_ERROR");
  }
  else
    return(ibuf);
}

