.ce 100
A Mathematical Description of GDB
.sp
Mark B. Phillips
July 23, 1989
.ce 0
.sp
This is a mathematical description of GDB, my Geometry DataBase
subroutine library.  It describes in mathematical language, almost
exactly what GDB does.  The "almost exactly" is because this description
omits any mention of GDB "blocks", since they do not fit well into
this mathematical framework.  They are also less important and less used
than the general structure explained here, however, and can be
understood easily by reading the GDB documentation.

.ce
-----------------------------------------------

Let
.in +8
.nf
T = {t1,t2,...,t16}
A = {ascii strings}
Z+ = {positive integers}
.in -8
.fi

By a "typed metric space" we mean a metric space X together with a
function t: X --> T.  t is called the "type" function of X, and t(x)
is called the "type" of the element x of X.  Two functions f: S --> T
and g: S --> X on a set S are said to be "compatible" with X if f(s)
= t(g(s)) for all s in S.

A "geometry database" is a (ordered) finite set G together with a
typed metric space X and functions

.in +8
.nf
type: G --> T
name: G --> A
data: G --> X
size: T --> Z+
.in -8
.fi

where the functions type and data are compatible with X.  An element
of G is called an "entity".

GDB maintains a geometry database G.  The reasons for this method of
organization grew out of my experience.  This seems to be the most
flexible, usable method which also works well for the sorts of
programs I have written.  The current version (2.2) of GDB has been in
use for over a year now in several different programs which have been
used by many people, and it has proven to be very effective.  In
contrast to most other programs or packages I have worked on, I am not
aware of any bugs in GDB whatsoever.

Initially, when a GDB program starts, G is empty.  You add to G by
calling gdb_add_entity.  You remove from G by calling
gdb_delete_entity.

Before doing any of this (i.e. writing calls to GDB routines),
however, you must set things up.  There are several choices you must
make, and a few routines you must supply to GDB.  First of all, you
must choose your typed metric space X.  Suppose, for example, that
you are writing a program which draws points and line segments.  Then
X might be the collection of all points and line segments in a
certain subset of the plane.  The "type" of an element of X would
simply tell whether it is a point or a line segment; you might use
type t1 for points and type t2 for lines segments.

You should make a typedef for each type in X (the number of typedefs is
therefore the cardinality of t(X)).  Elements of X will be represented
in your program by variables of these data types.  Notice that the
phrase "the type of an object x" now has two meanings: t(x) = the value
at x of the type function of X (this is an element of T), and the C
data type used to store x.  This ambiguity is deliberate, because t(x)
is used as a flag to indicate the data type of x. All elements in X
with the same type value will be stored as the same data type.

NOTE: In GDB, the elements of T are called GDB_ENTITY_TYPE_1 through
GDB_ENTITY_TYPE_16.  I refer to these values here as t1 through t16
simply for readability.  The data type of these constants is
gdb_Entity_type.

You must then write procedures which GDB can use to manipulate these
data types.  In particular, you must provide the following three
routines (see the GDB manual page for the exact specifications):

.in +2
gdb_entity_data_size:
.in +2
This function implements the "size" function in the definition above.
It should return the number of bytes occupied by each of your types.
GDB uses this number internally to tell how much space must be set
aside to store your objects.
.in -2

gdb_entity_data_copy:
.in +2
This function should copy an element in X to another element of
the same type.  GDB uses this function to make copies of objects.
.in -2

gdb_entity_data_dist:
.in +2
This function should implement the metric in X; it returns the
distance between two elements.  GDB uses this in doing "distance
searches", which are described below.
.in -2
.in -2

Once you have set up your typedefs and written these functions, you
can begin writing code that uses GDB.  You can always add new data
types later, as long as you remember to modify these three functions
to handle the new types.

You add an entity to the database by calling gdb_add_entity.  In this
call, you specify the element of X for that entity (by a pointer to the
object), its type, and a name for it.  The name can be any ascii string
and simply serves to identify the entity.  gdb_add_entity adds to G an
entity with the given type, name, and object (the "data"), and gives
you back a value which is called the "handle" of the new entity.  This
is really a pointer to where the entity is stored, but you should think
of it as embodying the entity itself.  The C data type of this handle
is called gdb_Entity to emphasize this role.  According to the above
definition, the only structure on G consists of the type, name, and
data functions.  Hence in your program, the only things you can do with
an entity handle are to find its type, name, and data.  The type
function is gdb_entity_type, the name function is gdb_entity_name, and
the data function is gdb_entity_data.  Each of these takes an entity
handle and gives you back the corresponding value.

You remove entities from G by calling gdb_delete_entity.  This
procedure takes an argument which is an entity's handle, and deletes
that entity from G.

To free you from having to keep track of the handles of all the
entities in the database, GDB provides a way to retrieve an entity's
handle based on information about that entity.  This is what the
procedure gdb_retrieve_entity does.  You can use gdb_retrieve_entity in
several different ways, depending on how you want to search for an
entity.  The simplest way is to do a "name search"; you specify a name,
and gdb_retrieve_entity gives you the handle of the entity with that
name.  You can also do a "sequential search", in which
gdb_retrieve_entity returns entities in the order in which they were
entered into G.  You can also search for an entity whose data is close
to a specified element of X.  This last type of search is the most
sophisticated, and it is GDB's ability to do this that makes it
well suited for maintaining databases of geometric objects.
Indeed, this is the only thing about GDB that has to do with geometry
(or rather the metric on X).  The main use intended for this capability
is to search the database for an object near a point on the screen
where the user has pointed.

.ce
-----------------------------------------------

The rest of this document contains various hints and suggestions for
writing GDB programs.  It is not intended as a substitute for the GDB
manual pages, which should be consulted for exact details.

For the sake of readability, you should define constants (with
#define) to identify the types you use.  The values of these
constants will be elements of T.  For example, in a program to deal
with points and line segments, you might write

.in +8
.nf
#define POINT	GDB_ENTITY_TYPE_1
#define SEGMENT	GDB_ENTITY_TYPE_2
.in -8
.fi

You can then give an object's type as either POINT or SEGMENT.  You
should have no reason to refer to the symbols GDB_ENTITY_TYPE_n
outside the definitions of these constants.

The reason GDB uses these pre-defined constants, instead of just the
integers 1 through 16, is so that it can do fast bitwise comparisons
among them.  The values of these constants are actually the first 16
powers of 2; one for each bit in a 16 bit int.  This is also why there
is a limit of 16 types.

GDB requires that you use pointers a lot.  It is therefore a good idea
to run lint on your program frequently to try to catch pointer errors.
You should also use type casting to guarantee that pointers are
converted to the correct type before being used.  Many pointers that
GDB procedures expect as arguments or return as values are delcared to
be pointers to chars.  This is because GDB procedures must be prepared
to deal with different types of pointers, and char * is the most
general type of pointer (any pointer can be cast to char *, and a char
* can be cast to point to anything).  It is therefore a good idea to
put "(char *)" in front of any such pointer arguments which are not
already pointers to chars.  For example,

.in +8
.nf
Point p;
gdb_Entity e;

gdb_add_entity(POINT, (char*)&p, "point 1", &e);
.in -8
.fi

(unless, of course, Point is an array data type, in which case the '&'
in front of the p should be omitted).  If you check the GDB manual
page, you will see that gdb_add_entity expects a char * as the second
argument (the "data").  This is because in general the type of this
pointer will be different on different calls.  Notice that the pointer
to e should not be cast to char *, because it is declared to be
gdb_Entity *.

The function gdb_entity_data returns a char * which points to an
element of X stored as one of your data types.  Before referencing
this pointer, you should cast it to the appropriate type.  This is
best done at on the same line as the call to gdb_entity_data, as in:

.in +8
.nf
Segment *s;
gdb_Entity e;

s = (Segment *)gdb_entity_data(e);
.in -8
.fi

All this type casing is probably not really necessary, because for the
most part, all pointers are equivalent.  But adhering to these
guidelines will force you to think about the types and will cut down
on your errors.  And it will allow lint to give you more meaningful
messages (and fewer meaningless ones).

When adding an entity to the database, you must pick the name.  It is
up to you to make sure that the name you pick is not the name of some
other entity already in the database.  You can use the function
gdb_generate_unique_name to create names which are guaranteed to be
unique.

When GDB adds an entity to the database, what it actually puts in the
database as the entity's data is a copy of the object you give it.
If you later modify the object you passed to gdb_add_entity, it will
not affect the entity's data.  You can, however, directly modify an
entity's data by using the pointer returned by gdb_entity_data.

gdb_entity_data_size should be a big switch statement which just
returns sizeof(dn) where dn is the typedef name which you associate
with type tn.

X does not really have to be a metric space.  The only part of GDB
which makes use of the metric is the distance search capability of
gdb_retrieve_entity.  If you do not plan to use this, you can think of
X as just a set, and you can provide a dummy version of
gdb_entity_data_dist.  (You will have to provide at least a dummy to
prevent an incomplete load.)

Another possibility is that you only care about computing the distances
between certain types of objects.  In this situation, you can write
gdb_entity_data_dist to return the special value GDB_INFINITY if
it is called with a pair of objects for whose types no distance
is defined.

The main thing to keep in mind is that you have complete control over
the types that will be passed to gdb_entity_data_dist; they are
determined by the object you pass to gdb_retrieve_entity when doing a
distance search.  GDB will try to compute the distance from this object
to all objects in the database.
