/**********************************************************************
 * $Id$
 *
 *                           hconvert.c
 *
 * This file contains the conversion routines to map between the
 * Poincare, Klein and UHP R2 planes.  It also contains auxillary
 * functions to compute norms and distances
 *
 ********************************************************************/
/*
 * hconvert.c
 * 
 * This module contains functions for mapping between the "Poincare",
 * "Klein" and "UHP" versions of R2.  That is, the Poincare disc and the
 * Klein disc are thought of as embedded in R2, and the mapping between
 * them is given as the identity outside the disc, and the appropriate
 * isometry between hyperbolic spaces on the disc.  The upper half plane 
 * is also regarded as embedded in R2 and the map between the Poincare
 * plane and the UHP plane is the Mobius transformation taking the unit 
 * disc to the upper half plane.  The map between the Klein R2 and the
 * UHP R2 is then the composition of these maps.
 */

/**************************************************************************
 *     Copyright (C) 1990 by Mark B. Phillips and Robert R. Miner	  *
 * 									  *
 * Permission to use, copy, modify, and distribute this software, its	  *
 * documentation, and any images it generates for any purpose and without *
 * fee is hereby granted, provided that					  *
 * 									  *
 * (1) the above copyright notice appear in all copies and that both that *
 *     copyright notice and this permission notice appear in supporting	  *
 *     documentation, and that the names of Mark B.  Phillips, Robert R.  *
 *     Miner, or the University of Maryland not be used in advertising or *
 *     publicity pertaining to distribution of the software without	  *
 *     specific, written prior permission.				  *
 *									  *
 * (2) Explicit written credit be given to the authors Mark B.  Phillips  *
 *     and Robert R. Miner in any publication which uses part or all of	  *
 *     any image produced by this software.				  *
 *									  *
 * This software is provided "as is" without express or implied warranty. *
 **************************************************************************/

#include "hcore.h"
  
/*-----------------------------------------------------------------------
 * Function:	PoincareToKlein
 * Description:	map R^2 --> R^2 via identity outside the unit disc
 *              and by the map taking Poincare coordinates to Klein
 *              coordinates inside the disc
 * Args  IN:	z: point in the "Poincare" R^2
 *      OUT:	w: point in the "Klein" R^2
 * Returns:	success status
 * Author:	lena
 * Date:	Thu Apr  5 12:22:07 1990
 * Notes:	uses the transformation:  w = 2z / (1 + |z|^2) for z
 *		inside the unit disk, and uses the identity map for z
 *		outside the unit disk.
 */

PoincareToKlein(w, z)
R2Point w,z;
{
  double n;

  n = normsq(z);
  if (n < 1.0) {
    w[0] = (2 * z[0])/(1 + n);
    w[1] = (2 * z[1])/(1 + n);
  }
  else {
    w[0] = z[0];
    w[1] = z[1];
  }
  return(0);
}

/*-----------------------------------------------------------------------
 * Function:	KleinToPoincare
 * Description:	map R^2 --> R^2 via identity outside the unit disc
 *              and by the map taking Klein coordinates to Poincare
 *              coordinates inside the disc
 * Args  IN:	z: point in the "Klein" R^2
 *      OUT:	w: point in the "Poincare" R^2
 * Returns:	
 * Author:	lena
 * Date:	Thu Apr  5 12:34:47 1990
 * Notes:        uses the transformation:
 *               
 *                w = z * (1 + |w|^2) / 2,
 *
 *               where
 *                         (1-sqrt(1-|z|^2)) / |z|   if |z|>.5
 *                |w| = {                                       }.
 *                         |z| / (1+sqrt(1-|z|^2))   if |z|<=.5
 *               
 *               NOTE: These two expressions for |w| are equal; we
 *               chose between them to best deal with the extreme
 *               cases |w|~0 and |w|~1.
 *               ALSO: z is assumed to be finite.
 */
KleinToPoincare(w, z)
R2Point w,z;
{
  double nw, nz, d, tmp;

  nz = normsq(z);
  if (nz < 1) {
    if (nz > .5) {
      tmp = 1 - sqrt(1 - nz);
      nw = SQR(tmp) / nz;
    }
    else {
      tmp = 1 + sqrt(1 - nz);
      nw = nz / SQR(tmp);
    }
    d = (1 + nw) / 2;
    w[0] = d * z[0];
    w[1] = d * z[1];
  }
  else {
    w[0] = z[0];
    w[1] = z[1];
  }
}

/*-----------------------------------------------------------------------
 * Function:	UHPToPoincare
 * Description:	maps R2 --> R2 so that the UHP goes to the Poincare disc
 * Args  IN:	z: point in "UHP" R2
 *      OUT:	w: point in "Poincare" R2
 * Returns:	
 * Author:	lena
 * Date:	Thu Apr  5 13:07:20 1990
 * Notes:       uses the Mobius transform w = (z-i) / (z+i)
 *              the point -i is sent to (0, 99999.0)
 */
UHPToPoincare(w, z)
R2Point w,z;
{
  double n, d;

  if ((z[0] == 0.0) &&(z[1] == -1.0)) {
    w[0] = 0.0;
    w[1] = 99999.0;
  }
  else {
    n = normsq(z);
    d = n  + 2 * z[1] + 1;
    w[0] = (n - 1) / d;
    w[1] = -2 * z[0] / d;
  }
}

/*-----------------------------------------------------------------------
 * Function:     PoincareToUHP
 * Description:  maps R2 -->R2 so that Poincare disc goes to UHP
 * Arguments IN: z: point in "Poincare" R2
 *          OUT: w; point in "UHP" R2
 * Returns:      nothing
 * Author:	 lena
 * Date:	 Thu Apr  5 12:34:47 1990
 * Notes:        uses the transformation: w = i(1+z) / (1-z)
 *               the point z = 1 goes to (0,99999.0)
 */
PoincareToUHP(w, z)
R2Point w,z;
{
  double n, d;

  if ((z[0] == 1.0) && (z[1] == 0.0)) {
    w[0] = 0.0;
    w[1] = 99999.0;
  }
  else {
    n = normsq(z);
    d = 1 - 2 * z[0] + n;
    w[0] = -2 * z[1] /  d;
    w[1] = (1 - n) / d;
  }
}


/*-----------------------------------------------------------------------
 * Function:	UHPToKlein
 * Description:	converts UHP R2 to Klein R2
 * Args  IN:	z: point in UHP R2
 *      OUT:	w: point in Klein R2
 * Returns:	
 * Author:	lena
 * Date:	Thu Apr  5 22:44:33 1990
 * Notes:	
 */
UHPToKlein(w, z)
R2Point w,z;
{
  R2Point a;

  UHPToPoincare(a, z);
  PoincareToKlein(w, a);
}

/*-----------------------------------------------------------------------
 * Function:	KleinToUHP
 * Description:	converts Klein R2 to UHP R2
 * Args  IN:	z: point in Klein R2
 *      OUT:	w: point in UHP R2
 * Returns:	
 * Author:	lena
 * Date:	Thu Apr  5 22:51:02 1990
 * Notes:	
 */
KleinToUHP(w, z)
R2Point  w, z;
{
  R2Point a;
  
  KleinToPoincare(a, z);
  PoincareToUHP(w, a);
}
