/* $Id$
 *
 * hutil.c
 *
 * This module contains functions for doing various Euclidean
 * and hyperbolic geometric calculations.  Notably, those related to
 *    
 *   norms and distances
 *   hyperbolic geodesics
 */

/**************************************************************************
 *     Copyright (C) 1990 by Mark B. Phillips and Robert R. Miner	  *
 * 									  *
 * Permission to use, copy, modify, and distribute this software, its	  *
 * documentation, and any images it generates for any purpose and without *
 * fee is hereby granted, provided that					  *
 * 									  *
 * (1) the above copyright notice appear in all copies and that both that *
 *     copyright notice and this permission notice appear in supporting	  *
 *     documentation, and that the names of Mark B.  Phillips, Robert R.  *
 *     Miner, or the University of Maryland not be used in advertising or *
 *     publicity pertaining to distribution of the software without	  *
 *     specific, written prior permission.				  *
 *									  *
 * (2) Explicit written credit be given to the authors Mark B.  Phillips  *
 *     and Robert R. Miner in any publication which uses part or all of	  *
 *     any image produced by this software.				  *
 *									  *
 * This software is provided "as is" without express or implied warranty. *
 **************************************************************************/

#include "hcore.h"

/*-----------------------------------------------------------------------
 * Function:	R2distsq
 * Description:	computes the square of the euc. dist. from x to y
 * Args  IN:	x,y: R2points
 *      OUT:	
 * Returns:	d: distance
 * Author:	lena
 * Date:	Mon Apr  9 16:08:22 1990
 * Notes:	
 */
double R2distsq(x,y)
R2Point x,y;
{
  return(SQR(x[0]-y[0]) + SQR(x[1]-y[1]));
}


/*-----------------------------------------------------------------------
 * Function:	R2dist
 * Description:	computes Euclidean distance between 2 R2 points
 * Args  IN:	x,y: R2points
 *      OUT:	
 * Returns:	d: their distance
 * Author:	lena
 * Date:	Mon Apr  9 16:06:34 1990
 * Notes:	
 */
double R2dist(x,y)
R2Point x,y;
{
  return(sqrt(R2distsq(x,y)));
}

/*-----------------------------------------------------------------------
 * Function:	normsq
 * Description:	computes the square of the euclidean norm of x
 * Args  IN:	x: and R2Point
 *      OUT:	
 * Returns:	n: the norm squared
 * Author:	lena
 * Date:	Mon Apr  9 16:10:44 1990
 * Notes:	
 */
double normsq(x)
R2Point x;
{
  return(SQR(x[0]) + SQR(x[1]));
}

/*-----------------------------------------------------------------------
 * Function:	norm
 * Description:	returns the norm of an R2 point thought of as a vector
 * Args  IN:	x: an R2Point
 *      OUT:	
 * Returns:	n: the norm
 * Author:	lena
 * Date:	Wed Apr 11 21:11:16 1990
 * Notes:	
 */
double norm(x)
R2Point x;
{
  return(sqrt(normsq(x)));
}       



/*-----------------------------------------------------------------------
 * Function:	UHPGeoCenter()
 * Description:	computes the center of the circle on which a geodesic
 *              arc lies.  If the arc is vertical, it returns (0,t)
 *              where t is the x intercept of the vertical geodesic
 * Args  IN:	a,b: the endpoints of the geodesic segment
 *      OUT:	c: the center, or -1 if vertical
 * Returns:	
 * Author:	lena
 * Date:	Mon Apr  9 12:22:33 1990
 * Notes:       The center for non-vertical geodesics is given by the
 *              formula
 *                          c = ( (|b|^2 - |a|^2) / 2 Re(b-a) , 0)
 */
UHPGeoCenter(c,a,b)
R2Point a,b,c;
{
  if ((a[0] == b[0]) || (UHP_INFINITY(a)) || (UHP_INFINITY(b))) {
    c[0] = 0.0;
    c[1] = -1.0;
  }
  else {
    c[0] = (SQR(b[0])+SQR(b[1])-SQR(a[0])-SQR(a[1])) / (2*(b[0] - a[0]));
    c[1] = 0.0;
  }
}

/*-----------------------------------------------------------------------
 * Function:	PoincareGeoCenter
 * Description:	finds the center of the circle on which a poincare 
 *              geodesic segment lies
 * Args  IN:	a,b: the segment endpoints
 *      OUT:	c: the center
 * Returns:	
 * Author:	lena
 * Date:	Mon Apr  9 12:29:32 1990
 * Notes:	If a,b lie on a diameter, (0,0) is returned for the center.
 *              Otherwise, the fact that the Klein pole and the Poincare
 *              pole are the same is used.  The Klein pole is the cross-product
 *              of the vectors representing a,b in homogenous coordinates.
 *              If x,y = KleinToPoincare(a,b), the formula becomes
 *
 *              c = ((y2 -x2)/(x1 y2-x2 y1), (x1-y1) / (x1 y2 - x2 y1))              
 */
PoincareGeoCenter(c,a,b)
R2Point a,b,c;
{
  double denom;
  R2Point x,y;

  if ((a[0]*b[1] - a[1]*b[0]) == 0) {
    c[0] = 0.0;
    c[1] = 0.0;
  }
  else {
    PoincareToKlein(x,a);
    PoincareToKlein(y,b);
    denom = x[0] * y[1] - x[1] * y[0];
    c[0] = (y[1] - x[1])/denom;
    c[1] = (x[0] - y[0])/denom;
  }
}
