static char WHAT[]="@(#)kaleido.c	2.63 (rl) 92/07/30";
/**********************************************************
* kaleido
*
*	Kaleidoscopic construction of uniform polyhedra
*
*	Author:
*		Dr. Zvi Har'El,
*		Deptartment of Mathematics,
*		Technion, Israel Institue of Technology,
*		Haifa 32000, Israel.
*		E-Mail: rl@gauss.technion.ac.il
**********************************************************/
#include "kaleido.h"
#include "uniform.h"

char *kaleido[]  = {
	"Kaleidoscopic Construction of Uniform Polyhedra, Version 2.63 (92/07/30)\n",
	"Author: Dr. Zvi Har'El <rl@gauss.technion.ac.il>\n",
	"Usage: ",
	"", /*stick program name here */
	" [options], where the options are:\n",
	"-l: List polyhedron names, symbols and reference figures.\n",
	"-v: display Vertex and face coordinates.\n",
	"-x: display successive approXimations.\n",
	"-d digits: number of significant Digits.\n",
	"-a degrees: rotation axis Azimouth.\n",
	"-e degrees: rotation axis Eleveation.\n",
	"-f degrees: angle of rotation until Freezing.\n",
	"-[pP] prefix: prefix for Pic snapshots.\n",
#ifndef NOSCOPE
	"-g: display polyhedron on a Graphics monitor\n",
	"    (hit Enter for next, Backspace for previous, Tab for dual,\n",
	"    S for a snapshot, F to freeze, Esc to exit).\n",
	"-n: display Numerical data in addition to graphical.\n",
	"-L: use a Light background.\n",
	"-[RGB]: display colors: Red, Green, Blue, or combinations.\n",
	"-m mode: video Mode.\n",
	"-r edges: drawing Rate: edges per revolution.\n",
#endif
	"-i file|-: display only Input polyhedra (Each line should name\n",
	"    one polyhedron, in term of it Wythoff symbol (3 rationals and a |)\n",
	"    or its index (# and an integer in the range 1 through ",
	0
};
char opt[0177];
rational frax;

main (argc, argv)
int argc;
char **argv;
{
	vector *v, *f;
	short **adj;/*vertices adjecent to a vertex*/
	short **incid;/*faces incident with a vertex*/
	char *firstrot, *rot, *snub, *ftype;
	int V, E, N, M, F, chi, D, g, K, Fi[5];
	int newV, newF;
	int i, k, j, j0, index, twos, bars, onesided, even, hemi;
	int line = 0, more_lines = 23, mode = -1, rate = 25200, digits = 6;
	char *prefix = 0, *Prefix = 0;
	char *s, *t;/* general purpose string pointers */ 
	char polyname[80], dualname[80], polyform[20], config[30];
	double p[4], n[5], m[5], n0 ,minr;
	double alpha[5], gamma[5], cos2gamma[5], sin2gamma[5];
	double cosa, sina;
	static char *group[] = {"di", "tetra", "octa", "icosa"};
	static char *alias[] = {"D", "A4", "S4", "A5"};
	double freeze = 0;
	double azimuth = AZ, elevation = EL;
	FILE *input = 0;

	kaleido[3] = *argv;
	while (--argc) 
		if (*(s = *++argv) == '-') {
			while (*++s) switch (*s) {
				char *argp;
			case 'l': case 'v': case 'x':
				opt[*s] = 1; break;
			case 'd':
				digits = atoi (VAL); break;
			case 'f':
				freeze = atof (VAL) / DEG; opt['f'] = 1; break;
			case 'a':
				azimuth = atof (VAL) / DEG; break;
			case 'e':
				elevation = atof (VAL) / DEG; break;
			case 'p':
				prefix = VAL; break;
			case 'P':
				Prefix = VAL; break;
#ifndef NOSCOPE
			case 'g': case 'n': case 'L': case 'R': case 'G': case 'B':
				opt[*s] = 1; break;
			case 'm':
				mode = atoi (VAL); break;
			case 'r':
				rate = atoi (VAL); break;
#endif
			case 'i':
				if (!strcmp (VAL, "-")) input = stdin;
				else if (!(input = fopen (argp, "r"))) {
					fprintf (stderr, "%s: cannot open: %s\n", kaleido[3], argp);
					usage();
				}
				break;
			default:
				fprintf (stderr, "%s: illegal  option: -%c\n",
					kaleido[3], *s);
				usage ();
			}
		} else usage ();
	if (!prefix && !Prefix) {
		extern int getpid();
		static char defaultprefix[9] = "pic";
		sprintf (defaultprefix + 3, "%d", getpid());
		Prefix = prefix = defaultprefix;
	} else if (!prefix) prefix = Prefix;
	else if (!Prefix) Prefix = prefix;
	if (opt['g']) {
#ifndef NOSCOPE
		if (!opt['R'] && !opt['G'] && !opt['B'])
			opt['R'] = opt['G'] = opt['B'] = 1;
		if (opt['x'] || opt['v']) opt['n'] = 1;
		if (!gcont && !opt['n'] && input!=stdin) gcont = 1;
		if (gcont &&  !ginit(mode,opt['R'],opt['G'],opt['B'],opt['L'])) {
			fprintf (stderr, "%s: cannot initialize graphics\n", kaleido[3]);
			exit (1) ;
		}
#endif
	} else {
		opt['n'] = 1;	
		if (opt['f']) rotframe (azimuth, elevation, freeze);
	}
	if (!input) index = 0;
	if (s = getenv ("LINES"))
		more_lines = atoi (s) - 1;
	for (;;) {
/*
* Get Wythoff symbol (p|qr, pq|r, pqr| or |pqr), and find the Moebius
* triangle (2 3 K) (or (2 2 k)) of the Schwarz triangle (pqr),
* the order g of its symmetry group, its Euler characteristic chi, 
* and its covering density D.
* g is the number of copies of (2 3 K) covering the sphere, i.e.,
*
*		g * pi * (1/2 + 1/3 + 1/K - 1) = 4 * pi
*		
* D is the number of times g copies of (pqr) cover the sphere, i.e.
*
*		D * 4 * pi = g * pi * (1/p + 1/q + 1/r - 1)
*
* chi is V - E + F, where F = g is the number of triangles, E = 3*g/2 is the
* number of triangle edges, and V = Vp+ Vq+ Vr, with Vp = g/(2*np) being the
* number of vertices with angle pi/p (np is the numerator of p).
*/
		K = 2; snub = 0; even = -1; twos = bars = onesided = hemi = 0;
		s = polyform;
		if (input) index = data (input, p);
		if (index == last_uniform - 1) *s++ = '|';
		for (j = 0; j < 4; j++) {
			if (index != -1 ? p[j] = uniform[index].Wythoff[j] : p[j]) {
				if (j && p[j-1]) *s++ = ' ';
				s = sprintfrac (s, p[j]);
				if (p[j] <=1) break;
				else if (p[j] != 2) {
					if ((k = numerator (p[j])) > K) {
						if (K == 4) break;
						K = k;
					} else if (k < K && k == 4) break;
				} else twos++;
			} else {
				*s++ = '|';
				bars++;
			}
		}
		*s++= '\0';
		if (twos < 2) g = 24 * K / (6 - K);
		else {/*dihedral*/
			g = 4 * K;
			K = 2;
		}
		if (index != last_uniform - 1) {
			D = chi = - g;
			for (j = 0; j < 4; j++) if (p[j]) {
				chi += i = g / numerator (p[j]);
				D += i * denominator (p[j]);
			}
			chi /= 2;
			D /= 4;
			if (bars != 1 || j < 4 || twos < 2 && K > 5 || D < 1) {
					fputs (polyform, stderr);
					fputs (": impossible polyhedron\n", stderr);
					continue;
			}		
		}
		errno = 0;
/*
* Decompose Schwarz triangle into N right triangles and compute the number of
* vertices V and the vertex order M.
* V is computed from the number g of Schwarz triangles in the cover, divided
* by the number of triangles which share a vertex. It is halved for one-sided
* polyhedra, because the kaleidoscopic construction really produces a double
* orientable covering of such polyhedra. All q' q|r are of the "hemi" type,
* i.e. have equatorial {2r} faces, and therefore are (except 3/2 3|3 and the
* dihedra 2 2|r) one-sided. A well known example is 3/2 3|4, the "one-sided
* heptahedron". Also, all p q r| with one even denominator have a crossed
* parallelogram as a vertex figure, and thus are one-sided as well.
*/
		if (!p[1]) { /* p|q r */
			N = 2;
			M = 2 * numerator (p[0]);
			V = g / M;
			for (j = 0; j < 2; j++) {
				n[j] = p[j+2];
				m[j] = p[0];
			}
			s = rot = (char*) malloc (M * sizeof (char));
			for (i = M / 2; i--;) {
				*s++ = 0;
				*s++ = 1;
			}
		} else if (!p[2]) { /* p q|r */
			N = 3;
			M = 4;
			V = g / 2;
			n[0] = 2 * p[3];
			m[0] = 2;
			s = rot = (char*) malloc (M * sizeof (char));
			for (j = 1; j < 3; j++) {
				n[j] = p[j-1];
				m[j] = 1;
				*s++ = 0;
				*s++ = j;
			}
			if (p[0] == compl (p[1])) {/* "hemi" */
				hemi = 1;
				D = 0;
				if (p[0] != 2 && !(p[3] == 3 && (p[0] == 3 || p[1] == 3))) {
					onesided = 1;
					V /= 2;
					chi /= 2;
				}
			}
		} else if (!p[3]) { /* p q r| */
			M = N = 3;
			V = g;
			s = rot = (char*) malloc (M * sizeof (char));
			for (j = 0; j < 3; j++) {
				if (!(denominator (p[j]) % 2)) { /*needs postprocessing*/
					even = j;/*memorize the removed face*/
					onesided = 1;
					D = 0;
					V /= 2;
					chi -= g / numerator (p[j]) / 2;
				}
				n[j] = 2 * p[j];
				m[j] = 1;
				*s++ = j;
			}
		} else { /* |p q r - snub polyhedron*/
			N = 4;
			M = 6;
			V = g / 2;/* Only "white" triangles carry a vertex */
			m[0] = n[0] = 3;
			s = rot = (char*) malloc (M * sizeof (char));
			t = snub = (char*) malloc (M * sizeof (char));
			for (j = 1; j < 4; j++) {
				n[j] = p[j];
				m[j] = 1;
				*s++ = 0;
				*s++ = j;
				*t++ = 1;
				*t++ = 0;
			} 
		}
/*
* Sort the fundamental triangles (using bubble sort) according to
* decreasing n[i], while pushing the trivial triangles (n[i] = 2)
* to the end.
*/
		j0 = N - 1;
		while (j0) {
			int last;
			double temp;
			last = j0;
			j0 = 0;
			for (j = 0; j < last; j++) {
				if ((n[j] < n[j+1] || n[j] == 2) && n[j+1] != 2) {
					temp = n[j];
					n[j] = n[j+1];
					n[j+1] = temp;
					temp = m[j];
					m[j] = m[j+1];
					m[j+1] = temp;
					for (i = 0; i < M; i++) {
						if (rot[i] == j) rot[i] = j+1;
						else if (rot[i] == j+1) rot[i] = j;
					}
					if (even != -1) {
						if (even == j) even = j+1;
						else if (even == j+1) even = j;
					}
					j0 = j;
				}
			}
		}
/*
*  Get rid of repeated triangles. 
*/
		for (j0 = 0; j0 < N && n[j0] != 2;j0++) {
			for (j = j0+1; j < N && n[j]==n[j0]; j++)
				m[j0] += m[j];
			k = j - j0 - 1;
			if (k) {
				for (i = j; i < N; i++) {
					n[i - k] = n[i];
					m [i - k] = m[i];
				}
				N -= k;
				for (i = 0; i < M; i++) {
					if (rot[i] >= j) rot[i] -= k;
					else if (rot[i] > j0) rot[i] = j0;
				}
				if (even >= j) even -= k;
			}
		}
/*
* Get rid of trivial triangles.
*/
		if (j0 < N) {
			N = j0;
			for (i = 0; i < M; i++) {
				if (rot[i] >= N) {
					for (j = i + 1; j < M; j++) {
						rot[j-1] = rot[j];
						if (snub) snub[j-1] = snub[j];
					}
					M--;
				}
			}
		}
		rot = (char*) realloc (rot, M * sizeof (char));
		if (snub) snub = (char*) realloc (snub, M * sizeof (char));
/*
* Get the polyhedron name, using table or guesswork.
* Ideally, we should try to locate the Wythoff symbol in the table
* (unless, of course, it is dihedral), after doing few normalizations,
* such as sorting angles and splitting isoceles triangles.
*/
		if (index != -1) {/*tabulated*/
			sprintf (polyname, "%d) %s", index + 1, uniform[index].name);
			sprintf (dualname, "%d*) %s", index + 1, uniform[index].dual);
		} else if (twos >= 2) {/*dihedral nontabulated*/
			double temp;
			if (!p[0]) { /* snub dihedral - antiprism */
				temp = n[0] == 3 ? n[1] : n[0];
				sprintfrac (polyname, temp < 2 ? compl (temp) : temp);
				strcat (polyname, "-gonal ");
				strcpy (dualname, polyname);
				if (temp < 2) {
					strcat (polyname, "crossed ");
					strcat (dualname, "concave ");
				}
				strcat (polyname, "antiprism");
				strcat (dualname, "deltohedron");/*has deltoidal faces*/
					/*somehow, Cundy & Rollett prefer trapezohedron*/
			} else if (!p[3] || !p[2] && p[3] == 2) {
				temp = n[1] < 2 ? compl (n[1]) : n[0] == 4 ? n[1] : n[0];
				sprintfrac (polyname, temp);
				strcat (polyname, "-gonal ");
				strcpy (dualname, polyname);
				strcat (polyname, "prism");
				strcat (dualname, "dipyramid");
			} else {
				sprintfrac (polyname, n[0]);
				strcat (polyname, "-gonal ");
				strcpy (dualname, polyname);
				strcat (polyname, "dihedron");
				strcat (dualname, "hosohedron");
			}
		} else {/*other nontabulated*/
			strcpy (polyname, group[K-2]);
			strcat (polyname, "hedral ");
			if (onesided) strcat (polyname, "one-sided ");
			else if (D == 1) strcat (polyname, "convex ");
			else strcat (polyname, "nonconvex ");
			strcpy (dualname, polyname);
			strcat (polyname, "isogonal polyhedron");
			strcat (dualname, "isohedral polyhedron");
		}
/*
* Print polyhedron name, Wythoff symbol, and reference figures.
*/
		if (opt['n']) {
			if (index != -1) {
				printf ("%d) %s %s", index + 1, uniform[index].name,
					polyform);
				if (uniform[index].Coxeter)
					printf (" [%d,%d]",uniform[index].Coxeter,
						uniform[index].Wenninger);
				putchar ('\n');
			} else printf ("%s %s\n", polyname, polyform);
		}
		if (opt['l']) {
			free (rot);
			if (snub) free (snub);
			if (!input && ++index == last_uniform) break;
			if (++line == more_lines) {
				more ();
				line = 0;
			}
			continue;
		}
/*
* Solve the right spherical triangles:
* First, we find initial approximations.
*/
		for (j = 0; j < N; j++) {
			alpha[j] = M_PI / n[j];
			gamma[j] = M_PI / 2 - alpha[j];
		}
/*
* Next, iteratively find closer approximations for gamma[0] and compute
* other gamma[j]'s from Napier's equations.
*/
		for (;;) {
			double delta = M_PI, sum = 0;
			for (j = 0; j < N; j++) {
				if (opt['x']) printf ("%-*.*f",
					digits + 5, digits, DEG * gamma[j]);
				delta -= m[j] * gamma[j];
			}
			if (opt['x']) printf ("(%g)\n", DEG * delta);
			if (abs(delta) < 8 * DBL_EPSILON) break;
			for (j = 0; j < N; j++) sum += m[j] * tan (gamma[j]);
			gamma[0] += delta * tan (gamma[0]) / sum;
			if (gamma[0] < 0 || gamma[0] > M_PI) {
				errno = -1;
				break;
			}
			cosa = cos (alpha[0]) / sin (gamma[0]);
			for (j = 1; j < N; j++)
				gamma[j] = asin (cos (alpha[j]) / cosa);
			if (errno) break;
		}
		if (errno) {
			fputs (polyform, stderr);
			fputs (": impossible polyhedron\n", stderr);
			errno = 0;
			free (rot);
			if (snub) free (snub);
			continue;
		}
		sina = sqrt (1 - cosa * cosa);
		if (hemi && N > 1) /* gamma[0] is pi/2 */
			minr = fabs (cosa * cos (gamma[1]) / sin (alpha[1]));
		else minr = fabs (cosa * cos (gamma[0]) / sin (alpha[0]));
			/*smallest non-zero inradius*/
/*
* Postprocess pqr| where r has an even denominator (cf. Coxeter &al. Sec.9).
* Remove the {2r} and add a retrograde {2p} and retrograde {2q}.
*/
		if (even != -1) {
			M = N = 4;
			for (j = even + 1; j < 3; j++) {
				n[j-1] = n[j];
				alpha[j-1] = alpha[j];
				gamma[j-1] = gamma[j];
			}
			n[2] = compl (n[1]);
			alpha[2] = M_PI - alpha[1];
			gamma[2] = - gamma[1];
			n[3] = compl (n[0]);
			m[3] = 1;
			alpha[3] = M_PI - alpha[0];
			gamma[3] = - gamma[0];
			rot = (char*) realloc (rot, M * sizeof (char));
			rot[0] = 0;
			rot[1] = 1;
			rot[2] = 3;
			rot[3] = 2;
		}

/*
* Postprocess the last polyhedron |3/2 5/3 3 5/2 by taking a |5/3 3 5/2,
* replacing the three snub triangles by four equatorial squares and adding
* the missing {3/2} (retrograde triangle, cf. Coxeter &al. Sec. 11).
* Note that it is not really necessary to assume that the `snub` squares are
* equatorial, and gamma[0] could be computed from asin(cos(alpha[0])/cosa).
*/
		if (index == last_uniform - 1) {
			N = 5;
			M = 8;
			hemi = 1;
			D = 0;
			for (i = 3; i; i--) {
				m[i] = 1;
				n[i] = n[i-1];
				alpha[i] = alpha[i-1];
				gamma[i] = gamma[i-1];
			}
			m[0] = n[0] = 4;
			alpha[0] = M_PI / 4;
			gamma[0] = M_PI / 2;
			m[4] = 1;
			n[4] = compl (n[1]);
			alpha[4] = M_PI - alpha[1];
			gamma[4] = - gamma[1];
			rot = (char*) realloc (rot, M * sizeof (char));
			snub = (char*) realloc (snub, M * sizeof (char));
			for (i = 1; i < 6; i+=2) rot[i]++;
			rot[6] = 0;
			rot[7] = 4;
			snub[6] = 1;
			snub[7] = 0;
		}
/*
* Compute edge and face counts.
*/
		F = E = 0;
		for (j = 0; j < N; j++) {
			int temp;
			E += temp = V * numerator (m[j]);
			F += Fi[j] = temp / numerator (n[j]);
		}
		E /= 2;
/*
* Update D in the few cases the density of the polyhedron is meaningful but
* different than the density of the corresponding Schwarz triangle
* (cf. Coxeter &al., p. 418 and p. 425).
* In these cases, spherical faces of one type are concave (bigger than a 
* hemisphere), and the actual density is the number of these faces less the
* computed density. 
* Note that if j != 0, the relation gamma[j] == asin(cos(alpha[j])/cosa)
* implies gamma[j] cannot be obtuse.
* Also, compute chi for the only non-Wythoffian polyhedron.
*/
		if (D && gamma[0] > M_PI / 2) D = Fi[0] - D;
		if (index == last_uniform - 1) chi = V - E + F;
/*
* Generate vertex configuration symbol.
*/
			t = config;
			*t++ = '(';
			for (i = 0; i < M; i++) {
				if (i) *t++ = '.';
				t = sprintfrac (t, n[rot[i]]);
			}
			if ((i = denominator (m[0])) == 1) sprintf (t, ")");
			else sprintf (t, ")/%d", i);
/*
* Print combinatorial description and solution.
*/
		if (opt['n']) {
			printf ("\tdual: %s\n\t%s, %shedral group %s",
				index != -1 ? uniform[index].dual: dualname,
				config, group[K-2], alias[K-2]);
			if (K == 2) printf ("%d", g / 4);
			printf (", chi=%d", chi);
			if (D) printf (", D=%d", D);
			else if (onesided) printf (", one-sided");
			printf ("\n\tV=%d, E=%d, F=%d=", V, E, F);
			for (j = 0; j < N; j++) {
				if (j) putchar ('+');
				printf ("%d{", Fi[j]);
				printfrac (n[j]);
				putchar ('}');
			}
			printf ("\n%6s%6s%*s%*s%*s%*s%*s%*s%*s%*s\n",
				"", "alpha", digits + 3, "gamma", digits + 1, "a",
				digits + 1, "b", digits + 1, "c", digits + 3, "rho/R",
				digits + 3, "r/rho", digits + 3, "l/rho", digits + 3, "h/r");
			for (j = 0; j < N; j++) {
				double cosc = cos (gamma[j]) / sin (alpha[j]);
				char temp[6];
				temp[0]='{';
				sprintfrac (temp + 1, n[j]);
				printf ("%5s}%6.1f%*.*f%*.*f%*.*f%*.*f%*.*f%*.*f%*.*f",
					temp, DEG * alpha[j],
					digits + 3, digits - 2, DEG * gamma[j],
					digits + 1, digits - 4, DEG * acos (cosa), 
					digits + 1, digits - 4, DEG * acos (cosa * cosc),
					digits + 1, digits - 4, DEG * acos (cosc),
					digits + 3, digits, cosa,
					digits + 3, digits, cosc,
					digits + 3, digits, sina / cosa);
				if (abs(cosc) < 1e-6) printf ("%*s\n", digits + 3, "infinity");
				else printf ("%*.*f\n",
					digits + 3, digits, sqrt (1 - cosc * cosc) / cosc);
			}
			putchar ('\n');
		}
		if (!opt['v'] && !opt['g'] && !opt['f']) {
			free (rot);
			if (snub) free (snub);
			if (!input && ++index == last_uniform) break;
			if (input != stdin) more();
			continue;
		}
		if (opt['n']) more ();
/*
* Compute polyhedron vertices and vertex adjecency lists.
*/
		if (!(v = (vector*) malloc (V * sizeof (vector))) ||
			!(adj = (short**) matalloc (M, V * sizeof (short))) ||
			!(firstrot = (char*) malloc (V * sizeof (char)))) {
			no_space ();
			free (rot);
			if (!input && ++index == last_uniform) break;
			continue;
		}
		for (j = 0; j < N; j++) {
			cos2gamma[j] = cos (2 * gamma[j]);
			sin2gamma[j] = sin (2 * gamma[j]);
		}
		v[0].x = 0;
		v[0].y = 0;
		v[0].z = 1;
		firstrot[0] = 0;
		adj[0][0] = 1;
		v[1].x = 2 * cosa * sina;
		v[1].y = 0;
		v[1].z = 2 * cosa * cosa - 1;
		if (!snub) {
			firstrot[1] = 0;
			adj[0][1] = -1;/*start the other side*/
			adj[M-1][1] = 0;
		} else {
			firstrot[1] = snub[M-1] ? 0 : M-1 ;
			adj[0][1] = 0;
		}
		newV = 2;
		for (i = 0; i < newV; i++) {
			vector temp, rotate();
			int last, one, start, limit;
			if (adj[0][i] == -1) {
				one = -1; start = M-2; limit = -1;
			} else {
				one = 1; start = 1; limit = M;
			}
			k = firstrot[i];
			for (j = start; j != limit; j += one) {
				temp = rotate (v[adj[j-one][i]], v[i], cos2gamma[rot[k]],
							one * sin2gamma[rot[k]]);
				for (j0=0; j0<newV && !same(v[j0],temp,BIG_EPSILON); j0++)
					;
				adj[j][i] = j0;
				last = k;
				if (++k == M) k = 0;
				if (j0 == newV) { /*new vertex*/
					if (newV == V) {
						newV = 0;
						break;
					}
					v[newV++] = temp;
					if (!snub) {
						firstrot[j0] = k;
						if (one > 0) {
							adj[0][j0] = -1;
							adj[M-1][j0] = i;
						} else {
							adj[0][j0] = i;
						}
					} else {
						firstrot[j0] = !snub[last] ? last : 
							!snub[k] ? (k+1)%M : k ;
						adj[0][j0] = i;
					}
				}
			}
			if (!newV) break;
		}	
		if (!newV) {
			if (index != -1) fprintf (stderr, "%d) ", index + 1);
			fputs (polyform, stderr);
			fputs (": too many vertices\n", stderr);
			if (!input && ++index == last_uniform) break;
			free (v);
			matfree (adj, M);
			free (firstrot);
			free (rot);
			if (snub) free (snub);
			continue;
		}
/*
* Compute polyhedron faces (dual vertices)
* For orientable polyhedra, we can distinguish between the two faces meeting
* at a given directed edge and identify the face on the left and the face on
* the right, as seen from the outside.
* For one-sided polyhedra, the vertex figure is a papillon (in Coxeter &al.
* terminology, a crossed parallelogram) and the two faces meeting at an edge
* can be identified as the side face (n[1] or n[2]) and
* the diagonal face (n[0] or n[3]).
*/
		if (!(f = (vector*) malloc (F * sizeof (vector))) ||
			!(ftype = (char*) malloc (F * sizeof (char))) ||
			!(incid = (short**) matalloc (M, V * sizeof (short)))) {
			no_space ();
			free (v);
			matfree (adj, M);
			free (firstrot);
			free (rot);
			if (snub) free (snub);
			if (!input && ++index == last_uniform) break;
			continue;
		}
		for (i = M; --i>=0; )
			for (j = V; --j>=0; )
				incid[i][j] = -1;
		newF = 0;/*face number*/
		for (i = 0; i < V; i++) {
			for (j = 0; j < M; j++) {
				int i0, j0;
				int pap;/*papillon edge type*/
				if (incid[j][i] != -1) continue;
				incid[j][i] = newF;
				f[newF] = pole (minr, v[i], v[adj[j][i]],
					v[adj[(j+1)%M][i]]);
				ftype[newF] = rot [(firstrot[i] + (adj[0][i]
					< adj[M-1][i] ? j : 2 * M - 2 - j)) % M];
				if (onesided) pap = (firstrot[i] + j) % 2;
				i0 = i;
				j0 = j;
				for (;;) {
					k = i0;
					if ((i0 = adj[j0][k]) == i) break;
					for (j0 = 0; adj[j0][i0] != k; j0++)
						;
					if (onesided && (j0 + firstrot[i0]) % 2 == pap) {
						incid [j0][i0] = newF;
						if (++j0 >= M) j0 = 0;
					} else {
						if (--j0 < 0) j0 = M - 1;
						incid [j0][i0] = newF;
					}
				}
				newF++;
			}	
		}
		free (firstrot);
		free (rot);
		if (snub) free (snub);
/*
* Print vertices and faces
*/
		if (opt['v']) {
			printf ("vertices:\n");
			for (i = 0; i < V; i++) {
				printf ("v%-3d (%*.*f,%*.*f,%*.*f)",
					i + 1, digits + 3, digits, v[i].x, digits + 3,
					digits, v[i].y, digits + 3, digits, v[i].z);
				for (j = 0; j < M; j++) printf (" v%-3d",adj[j][i] + 1);
				printf ("\n%*s", 3 * digits + 20, "");
				for (j = 0; j < M; j++) printf (" f%-3d",incid[j][i] + 1);
				putchar ('\n');
				if (!((i + 1) % (more_lines / 2))) more();
			}
			if (V % (more_lines / 2)) more ();

			printf ("faces (RHS=%*.*f", digits + 2, digits, minr);
			printf ("):\n");
			for (i = 0; i < F; i++){
				printf ("f%-3d (%*.*f,%*.*f,%*.*f) {",
					i + 1, digits + 3, digits, f[i].x, digits + 3,
					digits, f[i].y, digits + 3, digits, f[i].z);
				printfrac (n[ftype[i]]);
				putchar ('}');
				if (hemi && !ftype[i]) putchar ('*');
				putchar ('\n');
				if (!((i + 1) % more_lines)) more();
			}
			if (F % more_lines) more ();
			putchar ('\n');
		}
/*
* Compute edge lists and graph polyhedron and dual
* If the polyhedron is of the "hemi" type, each edge has one finite vertex
* and one ideal vertex. We make sure the latter is always the out-vertex,
* so that the edge becomes a ray (half-line).
* Each ideal vertex is represented by a unit vector, and the direction of
* the ray is either parallel or anti-parallel this vector. We flag this in the
* array anti[E] below.
*/
		if (opt['g'] || opt ['f']) {
			int c;
			short **e, *s, *t;
			char *anti, *u;
			double rotangle;
			int totalsteps, freezesteps;
			char *fn;
			if (!(e = (short**) matalloc (2, E * sizeof (short)))) {
				no_space ();
				free (v);
				matfree (adj, M);
				matfree (incid, M);
				free (f);
				free (ftype);
				if (!input && ++index == last_uniform) break;
				continue;
			}
#ifndef NOSCOPE
			if (opt['g']) {
				if (!gcont &&  !ginit(mode,opt['R'],opt['G'],opt['B'],opt['L'])) {
					fprintf (stderr, "%s: cannot initialize graphics\n",
						kaleido[3]);
					exit (1) ;
				}
				totalsteps = rate / E;
				rotangle  = 2 * M_PI / totalsteps;
				freezesteps = mod (freeze / rotangle, totalsteps);
				rotframe (azimuth, elevation, rotangle);
			}
#endif
			for (;;) {
				if (index != -1) sprintf (polyname, "%d) %s",
					index + 1, uniform[index].name); 
				s = e[0];
				t = e[1];
				for (i = 0; i < V; i++)
					for (j = 0; j < M; j++)
						if (i < adj[j][i]) {
							*s++ = i;
							*t++ = adj[j][i];
						}
				if (opt['g']) {
#ifndef NOSCOPE
					c = scope(v, V, e, E, (char*)0, polyname, config, polyform, 
						totalsteps, freezesteps, prefix);
					if (c!= '\t') break;
#endif
				} else {
					fn = picfile (v, V, e, E, (char*)0, polyname, config,
						polyform, prefix);
					if (!fn) fprintf (stderr, "%s: write error\n", polyname);
					else {
						fprintf (stderr, "%s: saved on %s\n", polyname, fn);
						free (fn);
					}
				}
				if (index != -1) sprintf (polyname, "%d*) %s",
					index + 1, uniform[index].dual); 
				s = e[0];
				t = e[1];
				if (!hemi) anti = 0;
				else if (!(u = anti = (char*) malloc (E * sizeof (char)))) {
					no_space ();
					break;
				}
				for (i = 0; i < V; i++)
					for (j = 0; j < M; j++)
						if (i < adj[j][i])
							if (!hemi) {
								*s++ = incid[(j+M-1)%M][i];
								*t++ = incid[j][i];
							} else {
								if (ftype[incid[j][i]]) {
									*s = incid[j][i];
									*t = incid[(j+M-1)%M][i];
								} else {
									*s = incid[(j+M-1)%M][i];
									*t = incid[j][i];
								}
								*u++ = dot (f[*s++], f[*t++]) > 0;
							}
				if (opt['g']) {
#ifndef NOSCOPE
					c = scope(f, F, e, E, anti, dualname, config, polyform,
						totalsteps, freezesteps, Prefix);
					if (hemi) free (anti);
					if (c!= '\t') break;
#endif
				} else {
					fn = picfile (f, F, e, E, anti, dualname, config,
						polyform, Prefix);
					if (!fn) fprintf (stderr, "%s: write error\n", dualname);
					else {
						fprintf (stderr, "%s: saved on %s\n",dualname, fn);
						free (fn);
					}
					if (hemi) free (anti);
					more ();
					break;
				}
			}
			matfree (e, 2);
#ifndef NOSCOPE
			if (opt['g']) {	
				if(!gcont) gend ();
				if (c == '\033') break;
				if (!input) {
					if ((c == '\n' || c == '\r') && ++index == last_uniform)
						index = 0;
					else if (c == '\b' && --index < 0) index = last_uniform - 1;
				}
			}
#endif

		} 
		free (v);
		matfree (adj, M);
		matfree (incid, M);
		free (f);
		free (ftype);
		if (!input && !opt['g'] && ++index == last_uniform) break;
	}
#ifndef NOSCOPE
	if(opt['g'] && gcont) gend();
#endif
	exit (0);
}

char*
picfile (v, V, e, E, anti, title, subtitle, subsub, prefix)
vector *v;
short **e;
int V, E;
char *anti;
char *title, *subtitle, *subsub, *prefix;
/* generate a pic file */
{
		int i;
		FILE *fp = 0;
		vector temp, *new;
		short *s, *t;
		char *u, *fn;

		if (!(fn = malloc (strlen (prefix) + 5))) {
			no_space ();
			return 0;
		}
		for (i = 0; i < 1000; i++) {
			sprintf (fn, "%s.%03d", prefix, i);
			if (access (fn, 0)) {/*file doesn't exist*/
				fp = fopen (fn, "w");
				break;
			}
		}
		if (!fp) {
			free (fn);
			return 0;
		}
		if (opt['g']) new = v;
		else {
			if (!(new = (vector*) malloc (V * sizeof (vector)))) {
				no_space ();
				fclose (fp);
				return 0;
			}
			rotarray (new, v, V);
		}
		fprintf (fp, ".PS\n.\\\" %s.\\\" %s.\\\" %s\n",
			kaleido[0], kaleido[1], title);
		fprintf (fp, ".ps 8\nbox ht 2.5 wid 2.5 at 0,0\n");
		fprintf (fp, "\"%s \" at last box.se rjust above\n", subtitle);
		fprintf (fp, "\" %s\" at last box.sw ljust above\n", subsub);
		for (i = 0; title[i] && title[i] != ')'; i++)
			;
		if (title[i])
			fprintf (fp, "\"Fig. %.*s\" at last box.s below\n",
				i, title);
		s = e[0];
		t = e[1];
		u = anti;
		for (i = E; !ferror (fp) && i--; ) {
			int j = *s++, k = *t++;
			if (!anti) {
				pic (fp, new[j], new[k], j, k);
			} else if (*u++) {
				temp = scale (.5, new[j]);
				pic (fp, temp, diff (temp, new[k]), j, k);
			} else {
				temp = scale (.5, new[j]);
				pic (fp, temp, sum (temp, new[k]), j, k);
			}
		}
		fprintf (fp, ".PE\n");
		i = ferror (fp);
		fclose (fp);
		if (!opt['g']) free (new);
		if (!i) return fn;
		free (fn);
		return 0;
}


pic (fp, a, b, j, k)
FILE *fp;
vector a, b;
int j, k;
/* generate pic command to draw a segment using variable pointsize */
{
	int i;
	double aa, bb;
	if (a.z < b.z) {/*swap endpoints to get increasing point size*/
		vector temp;
		temp = a;
		a = b;
		b = temp;
		i = j;
		j = k;
		k = i;
	}
	aa = 8 * (1 - a.z);
	bb = 8 * (1 - b.z);
	fprintf (fp, "move to %g,%g #v%d\n", a.x, a.y, j);
	for (i = (int) aa + 1; i < (int) bb + 1; i++) {
		double f;
		f = ((double) i - aa) / (bb - aa);
		fprintf (fp, ".ps %d\nline to %g,%g\n", i, 
			a.x + (b.x - a.x) * f,
			a.y + (b.y - a.y) * f);
	}
	fprintf (fp, ".ps %d\nline to %g,%g #v%d\n", i, b.x, b.y, k);
}

usage()
{
	char **p;
	for (p = kaleido; *p; p++) fputs (*p, stderr);
	fprintf (stderr, "%d.\n", last_uniform);
	exit (1);
}

int
data (fp, p)
/* Read input data: Wythoff symbol or a index to uniform[] */
FILE *fp;
double p[4];
{
	int i, c, n, d;
	if (isatty(fileno(stderr)) && isatty(fileno(fp)))
		fputs ("Enter data: ",stderr);
reenter:
	for (i = 0; i< 4; i++) {
		while ((c = getc (fp)) != EOF && isspace(c))
			;
		if (c == EOF) {
			if (i) break;
			if (isatty(fileno(stderr)) && isatty(fileno(fp)))
				fputs ("EOF\n", stderr);
#ifndef NOSCOPE
			if(opt['g'] && gcont) gend();
#endif
			exit (0);
		} else if (c == '#') {
			if (i || !isdigit (c = getc (fp))) break;
			ungetc (c, fp);
			fscanf (fp, "%d", &n);
			while ((c = getc(fp)) != EOF && c != '\n')
				;
			if (n >= 1 && n <= last_uniform) return n - 1;
			fprintf (stderr, "No such polyhedron: #%d\n", n);
			if (isatty(fileno(stderr)) && isatty(fileno(fp)))
				fputs ("Reenter data: ", stderr);
			i = -1;
			continue;
		} else if (c == '|'){
			p[i] = 0;
			continue;
		} else if (!isdigit(c)) break;
		ungetc (c, fp);
		fscanf (fp, "%d", &n);
		if (!n) { c = '0'; break; }
		if ((c = getc(fp)) != '/') {
			if (c != EOF) ungetc (c, fp);
			p[i] = n;
			continue;
		}
		if (!isdigit (c = getc (fp))) break;
		ungetc (c, fp);
		fscanf (fp, "%d", &d);
		if (!d) { c = '0'; break; }
		p[i] = (double) n / d;
	}
	while ((d = getc(fp)) != EOF && d != '\n')
		;
	if (i == 4) return -1;
	fputs ("Data error, saw ", stderr);
	if (c == EOF) fputs ("EOF\n", stderr);
	else if (isprint (c)) fprintf (stderr, "'%c'\n", c);
	else fprintf (stderr, "'\\0%o'\n", c);
	if (isatty(fileno(stderr)) && isatty(fileno(fp))) {
		fputs ("Reenter data: ", stderr);
		goto reenter;
	}
#ifndef NOSCOPE
	if(opt['g'] && gcont) gend();
#endif
	exit (1);
}


mod (i, j)
/*
* compute the mathematical modulus function.
*/
{    
	return (i%=j)>=0?i:j<0?i-j:i+j;
}

frac (x)
double x;
/*
* Find the numerator and the denominator using the Euclidean algorithm.
*/
{
	static rational zero = {0,1}, inf = {1,0};
	rational r0, r;
	long f;
	double s = x;
	r = zero;
	frax = inf;
	for (;;) {
		if (abs(s) > (double) MAXLONG) return;
		f = (long) floor (s);
		r0 = r;
		r = frax;
		frax.n = frax.n * f + r0.n;
		frax.d = frax.d * f + r0.d;
		if (x == (double)frax.n/(double)frax.d) return;
		s = 1 / (s - f); 
	}
}

printfrac (x)
double x;
/*
* Print a fraction.
*/
{
	frac (x);
	printf ("%d",frax.n);
	if (frax.d!=1) printf ("/%d",frax.d);
}

char*
sprintfrac (s, x)
char *s;
double x;
{
	frac (x);
	sprintf (s, "%d", frax.n);
	s += strlen (s);
	if (frax.d!=1) {
		sprintf (s, "/%d",frax.d);
		s += strlen (s);
	}
	return s;
}

double
dot (a, b)
vector a, b;
{
	return a.x * b.x + a.y * b.y + a.z * b.z;
}

vector
scale (k, a)
double k;
vector a;
{
	a.x *= k;
	a.y *= k;
	a.z *= k;
	return a;
}

vector
diff (a, b)
vector a,b;
{
	a.x -= b.x;
	a.y -= b.y;
	a.z -= b.z;
	return a;
}

vector
cross (a, b)
vector a, b;
{
	vector p;
	p.x = a.y * b.z - a.z * b.y;
	p.y = a.z * b.x - a.x * b.z;
	p.z = a.x * b.y - a.y * b.x;
	return p;
}

vector
sum (a, b)
vector a, b;
{
	a.x += b.x;
	a.y += b.y;
	a.z += b.z;
	return a;
}

vector
sum3 (a, b, c)
vector a, b, c;
{
	a.x += b.x + c.x;
	a.y += b.y + c.y;
	a.z += b.z + c.z;
	return a;
}

vector
rotate (vertex, axis, cosine, sine)
vector vertex, axis;
double cosine, sine;
{
	vector p;
	p = scale (dot (axis, vertex), axis);
	return sum3 (p, scale(cosine, diff (vertex, p)),
		scale(sine, cross (axis, vertex)));
}

vector x, y, z;

rotframe (azimuth, elevation, angle)
double azimuth, elevation, angle;
/* rotate the standard frame */
{
	static vector x0 = {1,0,0}, y0 = {0,1,0}, z0 = {0,0,1}; 
	vector axis;
	double cosine, sine;

	axis = rotate (rotate (x0, y0, cos (elevation),
		sin (elevation)), z0, cos (azimuth), sin (azimuth));
	cosine = cos (angle);
	sine = sin (angle);
	x = rotate (x0, axis, cosine, sine);
	y = rotate (y0, axis, cosine, sine);
	z = rotate (z0, axis, cosine, sine);
}

rotarray (new, old, n)
vector *new, *old;
int n;
/* rotate an array of n vectors */
{
	while (n--) {
		*new++ = sum3 (scale (old->x, x),
			scale (old->y, y), scale (old->z, z));
		old++;
	}
}

int
same (a, b, epsilon)
vector a, b;
double epsilon;
{
	return abs (a.x - b.x) < epsilon && abs (a.y - b.y) < epsilon
		&& abs (a.z - b.z) < epsilon;
}

vector
pole (r, a, b, c)
double r;
vector a, b, c;
/*
* Compute the polar reciprocal of the plane containing a, b and c:
*
* If this plane does not contain the origin, return p such that
*	dot(p,a) = dot(p,b) = dot(p,b) = r. 
*
* Otherwise, return p such that
*	dot(p,a) = dot(p,b) = dot(p,c) = 0
* and
*	dot(p,p) = 1.
*/
{
	vector p;
	double k;
	p = cross (diff (b, a), diff (c, a));
	k = dot (p, a);
	if (abs (k) < 1e-6) return scale (1 / sqrt (dot (p, p)), p);
	else return scale (r/ k , p);
}

char **
matalloc (rows, row_size)
int rows, row_size;
{
	char **mat;
	int i = 0;
	if (!(mat = (char**) malloc (rows * sizeof (char*)))) return 0;
	while ((mat[i] = (char*) malloc (row_size)) && ++i < rows)
		;
	if (i == rows) return mat;
	while (--i >= 0) free (mat[i]);
	free (mat);
	return 0;
}

matfree (mat, rows)
char** mat;
{
	while (--rows >= 0) free (mat[rows]);
	free (mat);
}

more ()
{
	int c;
	if (isatty(fileno(stdout)) && isatty(fileno(stdin))) {
		fputs ("Enter for more...", stdout);
		while ((c = getchar ()) != EOF && c != '\n') ;
	}
}
