(* Copyright (c) 1992 The Geometry Center; University of Minnesota
   1300 South Second Street;  Minneapolis, MN  55454, USA;
   
This is file OOGL.m.  OOGL.m is free software.  You can redistribute
it and/or modify it only under the terms given in the file COPYING,
which you should have received along with this file.  this and other
related software may be obtained via anonymous ftp from geom.umn.edu;
email: software@geom.umn.edu. *)

(* Author: Nils McCarthy *)

BeginPackage["OOGL`"]

WriteOogl[x___] :=(
Print["WriteOogl has been renamed to WriteOOGL."]
)

OoglFormat[x___] :=(
Print["OoglFormat has been renamed to OOGLFormat."]
)

OOGLFormat::usage = "
OOGLFormat[x] gives an OOGL representatio of x.  X can be either a list
of polygons and/or lines, or a Grpahics3D object.  Things other than
Polygons or Lines in x will be ignored."

DivideQuads::usage=  "
An option to give to WriteOOGL to get it to divide quadrilaterals on
graphics3d objects.
"

MergeVertices::usage =  "
An option to give to WriteOOGL to get it not to collect common Vertices
on graphics3d objects.
"

WriteOOGL::usage=  "
WriteOOGL [<filename or stream>, -Graphics3D-]\n
WriteOOGL [<filename or stream>, -SurfaceGraphics-] writes the given
Graphics3D or SurfaceGraphics object to the given filename or
stream in OOGL format.  In the Graphics3D object, things other than
Polygons or Lines will be ignored.  WriteOOGL takes the options
DivideQuads and MergeVertices."

WriteOOGL /: Options[WriteOOGL] := {
     MergeVertices -> False,
     DivideQuads -> False
};

Geomview::usage= "
Geomview [-Graphics3D or SurfaceGraphics-]\n
Geomview [\"name\", -Graphics3D or SurfaceGraphics-]
writes the given graphics object to Geomview, starting a copy of geomview
if necessary.  Only Polygons and Lines are drawn from Graphics3D objects.
Geomview[\"name\", ...] creates geomview object with that name;
default is \"Mathematica\".  Takes options DivideQuads and MergeVertices."

Geomview /: Options[Geomview] := {
	MergeVertices -> False,
	DivideQuads -> False
};

Begin["`Private`"]

toGV = "!togeomview ";

Geomview[ name_String, data_, options___Rule ] := Block[
	  {out = OpenWrite[toGV]},
		WriteString[out, StringJoin["(geometry \"", name, "\" {\n"]];
		WriteOOGL[ out, data, options ];
		WriteString[out, "\n})"];
		Close[out];
		data
	];

Geomview[ data_, options___Rule ] := Geomview[ "Mathematica", data, options ];



WriteOOGL [outstream_OutputStream, surface_SurfaceGraphics, options___Rule] :=(
    Block[{ pts = surface[[1]],  (** the big list of points **)
            colors = surface[[2]], (** the big list of colors if iscolor **)
            xpoints = Length[surface[[1]]], (** num of points in the x directin **)
            ypoints = Length[surface[[1]][[1]]], (** ' ' in the y direction **)
            iscolor = (Length[surface]==3),
            height = Sqrt[xpoints * ypoints],
            option = surface[Length[surface]]}, (** options. currently not used **)
    (
        WriteString[outstream,
            If[iscolor, "CZMESH\n\n", "ZMESH\n\n"]];
        WriteString[outstream, TableForm[CForm[#1]& /@
                                Flatten[{xpoints, ypoints,
            Table[ (** a table of all the points **)
                (If[iscolor, {pts[[x]][[y]] * height,
                            offcolor[colors[[If[x==xpoints,x-1,x]]]
                                        [[If[y==ypoints, y-1, y]]]], 1}
                        ,{pts[[x]][[y]] * height}]),
                {x, 1, xpoints}, {y, 1, ypoints}]
        }]]]
    )]
)

WriteOOGL [filename_String, obj_, options___Rule] :=(
    Block[{strm},(
        strm=OpenWrite[filename];
        WriteOOGL[strm, obj, options];
        Close[strm];
    )]
)

offnumtostring [n_] := ToString[CForm[N[n]]]


WriteOOGL [outstream_, gr_Graphics3D, options___Rule] :=(
    WriteString[outstream, OOGLFormat[gr, options]]
)

OldWriteOOGL [outstream_OutputStream,Graphics3D[gr___],options___Rule] :=(
    WriteString[outstream, "LIST\n\n"];
    
    Block [{x = Map[(Cases[Flatten[{gr}],Blank[#1]])&,
                     {Polygon, Line}]}, (** separate it out into the
                                            polygons and lines. possibly
                                            other things in the future **)
        WriteString[outstream, (** put out the actual OOGLs **)
         (StringJoin[
	    If[x[[1]] == {}, "", (StringJoin["{ =\n", OOGLFormat[x[[1]],options], "\n}\n"])],
            If[x[[2]] == {},"",StringJoin["{ =\n", OOGLFormat[x[[2]]], "\n}\n"]]
	])
       ]
    ]
)

(** color utilities **)
offcolor[foo_] := (StringJoin[
    " ", offnumtostring[offred[foo]], " ", offnumtostring[offgreen[foo]], " ",
     offnumtostring[offblue[foo]]
])

offred[RGBColor[r_, g_, b_]] := r;
offgreen[RGBColor[r_, g_, b_]] := g;
offblue[RGBColor[r_, g_, b_]] := b;
offred[GrayLevel[x_]] := x;
offgreen[GrayLevel[x_]] := x;
offblue[GrayLevel[x_]] := x;

OOGLFormat[{polys__Polygon},options___Rule] :=(
    (** a list of polygons.. maybe shared vertices.
        use off. **)
    polylist = Flatten[{polys}];
    If[DivideQuads /. Flatten[{options}] /. Options[OOGLFormat],
        polylist = Flatten[(
            If[Length[(#1)[[1]]]!=4, #1,
                {Polygon[{#1[[1]][[1]],#1[[1]][[2]],#1[[1]][[3]]}],
                 Polygon[{#1[[1]][[3]],#1[[1]][[4]],#1[[1]][[1]]}]}]
        ) & /@ polylist]
    ];
    If[(MergeVertices /. {options} /. Options[WriteOOGL]) == False,
        ToString[OutputForm[TableForm[Join[
            {"OFF"},
            {{Apply[Plus,(Length[#1[[1]]]& /@ polylist)], mylen=Length[polylist], 1}},
            {""},
            {offnumtostring[#1[[1]]],offnumtostring[#1[[2]]],offnumtostring[#1[[3]]]}& /@ Apply[Join,(#1[[1]] & /@ polylist)],
            {""},
            noat = -1; (** start at zero. **)
            (
                Flatten[{Length[#1[[1]]],
                    Range[1+noat, (noat += Length[#1[[1]]])]}]
            )& /@ polylist,
            {""}
        ]]]],
        Block[{allVertices},(
            allVertices = Apply[Union,#1[[1]]& /@ polylist];
            ToString[OutputForm[TableForm[Join[
                {"OFF"},
                {{ Length[allVertices], Length[polylist], 1}},
                {""},
                ( {offnumtostring[#1[[1]]], offnumtostring[#1[[2]]], offnumtostring[#1[[3]]]} & /@ allVertices ), (** a list of all the Vertices **)
                {""},
                ( Join[ {Length[#1[[1]]]}, Function[x,Position[allVertices,x][[1]][[1]]-1] /@ #1[[1]] (** and then all the polygons **)
                ] & /@ polylist ),
		{""}
            ]]]]
        )]
    ]
);

OOGLFormat[{lines__Line}, options___Rule] :=(
    linelist = Flatten[{lines}];
    ToString[OutputForm[TableForm[Join[
        {"VECT"},
        {{Length[linelist], Length[Flatten[((#1)[[1]])& /@ linelist, 1]],
                0}}, (** the number of lines, and the total number of Vertices **)
        {},
        {Length[((#1)[[1]])]& /@ linelist}, (** the number of Vertices for each line. **)
        {},
        {(0)& /@ linelist}, (** the color values for each **)
        {},
        (offnumtostring /@ #1)& /@ Flatten[((#1)[[1]])& /@ linelist, 1] (** and then all the points. **)
    ]]]]
)


OOGLFormat[{polysandlines__}, options___Rule] :=(
    Block[{polys,lines}, (
        polys = Cases[Flatten[{polysandlines}],_Polygon];
	lines = Cases[Flatten[{polysandlines}],_Line];
	If [polys == {},
	    OOGLFormat[lines, options],
	    If [lines == {},
		OoglFormat[polys, options],
		StringJoin[ (** both lines and polys **)
		    "LIST\n\n{ =\n",
		    OOGLFormat[polys,options],
		    "\n}\n\n{ =\n",
		    OOGLFormat[lines,options],
		    "\n}\n\n"
		]
	    ]
	]
    )]
)

OOGLFormat[x_Graphics3D, options___Rule] :=(
    OOGLFormat[x[[1]]]
)

(* Given junk, silently ignore it. *)
OOGLFormat[x_, options___Rule] := {}

End[]
EndPackage[]
