 /*
  * Khoros: $Id: update.c,v 1.4 1992/03/20 22:43:40 dkhoros Exp $
  */

#if !defined(lint) && !defined(SABER)
static char rcsid[] = "Khoros: $Id: update.c,v 1.4 1992/03/20 22:43:40 dkhoros Exp $";
#endif

 /*
  * $Log: update.c,v $
 * Revision 1.4  1992/03/20  22:43:40  dkhoros
 * VirtualPatch5
 *
  */ 


/*
 *----------------------------------------------------------------------
 *
 * Copyright 1990, University of New Mexico.  All rights reserved.
 *
 * Permission to copy and modify this software and its documen-
 * tation only for internal use in your organization is hereby
 * granted, provided that this notice is retained thereon and
 * on all copies.  UNM makes no representations as too the sui-
 * tability and operability of this software for any purpose.
 * It is provided "as is" without express or implied warranty.
 * 
 * UNM DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FIT-
 * NESS.  IN NO EVENT SHALL UNM BE LIABLE FOR ANY SPECIAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY OTHER DAMAGES WHAT-
 * SOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER
 * IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS
 * ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PER-
 * FORMANCE OF THIS SOFTWARE.
 * 
 * No other rights, including for example, the right to redis-
 * tribute this software and its documentation or the right to
 * prepare derivative works, are granted unless specifically
 * provided in a separate license agreement.
 *---------------------------------------------------------------------
 */

#include "unmcopyright.h"	 /* Copyright 1990 by UNM */
#include "cantata.h"


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>								<<<<
   >>>>	    file name:  update.c				<<<<
   >>>>								<<<<
   >>>>   description:  This file is to update the widgets	<<<<
   >>>>			on the glyph.				<<<<
   >>>>								<<<<
   >>>>      routines:  xvl_update_dav()			<<<<
   >>>>			xvl_update_optional()			<<<<
   >>>>			xvl_update_modified()			<<<<
   >>>>			xvl_update_selected()			<<<<
   >>>>			xvl_update_mapping()			<<<<
   >>>>			xvl_update_mapped()			<<<<
   >>>>			xvl_update_clipboard()			<<<<
   >>>>			xvl_update_input()			<<<<
   >>>>			xvl_update_undo()			<<<<
   >>>>			xvl_update_glyph_label()		<<<<
   >>>>			xvl_update_distributed()		<<<<
   >>>>			xvl_update_negotiator()			<<<<
   >>>>								<<<<
   >>>>			xvl_update_all_optional()		<<<<
   >>>>			xvl_update_all_dav()			<<<<
   >>>>			xvl_update_all_modified()		<<<<
   >>>>			xvl_update_all_selected()		<<<<
   >>>>			xvl_update_all_mapping()		<<<<
   >>>>			xvl_update_all_mapped()			<<<<
   >>>>			xvl_update_all_clipboard()		<<<<
   >>>>								<<<<
   >>>> modifications:						<<<<
   >>>>								<<<<
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< */



/************************************************************
*
* Routine Name:  xvl_update_dav
*
*      Purpose:  This routine is used to change the dav label
*		 to be shown available or not.  If dav is true
*		 then the dav widget is mapped, otherwise it is
*		 unmapped.
*
*        Input:  node      - the node to be changed.
*
*
*   Written By: Mark Young
*
*************************************************************/


xvl_update_dav(node)

Node		*node;
{
	Workspace *workspace, *attributes;


	if (node->dav_widget == NULL)
	   return;

	workspace = node->glyph->workspace;
	attributes = xvl_get_attributes(workspace);
	if (node->dav && attributes->show_dav)
	   XtMapWidget(node->dav_widget);
	else
	   XtUnmapWidget(node->dav_widget);
}



/************************************************************
*
* Routine Name:  xvl_update_optional
*
*      Purpose:  This routine is used to change the optionally
*		 selected pixmap according to if a node is
*		 selected or not.  This should not be confused
*		 with when a glyph is selected in a workspace.
*
*        Input:  node      - the node to be changed.
*
*
*   Written By: Mark Young
*
*************************************************************/


xvl_update_optional(node)

Node		*node;
{
	int	i;
	Arg	arg[5];
	Pixmap  bitmap;
	Workspace *workspace, *attributes;


	workspace = node->glyph->workspace;
	attributes = xvl_get_attributes(workspace);
	if (node->selected)
	{
	   if (attributes->small_glyphs == True)
	      bitmap = ios_pix;
	   else
	      bitmap = io_pix;
	}
	else
	{
	   if (attributes->small_glyphs == True)
	      bitmap = io_opts_pix;
	   else
	      bitmap = io_opt_pix;

	}
	i = 0;
	XtSetArg(arg[i], XtNbitmap, bitmap); 	i++;
	XtSetValues(node->widget, arg, i);
}



/************************************************************
*
* Routine Name:  xvl_update_modified
*
*      Purpose:  This routine is used to change the modified label
*		 to be shown available or not.  If modified is true
*		 then the modified widget is mapped, otherwise it is
*		 unmapped.
*
*        Input:  glyph      - the glyph to be changed.
*
*
*   Written By: Mark Young
*
*************************************************************/


xvl_update_modified(glyph)

Glyph		*glyph;
{
	int	  running;
	Workspace *workspace, *attributes;


	if (glyph->modified == True && 
	    xvl_check_if_loop_node(glyph, glyph->input) == False)
	{
	   xvl_clear_dav(glyph, NULL);
	}

	/*
	 *  Comment glyphs and glyphs that are of run type SOURCE are immunue
	 *  to being modified and therefore should not reflect a modified
	 *  state.
	 */
	if (glyph->type == COMMENT || (glyph->exec_type == NORUN &&
	     glyph->run_type == SOURCE))
	   return;

	/*
	 *  Since the glyph is modified we need to clear it's data avaiables
	 *  since they are not currently available.
	 */
	workspace = glyph->workspace;
	running = xvl_check_if_glyphlist(glyph, workspace->running);

	if (running == False)
	{
	   attributes = xvl_get_attributes(workspace);
	   if (glyph->modified && attributes->show_modified)
	   {
	      XSetWindowBackgroundPixmap(display, XtWindow(glyph->back),
			attributes->modified_pix);
	   }
	   else
	   {
	      XSetWindowBackgroundPixmap(display, XtWindow(glyph->back), None);
	      XSetWindowBackground(display, XtWindow(glyph->back),
			wresource.glyph_bg);
	   }
	   XClearWindow(display, XtWindow(glyph->back));
	}
}



/************************************************************
*
* Routine Name:  xvl_update_selected
*
*      Purpose:  This routine is used to change the selected
*		 pixmap according to if a glyph is in the workspace's
*		 selected list or nor.
*
*        Input:  glyph   - the glyph to be updated.
*
*
*   Written By: Mark Young
*
*************************************************************/


xvl_update_selected(glyph)

Glyph		*glyph;
{
	int	  i;
	Workspace *workspace;
	Position  xpos, ypos;
	Dimension border_width;
	Arg	  args[MaxArgs];


	workspace = glyph->workspace;
	if (xvl_check_if_glyphlist(glyph, workspace->selected))
	{
	   border_width = 5;
	   xpos = glyph->xpos -3;
	   ypos = glyph->ypos -3;
	}
	else
	{
	   border_width = 2;
	   xpos = glyph->xpos;
	   ypos = glyph->ypos;
	}

	i = 0;
	XtSetArg(args[i], XtNborderWidth, border_width);        i++;
	if (glyph->xpos != -1 && glyph->ypos != -1)
	{
	   XtSetArg(args[i], XtNx, xpos);			i++;
	   XtSetArg(args[i], XtNy, ypos);			i++;
	   XtSetArg(args[i], XtNhorizDistance, (int) xpos);	i++;
	   XtSetArg(args[i], XtNvertDistance,  (int) ypos);	i++;
	}
	XtSetValues(glyph->toplevel, args, i);
	(void) XtConfigureWidget(glyph->toplevel, xpos, ypos, glyph->width,
		glyph->height, border_width);
}



/************************************************************
*
* Routine Name:  xvl_update_mapping
*
*      Purpose:  This routine is used to change the glyph's form
*		 widget to show whether the glyph's subform or form
*		 is currently mapping or not.  If mapping is true
*		 then the form widget's color is inverted, otherwise
*		 it is set to be normal.
*
*        Input:  glyph      - the glyph to be changed.
*
*
*   Written By: Mark Young
*
*************************************************************/


xvl_update_mapping(glyph)

Glyph		*glyph;
{
	Workspace *attributes, *macro;


	if (glyph->xpos == -1 && glyph->ypos == -1)
	   return;

	/*
	 *  If we want the forms library to map and unmap our glyphs then
	 *  we need to set the glyph toplevel for the form or subform.
	 */
	attributes = xvl_get_attributes(glyph->workspace);
	if (glyph->type == PROCEDURE)
	{
	   macro = glyph->val.macro;
	   if (attributes->leave_glyph == False)
	   {
	      macro->menuform->glyph_toplevel = glyph->toplevel;
	      if (macro->menuform->glyph_state)
		 XtMapWidget(glyph->toplevel);
	      else
		 XtUnmapWidget(glyph->toplevel);
	   }
	   else
	   {
	      macro->menuform->glyph_toplevel = NULL;
	      XtMapWidget(glyph->toplevel);
	   }
	}
	else if (glyph->type == GLYPH || glyph->type == CONTROL ||
		 glyph->type == COMMENT || glyph->type == COMMAND)
	{
	   if (attributes->leave_glyph == False)
	   {
	      glyph->val.subform->glyph_toplevel = glyph->toplevel;
	      if (glyph->val.subform->glyph_state)
		 XtMapWidget(glyph->toplevel);
	      else
		 XtUnmapWidget(glyph->toplevel);
	   }
	   else
	   {
	      glyph->val.subform->glyph_toplevel = NULL;
	      XtMapWidget(glyph->toplevel);
	   }
	}
	else if (glyph->type == IMAGES)
	{
	      /* not sure what to do at this time */
	}
}



/************************************************************
*
* Routine Name:  xvl_update_mapped
*
*      Purpose:  This routine is used to change the glyph's form
*		 widget to show whether the glyph's subform or form
*		 is currently mapped or not.  If mapped is true
*		 then the form widget's color is inverted, otherwise
*		 it is set to be normal.
*
*        Input:  glyph      - the glyph to be changed.
*
*
*   Written By: Mark Young
*
*************************************************************/


xvl_update_mapped(glyph)

Glyph		*glyph;
{
	int	  mapped;
	Pixel	  fg, bg;
	Workspace *macro;


	if (glyph->type == PROCEDURE)
	{
	   macro = glyph->val.macro;
	   if (macro == NULL)
	      return;

	   if (macro->menuform->glyph_state == True)
	     mapped = True;
	   else
	     mapped = False;
	}
	else if (glyph->type == GLYPH || glyph->type == CONTROL ||
		 glyph->type == COMMENT || glyph->type == COMMAND)
	{
	   if (glyph->val.subform->glyph_state == True)
	     mapped = True;
	   else
	     mapped = False;
	}
	else return;

	if (wresource.form_fg == ~0L || wresource.form_bg == ~0L)
	{
	   xvl_init_color(glyph->form, &wresource.form_fg, &wresource.form_bg,
			NULL);
	}
	fg = wresource.form_fg;
	bg = wresource.form_bg;

	if (mapped == False)
	   xvf_change_colors(glyph->form, fg, bg);
	else
	   xvf_change_colors(glyph->form, bg, fg);
}



/************************************************************
*
* Routine Name:  xvl_update_clipboard
*
*      Purpose:  This routine is used to change the workspace
*		 clipboard widget to show whether the clipboard
*		 is currently mapped or not.  If mapped is true
*		 then the clipboard widget's color is inverted, otherwise
*		 it is set to be normal.
*
*        Input:  glyph      - the glyph to be changed.
*
*
*   Written By: Mark Young
*
*************************************************************/


xvl_update_clipboard(workspace)

Workspace	*workspace;
{
	Pixel	  fg, bg;


	if (clipboard == NULL)
	   return;

	if (wresource.glyph_fg == ~0L || wresource.glyph_bg == ~0L)
	{
	   xvl_init_color(workspace->clipboard, &wresource.glyph_fg,
			&wresource.glyph_bg, NULL);
	}
	fg = wresource.glyph_fg;
	bg = wresource.glyph_bg;

	if (clipboard->menuform->glyph_state == True)
	   xvf_change_colors(workspace->clipboard, bg, fg);
	else
	   xvf_change_colors(workspace->clipboard, fg, bg);
}



/************************************************************
*
* Routine Name:  xvl_update_input
*
*      Purpose:  This routine is used to update the undo status
*		 of the workspaces undo button.
*
*        Input:  glyph      - the glyph to be changed.
*
*
*   Written By: Mark Young
*
*************************************************************/


xvl_update_input(glyph)

Glyph	*glyph;
{
	Node	  *node;
	NodeList  *output;


	/*
	 *  We need to update the dav (data available) flags for the
	 *  glyph's children.  The other thing is we need to do is
	 *  set each of those input connection's glyph "input" to
	 *  the node that asserted it.
	 */
	output = glyph->output_list;
	while (output != NULL)
	{
	   node = output->node;
	   if (node->selected == True)
	   {
	      xvl_update_input_node(node);
	   }
	   output = output->next;
	}
}

xvl_update_input_node(node)

Node *node;
{
	NodeList *links;


	node->dav = True;
	xvl_update_dav(node);

	links = node->links;
	while (links != NULL)
	{
	   node = links->node;
	   node->dav = True;
	   xvl_update_dav(node);
	   node->glyph->input = node;

	   links = links->next;
	}
}



/************************************************************
*
* Routine Name:  xvl_update_undo
*
*      Purpose:  This routine is used to update the undo status
*		 of the workspaces undo button.
*
*        Input:  glyph      - the glyph to be changed.
*
*
*   Written By: Mark Young
*
*************************************************************/


xvl_update_undo(workspace)

Workspace	*workspace;
{
	int	  index;


	/*
	 *  Update the undo state.  If undo is not NULL and there is a
	 *  workspace containing temporarily deleted objects then go
	 *  ahead and then to see if any glyphs are currently in the
	 *  workspace.  If so then change the undo button from in-active
	 *  to active, this allows the user to undo the delete if desired.
	 */
	if (workspace->undo != NULL && workspace->undo_workspace != NULL)
	{
	   if (workspace->undo->button_index != -1)
	   {
	      index = workspace->undo->button_index;
	      if (workspace->undo_workspace->glyphs != NULL)
	      {
	         xvf_change_input(workspace->menuform,index,xvf_activation_chng,
			"1", 0);
	      }
	      else
	      {
	         xvf_change_input(workspace->menuform,index,xvf_activation_chng,
			"0", 0);
	      }
	   }
	}
	else if (workspace->undo != NULL)
	{
	   if (workspace->undo->button_index != -1)
	   {
	      index = workspace->undo->button_index;
	      xvf_change_input(workspace->menuform,index,xvf_activation_chng,
			"0", 0);
	   }
	}
}



/************************************************************
*
* Routine Name: xvl_update_glyph_label
*
*      Purpose: This routine is used to update the label on
*		the glyph.  It sees if the name has changed.
*		The only glyph's that have this capibility is
*		the comment and command glyph.
*
*       Output: None
*
*    CALLED BY: xvl_update_glyph()
*
*   WRITTEN BY: Mark Young
*
*
*************************************************************/


xvl_update_glyph_label(glyph)

Glyph *glyph;
{
	char *label;


	label = xvl_find_subform_label(glyph->val.subform);
	if ((label != NULL && glyph->label_str == NULL) ||
	    (label == NULL && glyph->label_str != NULL))
	{
	   if (glyph->label_str != NULL) free(glyph->label_str);

	   if (label != NULL)
	      glyph->label_str = label;
	   else
	      glyph->label_str = xvf_strcpy(glyph->glyph_name);

	   xvl_create_glyph(glyph);
	}
	else if (label != NULL)
	{
	   if (strcmp(label, glyph->label_str) != 0)
	   {
	      free(glyph->label_str);
	      glyph->label_str = label;
	      xvl_create_glyph(glyph);
	   }
	   else free(label);
	}
}



/************************************************************
*
* Routine Name:  xvl_update_distributed
*
*      Purpose:  This routine is used to map the glyph's remote
*		 execution button *widget which allows the user to
*		 to specify the remote machine in which to execute
*		 a process.
*
*        Input:  glyph      - the glyph to be changed.
*
*
*   Written By: Mark Young
*
*************************************************************/


xvl_update_distributed(glyph)

Glyph		*glyph;
{
	char	  temp[MaxLength];
	Workspace *workspace, *attributes, *macro;


	/*
	 *  A glyph will be abled to be scheduled remotely if it is runnable
	 *  and it has a remote Widget.
	 */
	if (glyph->remote == NULL)
	   return;

	/*
	 *  Update the glyph's location which will dictate how the input and
	 *  output names are built.
	 */
	xvl_update_glyph_location(glyph);

	/*
	 *  Get the attributes for this workspace so that we can see if we
	 *  are to enable or un-enable remote execution.
	 */
	workspace = glyph->workspace;
	attributes = xvl_get_attributes(workspace);

	/*
	 *  Map or unmap the remote execution icon according to if we have
	 *  remote execution enabled or not.
	 */
	if (attributes->remote_execution)
	{
	   XtMapWidget(glyph->remote);
	   (void) xvl_draw_label(glyph, xvl_get_machlabel(glyph, temp));
	   if (khoros_phantomd_pid == 0)
	   {
	      xvl_start_phantomd();
	   }
	}
	else
	{
	   XtUnmapWidget(glyph->remote);
	   (void) xvl_clear_label(glyph, xvl_get_machlabel(glyph, temp));
	}

	if (glyph->type == PROCEDURE)
	{
	   macro = glyph->val.macro;
	   if (macro->inherit_attributes == True)
	      xvl_set_distributed(macro);
	}
}

xvl_update_glyph_location(glyph)

Glyph	*glyph;
{
	NodeList    *input_list, *output_list;


	if (glyph->type == CONTROL || glyph->type == GLYPH ||
            glyph->type == COMMAND)
	{
	   /*
	    *  update the input lists input filenames to reflect the fact
	    *  that we have changed hosts.
	    */
	   input_list = glyph->input_list;
	   while (input_list != NULL)
	   {
	      xvl_update_filename(input_list->node);
	      input_list = input_list->next;
	   }

	   output_list = glyph->output_list;
	   while (output_list != NULL)
	   {
	      /*
	       *  xvl_copy() data from this machine to the remote 
	       */
	      input_list = output_list->node->links;
	      while (input_list != NULL)
	      {
		 xvl_update_filename(input_list->node);
	         input_list = input_list->next;
	      }
	      output_list = output_list->next;
	   }
	}
}



/************************************************************
*
* Routine Name:  xvl_update_negotiator
*
*      Purpose:  This routine is used to change the glyph's stream
*		 based execution transports.
*
*        Input:  glyph      - the glyph to be changed.
*
*
*   Written By: Mark Young
*
*************************************************************/


xvl_update_negotiator(glyph)

Glyph		*glyph;
{
	char	  temp[MaxLength];
	Workspace *workspace, *attributes, *macro;


	/*
	 *  Get the attributes for this workspace so that we can see if we
	 *  are to enable or un-enable remote execution.
	 */
	workspace = glyph->workspace;
	attributes = xvl_get_attributes(workspace);

	/*
	 *  Set or unset the glyph's stream based transports according
	 *  to whether we want to use stream based executions.
	 */
	if (attributes->stream_execution)
	{
	}
	else
	{
	}

	if (glyph->type == PROCEDURE)
	{
	   macro = glyph->val.macro;
	   if (macro->inherit_attributes == True)
	      xvl_set_negotiator(macro);
	}
}



/************************************************************
*
* Routine Name:  xvl_update_all_optional
*
*      Purpose:  This routine is used to change/update all the
*		 optional widgets.  This is done by racing thru
*		 the workspace's glyphs and in turn each of the
*		 glyph's input and output list calling
*		 xvl_update_optional() to actually change the
*		 widget to show whether the input or output is
*		 currently selected or not.
*
*        Input:  workspace - the workspace for which we wish to
*			     update the optional widgets.
*
*
*   Written By: Mark Young
*
*************************************************************/


xvl_update_all_optional(workspace)

Workspace *workspace;
{
	Glyph *glyph;
	GlyphList *glyphlist;

	Node *node;
	NodeList *nodelist;


	glyphlist = workspace->glyphs;
	while (glyphlist != NULL)
	{
	   glyph = glyphlist->glyph;

	   nodelist = glyph->input_list;
	   while (nodelist != NULL)
	   {
	      node = nodelist->node;
	      xvl_update_optional(node);
	      nodelist = nodelist->next;
	   }

	   nodelist = glyph->output_list;
	   while (nodelist != NULL)
	   {
	      node = nodelist->node;
	      xvl_update_optional(node);
	      nodelist = nodelist->next;
	   }
	   glyphlist = glyphlist->next;
	}
}



/************************************************************
*
* Routine Name:  xvl_update_all_dav
*
*      Purpose:  This routine is used to change/update all the
*		 dav widgets.  This is done by racing thru
*		 the workspace's glyphs and in turn each of the
*		 glyph's input and output list calling xvl_update_dav()
*		 to actually map or unmap the dav widget.
*
*        Input:  workspace - the workspace for which we wish to
*			     update the data available widgets.
*
*
*   Written By: Mark Young
*
*************************************************************/


xvl_update_all_dav(workspace)

Workspace *workspace;
{
	Glyph *glyph;
	GlyphList *glyphlist;

	Node *node;
	NodeList *nodelist;

	glyphlist = workspace->glyphs;
	while (glyphlist != NULL)
	{
	   glyph = glyphlist->glyph;

	   nodelist = glyph->input_list;
	   while (nodelist != NULL)
	   {
	      node = nodelist->node;
	      xvl_update_dav(node);
	      nodelist = nodelist->next;
	   }

	   nodelist = glyph->output_list;
	   while (nodelist != NULL)
	   {
	      node = nodelist->node;
	      xvl_update_dav(node);
	      nodelist = nodelist->next;
	   }
	   glyphlist = glyphlist->next;
	}
}



/************************************************************
*
* Routine Name:  xvl_update_all_modified
*
*      Purpose:  This routine is used to change/update all the
*		 modified widgets.  This is done by racing thru
*		 the workspace's glyphs and calling xvl_update_modified().
*
*        Input:  workspace - the workspace for which we wish to
*			     update the modified widgets.
*
*
*   Written By: Mark Young
*
*************************************************************/


xvl_update_all_modified(workspace)

Workspace *workspace;
{
	Glyph *glyph;
	GlyphList *glyphlist;

	glyphlist = workspace->glyphs;
	while (glyphlist != NULL)
	{
	   glyph = glyphlist->glyph;
	   xvl_update_modified(glyph);
	   glyphlist = glyphlist->next;
	}
}



/************************************************************
*
* Routine Name:  xvl_update_all_selected
*
*      Purpose:  This routine is used to change/update all the
*		 selected widgets.  This is done by racing thru
*		 the workspace's glyphs and calling xvl_update_selected().
*
*        Input:  workspace - the workspace for which we wish to
*			     update the selected widgets.
*
*
*   Written By: Mark Young
*
*************************************************************/


xvl_update_all_selected(workspace)

Workspace *workspace;
{
	Glyph *glyph;
	GlyphList *glyphlist;

	glyphlist = workspace->glyphs;
	while (glyphlist != NULL)
	{
	   glyph = glyphlist->glyph;
	   xvl_update_selected(glyph);
	   glyphlist = glyphlist->next;
	}
}



/************************************************************
*
* Routine Name:  xvl_update_all_mapping
*
*      Purpose:  This routine is used to change/update all the
*
*        Input:  workspace - the workspace for which we wish to
*			     update the mapping.
*
*
*   Written By: Mark Young
*
*************************************************************/


xvl_update_all_mapping(workspace)

Workspace *workspace;
{
	Glyph *glyph;
	GlyphList *glyphlist;

	glyphlist = workspace->glyphs;
	while (glyphlist != NULL)
	{
	   glyph = glyphlist->glyph;
	   xvl_update_mapping(glyph);
	   glyphlist = glyphlist->next;
	}
}



/************************************************************
*
* Routine Name:  xvl_update_all_mapped
*
*      Purpose:  This routine is used to change/update all the
*		 glyph form widgets.  This is done by racing thru
*		 the workspace's glyphs and calling xvl_update_mapped().
*		 The glyph form widget show whether a glyph's subform
*		 is currently mapped or not.
*
*        Input:  workspace - the workspace for which we wish to
*			     update the mapped widgets.
*
*
*   Written By: Mark Young
*
*************************************************************/


xvl_update_all_mapped(workspace)

Workspace *workspace;
{
	Glyph *glyph;
	GlyphList *glyphlist;

	glyphlist = workspace->glyphs;
	while (glyphlist != NULL)
	{
	   glyph = glyphlist->glyph;
	   xvl_update_mapped(glyph);
	   glyphlist = glyphlist->next;
	}
}



/************************************************************
*
* Routine Name:  xvl_update_all_clipboard
*
*      Purpose:  This routine is used to change/update all the
*		 glyph clipboard widgets on all the workspaces.
*		 This is done by racing thru the workspace's glyphs
*		 and calling xvl_update_clipboard() with all the
*		 workspaces and child workspaces.
*
*        Input:  workspace - the workspace for which we wish to
*			     update the clipboard widgets.
*
*
*   Written By: Mark Young
*
*************************************************************/


xvl_update_all_clipboard(workspace)

Workspace *workspace;
{
	Glyph *glyph;
	GlyphList *glyphlist;

	xvl_update_clipboard(workspace);
	glyphlist = workspace->glyphs;
	while (glyphlist != NULL)
	{
	   glyph = glyphlist->glyph;
	   if (glyph->type == PROCEDURE)
	      xvl_update_all_clipboard(glyph->val.macro);

	   glyphlist = glyphlist->next;
	}
}
