/*
 * Copyright (c) 1990, 1991 Stanford University
 *
 * Permission to use, copy, modify, and distribute this software and 
 * its documentation for any purpose is hereby granted without fee, provided
 * that (i) the above copyright notices and this permission notice appear in
 * all copies of the software and related documentation, and (ii) the name
 * Stanford may not be used in any advertising or publicity relating to
 * the software without the specific, prior written permission of
 * Stanford.
 * 
 * THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND, 
 * EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY 
 * WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.  
 *
 * IN NO EVENT SHALL STANFORD BE LIABLE FOR ANY SPECIAL, INCIDENTAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY KIND, OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER OR NOT
 * ADVISED OF THE POSSIBILITY OF DAMAGE, AND ON ANY THEORY OF LIABILITY,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 */

/* $Header: /Source/Media/collab/DTR/RCS/canvas.c,v 1.10 92/01/09 12:40:34 drapeau Exp Locker: derek $ */
/* $Log:	canvas.c,v $
 * Revision 1.10  92/01/09  12:40:34  drapeau
 * > Slight modifications to the code to make it ANSI-compliant.
 * 
 * Revision 1.0  92/01/06  17:41:10  drapeau
 * Made a number of cosmetic changes to make code easier to read and
 * to conform to programming specifications.
 * 
 * Revision 0.26  91/09/18  22:47:14  derek
 * The following things are done:
 * 1.	The Makefile is changed corresponding to the changes in collab/.
 * 2.	Copyright messages included.
 * 3.	Some minor bugs fixed.
 * 
 * Revision 0.25  91/08/27  18:05:58  derek
 * The SizeToFit bug is fixed.
 * 
 * Revision 0.24  91/08/27  17:22:25  derek
 * The zooming bug is fixed in this version.
 * 
 * Revision 0.23  91/08/21  17:46:53  derek
 * The big canvas "mis-placed" bug is fixed.
 * 
 * Revision 0.22  91/08/21  11:34:07  derek
 * The following changes are made:
 * 1.	Now the duration and size of the recorded sound will be displayed
 * 	during recording.
 * 2.	I have changed GetSelection() corresponding to the request of Tek joo
 * 3.	Info Panel is added to the application.
 * 4.	Fixed SizeToFitHandler() so that when no file or buffer is currently
 * 	loaded, it would not do anything (except giving a warning
 * 	notice_prompt).
 * 5.	Inplemented the `Close' Menu option in the document menu.
 * 6.	Fixed the bug in which after ClearAll and I press PreviewEdit,
 * 	the edit wont be played.
 * 7.	I have made the changes corresponding to the change in OpenPanel's
 * 	name.  (from OpenPanel to Browse).
 * 8.	Incorporated BrowseCheck to check command line arg.
 * 9.	Changed most EditingStatusMessages to NoticePrompts.
 * 10.	SoundFileSaveAsPopUp and EditListSaveAsPopUp are removed 
 * 	from the application.
 * 
 * Revision 0.21  91/08/08  21:44:27  derek
 * Fixed a number of bugs.
 * 
 * Revision 0.20  91/08/08  11:01:59  derek
 * This is a cleaner version.  I have removed lots of printf/fprintf 
 * statements from it, and have also cleaned up the code using xsaber.
 * This version should run substantially faster.
 * 
 * Revision 0.19  91/08/07  16:23:55  derek
 * The Edit list part of DTR is done.  OpenPanel is also incorporated.
 * 
 * Revision 0.18  91/08/06  12:41:05  derek
 * Edit list panel is done.  Still need to link it to the network code.
 * 
 * Revision 0.17  91/07/29  15:10:07  derek
 * The playing w/o stopping error is fixed.
 * 
 * Revision 0.16  91/07/26  13:18:12  derek
 * Some saving bugs fixed.
 * 
 * Revision 0.15  91/07/24  12:51:48  derek
 * Disk editing is done.  Now the application can record sound infinitely,
 * as long as there is disk space available.  Command line args are also
 * supported.
 * 
 * Revision 0.14  91/07/23  21:21:26  derek
 * This version is not ready for release.  Disk space editing is half-done:
 * the application can play an infinite sound and the canvases can handle
 * infinite sound files.  The app is pretty bug free too, I think.  The
 * weakness is that it cannot record sound infinitely.  
 * 
 * Revision 0.13  91/06/26  15:54:57  derek
 * I have reformatted the code to conform coding specs.
 * 
 * Revision 0.12  91/06/20  19:55:15  derek
 * The network part should be working.  Also fixed numerous minor parts
 * involving the canvas and the display.
 * 
 * Revision 0.10  1991/04/25  01:45:11  derek
 * This version is checked in on 4/24/91
 * */
static char rcsid[] = "$Header: /Source/Media/collab/DTR/RCS/canvas.c,v 1.10 92/01/09 12:40:34 drapeau Exp Locker: derek $";

#include "dtr.h"
#include "dtr_ui.h"

/*
 * Various Windows.
 */
extern  dtr_mainWindow_objects       *dtr_mainWindow;

/*
 * Scrollbars
 */
Scrollbar	WaveCanvasScrollbar;
Scrollbar       ScaleCanvasScrollbar;
int        	ScrollbarPixelsPerUnit;
int             ScrollbarViewStart;
int 		ScrollbarViewEnd;

/*
 * Global variables for Wave Canvas;
 */
Window     	xidWave;
Xv_Window  	paintWinWave;
Display    	*dpyWave;
GC         	gcWave;
int        	viewStart;
unsigned long *	WaveCanvasColormap;

/*
 *  Global variables for VU Meter Canvas
 */
int             oldVUMeterLevel;
Window		xidVUMeter;
Xv_window	paintWinVUMeter;
Display		*dpyVUMeter;
GC		gcVUMeter;
unsigned long * VUMeterCanvasColormap;
#define REDLEVELSTART 72

/*
 *  Global variables for Scale canvas.  The subscript "Scale" in 
 *  these variables refer to the Scale canvas, but not literally
 *  to some "scale".
 */
Window          xidScale;
Xv_Window       paintWinScale;
Display         *dpyScale;
GC              gcScale;
unsigned long * ScaleCanvasColormap;
int             ScaleAxisHeight;
int             ScaleTimeHeight;


Notify_value
  WaveCanvasScrollbarHandler(win, event, arg, type)
Xv_window win;
Event     *event;
Notify_arg arg;
Notify_event_type type;
{
  int    ie;
  int    newViewStart;
  
  ie = event_id(event);
  
  switch(ie) 
  {
   case SCROLLBAR_REQUEST:
    UpdateCanvasFrame();
    newViewStart = xv_get(WaveCanvasScrollbar, SCROLLBAR_VIEW_START);
    xv_set(ScaleCanvasScrollbar, SCROLLBAR_VIEW_START, newViewStart, NULL);
    break;
   default:
    break;
  }
  return notify_next_event_func(win, (Notify_event)event, arg, type);
}


void
  InitWaveCanvas()
{
  Xv_opaque       wcanvas = dtr_mainWindow->waveCanvas;
  
  EVENT("Init_Wave_Canvas");
  
  paintWinWave = canvas_paint_window(wcanvas);
  xidWave = (Window) xv_get(paintWinWave, XV_XID);
  dpyWave = (Display *) xv_get(wcanvas, XV_DISPLAY);
  gcWave = XCreateGC(dpyWave, xidWave, 0, 0);
  
  xv_set(paintWinWave, WIN_CONSUME_EVENTS, WIN_NO_EVENTS,	    /*  Specifies that only the left- and the right-..  */
	 LOC_WINENTER,						    /*  ...button eventw will be registered, in fact... */
	 LOC_WINEXIT,						    /*  ...useless because middle buttons events will.. */
	 WIN_ASCII_EVENTS,					    /*  ...appear anyways.                              */
	 LOC_DRAG, MS_LEFT, 
	 MS_MIDDLE, NULL, NULL);
  
  WaveCanvasColormap = GetColormap(wcanvas);			    /*  Get colormap.                                   */
  XSetForeground(dpyWave, gcWave, 
		 GetColorIndex(WaveCanvasColormap, "White"));
  
  oldLeftFloatMarker = -1.0;					    /*  Init left and right wave canvas markers.        */
  oldRightFloatMarker = -1.0;
  WaveCanvasRightMarkerSet = FALSE;
  
  WaveCanvasWidth = (int) xv_get(wcanvas, CANVAS_WIDTH, NULL);

  WaveCanvasHeight = (int) xv_get(wcanvas, CANVAS_HEIGHT, NULL);
  WaveWindowWidth = (int) xv_get(wcanvas, XV_WIDTH, NULL);
  WaveWindowHeight = (int) xv_get(wcanvas, XV_HEIGHT, NULL);
  WaveCanvasMidHeight = WaveCanvasHeight/2;
  WaveScopeWidth = WaveCanvasMidHeight - 4;
  WaveCanvasInitReady = TRUE;
  leftButtonHoldPoint = NOTREADY;
  
  DefaultWaveCanvasDataSamplingInterval = DEVICE_SAMPLE_RATE /	    /*  Define Default DataSampling Interval --- we...  */
    (WaveWindowWidth /						    /*  ...want the length of the wave window to be...  */
     10.0);							    /*  ...equal to 10 seconds.                         */
  
  InitScaleCanvas();						    /*  Process to initialize the scale canvas.         */

  FrameStartingSecond = 0.0;					    /*  Initialize FrameStartingSecond.                 */
}


/*
 *  Initialize wave canvas scrollbar (and implicitly initialize
 *  the scale canvas scrollbar also)
 */
void
  InitWaveCanvasScrollbar()
{
  Notify_value  WaveCanvasScrollbarHandler();
  Canvas    canvas = dtr_mainWindow->waveCanvas;
  
  EVENT("Init_Wave_Canvas_Scrollbar");
  
  xv_set(canvas, CANVAS_WIDTH, MAX_ZOOM*xv_get(canvas, XV_WIDTH,    /*  Set Canvas parameters to allow for scrolling.   */
					       NULL),
	 CANVAS_HEIGHT, xv_get(canvas, XV_HEIGHT, NULL),
	 CANVAS_AUTO_EXPAND, FALSE,
	 CANVAS_AUTO_SHRINK, FALSE,
	 NULL);  
  
  Zoom = 1.0;							    /*  Create scrollbar for Wave Canvas.               */
  ScrollbarPixelsPerUnit = 10;
  WaveCanvasDataSamplingInterval = DefaultWaveCanvasDataSamplingInterval /
    Zoom;
  WaveCanvasScrollbar = xv_create(dtr_mainWindow->waveCanvas, SCROLLBAR,
				  SCROLLBAR_DIRECTION, SCROLLBAR_HORIZONTAL, 
				  SCROLLBAR_PIXELS_PER_UNIT, ScrollbarPixelsPerUnit,
				  SCROLLBAR_VIEW_LENGTH, 
				  irint(xv_get(canvas, XV_WIDTH, NULL)/
					((double)ScrollbarPixelsPerUnit)),
				  SCROLLBAR_OBJECT_LENGTH, 
				  irint(xv_get(canvas, XV_WIDTH, NULL)/
					((double)ScrollbarPixelsPerUnit)),
				  SCROLLBAR_PAGE_LENGTH,
				  irint(xv_get(canvas, XV_WIDTH, NULL)/
					((double)ScrollbarPixelsPerUnit)),
				  NULL);
  
  notify_interpose_event_func(xv_get(WaveCanvasScrollbar, 
				     SCROLLBAR_NOTIFY_CLIENT), 
			      WaveCanvasScrollbarHandler,
			      NOTIFY_SAFE); 
}


/*
 *  This function is called by InitWaveCanvas() (not by main()) to
 *  initialize the Scale Canvas.
 */
void
  InitScaleCanvas()
{
  Xv_opaque            scanvas = dtr_mainWindow->scaleCanvas;
  int                  newCanvasWidth;

  EVENT("Init_Scale_Canvas");
  
  ScaleAxisHeight = 17;
  ScaleTimeHeight = 33;
  paintWinScale = canvas_paint_window(scanvas);
  xidScale = (Window)xv_get(paintWinScale, XV_XID);
  dpyScale = (Display *) xv_get(scanvas, XV_DISPLAY);
  
  gcScale = XCreateGC(dpyScale, xidScale, 0, 0);
  
  ScaleCanvasColormap = GetColormap(scanvas);			    /*  Get colormap.                                   */
  ScaleCanvasPenColor("black");
}


/*
 *  Initialize the Scale canvas Scrollbar which is not
 *  going to be shown.
 */
void
  InitScaleCanvasScrollbar()
{
  Notify_value  ScaleCanvasScrollbarHandler();
  Canvas    scanvas = dtr_mainWindow->scaleCanvas;
  
  EVENT("Init_Scale_Canvas_Scrollbar");
  
  xv_set(scanvas, CANVAS_WIDTH, MAX_ZOOM*xv_get(scanvas, XV_WIDTH,  /*  Set Canvas parameters to allow for scrolling.   */
					       NULL),
	 CANVAS_HEIGHT, xv_get(scanvas, XV_HEIGHT, NULL),
	 CANVAS_AUTO_EXPAND, FALSE,
	 CANVAS_AUTO_SHRINK, FALSE,
	 NULL);  
  
  ScaleCanvasScrollbar =xv_create(dtr_mainWindow->scaleCanvas,   /*  Create Scrollbar for ScaleCanvas.               */
				  SCROLLBAR,
				  SCROLLBAR_DIRECTION, 
				  SCROLLBAR_HORIZONTAL, 
				  NULL);
  
  xv_set(ScaleCanvasScrollbar, XV_SHOW, FALSE, NULL);		    /*  Do not show scrollbar.                          */
}


/*
 *  Update both the wave canvas and the scale canvas, as well as the 
 *  scrollbar.
 */
void
  UpdateCanvasesAndScrollbarParameters()
{
  int  newCanvasWidth;
  
  if (irint(Buffer.hdr.data_size / WaveCanvasDataSamplingInterval) 
      < WaveWindowWidth)  
  {
    newCanvasWidth = WaveWindowWidth;				    /*  newCanvasWidth is the virtual canvas width: ie..*/
  }								    /*  ...it is usually part of the real canvas which..*/
  else								    /*  ...is MAX_ZOOM * WaveWindowWidth in length.     */
  {
    newCanvasWidth = WaveWindowWidth * Zoom;
  }
  
  WaveCanvasWidth = newCanvasWidth;

  xv_set(WaveCanvasScrollbar,					    /*  Update Scrollbars.                              */
	 SCROLLBAR_PIXELS_PER_UNIT, ScrollbarPixelsPerUnit, 
	 SCROLLBAR_VIEW_LENGTH, irint(((double)WaveWindowWidth)/
				      ScrollbarPixelsPerUnit),
	 SCROLLBAR_OBJECT_LENGTH, irint((double) newCanvasWidth / 
					ScrollbarPixelsPerUnit),
	 SCROLLBAR_PAGE_LENGTH, irint(((double)WaveWindowWidth)/
				      ScrollbarPixelsPerUnit),
	 NULL);
  xv_set(ScaleCanvasScrollbar, 
	 SCROLLBAR_PIXELS_PER_UNIT, ScrollbarPixelsPerUnit, 
	 SCROLLBAR_VIEW_LENGTH, irint(((double)WaveWindowWidth)/
				      ScrollbarPixelsPerUnit),
	 SCROLLBAR_OBJECT_LENGTH, irint((double) newCanvasWidth / 
					ScrollbarPixelsPerUnit),
	 SCROLLBAR_PAGE_LENGTH, irint(((double)WaveWindowWidth)/
				      ScrollbarPixelsPerUnit),
	 NULL);
}    


void
  RepaintWaveCanvas()
{
  EVENT("Repaint_Wave_Canvas");
  
  WaveCanvasRepaintHandler((Canvas)dtr_mainWindow->waveCanvas, 
			   paintWinWave, (Display *)dpyWave,
			   xidWave, NULL);
  
  ScaleCanvasRepaintHandler((Canvas)dtr_mainWindow->scaleCanvas,
			    paintWinScale, (Display *)dpyScale,
			    xidScale, NULL);
}


void
  InitVUMeterCanvas()
{
  Xv_opaque		vucanvas = dtr_mainWindow->vuMeterCanvas;
  XGCValues		gcv;
  int			dash_offset = 0;
  static char		long_dashed[2] = {10,2};
  
  EVENT("Init_VU_Meter_Canvas");
  
  paintWinVUMeter = canvas_paint_window(vucanvas);
  xidVUMeter = (Window)xv_get(paintWinVUMeter, XV_XID);
  dpyVUMeter = (Display *) xv_get(vucanvas, XV_DISPLAY);
  
  gcv.line_style = LineOnOffDash;
  gcv.line_width = 20;
  gcVUMeter = XCreateGC(dpyVUMeter, xidVUMeter, GCLineStyle, &gcv);
  XSetDashes(dpyVUMeter, gcVUMeter, dash_offset, long_dashed, 2);
  XSetLineAttributes(dpyVUMeter, gcVUMeter, gcv.line_width, 
		     gcv.line_style, CapButt, JoinRound);
  
  VUMeterCanvasColormap = GetColormap(vucanvas);		    /*  Get colormap.                                   */
  
  oldVUMeterLevel = 0;						    /*  Initialize VU Meter level.                      */
  
  VUMeterCanvasPenColor("Forest Green");			    /*  Draw some dark red and green colors onto the... */
  XDrawLine(dpyVUMeter, xidVUMeter, gcVUMeter,			    /*  ...meter initially.                             */
	    7+0, 15,
	    7+REDLEVELSTART, 15);
  VUMeterCanvasPenColor("brown");
  XDrawLine(dpyVUMeter, xidVUMeter, gcVUMeter, 
	    7+REDLEVELSTART, 15,
	    7+128, 15);
  
}


/*
 *  DisplayVUMeter(): 
 *  Brief description of the algorithm:
 *  If the new sound level is higher than before, paint "difference" area with
 *  pen_color (green).  Else, if new sound level is lower than before,
 *  paint "difference" area with black.
 */
void
  DisplayVUMeter(startpos)
int  startpos;
{
  register int		i;
  int			level;
  int			end;
  unsigned char		data;
  static unsigned char  soundData[SCOPE_WIDTH];
  
  EVENT("Display_VU_Meter");
  
  if (startpos < 0)  
  {
    i = -startpos;
    startpos = 0;
  }
  else i = startpos;
  
  end = SCOPE_WIDTH + startpos;
  
  if (FileReady)
  {
    soundfd = open(Buffer.filename, O_RDONLY);
    lseek(soundfd, Buffer.hdr_size + i, L_SET);
    read(soundfd, &soundData, SCOPE_WIDTH);
    close(soundfd);
    
    for (level = 0; i < end && i < Buffer.hdr.data_size; i++)	    
    {
      data = soundData[i - startpos];
      if (abs((int)audio_u2c(data)) > level)			    /*  Taking the MAX.                                 */
	level = abs((int)audio_u2c(data));
    }
  }
  else 
  {
    for (level = 0; i < end && i < Buffer.hdr.data_size; i++)	    
    {
      if (abs((int)audio_u2c(Buffer.data[i])) > level)		    /*  Taking the MAX.                                 */
	level = abs((int)audio_u2c(Buffer.data[i]));
    }
  }
  
  level = ((level/12)+1)*12;
  
  if (oldVUMeterLevel < level)
  {
    if (oldVUMeterLevel > REDLEVELSTART) 
    {
      VUMeterCanvasPenColor("Red");   
      XDrawLine(dpyVUMeter, xidVUMeter, gcVUMeter, 
		7+oldVUMeterLevel, 15,
		7+level, 15);
    }
    else if (level > REDLEVELSTART) 
    {
      VUMeterCanvasPenColor("Green");
      XDrawLine(dpyVUMeter, xidVUMeter, gcVUMeter, 
		7+oldVUMeterLevel, 15,
		7+REDLEVELSTART, 15);
      VUMeterCanvasPenColor("Red");   
      XDrawLine(dpyVUMeter, xidVUMeter, gcVUMeter, 
		7+REDLEVELSTART, 15,
		7+level, 15);
    }
    else 
    {
      VUMeterCanvasPenColor("Green");
      XDrawLine(dpyVUMeter, xidVUMeter, gcVUMeter, 
		7+oldVUMeterLevel, 15,
		7+level, 15);
    }
  } 
  else if (oldVUMeterLevel > level) 
  {
    if (level < REDLEVELSTART && REDLEVELSTART <= oldVUMeterLevel) 
    {
      VUMeterCanvasPenColor("Forest Green");
      XDrawLine(dpyVUMeter, xidVUMeter, gcVUMeter, 
		7+level, 15,
		7+REDLEVELSTART, 15);
      VUMeterCanvasPenColor("brown");
      XDrawLine(dpyVUMeter, xidVUMeter, gcVUMeter, 
		7+REDLEVELSTART, 15,
		7+oldVUMeterLevel, 15);
    }
    else if (level >= REDLEVELSTART) 
    {
      VUMeterCanvasPenColor("brown");
      XDrawLine(dpyVUMeter, xidVUMeter, gcVUMeter, 
		7+level, 15,
		7+oldVUMeterLevel, 15);
    } 
    else 
    {
      VUMeterCanvasPenColor("Forest Green");
      XDrawLine(dpyVUMeter, xidVUMeter, gcVUMeter, 
		7+level, 15,
		7+oldVUMeterLevel, 15);
    }
  }
  
  oldVUMeterLevel = level;
}								    /* end function DisplayVUMeter */


/*
 *  Clear Wave Canvas, reset the scrollbars.
 */
void
  ResetWaveCanvas()
{
  ClearWaveCanvas();						    /*  Clear Wave Canvas and the Scale Canvas.         */
  ClearScaleCanvas();
  xv_set(WaveCanvasScrollbar, SCROLLBAR_VIEW_START, 0, NULL);	    /*  Reset the canvas scrollbar.                     */
  ScrollbarViewStart = ScrollbarViewEnd = 0;
  if (xv_get(dtr_mainWindow->zoomSlider, PANEL_VALUE) != 10)	    /*  Reset the zoom slider.                          */
    xv_set(dtr_mainWindow->zoomSlider, PANEL_VALUE, 10, NULL);
  Zoom = 1.0;
  oldLeftFloatMarker = oldRightFloatMarker = -1.0;		    /*  Reset Various Selection parameters.             */
  WaveCanvasRightMarkerSet = FALSE;
  leftButtonHoldPoint = NOTREADY;
}								    /* end function ResetWaveCanvas */


void
  UpdateVUMeter()
{
  int		display_position;
  unsigned	s;
  int		i;
  short		try;
  short		last;
  unsigned char data;
  
  EVENT("UpdateVUMeter");
  
  if (Buffer.hdr.data_size == 0)
    return;
  
  if (ActiveFlag & RECORD)					    /*  Recording.                                      */
  {	
    display_position =						    /*  Display the next piece of data.  We only want   */
      (Buffer.hdr.data_size - SCOPE_WIDTH) & ~SCOPE_MASK;	    /*  to display full panels of data.                 */
    if (display_position < 0) 
    {
      display_position = 0;
    }
  } 
  else if (ActiveFlag & PLAY)					    /*  Playing.                                        */
  {
    s = 0;							    /*  Now display the scope waveform.  We want to...  */
    (void) audio_get_play_samples(Audio_fd, &s);		    /*  ...display starting at a multiple of 256 (the.. */
    if (Buffer.play.start != Buffer.play.end)			    /*  ...scope window size).  If we are not at the... */
    {								    /*  ...next such boundary, return w/o doing...      */
      display_position = (Buffer.play.start +			    /*  ...anything.                                    */
			  (s % (Buffer.play.end - 
				Buffer.play.start)))
	& ~SCOPE_MASK;
    }
    else 
    {
      display_position = Buffer.play.start & ~SCOPE_MASK;
    }
  } 
  else 
  {
    display_position = Buffer.display.last;			    /*  Inactive.                                       */
    DisplayVUMeter(display_position);				    /*  Repair damage.                                  */
    return;
  }
  
  if (FileReady)
  {
    data = GetFileData(display_position);
    last = audio_u2s(data);
  }
  else
  {
    last = audio_u2s(Buffer.data[display_position]);		    /*  Look for a rising edge at a zero-crossing.      */
  }
  
  for (i = display_position - 1;
       (i >= 0) && (i > (display_position - (SCOPE_WIDTH / 2)));
       i--)
  {
    if (FileReady)
    {
      data = GetFileData(i);
      try = audio_u2s(data);
    }
    else
    {
      try = audio_u2s(Buffer.data[i]);
    }
    
    if ((try < last) && (last >= 0) && (try <= 0)) 
    {
      if (last < ((try < 0) ? -try : try))
	i++;
      display_position = i;
      break;
    }
    last = try;
  }
  
  if (display_position != Buffer.display.last) 
  {
    DisplayVUMeter(display_position);
    Buffer.display.last = display_position;
  }
}


void
  ClearWaveCanvas()
{
  extern   GC gcWave;
  double   scaleMarkerInterval;
  double   x;

  Zoom = 1.0;
  
  XClearWindow(dpyWave, xidWave);
  scaleMarkerInterval = WaveWindowWidth / 10.0 * Zoom;		    /*  Draw the horizontal time axis.                  */
  WaveCanvasPenColor("Red");
  
  XDrawLine(dpyWave, xidWave, gcWave, 0, WaveCanvasMidHeight-15, 
	    WaveCanvasWidth, WaveCanvasMidHeight-15);
  for(x = 0.0 ; x < (double) WaveCanvasWidth ; x += scaleMarkerInterval)
    XDrawLine(dpyWave, xidWave, gcWave, 
	      irint(x), WaveCanvasMidHeight-15+5,
	      irint(x), WaveCanvasMidHeight-15-5);
}


void
  ClearScaleCanvas()
{
  extern   GC gcScale;
  
  XClearWindow(dpyScale, xidScale);
  ScaleCanvasPenColor("black");
  XDrawLine(dpyScale, xidScale, gcScale, 0, ScaleAxisHeight, 
	    WaveCanvasWidth-1, ScaleAxisHeight);
  Zoom = 1.0;
  ScaleCanvasRepaintHandler((Canvas)dtr_mainWindow->scaleCanvas,
			    paintWinScale, dpyScale, xidScale, NULL);
}


/*
 *  Check if the user has clicked the mouse at a place outside
 *  the scope of the soundbuffer.
 */
BOOL
  WaveCanvasOutOfSoundScope(winX)
int winX;
{
  int bufferX;
  
  if (WaveCanvasDataSamplingInterval == 0.0)
    return (TRUE);
  
  bufferX = irint(winX * WaveCanvasDataSamplingInterval);
  
  if (bufferX > Buffer.hdr.data_size || bufferX < 0)
    return(TRUE);
  return(FALSE);
}


/*
 *  The winX here is the overall x-coord under the current Zoom value.
 */
void
  WaveCanvasMarkerHandler(winX, oldmarker)
int winX;
double *oldmarker;
{
  int     oldLeftWinX, oldRightWinX, LeftWinX, RightWinX;
  
  EVENT("Marker_Handler");
  
  if (WaveCanvasOutOfSoundScope(winX))
    return;
  
  oldLeftWinX = irint(oldLeftFloatMarker /			    /*  Save old markers.                               */
		      WaveCanvasDataSamplingInterval);
  oldRightWinX = irint(oldRightFloatMarker /
		       WaveCanvasDataSamplingInterval);
  
  if (oldmarker == &oldLeftFloatMarker)				    /*  Left button or Right button pressed.            */
  {
    oldLeftFloatMarker = oldRightFloatMarker = winX * 
      WaveCanvasDataSamplingInterval;
  } 
  else 
  {
    oldRightFloatMarker = winX * WaveCanvasDataSamplingInterval;
  }

  LeftWinX = irint( oldLeftFloatMarker /			    /*  Draw markers.                                   */
		   WaveCanvasDataSamplingInterval );
  RightWinX = irint( oldRightFloatMarker / 
		    WaveCanvasDataSamplingInterval );		    
  
  if (!WaveCanvasRightMarkerSet && oldmarker == 
      &oldRightFloatMarker)
    ReverseWaveCanvasArea(winX, winX);

/**********
	ReverseWaveCanvasArea(winX, winX);
**********/
  
  if (LeftWinX > oldRightWinX || RightWinX < oldLeftWinX) 
  {
    if (WaveCanvasRightMarkerSet) 
    {
      ReverseWaveCanvasArea(oldLeftWinX, oldRightWinX);
      if (oldmarker == &oldLeftFloatMarker)
	ReverseWaveCanvasArea(winX, winX);
/**********
	ReverseWaveCanvasArea(winX, winX);
**********/
    }
  }
  else if (WaveCanvasRightMarkerSet) 
  {
    if (LeftWinX > oldLeftWinX)
    {
      ReverseWaveCanvasArea(oldLeftWinX, LeftWinX-1);
    } 
    else if (LeftWinX < oldLeftWinX) 
    {
      ReverseWaveCanvasArea(LeftWinX, oldLeftWinX-1);
    }
    
    if (RightWinX > oldRightWinX) 
    {
      ReverseWaveCanvasArea(oldRightWinX + 1, RightWinX);
    } 
    else if (RightWinX < oldRightWinX) 
    {
      ReverseWaveCanvasArea(RightWinX + 1, oldRightWinX);
    }
  }
  
  if (oldmarker == &oldRightFloatMarker)
    WaveCanvasRightMarkerSet = TRUE;
  
  Buffer.play.start = irint(oldLeftFloatMarker);		    /*  Update start and end points.                    */
  Buffer.play.end = irint(oldRightFloatMarker);
  
  UpdateMessageDisplay();					    /*  Update message display (sound selection size,.. */
								    /*  ...duration etc.) */
}


/*
 *  Takes cares of case when user tries to drag leftwards (instead of 
 *  rightwards).
 */
void
  WaveCanvasBackwardMarkerHandler(winX, oldmarker)
int  winX;
double *oldmarker;
{
  int     oldLeftWinX, oldRightWinX, LeftWinX, RightWinX;
  
  EVENT("Wave_Canvas_Marker_Handler");
  
  if (WaveCanvasOutOfSoundScope(winX))
    return;
  
  oldLeftWinX = irint( oldLeftFloatMarker /			    /*  Save old markers.                               */
		      WaveCanvasDataSamplingInterval );
  oldRightWinX = irint( oldRightFloatMarker / 
		       WaveCanvasDataSamplingInterval );
  
  if (oldmarker == &oldLeftFloatMarker) 
  {								    /*  Left button or right button pressed.            */
    oldLeftFloatMarker = winX * WaveCanvasDataSamplingInterval;
  } 
  
  LeftWinX = irint( oldLeftFloatMarker /			    /*  Draw markers.                                   */
		   WaveCanvasDataSamplingInterval );
  RightWinX = irint( oldRightFloatMarker / 
		    WaveCanvasDataSamplingInterval );
  
  if (!WaveCanvasRightMarkerSet && oldmarker == &oldRightFloatMarker)
    ReverseWaveCanvasArea(winX, winX);
  
  if (LeftWinX > oldRightWinX || RightWinX < oldLeftWinX) 
  {
    if (WaveCanvasRightMarkerSet)
      ReverseWaveCanvasArea(oldLeftWinX, oldRightWinX);
    if (oldmarker == &oldLeftFloatMarker)
      ReverseWaveCanvasArea(winX, winX);
  }
  else if (WaveCanvasRightMarkerSet) 
  {
    if (LeftWinX > oldLeftWinX && oldLeftWinX != 0) 
    {
      ReverseWaveCanvasArea(oldLeftWinX, LeftWinX-1);
    } 
    else if (LeftWinX < oldLeftWinX) 
    {
      ReverseWaveCanvasArea(LeftWinX, oldLeftWinX-1);
    }
    
    if (RightWinX > oldRightWinX) 
    {
      ReverseWaveCanvasArea(oldRightWinX + 1, RightWinX);
    } 
    else if (RightWinX < oldRightWinX) 
    {
      ReverseWaveCanvasArea(RightWinX + 1, oldRightWinX);
    }
  }
  
  WaveCanvasRightMarkerSet = TRUE;
  
  Buffer.play.start = irint(oldLeftFloatMarker);		    /*  Update start and end points.                    */
  Buffer.play.end = irint(oldRightFloatMarker);
  
  UpdateMessageDisplay();					    /*  Update message display (sound selection size... */
								    /*  ...and duration).                               */
}


int
  GetColorIndex(colormap, colorString)
unsigned long *colormap;
char *colorString;
{
  return(colormap[gcm_color_index(colorString)]);
}


/*
 *  An inverse mapping to retrieve the true colormap defined
 *  by guide.
 */
unsigned long *
  GetColormap(win)
Xv_opaque win;
{
  unsigned long *colors;
  Cms  cms;
  Xv_Screen  screen;
  
  screen = (Xv_Screen) XV_SCREEN_FROM_WINDOW(win);
  cms = (Cms) xv_find(screen, CMS, CMS_NAME, gcm_colormap_name(),
		      XV_AUTO_CREATE, FALSE, 0);
  colors = (unsigned long *) xv_get(cms, CMS_INDEX_TABLE);
  return (colors);
}


/*
 *  Set Foreground color of wave canvas.
 */
void
  WaveCanvasPenColor(colorStr)
char *colorStr;
{
  XSetForeground(dpyWave, gcWave, 
		 GetColorIndex(WaveCanvasColormap, colorStr));
}


/*
 *  Set Foreground color of wave canvas.
 */
void
  ScaleCanvasPenColor(colorStr)
char *colorStr;
{
  XSetForeground(dpyScale, gcScale,
		 GetColorIndex(ScaleCanvasColormap, colorStr));
}


/*
 *  Set Foreground color of VU Meter.
 */
void
  VUMeterCanvasPenColor(colorStr)
char *colorStr;
{
  XSetForeground(dpyVUMeter, gcVUMeter,
		 GetColorIndex(VUMeterCanvasColormap, colorStr));
}


/*
 *  Reset VU Meter parameters.
 */
void
  ResetVUMeter()
{
  oldVUMeterLevel = 0;
}


void
  ClearVUMeterCanvas()
{
  XClearWindow(dpyVUMeter, xidVUMeter);
  
  VUMeterCanvasPenColor("Forest Green");			    /*  Draw some dark red and green colors onto the... */
  XDrawLine(dpyVUMeter, xidVUMeter, gcVUMeter,			    /*  ...meter initially.                             */
	    7+0, 15,
	    7+REDLEVELSTART, 15);
  VUMeterCanvasPenColor("brown");
  XDrawLine(dpyVUMeter, xidVUMeter, gcVUMeter, 
	    7+REDLEVELSTART, 15,
	    7+132, 15);
}


/*
 *  This function is called when user selects "Reset Markers"
 *  from the function menu.
 */
void
  ResetWaveCanvasMarkers()
{
  EVENT("Reset_Wave_Canvas_Markers");
  
  ReverseWaveCanvasArea(irint(oldLeftFloatMarker/
			      WaveCanvasDataSamplingInterval),
			irint(oldRightFloatMarker/
			      WaveCanvasDataSamplingInterval));
  WaveCanvasRightMarkerSet = FALSE;
  leftButtonHoldPoint = NOTREADY;
  UpdateMessageDisplay();
  UpdateSelection();
  Buffer.play.start = 0;
  Buffer.play.end = Buffer.hdr.data_size - 1;
}


/*
 *  When user chooses "Clear All" from the menu.
 */
void
  ClearAllSound()
{
  ResetWaveCanvas();
  ResetGlobalWaveCanvas();
  ClearMessageDisplay();
  InitBuffer();
  SoundBufferReady = FALSE;
  SameSoundFile = FALSE;
  FileReady = FALSE;
  BufferSaved = TRUE;
  currentSelectionReadyToPlay = FALSE;
  sprintf(currentSoundFile,"Untitled");
  UpdateHeader(FALSE);
  UpdateSelection();
  strcpy(Buffer.filename, "");
}								    /* end function ClearAllSound */


/*
 *  For Highlighting or Unhighlighting a portion of the canvas.
 */
void
  ReverseWaveCanvasArea(left, right)
int  left, right;
{
  int   startReverseX;
  int   endReverseX;
  
  startReverseX = irint(left - (FrameStartingSecond * DEVICE_SAMPLE_RATE /
				WaveCanvasDataSamplingInterval));
  if (startReverseX < 0)
    startReverseX = 0;

  endReverseX = irint(right - (FrameStartingSecond *  DEVICE_SAMPLE_RATE /
			       WaveCanvasDataSamplingInterval));
  
  if (endReverseX > WaveCanvasWidth)
    endReverseX = WaveCanvasWidth;
  
  XSetFunction(dpyWave, gcWave, GXinvert);
  XFillRectangle(dpyWave, xidWave, gcWave, 
		 startReverseX, 0,
		 endReverseX - startReverseX + 1, WaveWindowHeight - 1);
  XSetFunction(dpyWave, gcWave, GXcopy);
}


void
  CenterWaveCanvasHighlightedSegment()
{
  int              size;
  extern Scrollbar WaveCanvasScrollbar;
  extern int       ScrollbarPixelsPerUnit;
  int              newScrollbarViewStart;
  int              midPt;
  int              maxScrollbarViewStart;
  
  if (WaveCanvasRightMarkerSet) 
  {
    size = irint(oldRightFloatMarker - oldLeftFloatMarker);

    maxScrollbarViewStart = irint(floor(10.0 * DEVICE_SAMPLE_RATE /
					WaveCanvasDataSamplingInterval / 
					ScrollbarPixelsPerUnit));

    midPt = irint((((oldLeftFloatMarker + oldRightFloatMarker) / 2.0)
		   - (FrameStartingSecond * DEVICE_SAMPLE_RATE)) /
		  WaveCanvasDataSamplingInterval);

    if (midPt > WaveWindowWidth/2)				    /*  If midpt of highlighted portion is greater...   */
    {								    /*  ...than the width of wave window, the slider... */
      if (irint(size/WaveCanvasDataSamplingInterval) <		    /*  ...needs to be adjusted.                        */
	  WaveWindowWidth) 
      {
	newScrollbarViewStart = (midPt - (WaveWindowWidth / 2)) / 
	  ScrollbarPixelsPerUnit;
      }
      else 
      {
	newScrollbarViewStart = irint((oldLeftFloatMarker - (FrameStartingSecond * DEVICE_SAMPLE_RATE)) /
				      WaveCanvasDataSamplingInterval /
				      ScrollbarPixelsPerUnit);
      }

      if (newScrollbarViewStart < 0)
	newScrollbarViewStart = 0;
      if (newScrollbarViewStart > maxScrollbarViewStart)
	newScrollbarViewStart = maxScrollbarViewStart;
      
      xv_set(WaveCanvasScrollbar, SCROLLBAR_VIEW_START,
	     newScrollbarViewStart, NULL);
    } 
    else 
    {
      xv_set(WaveCanvasScrollbar, SCROLLBAR_VIEW_START, 0, NULL);
    }
  }
}


/*
 *  Each Frame contains 10 seconds of data.  If the
 *  segment to be displayed exceeds 10 seconds, this 
 *  function cuts out the excess portions.
 */
void
  HighlightWaveCanvas()
{
  int    startReverseX;
  int    endReverseX;

  startReverseX = irint((oldLeftFloatMarker -
			 (FrameStartingSecond * DEVICE_SAMPLE_RATE)) /
			WaveCanvasDataSamplingInterval);
  if (startReverseX < 0)
    startReverseX = 0;
  
  endReverseX = irint((oldRightFloatMarker -
		       (FrameStartingSecond * DEVICE_SAMPLE_RATE)) /
		      WaveCanvasDataSamplingInterval);
  if (endReverseX > WaveCanvasWidth)
    endReverseX = WaveCanvasWidth;
  
  ReverseWaveCanvasArea(startReverseX, endReverseX);
}


void
  CalculateLevelAndDisplayVUMeter(level)
int  level;
{
  level = ((level/12)+1)*12;
  
  if (oldVUMeterLevel < level) {
    if (oldVUMeterLevel > REDLEVELSTART) 
    {
      VUMeterCanvasPenColor("Red");   
      XDrawLine(dpyVUMeter, xidVUMeter, gcVUMeter, 
		7+oldVUMeterLevel, 15,
		7+level, 15);
    }
    else if (level > REDLEVELSTART) 
    {
      VUMeterCanvasPenColor("Green");
      XDrawLine(dpyVUMeter, xidVUMeter, gcVUMeter, 
		7+oldVUMeterLevel, 15,
		7+REDLEVELSTART, 15);
      VUMeterCanvasPenColor("Red");   
      XDrawLine(dpyVUMeter, xidVUMeter, gcVUMeter, 
		7+REDLEVELSTART, 15,
		7+level, 15);
    }
    else 
    {
      VUMeterCanvasPenColor("Green");
      XDrawLine(dpyVUMeter, xidVUMeter, gcVUMeter, 
		7+oldVUMeterLevel, 15,
		7+level, 15);
    }
  } 
  else if (oldVUMeterLevel > level) 
  {
    if (level < REDLEVELSTART && REDLEVELSTART <= oldVUMeterLevel) 
    {
      VUMeterCanvasPenColor("Forest Green");
      XDrawLine(dpyVUMeter, xidVUMeter, gcVUMeter, 
		7+level, 15,
		7+REDLEVELSTART, 15);
      VUMeterCanvasPenColor("brown");
      XDrawLine(dpyVUMeter, xidVUMeter, gcVUMeter, 
		7+REDLEVELSTART, 15,
		7+oldVUMeterLevel, 15);
    }
    else if (level >= REDLEVELSTART) 
    {
      VUMeterCanvasPenColor("brown");
      XDrawLine(dpyVUMeter, xidVUMeter, gcVUMeter, 
		7+level, 15,
		7+oldVUMeterLevel, 15);
    } 
    else 
    {
      VUMeterCanvasPenColor("Forest Green");
      XDrawLine(dpyVUMeter, xidVUMeter, gcVUMeter, 
		7+level, 15,
		7+oldVUMeterLevel, 15);
    }
  }
  
  oldVUMeterLevel = level;					    
}								    /* end function CalculateLevelAndDisplayVUMeter */
