/*
 * Copyright (c) 1990, 1991 Stanford University
 *
 * Permission to use, copy, modify, and distribute this software and 
 * its documentation for any purpose is hereby granted without fee, provided
 * that (i) the above copyright notices and this permission notice appear in
 * all copies of the software and related documentation, and (ii) the name
 * Stanford may not be used in any advertising or publicity relating to
 * the software without the specific, prior written permission of
 * Stanford.
 * 
 * THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND, 
 * EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY 
 * WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.  
 *
 * IN NO EVENT SHALL STANFORD BE LIABLE FOR ANY SPECIAL, INCIDENTAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY KIND, OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER OR NOT
 * ADVISED OF THE POSSIBILITY OF DAMAGE, AND ON ANY THEORY OF LIABILITY,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 */

/*$Header: /Source/Media/collab/DTR/RCS/editlist.c,v 1.14 92/05/29 12:42:13 drapeau Exp $*/
/* $Log:	editlist.c,v $
 * Revision 1.14  92/05/29  12:42:13  drapeau
 * Modified code to track new name of the MAEstro "Selection" structure;
 * it is now named "MAESelection".
 * 
 * Revision 1.13  92/03/02  16:08:12  drapeau
 * Modified SetCurrentSelection() -- previous changes to accommodate
 * partial selections introduced rounding errors.  This change corrects
 * the rounding errors, which would cut off as much as a second from
 * the end of an edit's audio.
 * 
 * Revision 1.12  92/03/02  12:30:01  drapeau
 * Modified SetCurrentSelection() to handle requests from the network
 * for partial selections.  This function now takes into account
 * both the duration and offset fields of a Selection* structure
 * passed into the function as argument; if the duration differs
 * from the default selection, the requested duration will be set
 * (if possible).  If the offset is non-zero, this function will
 * attempt to accommodate the new offset value.
 * 
 * Revision 1.11  92/01/07  14:26:59  drapeau
 * * Modified SetEditList() to clean up the updating of current
 *   edit list name.
 * * Modified FormatEntry() to reflect new interface.  The current
 *   sound file for the current edit is no longer displayed directly on
 *   the scrolled list since it is shown in several other places.
 * * Modified SetCurrentSelection() to clean up the semantics and make
 *   more robust.  The function now allows for previewing of selections
 *   before they are added to the edit list, and also allows for direct
 *   loading of selection simply by dragging the mouse over the desired
 *   area on the Wave Canvas.  Before this modification, a button labeled
 *   "Load Selection" needed to be pressed before the selection was loaded.
 * * Simplified NewEditList() so that it simply clears out the current
 *   edit list and begins anew.  The author is no longer required to pick
 *   a filename for the new, empty edit list.
 * Also, Made a number of cosmetic changes to make code easier to read and
 * to conform to programming specifications.
 * 
 * Revision 1.10  91/09/18  22:47:23  derek
 * The following things are done:
 * 1.	The Makefile is changed corresponding to the changes in collab/.
 * 2.	Copyright messages included.
 * 3.	Some minor bugs fixed.
 * 
 * Revision 1.9  91/08/22  15:47:06  derek
 * The format of the DTR edit-list has been changed.  The header is 
 * changed from "#DTR Edit File#" to "#DTR Edit Document#".  Also,
 * EditNumber is added to every edit within the edit-list.
 * 
 * Revision 1.8  91/08/21  11:34:16  derek
 * The following changes are made:
 * 1.	Now the duration and size of the recorded sound will be displayed
 * 	during recording.
 * 2.	I have changed GetSelection() corresponding to the request of Tek joo
 * 3.	Info Panel is added to the application.
 * 4.	Fixed SizeToFitHandler() so that when no file or buffer is currently
 * 	loaded, it would not do anything (except giving a warning
 * 	notice_prompt).
 * 5.	Inplemented the `Close' Menu option in the document menu.
 * 6.	Fixed the bug in which after ClearAll and I press PreviewEdit,
 * 	the edit wont be played.
 * 7.	I have made the changes corresponding to the change in OpenPanel's
 * 	name.  (from OpenPanel to Browse).
 * 8.	Incorporated BrowseCheck to check command line arg.
 * 9.	Changed most EditingStatusMessages to NoticePrompts.
 * 10.	SoundFileSaveAsPopUp and EditListSaveAsPopUp are removed 
 * 	from the application.
 * 
 * Revision 1.7  91/08/16  18:10:37  derek
 * 
 * The following things are done:
 * 1.	I have fixed an openpanel bug in my code in which I 
 * 	made the app return the wrong values to OpenHandler and
 * 	the SaveHandler.
 * 2.	The flashing color of the play button has been changed from
 * 	Red to Green.
 * 3.	Fixed a quantization bug: Buffer.play.end, when converted
 * 	from endingTimeInSec, should not exceed Buffer_hdr_data_size - 1.
 * 
 * Revision 1.6  91/08/14  18:16:51  derek
 * The following things are done:
 * 1.	I changed the name of the "save as" and "save" menu options
 * 	to be more specific.  ie. to "save sound file" and "save edit list".
 * 2.	I removed the preview panel popup and replaced it with a preview
 * 	edit button.
 * 3.	I changed the message displayed by UpdateSpaceOnTmp() to consist
 * 	only of the time remaining and the amount of space left.  There is
 * 	no more percentage indicating the capacity of /usr/tmp.
 * 
 * Revision 1.5  91/08/08  21:44:35  derek
 * Fixed a number of bugs.
 * 
 * Revision 1.4  91/08/08  11:27:54  derek
 * I have fixed a minor bug.
 * 
 * Revision 1.3  91/08/08  11:02:10  derek
 * This is a cleaner version.  I have removed lots of printf/fprintf 
 * statements from it, and have also cleaned up the code using xsaber.
 * This version should run substantially faster.
 * 
 * Revision 1.2  91/08/07  16:24:13  derek
 * The Edit list part of DTR is done.  OpenPanel is also incorporated.
 * 
 * Revision 1.1  91/08/06  12:41:13  derek
 * Initial revision
 *  */
static char rcsid[] = "$Header: /Source/Media/collab/DTR/RCS/editlist.c,v 1.14 92/05/29 12:42:13 drapeau Exp $";

#include "dtr.h"              
#include "dtr_ui.h"

extern	dtr_editListPanelPopUp_objects    	*dtr_editListPanelPopUp;

Xv_font *listFont;

void
  InitEditList()
{
  int	i;
  
  EditList.soundFile     = (char   **) malloc(DEFAULT_EDITLIST_SIZE * sizeof(char *));
  EditList.label         = (char   **) malloc(DEFAULT_EDITLIST_SIZE * sizeof(char *));
  for(i = 0; i < DEFAULT_EDITLIST_SIZE ; i++)
  {
    EditList.soundFile[i] = (char *) malloc(80);
    EditList.label[i]     = (char *) malloc(40);
  }
  
  EditList.startingMin   = (int     *) malloc(DEFAULT_EDITLIST_SIZE * sizeof(int));
  EditList.startingSec   = (double  *) malloc(DEFAULT_EDITLIST_SIZE * sizeof(double));
  EditList.endingMin     = (int     *) malloc(DEFAULT_EDITLIST_SIZE * sizeof(int));
  EditList.endingSec     = (double  *) malloc(DEFAULT_EDITLIST_SIZE * sizeof(double));
  EditList.volume        = (int     *) malloc(DEFAULT_EDITLIST_SIZE * sizeof(int));
  EditList.capacity      = DEFAULT_EDITLIST_SIZE;
  EditList.numItems      = 0;
  
  currentSelection = NO_CURRENT_EDIT;
  unsavedChangesExist = FALSE;
  currentSelectionReadyToPlay = FALSE;
  SetScrollListFont();
  sprintf(absoluteEditListName, "Untitled");
}								    /* end function InitEditList */
  

void
  SetScrollListFont()
{
  listFont = (Xv_font *) xv_find(dtr_editListPanelPopUp->editListPanelPopUp, FONT,
				 FONT_FAMILY, FONT_FAMILY_LUCIDA_FIXEDWIDTH,
				 FONT_STYLE, FONT_STYLE_NORMAL,
				 FONT_SIZE, 12,
				 NULL);
}


/*
 *  This function is called when the current number of edits is close to the
 *  default max allowed number.
 */
void
  ReallocateMemoryForEditList()
{
  int	i;

  EditList.soundFile     = (char   **) realloc(EditList.soundFile,
					       (EditList.capacity + DEFAULT_EDITLIST_SIZE) * sizeof(char *));
  EditList.label         = (char   **) realloc(EditList.label,
					       (EditList.capacity + DEFAULT_EDITLIST_SIZE) * sizeof(char *));
  for(i = EditList.capacity ;					    
      i < EditList.capacity + DEFAULT_EDITLIST_SIZE ;
      i++)						
  {
    EditList.soundFile[i] = (char *) malloc(80);
    EditList.label[i]     = (char *) malloc(40);
  }

  EditList.startingMin   = (int     *) realloc(EditList.startingMin,
					       (EditList.capacity + DEFAULT_EDITLIST_SIZE) * sizeof(int));
  EditList.startingSec   = (double  *) realloc(EditList.startingSec,
					       (EditList.capacity + DEFAULT_EDITLIST_SIZE) * sizeof(double));
  EditList.endingMin     = (int     *) realloc(EditList.endingMin,
					       (EditList.capacity + DEFAULT_EDITLIST_SIZE) * sizeof(int));
  EditList.endingSec     = (double  *) realloc(EditList.endingSec,
					       (EditList.capacity + DEFAULT_EDITLIST_SIZE) * sizeof(double));
  EditList.volume        = (int     *) realloc(EditList.volume,
					       (EditList.capacity + DEFAULT_EDITLIST_SIZE) * sizeof(int));
  EditList.capacity += DEFAULT_EDITLIST_SIZE;
}  
 

int
  ReadEditListFromFile(fileName, caller)
char   *fileName;
char   caller;
{
  double	duration;
  FILE		*fp;
  int		size;
  int		counter;
  int           garbage;
  
  if (unsavedChangesExist == TRUE)
  {
    int    reply;
      
    sprintf(msg, "Unsaved changes in %s edit list exist, load new edit list anyway?", prog);
    reply = notice_prompt(dtr_editListPanelPopUp->editListUpperControlPanel, NULL,
			  NOTICE_MESSAGE_STRINGS,
			  msg,
			  NULL,
			  NOTICE_BUTTON, "Yes", 'Y',
			  NOTICE_BUTTON, "No", 'C',
			  NULL);
    if (reply == 'C')
      return FILE_NOT_OK;
  }
  
  fp = fopen (fileName, "r");					    /*  Attempt to open the specified file.             */
  if (fp == (FILE*)NULL)					    /*  Did the open succeed?                           */
  {								    /*  No, inform author and return w/o doing anything */
    sprintf(msg, "'%s'", fileName);
    notice_prompt(dtr_editListPanelPopUp->editListUpperControlPanel,
		  NULL,
		  NOTICE_MESSAGE_STRINGS,
		  "Unable to open editlist",
		  msg,
		  NULL,
		  NOTICE_BUTTON, "OK", 'O',
		  NULL);
    return FILE_NOT_OK;
  }
  
  fscanf (fp, "#%s Edit Document#\n", msg);			    /*  Is this file a DTR Editlist file?               */
  if (strcmp(msg, "DTR"))					    /*  No, this is not a DTR Editlist file.            */
  {								    /*  Inform the author of the mistake and return.    */
    if (caller == USER)
    {
      sprintf(msg, "'%s'", fileName);
      notice_prompt(dtr_editListPanelPopUp->editListUpperControlPanel,
		    NULL,
		    NOTICE_MESSAGE_STRINGS,
		    msg,
		    "is not a valid DTR edit list",
		    NULL,
		    NOTICE_BUTTON, "OK", 'O',
		    NULL);
    }
    return FILE_NOT_OK;						    /*  Exit regardless if called from network or by... */
								    /*  ...the author.                                  */
  }
  DeselectList();
  ResetCurrentSelection();
  EmptyList();
  fscanf (fp, "#Number of Edits:\t%d", &size);			    /*  Read the size of the edit list to be read in.   */
  EditList.numItems = size;
  if (size > EditList.capacity - 2)
  {
    for(counter = 1 ; counter < ((size+2)/DEFAULT_EDITLIST_SIZE)+1; /*  If not enough memory in edit list data...       */
	counter++)						    /*  ...structure, reallocate data repeatedly.       */
      ReallocateMemoryForEditList();
  }
  
  for (counter = 0; counter < size; counter ++)			    /* Read through the edit list file, loading it... */
  {								    /* ...into the edit list, entry by entry. */
    fscanf(fp, "\n#EditNumber:\t%d\n#SoundFile:\t%s\n#StartingMin:\t%d\n#StartingSec:\t%lf\n#EndingMin:\t%d\n#EndingSec:\t%lf\n#Volume:\t%d\n#Label:\t", 
	   &garbage,
	   EditList.soundFile[counter],
	   &EditList.startingMin[counter],
	   &EditList.startingSec[counter],
	   &EditList.endingMin[counter],
	   &EditList.endingSec[counter],
	   &EditList.volume[counter]);
    fgets(EditList.label[counter], 40, fp);
    duration = (EditList.endingMin[counter] - EditList.startingMin[counter]) * 60.0 
      + EditList.endingSec[counter] - EditList.startingSec[counter];
    
    xv_set (dtr_editListPanelPopUp->editListScrollList,
	    PANEL_LIST_INSERT, counter,
	    PANEL_LIST_STRING, counter,
	    FormatEntry(counter+1,
			EditList.label[counter],
			EditList.soundFile[counter],
			duration),
	    PANEL_LIST_FONT, counter, listFont,
	    NULL);
  }
  fclose(fp);
  UpdateNumEditsMessage();
  SetEditList(fileName);
  unsavedChangesExist = FALSE;
  return FILE_OK;
}								    /* end function ReadEditListFromFile */



void
  DeselectList()
{
  if (currentSelection != NO_CURRENT_EDIT)
  {
    xv_set (dtr_editListPanelPopUp->editListScrollList,
	    PANEL_LIST_SELECT, currentSelection, FALSE,
	    NULL);
    currentSelection = NO_CURRENT_EDIT;
  }
  xv_set(dtr_editListPanelPopUp->currentEditMessage,
	 PANEL_LABEL_STRING, "New Edit",
	 NULL);
  xv_set(dtr_editListPanelPopUp->deleteButton,
	 PANEL_INACTIVE, TRUE, NULL);				    /* De-activate the Delete and Modify buttons */
  xv_set(dtr_editListPanelPopUp->modifyButton,
	 PANEL_INACTIVE, TRUE, NULL);
}


int
  WriteEditListToFile(fileName)
char	*fileName;
{
  int	size;
  int	counter;
  FILE	*fp;

  size = EditList.numItems;
  
  DeselectList();
  fp = fopen (fileName, "w");					    /*  Attempt to open the edit list passed in for...  */
								    /*  ...writing.                                     */
  if (fp == (FILE*)NULL)					    /*  Did the open fail?                              */
  {								    /*  Yes, unable to open the file for writing. ...   */
    sprintf(msg, "'%s'", fileName);
    notice_prompt(dtr_editListPanelPopUp->editListUpperControlPanel,
		  NULL,
		  NOTICE_MESSAGE_STRINGS,
		  msg,
		  "is a bad/non-writable file name.",
		  NULL,
		  NOTICE_BUTTON, "OK", 'O',
		  NULL);
    return FILE_NOT_OK;
  }
  fprintf (fp, "#DTR Edit Document#\n");			    /*  Write the file header determining the type...   */
								    /*  ... of this file.                               */
  size = EditList.numItems;
  fprintf (fp, "#Number of Edits:\t%d\n\n", size);		    /*  Write number of edits made for this document.   */
  for (counter = 0; counter < size; counter ++)			    /*  Loop through each item in the edit list, ...    */
  {								    /*  ...writing contents of each item to the file.   */
    fprintf(fp, "#EditNumber:\t%d\n#SoundFile:\t%s\n#StartingMin:\t%d\n#StartingSec:\t%lf\n#EndingMin:\t%d\n#EndingSec:\t%lf\n#Volume:\t%d\n#Label:\t%s\n\n", 
	    counter + 1,
	    EditList.soundFile[counter],
	    EditList.startingMin[counter],
	    EditList.startingSec[counter],
	    EditList.endingMin[counter],
	    EditList.endingSec[counter],
	    EditList.volume[counter],
	    EditList.label[counter]);
  }
  fclose (fp);							    /* Close the file just written */
  unsavedChangesExist = FALSE;
  SetEditList(fileName);
  return FILE_OK;
}								    /* end function WriteEditListToFile */

  
void
  ResetCurrentSelection()
{
  xv_set(dtr_editListPanelPopUp->soundFileTextField,
	 PANEL_VALUE, "", NULL);
  xv_set(dtr_editListPanelPopUp->labelTextField,
	 PANEL_VALUE, "", NULL);
  xv_set(dtr_editListPanelPopUp->startingTimeMinuteTextField,
	 PANEL_VALUE, "0", NULL);
  xv_set(dtr_editListPanelPopUp->startingTimeSecondTextField,
	 PANEL_VALUE, "0.0", NULL);
  xv_set(dtr_editListPanelPopUp->endingTimeMinuteTextField,
	 PANEL_VALUE, "0", NULL);
  xv_set(dtr_editListPanelPopUp->endingTimeSecondTextField,
	 PANEL_VALUE, "0.0", NULL);
  xv_set(dtr_editListPanelPopUp->currentEditMessage,
	 PANEL_VALUE, "New Edit", NULL);
}

  
void
  EmptyList()
{
  int	counter;

  for(counter = EditList.numItems-1 ; counter > -1; counter--)
  {
    xv_set(dtr_editListPanelPopUp->editListScrollList,
	   PANEL_LIST_DELETE, counter,
	   NULL);
  }
  EditList.numItems = 0;
}


void
  UpdateNumEditsMessage()
{
  sprintf(msg, "%d", EditList.numItems);
  xv_set(dtr_editListPanelPopUp->totalNumberOfEditsMessage,
	 PANEL_LABEL_STRING, msg, NULL);
}


	 
char  *
  FormatEntry(entryNum, label, fileName, duration)
int	entryNum;
char	*label;
char	*fileName;
double	duration;
{
  int		mins;
  double	secs;
  static char	formattedEntry[100];

  mins = irint(floor(duration/60.0));
  secs = duration - mins * 60.0;
  sprintf(msg, "%d:%.1lf", mins, secs);
  sprintf(formattedEntry, "%4d.   %-35.35s  %s", entryNum, label, msg);
  return(formattedEntry);
}


void SetEditList(char* editListName)
{
  if (strcmp(editListName, "NoneSpecified") != 0)
  {
    sprintf(absoluteEditListName, "%s", editListName);
    UpdateEditListHeader(FALSE);
    realpath(editListName, absoluteEditListName);
  }
  else
  {
    strcpy(absoluteEditListName, "NoneSpecified");
  }
  return;
}								    /* end function SetEditList */


void SetCurrentSelection(int entry, MAESelection* selection)
{
  double					startingTimeInSec;
  double					endingTimeInSec;
  double					tempSeconds;
  double					duration;
  int						tempMinutes;
  extern dtr_editListPanelPopUp_objects*	dtr_editListPanelPopUp;
  extern dtr_mainWindow_objects*		dtr_mainWindow;
  
  if (entry != NO_CURRENT_EDIT)					    /*  If there is a current Edit, load selection...   */
  {								    /*  ...parameters from edit list.                   */
    sprintf(msg, "%d", entry + 1);				    /*  Display information on the Edit list panel.     */
    xv_set(dtr_editListPanelPopUp->currentEditMessage,
	   PANEL_LABEL_STRING, msg, NULL);
    
    xv_set(dtr_editListPanelPopUp->soundFileTextField,		    /*  Set SoundFile text field.                       */
	   PANEL_VALUE, EditList.soundFile[entry]);
    xv_set(dtr_editListPanelPopUp->labelTextField,
	   PANEL_VALUE, EditList.label[entry]);
    
    sprintf(msg, "%d", EditList.startingMin[entry]);		    /*  Set Starting time (minute) text field.          */
    xv_set(dtr_editListPanelPopUp->startingTimeMinuteTextField,
	   PANEL_VALUE, msg, NULL);
    
    sprintf(msg, "%.2lf", EditList.startingSec[entry]);		    /*  Set starting time (second) text field.          */
    xv_set(dtr_editListPanelPopUp->startingTimeSecondTextField,
	   PANEL_VALUE, msg, NULL);
    
    sprintf(msg, "%d", EditList.endingMin[entry]);		    /*  Set ending time (minute) text field.            */
    xv_set(dtr_editListPanelPopUp->endingTimeMinuteTextField,
	   PANEL_VALUE, msg, NULL);
    
    sprintf(msg, "%.2lf", EditList.endingSec[entry]);		    /*  Set ending time (second) text field.            */
    xv_set(dtr_editListPanelPopUp->endingTimeSecondTextField,
	   PANEL_VALUE, msg, NULL);
    
    xv_set(dtr_editListPanelPopUp->editListVolumeSlider,	    /*  Set volume slider.                              */
	   PANEL_VALUE, EditList.volume[entry], NULL);
    xv_set(dtr_mainWindow->playGain,
	   PANEL_VALUE, EditList.volume[entry], NULL);
    
    SetPlayGain(EditList.volume[entry]);			    /*  Set the play volume of the workstation.         */
    startingTimeInSec = EditList.startingMin[entry] * 60.0	    /*  now setting the buffer stuff.                   */
      + EditList.startingSec[entry];
    endingTimeInSec = EditList.endingMin[entry] * 60.0
      + EditList.endingSec[entry];
    sprintf(currentSoundFile, "%s", EditList.soundFile[entry]);
  }								    /* end if (entry != NO_CURRENT_EDIT)                */
  else								    /* No current edit, get current selection values... */
  {								    /* ...from text fields instead of from edit list.   */
    tempMinutes = atoi(xv_get(dtr_editListPanelPopUp->startingTimeMinuteTextField,
			      PANEL_VALUE));
    tempSeconds = atof(xv_get(dtr_editListPanelPopUp->startingTimeSecondTextField,
			      PANEL_VALUE));
    startingTimeInSec = (tempMinutes * 60.0 + tempSeconds);
    
    tempMinutes = atoi(xv_get(dtr_editListPanelPopUp->endingTimeMinuteTextField,
			      PANEL_VALUE));
    tempSeconds = atof(xv_get(dtr_editListPanelPopUp->endingTimeSecondTextField,
			      PANEL_VALUE));
    endingTimeInSec = (tempMinutes * 60.0 + tempSeconds);
  }
  SoundBufferReady = ReadSoundFile(TRUE);			    /* Used to read into file, not buffer */
  UpdateHeader(FALSE);
  
/*
  FileReady = ReadSoundFile(FALSE);
  SoundBufferReady = FALSE;
*/
  if (selection != (MAESelection*)NULL)				    /* Was this called from SetSelection()? */
  {								    /* Yes, take into account possible partial selection */
    duration = endingTimeInSec - startingTimeInSec;
    startingTimeInSec += (selection->offset / 1000.0);		    /* Account for non-zero offset */
    if (startingTimeInSec < 0)
      startingTimeInSec = 0;
  }
  Buffer.play.start = irint(startingTimeInSec
			    * DEVICE_SAMPLE_RATE);
  if (selection != (MAESelection*)NULL)				    /* Was this called from SetSelection()? */
  {								    /* Yes, take into account possible partial selection */
    endingTimeInSec += (selection->duration / 1000.0) - duration;   /* Account for duration other than default value */
    if (endingTimeInSec < 0)
      endingTimeInSec = 0;
  }
  Buffer.play.end   = irint(endingTimeInSec
			    * DEVICE_SAMPLE_RATE); 
  if (Buffer.play.end >= Buffer.hdr.data_size)			    /*  This makes sure that the round-off error...     */
  {								    /*  ...caused by the conversion from time to byte.. */
    Buffer.play.end = Buffer.hdr.data_size - 1;			    /*  ...wont cause any trouble.                      */
  }
  oldLeftFloatMarker = (double) Buffer.play.start;
  oldRightFloatMarker = (double) Buffer.play.end;
  leftButtonHoldPoint = 0;
  WaveCanvasRightMarkerSet = TRUE;
  
  oldLeftFloatGbWaveMarker = (double) Buffer.play.start;
  oldRightFloatGbWaveMarker = (double) Buffer.play.end;
  GbWaveLeftButtonHoldPoint = 0;
  GlobalWaveCanvasRightMarkerSet = TRUE;

  SameSoundFile = TRUE;

  if((SoundBufferReady == TRUE) || (FileReady == TRUE)) 
  {
    RepaintWaveCanvas();
    RepaintGlobalWaveCanvas();
    UpdateMessageDisplay();
    currentSelectionReadyToPlay = TRUE;
  }
  else
  {
    ClearAllSound();
  }
}								    /* end function SetCurrentSelection */


void
  AddSelection()
{
  int		i;
  double	duration;
  char		proposedPath[80];
  char          *soundFileTextField;
  
  i = EditList.numItems;
  
  if (EditList.numItems % DEFAULT_EDITLIST_SIZE == DEFAULT_EDITLIST_SIZE - 2)
    ReallocateMemoryForEditList();
  
  DeselectList();
  soundFileTextField = (char *) xv_get(dtr_editListPanelPopUp->soundFileTextField,
				       PANEL_VALUE);
  
  if (!NullFileName(soundFileTextField) || soundFileTextField == NULL)
  {
    realpath(soundFileTextField, proposedPath);
    strcpy(EditList.soundFile[i], proposedPath);
  
    if (strcmp(xv_get(dtr_editListPanelPopUp->soundFileTextField,   /*  If the current string in the soundfiletextfield */
		      PANEL_VALUE),				    /*  ...does not represent the realpath, we set...   */
	       proposedPath) != 0)				    /*  ...the text field value to be the realpath.     */
    {
      xv_set(dtr_editListPanelPopUp->soundFileTextField,
	     PANEL_VALUE, proposedPath, NULL);
    }
  }
  else
  {
    strcpy(EditList.soundFile[i], "");
  }
  
  strcpy(EditList.label[i], xv_get(dtr_editListPanelPopUp->labelTextField,
				   PANEL_VALUE));
  EditList.startingMin[i] = atoi(xv_get(dtr_editListPanelPopUp->startingTimeMinuteTextField,
					PANEL_VALUE));
  EditList.startingSec[i] = atof(xv_get(dtr_editListPanelPopUp->startingTimeSecondTextField,
					PANEL_VALUE));
  EditList.endingMin[i] = atoi(xv_get(dtr_editListPanelPopUp->endingTimeMinuteTextField,
				      PANEL_VALUE));
  EditList.endingSec[i] = atof(xv_get(dtr_editListPanelPopUp->endingTimeSecondTextField,
				      PANEL_VALUE));
  EditList.volume[i] =
    (int) xv_get(dtr_editListPanelPopUp->editListVolumeSlider,
		 PANEL_VALUE);
  EditList.numItems++;
  currentSelection = i;
  duration = (EditList.endingMin[i] - EditList.startingMin[i]) * 60.0 
    + EditList.endingSec[i] - EditList.startingSec[i];
  xv_set(dtr_editListPanelPopUp->editListScrollList,
	 PANEL_LIST_INSERT, i,
	 PANEL_LIST_STRING, i,
	 FormatEntry(i+1, EditList.label[i], EditList.soundFile[i], duration),
	 PANEL_LIST_FONT, i, listFont,
	 NULL);
  sprintf(msg, "%d", i+1);
  xv_set(dtr_editListPanelPopUp->currentEditMessage,
	 PANEL_LABEL_STRING, msg, NULL);
  UpdateNumEditsMessage();
  unsavedChangesExist = TRUE;
  UpdateEditListHeader(TRUE);
  xv_set(dtr_editListPanelPopUp->deleteButton,			    /*  Re-activate the buttons again.                  */
	 PANEL_INACTIVE, FALSE, NULL);
  xv_set(dtr_editListPanelPopUp->modifyButton,
	 PANEL_INACTIVE, FALSE, NULL);
  xv_set(dtr_editListPanelPopUp->editListScrollList,
	 PANEL_LIST_SELECT, i, TRUE, NULL);
}								    /* end function AddSelection */


void
  ModifySelection()
{
  double	duration;
  char		proposedPath[80];
  char          *soundFileTextField;
  
  if (currentSelection == NO_CURRENT_EDIT)
  {
    notice_prompt (dtr_editListPanelPopUp->editListUpperControlPanel, NULL,
		   NOTICE_MESSAGE_STRINGS,
		   "You need to select an edit before you can modify it.",
		   NULL,
		   NOTICE_BUTTON,	"OK", 100,
		   NULL);
    return;							
  }    
  soundFileTextField = (char *) xv_get(dtr_editListPanelPopUp->soundFileTextField,
				       PANEL_VALUE);
  
  if (!NullFileName(soundFileTextField) || soundFileTextField == NULL)
  {
    realpath(soundFileTextField, proposedPath);
    strcpy(EditList.soundFile[currentSelection], proposedPath);
  
    if (strcmp(xv_get(dtr_editListPanelPopUp->soundFileTextField,   /*  If the current string in the soundfiletextfield */
		      PANEL_VALUE),				    /*  ...does not represent the realpath, we set...   */
	       proposedPath) != 0)				    /*  ...the text field value to be the realpath.     */
    {
      xv_set(dtr_editListPanelPopUp->soundFileTextField,
	     PANEL_VALUE, proposedPath, NULL);
    }
  }
  else
  {
    strcpy(EditList.soundFile[currentSelection], "");
  }
  strcpy(EditList.label[currentSelection],
	 xv_get(dtr_editListPanelPopUp->labelTextField,
		PANEL_VALUE));
  EditList.startingMin[currentSelection] =
    atoi(xv_get(dtr_editListPanelPopUp->startingTimeMinuteTextField,
		PANEL_VALUE));
  EditList.startingSec[currentSelection] =
    atof(xv_get(dtr_editListPanelPopUp->startingTimeSecondTextField,
		PANEL_VALUE));
  EditList.endingMin[currentSelection] =
    atoi(xv_get(dtr_editListPanelPopUp->endingTimeMinuteTextField,
		PANEL_VALUE));
  EditList.endingSec[currentSelection] =
    atof(xv_get(dtr_editListPanelPopUp->endingTimeSecondTextField,
		PANEL_VALUE));
  EditList.volume[currentSelection] =
    xv_get(dtr_editListPanelPopUp->editListVolumeSlider,
	   PANEL_VALUE);
  duration = (EditList.endingMin[currentSelection] -
	      EditList.startingMin[currentSelection]) * 60.0 
		+ EditList.endingSec[currentSelection]
		  - EditList.startingSec[currentSelection];
  xv_set(dtr_editListPanelPopUp->editListScrollList,
	 PANEL_LIST_STRING, currentSelection,
	 FormatEntry(currentSelection+1, EditList.label[currentSelection],
		     EditList.soundFile[currentSelection],
		     duration),
	 PANEL_LIST_FONT, currentSelection, listFont,
	 NULL);
  xv_set(dtr_editListPanelPopUp->deleteButton,			    /*  Re-activate the buttons again.                  */
	 PANEL_INACTIVE, FALSE, NULL);
  xv_set(dtr_editListPanelPopUp->modifyButton,
	 PANEL_INACTIVE, FALSE, NULL);
  UpdateNumEditsMessage();
  unsavedChangesExist = TRUE;
  UpdateEditListHeader(TRUE);
}								    /* end function ModifySelection */


void
  DeleteSelection()
{ 
  int    savedCurrentSelection;
  
  if (currentSelection == NO_CURRENT_EDIT)
  {
    notice_prompt (dtr_editListPanelPopUp->editListUpperControlPanel, NULL,
		   NOTICE_MESSAGE_STRINGS,
		   "You need to select an edit before you can delete it.",
		   NULL,
		   NOTICE_BUTTON,	"OK", 100,
		   NULL);
    return;							
  }    
  savedCurrentSelection = currentSelection;
  
  DeselectList();
/******************
  if (savedCurrentSelection != 0)
  {
    xv_set(dtr_editListPanelPopUp->editListScrollList,
	   PANEL_LIST_SELECT, savedCurrentSelection - 1, TRUE,
	   NULL);
  }
********************/  
  xv_set(dtr_editListPanelPopUp->editListScrollList,
	 PANEL_LIST_DELETE, savedCurrentSelection,
	 NULL);
  DeleteItemFromEditList(savedCurrentSelection);
/**************
  if (savedCurrentSelection != 0)
  {
    xv_set(dtr_editListPanelPopUp->editListScrollList,
	   PANEL_LIST_SELECT, savedCurrentSelection - 1, FALSE,
	   NULL);
  }
***************/
  UpdateNumEditsMessage();
  unsavedChangesExist = TRUE;
  UpdateEditListHeader(TRUE);
  xv_set(dtr_editListPanelPopUp->deleteButton,
	 PANEL_INACTIVE, TRUE, NULL);
  xv_set(dtr_editListPanelPopUp->modifyButton,
	 PANEL_INACTIVE, TRUE, NULL);
}								    /* end function DeleteSelection */

  
void
  DeleteItemFromEditList(itemNumber)
int	itemNumber;
{
  int		i;

  if (itemNumber+1 == EditList.numItems)			    /*  If the last item is going to be deleted...      */
  {								    /*  ... then we dont have to shuffle any items up.. */
    EditList.numItems--;					    /*  ... in the list.                                */
    RedrawEditList(itemNumber);					    /*  Update number of items in the edit list.        */
    return;
  }
  for (i = itemNumber; i < EditList.numItems; i++)
  {
    strcpy(EditList.soundFile[i], EditList.soundFile[i+1]);
    strcpy(EditList.label[i], EditList.label[i+1]);
    EditList.startingMin[i] = EditList.startingMin[i+1];
    EditList.startingSec[i] = EditList.startingSec[i+1];
    EditList.endingMin[i] = EditList.endingMin[i+1];
    EditList.endingSec[i] = EditList.endingSec[i+1];
    EditList.volume[i] = EditList.volume[i+1];
  }
  EditList.numItems--;
  RedrawEditList(itemNumber);
  return;
}								    /* end function DeleteItemFromEditList */


void
  RedrawEditList(entryNum)
int   entryNum;
{
  int		i;
  double	duration;

  for(i = entryNum; i < EditList.numItems ; i++)
  {
    duration = (EditList.endingMin[i] -
		EditList.startingMin[i]) * 60.0 
		  + EditList.endingSec[i]
		    - EditList.startingSec[i];
    
    xv_set(dtr_editListPanelPopUp->editListScrollList,
	   PANEL_LIST_STRING, i, FormatEntry(i+1,
					     EditList.label[i],
					     EditList.soundFile[i],
					     duration),
	   PANEL_LIST_FONT, i, listFont,
	   NULL);
  }

  return;
}								    /* end function RedrawEditList */

	   
void
  DeleteAllSelections()
{
  DeselectList();
  ResetCurrentSelection();
  EmptyList();
  UpdateNumEditsMessage();
  unsavedChangesExist = TRUE;
  xv_set(dtr_editListPanelPopUp->deleteButton,			    /*  De-activate the Delete and Modify buttons.      */
	 PANEL_INACTIVE, TRUE, NULL);
  xv_set(dtr_editListPanelPopUp->modifyButton,
	 PANEL_INACTIVE, TRUE, NULL);
}								    /* end function DeleteAllSelections */


void
  NewEditList()
{
  DeleteAllSelections();
  sprintf(absoluteEditListName, "Untitled");
  UpdateEditListHeader(FALSE);
}

  
void
  DisableEditListPanel()
{
  extern  dtr_editListPanelPopUp_objects *dtr_editListPanelPopUp;

  xv_set(dtr_editListPanelPopUp->addButton,			    
	 PANEL_INACTIVE, TRUE, NULL);
  xv_set(dtr_editListPanelPopUp->modifyButton,
	 PANEL_INACTIVE, TRUE, NULL);
  xv_set(dtr_editListPanelPopUp->deleteButton,			    
	 PANEL_INACTIVE, TRUE, NULL);
  xv_set(dtr_editListPanelPopUp->deleteAllButton,
	 PANEL_INACTIVE, TRUE, NULL);
  
  xv_set(dtr_editListPanelPopUp->editListScrollList,
	 PANEL_INACTIVE, TRUE, NULL);
}


  
void
  EnableEditListPanel()
{
  extern  dtr_editListPanelPopUp_objects *dtr_editListPanelPopUp;

  xv_set(dtr_editListPanelPopUp->addButton,
	 PANEL_INACTIVE, FALSE, NULL);
  xv_set(dtr_editListPanelPopUp->modifyButton,
	 PANEL_INACTIVE, FALSE, NULL);
  xv_set(dtr_editListPanelPopUp->deleteButton,			    
	 PANEL_INACTIVE, FALSE, NULL);
  xv_set(dtr_editListPanelPopUp->deleteAllButton,
	 PANEL_INACTIVE, FALSE, NULL);
  
  xv_set(dtr_editListPanelPopUp->editListScrollList,
	 PANEL_INACTIVE, FALSE, NULL);
}
