/******************************************************************************\
*                                                                              *
* The software for `MaMA' was developed by:    H. K. Blokhuis and              *
*                                              S. L. Voogd                     *
*                                                                              *
* as part of their graduation for:     the Hoger Informatica Onderwijs (HIO)   *
* in Enschede.                             ---------------------------------   *
* The project was carried out at:          Sun Microsystems Nederland B.V.     *
* in Amersfoort.                           -------------------------------     *
*                                                                              *
*                                                                              *
* Copyright (c) 1992 Sun Microsystems Nederland B.V.                           *
*                                                                              *
* Permission to use, copy, modify, and distribute this software and            *
* its documentation for any purpose is hereby granted without fee, provided    *
* that (i) the above copyright notices and this permission notice appear in    *
* all copies of the software and related documentation, and (ii) the name      *
* Sun may not be used in any advertising or publicity relating to              *
* the software without the specific, prior written permission of               *
* Sun.                                                                         *
*                                                                              *
* THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND,           *
* EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY             *
* WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.             *
*                                                                              *
* IN NO EVENT SHALL SUN BE LIABLE FOR ANY SPECIAL, INCIDENTAL,                 *
* INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY KIND, OR ANY DAMAGES                *
* WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER OR NOT       *
* ADVISED OF THE POSSIBILITY OF DAMAGE, AND ON ANY THEORY OF LIABILITY,        *
* ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS          *
* SOFTWARE.                                                                    *
*                                                                              *
\******************************************************************************/

/*----------------------------------------------------------------------------*\
|                                                                              |
|        File : MaMA_actionlist.c                                              |
|                                                                              |
|    Contents : Functions performing on the actionlist                         |
|                                                                              |
\*----------------------------------------------------------------------------*/

#include <MaMA_include.h>
#include <MaMA_global.h>

void   itoa(int, char *);
void   reverse(char *);

extern Xv_font    *listFont;
extern int NumberOfDoc;
extern FILELIST *FileList;
extern ACTIONLIST *CurAction;

/******************************************************************************\
*                                                                              *
*        Name : PrintActionList                                                *
*                                                                              *
* Description : Prints the contents of the file list                           *
*                                                                              *
\******************************************************************************/

void
PrintActionList(ACTIONLIST *ActionListPtr)
{
    Print("--A--C--T--I--O--N--L--I--S--T--\n\n");
    while(ActionListPtr != NIL(ACTIONLIST *))
    {
        sprintf(Debug_Msg, "\tdir      : %s\n", ActionListPtr->Path->Dir);
        Print(Debug_Msg);
        sprintf(Debug_Msg, "\tfile     : %s\n", ActionListPtr->Path->File);
        Print(Debug_Msg);
        sprintf(Debug_Msg, "\tAction   : %d\n",
            ActionListPtr->Action->ActionType);
        Print(Debug_Msg);
        sprintf(Debug_Msg, "\tDur      : %d\n", ActionListPtr->Duration);
        Print(Debug_Msg);
        sprintf(Debug_Msg, "\tselected : %d\n\n", ActionListPtr->Selected);
        Print(Debug_Msg);
        ActionListPtr = ActionListPtr->next;
    }
    Print("--------------------------------\n");
}

/******************************************************************************\
*                                                                              *
*        Name : DeleteSelectedAction                                           *
*                                                                              *
* Description : Delete the first selected from the ActionList                  *
*                                                                              *
\******************************************************************************/

ACTIONLIST 
*DeleteSelectedAction(ACTIONLIST *ActionListPtr)
{
    ACTIONLIST *Tmp;
    ACTIONLIST *ReturnPtr;

    ReturnPtr = ActionListPtr;

    if (ActionListPtr == NIL(ACTIONLIST *))
    {
        return(ActionListPtr);
    }

    if (ActionListPtr->Selected)
    {
        return(ActionListPtr->next);
    }

    while (ActionListPtr->next != NIL(ACTIONLIST *))
    {
        if ((ActionListPtr->next)->Selected)
        {
            Tmp = ActionListPtr->next;
            ActionListPtr->next = (ActionListPtr->next)->next;
            free(Tmp);
            return(ReturnPtr);
        }
        ActionListPtr = ActionListPtr->next;
    }

    return(ReturnPtr);
}

/******************************************************************************\
*                                                                              *
*        Name : UnselectAction                                                 *
*                                                                              *
* Description : Set all actions to not selected                                *
*                                                                              *
\******************************************************************************/

void
UnselectAction(ACTIONLIST *ActionListPtr)
{
    while(ActionListPtr != NIL(ACTIONLIST *))
    {
        ActionListPtr->Selected = False;
        ActionListPtr = ActionListPtr->next;
    }
}

/******************************************************************************\
*                                                                              *
*        Name : SelectAction                                                   *
*                                                                              *
* Description : Select the row-th action                                       *
*                                                                              *
\******************************************************************************/

void
SelectAction(ACTIONLIST *ActionListPtr, int row)
{
    int    i;
    int    entry;

    entry = 0;
    CurAction = NIL(ACTIONLIST *);

    while(ActionListPtr != NIL(ACTIONLIST *))
    {
        ActionListPtr->Selected = False;

        if (entry == row)
        {
            ActionListPtr->Selected = True;
	    CurAction = ActionListPtr;

            /*
             * Set Action
             */
            xv_set(MaMA_Edit_Panel->Action_Settings,PANEL_VALUE,
                    ActionListPtr->Action->ActionType,NULL);
            
            /*
             * Set Min & Sec
             */
            xv_set(MaMA_Edit_Panel->Minutes_tf, PANEL_VALUE,
                (ActionListPtr->Duration / 60), NULL);
            xv_set(MaMA_Edit_Panel->Seconds_tf, PANEL_VALUE,
                (ActionListPtr->Duration % 60), NULL);

            /*
             * Set Dir & File & Select Proper Entry in the DocList
             */
            xv_set(MaMA_Edit_Panel->Dir_tf, PANEL_VALUE,
                ActionListPtr->Path->Dir, NULL);
            xv_set(MaMA_Edit_Panel->File_tf, PANEL_VALUE,
                ActionListPtr->Path->File, NULL);
            /*
             * Use FileInList() to select proper entry, if it still
             * exists (e.g. not deleted by the user). 
             */
            (void)FileInList(FileList, ActionListPtr->Path->File,
                ActionListPtr->Path->Dir);
    
            /*
             * Set The Rest
             */
            switch(ActionListPtr->Action->ActionType)
            {
                case Jump:
                    ShowLayer(TRUE, FALSE, FALSE, FALSE);

                    xv_set(MaMA_Edit_Panel->Link_Name_tf, PANEL_VALUE,
                        ActionListPtr->Action->Param->Hyperlink, NULL);
    
                    break;

                case GotoFP:
                    ShowLayer(FALSE, TRUE, FALSE, FALSE);

                    break;

                case GotoNP:
                    ShowLayer(FALSE, FALSE, TRUE, FALSE);

                    xv_set(MaMA_Edit_Panel->Repeat_tf, PANEL_VALUE,
                        ActionListPtr->Action->Param->Times, NULL);
                    break;

                case GotoPP:
                    ShowLayer(FALSE, FALSE, TRUE, FALSE);

                    xv_set(MaMA_Edit_Panel->Repeat_tf, PANEL_VALUE,
                        ActionListPtr->Action->Param->Times, NULL);
                    break;

                case GotoLP:
                    ShowLayer(FALSE, TRUE, FALSE, FALSE);

                    break;

                case ZoomIn:
                    ShowLayer(FALSE, FALSE, FALSE, TRUE);

                    xv_set(MaMA_Edit_Panel->Fit_Settings,PANEL_VALUE,
                        ActionListPtr->Action->Param->Fit, NULL);
                    break;

                case ZoomOut:
                    ShowLayer(FALSE, FALSE, FALSE, TRUE);

                    xv_set(MaMA_Edit_Panel->Fit_Settings,PANEL_VALUE,
                        ActionListPtr->Action->Param->Fit, NULL);
                    break;

                case Zoom100:
                    ShowLayer(FALSE, FALSE, FALSE, TRUE);

                    xv_set(MaMA_Edit_Panel->Fit_Settings,PANEL_VALUE,
                        ActionListPtr->Action->Param->Fit, NULL);
                    break;

                case OpenDoc:
                    ShowLayer(FALSE, TRUE, FALSE, FALSE);

                    break;

                case CloseDoc:
                    ShowLayer(FALSE, TRUE, FALSE, FALSE);

                    break;

                case QuitDoc:
                    ShowLayer(FALSE, TRUE, FALSE, FALSE);

                    break;

            } /*End of Switch*/
        }

        entry++;
        ActionListPtr = ActionListPtr->next;
    }
}

/******************************************************************************\
*                                                                              *
*        Name : ShowLayer                                                      *
*                                                                              *
* Description : Shows the specified layer                                      *
*                                                                              *
\******************************************************************************/

void
ShowLayer(int Link, int Empty, int Rep, int Zoom)
{
    xv_set(MaMA_Edit_Panel->Link_Layer, XV_SHOW, Link, NULL);
    xv_set(MaMA_Edit_Panel->Empty_Layer, XV_SHOW, Empty, NULL);
    xv_set(MaMA_Edit_Panel->Rep_Layer, XV_SHOW, Rep, NULL);
    xv_set(MaMA_Edit_Panel->Zoom_Layer, XV_SHOW, Zoom, NULL);
}

/******************************************************************************\
*                                                                              *
*        Name : GetNewAction                                                   *
*                                                                              *
* Description : Insert a new entry at the end of ActionList                    *
*                                                                              *
* ReturnValue : Pointer to the newly inserted Action                           *
*                                                                              *
\******************************************************************************/

ACTIONLIST 
*GetNewAction(ACTIONLIST *ActionListPtr)
{
    ACTIONLIST *NewAction;

    NewAction = (ACTIONLIST *)malloc((unsigned )sizeof(ACTIONLIST));
    NewAction->Path = (FILEINFO *)malloc((unsigned )sizeof(FILEINFO));
    NewAction->Action = (ACTION *)malloc((unsigned )sizeof(ACTION));
    NewAction->Action->Param = (PARAM *)malloc((unsigned )sizeof(PARAM));
    NewAction->Action->Param->WindowPos = 
        (POSITION *)malloc((unsigned )sizeof(POSITION));

    NewAction->next = NIL(ACTIONLIST *);
    NewAction->Selected = True;
 
    if (ActionListPtr == NIL(ACTIONLIST *))
    {
        return(NewAction);
    }

    /*
     * Place new entry at end of list
     */
    while(ActionListPtr->next != NIL(ACTIONLIST *))
    {
        ActionListPtr = ActionListPtr->next;
    }
    ActionListPtr->next = NewAction;

    return(NewAction);
}
 
/******************************************************************************\
*                                                                              *
*        Name : SwitchAction                                                   *
*                                                                              *
* Description : Change the order of the actions in the list by swapping        *
*               positions of ActionPtr and ActionPtr->next                     *
*                                                                              *
\******************************************************************************/

ACTIONLIST
* SwitchAction(ACTIONLIST *ActionPtr)
{
    ACTIONLIST *Next;

    Next = ActionPtr->next;
    ActionPtr->next = Next->next;
    Next->next = ActionPtr;

    return(Next);
}

/******************************************************************************\
*                                                                              *
*        Name : MoveUp                                                         *
*                                                                              *
* Description : Move action indicated by ActionPtr one place up in the list    *
*                                                                              *
\******************************************************************************/

ACTIONLIST
*MoveUp(ACTIONLIST *ActionPtr)
{
    if ( (ActionPtr == NIL(ACTIONLIST *)) 
         || 
         (ActionPtr->next == NIL(ACTIONLIST *)))
    {
        return(ActionPtr);
    }

    if ((ActionPtr->next)->Selected)
    {
        return(SwitchAction(ActionPtr));
    }

    ActionPtr->next = MoveUp(ActionPtr->next);

    return(ActionPtr);
}

/******************************************************************************\
*                                                                              *
*        Name : MoveDown                                                       *
*                                                                              *
* Description : Move action indicated by ActionPtr one place down in the list  *
*                                                                              *
\******************************************************************************/

ACTIONLIST
*MoveDown(ACTIONLIST *ActionPtr)
{
    if ( (ActionPtr == NIL(ACTIONLIST *)) 
         || 
         (ActionPtr->next == NIL(ACTIONLIST *)))
    {
        return(ActionPtr);
    }

    if (ActionPtr->Selected)
    {
        return(SwitchAction(ActionPtr));
    }

    ActionPtr->next = MoveDown(ActionPtr->next);

    return(ActionPtr);
}

/******************************************************************************\
*                                                                              *
*        Name : FillAction                                                     *
*                                                                              *
* Description : Fill in the structure that describes an action                 *
*                                                                              *
\******************************************************************************/

void
FillAction()
{
    int min,sec;
    
    /*
     * Fill in Duration
     */
    min = (int)xv_get(MaMA_Edit_Panel->Minutes_tf, PANEL_VALUE);

    sec = (int)(xv_get(MaMA_Edit_Panel->Seconds_tf, PANEL_VALUE));

    CurAction->Duration = (min * 60) + sec;

    /*
     * Fill in Action
     */
    CurAction->Action->ActionType = (int)xv_get(MaMA_Edit_Panel->
        Action_Settings, PANEL_VALUE);

    /*
     * Fill in Dir and File
     */
    strcpy(CurAction->Path->Dir, (char *)(xv_get(MaMA_Edit_Panel->Dir_tf,
        PANEL_VALUE)));
    strcpy(CurAction->Path->File, (char *)(xv_get(MaMA_Edit_Panel->File_tf,
        PANEL_VALUE)));

    /*
     * Fill in other parameters
     */
    switch (CurAction->Action->ActionType) 
    {
        case Jump:
            strcpy(CurAction->Action->Param->Hyperlink,
                (char *)xv_get(MaMA_Edit_Panel->Link_Name_tf, PANEL_VALUE));

            break;

        case GotoNP:
            CurAction->Action->Param->Times = 
                    (int)(xv_get(MaMA_Edit_Panel->Repeat_tf, PANEL_VALUE));

            break;

        case GotoPP:
            CurAction->Action->Param->Times = 
            (int)xv_get(MaMA_Edit_Panel->Repeat_tf, PANEL_VALUE);

            break;

        case ZoomIn:
            CurAction->Action->Param->Fit = 
            (int)xv_get(MaMA_Edit_Panel->Fit_Settings, PANEL_VALUE);
            
            break;

        case ZoomOut:
            CurAction->Action->Param->Fit = 
            (int)xv_get(MaMA_Edit_Panel->Fit_Settings, PANEL_VALUE);

            break;

        case Zoom100:
            CurAction->Action->Param->Fit = 
            (int)xv_get(MaMA_Edit_Panel->Fit_Settings, PANEL_VALUE);

            break;

    } /*End of Switch*/
}

/******************************************************************************\
*                                                                              *
*        Name : BuildActionString                                              *
*                                                                              *
* Description : Returns a string that will be inserted in the GUI Action List  *
*                                                                              *
\******************************************************************************/
 
char
*BuildActionString()
{
    int min,sec;
    char MinString[4];
    char SecString[4];
    char RepeatString[5];
    char ActionString[50];
    char ListString[100];
    char DocString[50];
    char FitString[10];
    char TimeString[21];
    char Spaces[15];
 
    strcpy(Spaces, "                 ");        /* For String Tuning          */
    strcpy(FitString, "    ");
    
    /*
     * Time
     */
    min = (CurAction->Duration) / 60;
    itoa(min, MinString);
    sec = (CurAction->Duration) % 60;
    itoa(sec, SecString);
    
    if (min < 10)
    {
        strcpy(TimeString, "0");
        strcat(TimeString, MinString);
    } else {
        strcpy(TimeString, MinString);
    }
 
    strcat(TimeString,":");
 
    if (sec < 10)
    {
        strcat(TimeString, "0");
    }
    strcat(TimeString, SecString);
    strcat(TimeString, "   ");
 
    /*
     * Document
     */
    strcpy(DocString, "   ");
    strcat(DocString, CurAction->Path->File);
 
    /*
     * The Rest
     */
    switch (CurAction->Action->ActionType)
    {
        case Jump:
 
            strcpy(ActionString, dgettext("MaMA_labels", "Jump To "));
            strncat(ActionString, CurAction->Action->Param->Hyperlink, 14);
  
            /*
             * Fill out with spaces
             */
            strncat(ActionString, Spaces,
                14 - (strlen(CurAction->Action->Param->Hyperlink)));
 
            break;
 
        case GotoFP:
            strcpy(ActionString, dgettext("MaMA_labels", 
                "Goto First Page       "));
 
            break;
 
        case GotoNP:
 
            strcpy(ActionString, dgettext("MaMA_labels", "Goto Next Page ("));
            itoa(CurAction->Action->Param->Times, RepeatString);
            strcat(ActionString, RepeatString);
            strcat(ActionString, "x)");
            strncat(ActionString, Spaces, 4-(strlen(RepeatString)));
 
            break;
 
        case GotoPP:
 
            strcpy(ActionString, dgettext("MaMA_labels", "Goto Prev Page ("));
            itoa(CurAction->Action->Param->Times, RepeatString);
            strcat(ActionString, RepeatString);
            strcat(ActionString, "x)");
            strncat(ActionString, Spaces, 4-(strlen(RepeatString)));
 
            break;
 
        case GotoLP:
            strcpy(ActionString, dgettext("MaMA_labels", 
                "Goto Last Page        "));
 
            break;
 
        case ZoomIn:
            strcpy(ActionString, dgettext("MaMA_labels", 
                "Zoom In               "));
 
            switch (CurAction->Action->Param->Fit)
            {
                case NONE:
                    strcpy(FitString, dgettext("MaMA_labels", "None"));
                    break;
 
                case WTP:
                    strcpy(FitString, dgettext("MaMA_labels", "WTP "));
                    break;
 
                case PTW:
                    strcpy(FitString, dgettext("MaMA_labels", "PTW "));
                    break;
            }
 
            break;
 
        case ZoomOut:
 
            strcpy(ActionString, dgettext("MaMA_labels", 
                "Zoom Out              "));
 
            switch (CurAction->Action->Param->Fit)
            {
                case NONE:
                    strcpy(FitString, dgettext("MaMA_labels", "None"));
                    break;
 
                case WTP:
                    strcpy(FitString, dgettext("MaMA_labels", "WTP "));
                    break;
 
                case PTW:
                    strcpy(FitString, dgettext("MaMA_labels", "PTW "));
                    break;
            }
 
            break;
 
        case Zoom100:
            strcpy(ActionString, dgettext("MaMA_labels", 
                "Zoom To 100%          "));
 
            switch (CurAction->Action->Param->Fit)
            {
                case NONE:
                    strcpy(FitString, dgettext("MaMA_labels", "None"));
                    break;
 
                case WTP:
                    strcpy(FitString, dgettext("MaMA_labels", "WTP "));
                    break;
 
                case PTW:
                    strcpy(FitString, dgettext("MaMA_labels", "PTW "));
                    break;
            }
 
            break;
 
        case OpenDoc:
            strcpy(ActionString, dgettext("MaMA_labels", 
                "Open Document         "));
 
            break;
 
        case CloseDoc:
            strcpy(ActionString, dgettext("MaMA_labels", 
                "Close Document        "));
 
            break;
 
        case QuitDoc:
            strcpy(ActionString, dgettext("MaMA_labels", 
                "Quit Document         "));
 
            break;
 
    } /*End of Switch*/
 
    sprintf(ListString, "%s %s%s%s", ActionString, TimeString,
		FitString, DocString);
 
    return(ListString);
}

/******************************************************************************\
*                                                                              *
*        Name : GetSelectedAction                                              *
*                                                                              *
* Description : Returns a pointer to the currently selected action             *
*               if none selected => return NULL                                *
*                                                                              *
\******************************************************************************/

ACTIONLIST
*GetSelectedAction(ACTIONLIST *ActionList)
{
    while (ActionList != NIL(ACTIONLIST *))
    {
        if (ActionList->Selected)
        {
            return(ActionList);
        }
        ActionList = ActionList->next;
    }
    return(NIL(ACTIONLIST *));
}

/******************************************************************************\
*                                                                              *
*        Name : itoa                                                           *
*                                                                              *
* Description : Convert an integer to a string                                 *
*                                                                              *
\******************************************************************************/

void
itoa(int n, char *s)
{
    int i, sign;

    if ((sign = n) < 0)
        n = -n;
    i = 0;
    do 
    {
        s[i++] = n % 10 + '0';
    } while ((n /= 10) > 0);

    if (sign < 0)
        s[i++] = '-';
    s[i] = '\0';
    reverse(s);
}

/******************************************************************************\
*                                                                              *
*        Name : reverse                                                        *
*                                                                              *
* Description : reverse string                                                 *
*                                                                              *
\******************************************************************************/

void
reverse(char *s)
{
    int c, i, j;

    for (i = 0, j = strlen(s)-1; i < j; i++, j--) 
    {
        c = s[i];
        s[i] = s[j];
        s[j] = c;
    }
}

/******************************************************************************\
*                                                                              *
*        Name : DeleteActionList                                               *
*                                                                              *
* Description : Free all entries of the given ActionList                       *
*                                                                              *
\******************************************************************************/

void
DeleteActionList(ACTIONLIST *ActionListPtr)
{
    ACTIONLIST *Tmp;

    Tmp = ActionListPtr;

    while(ActionListPtr != NIL(ACTIONLIST *))
    {
        Tmp = ActionListPtr;
        ActionListPtr = ActionListPtr->next;
        FreeAction(Tmp);
    }
}

/******************************************************************************\
*                                                                              *
*        Name : FreeAction                                                     *
*                                                                              *
* Description : Free an action                                                 *
*                                                                              *
\******************************************************************************/

void
FreeAction(ACTIONLIST * Action)
{
    free(Action->Action);
    free(Action);
}

/******************************************************************************\
*                                                                              *
*        Name : CopyActionList                                                 *
*                                                                              *
* Description : Returns a copy of the given ActionList                         *
*                                                                              *
\******************************************************************************/

ACTIONLIST *
CopyActionList(ACTIONLIST *Source_ActionList)
{
    ACTIONLIST *New_ActionList;
    ACTIONLIST *Return_ActionList;
    ACTIONLIST *Temp_ActionList;

    Return_ActionList = NIL(ACTIONLIST *);

    /*
     * copy ActionList
     */
    while(Source_ActionList != NIL(ACTIONLIST *))
    {
        New_ActionList = (ACTIONLIST *)malloc((unsigned )sizeof(ACTIONLIST));
        New_ActionList->Path = (FILEINFO *)malloc((unsigned)sizeof(FILEINFO));
        New_ActionList->Action = (ACTION *)malloc((unsigned )sizeof(ACTION));
        New_ActionList->Action->Param = 
            (PARAM *)malloc((unsigned )sizeof(PARAM));
        New_ActionList->Action->Param->WindowPos = 
            (POSITION *)malloc((unsigned )sizeof(POSITION));
/*        New_ActionList->Action->ActionType = (ACTIONTYPE *)malloc((unsigned )
	sizeof(ACTIONTYPE));
*/
        New_ActionList->next = NIL(ACTIONLIST *);


        New_ActionList->Duration = Source_ActionList->Duration;
        New_ActionList->Selected = Source_ActionList->Selected;

        strcpy(New_ActionList->Path->Dir, Source_ActionList->Path->Dir);
        strcpy(New_ActionList->Path->File, Source_ActionList->Path->File);

        New_ActionList->Action->ActionType = 
            Source_ActionList->Action->ActionType;

        /*
         * copy action type
         */
        switch(Source_ActionList->Action->ActionType)
        {
            case Jump:
                strcpy(New_ActionList->Action->Param->Hyperlink, 
                    Source_ActionList->Action->Param->Hyperlink);
                break;

            case GotoNP:
                New_ActionList->Action->Param->Times = 
                    (int )Source_ActionList->Action->Param->Times;
                break;

            case GotoPP:
                New_ActionList->Action->Param->Times = 
                    (int )Source_ActionList->Action->Param->Times;
                break;

            case ZoomIn:
                New_ActionList->Action->Param->Fit = 
                    Source_ActionList->Action->Param->Fit;
                break;

            case ZoomOut:
                New_ActionList->Action->Param->Fit = 
                    Source_ActionList->Action->Param->Fit;
                break;

            case Zoom100:
                New_ActionList->Action->Param->Fit = 
                    Source_ActionList->Action->Param->Fit;
                break;

            case GotoFP:
                break;

            case GotoLP:
                break;

            case OpenDoc:
                break;

            case CloseDoc:
                break;

            case QuitDoc:
                break;

            default:
                /* Unknown Action */
                break;
        }

        /*
         * Add New Action to the 2B-returned ActionList
         */
        if (Return_ActionList == NIL(ACTIONLIST *))
        {
            Return_ActionList = New_ActionList;
        } else {
            Temp_ActionList->next = New_ActionList;
        }
        Temp_ActionList = New_ActionList;

        Source_ActionList = Source_ActionList->next;
    }

    return(Return_ActionList);
}

/******************************************************************************\
*                                                                              *
*        Name : ActionOk                                                       *
*                                                                              *
* Description : Check if an Action is properly defined (in the GUI)            *
*                                                                              *
* ReturnValue : 1, Action is OK                                                *
*               0, Action not properly defined                                 *
*                                                                              *
\******************************************************************************/

int
ActionOk()
{
    int min_value;
    int sec_value;
         
    /*
     * Check if a target document is selected
     */
    if (strcmp(((char *)xv_get(MaMA_Edit_Panel->Dir_tf,
        PANEL_VALUE, NULL)), "") == 0)
    {
        notice_prompt (MaMA_Edit_Panel->Edit_Panel, NULL,
            NOTICE_MESSAGE_STRINGS,
            dgettext("MaMA_labels", "You need to specify a Target Document!"),
            NULL,
            NOTICE_BUTTON, "OK", 100,
            NULL);
        return(0);
    }
 
    /*
     * Check if a linkname is specified when the action is 'Jump...'
     */
    if (
        (int)(xv_get(MaMA_Edit_Panel->Action_Settings,
            PANEL_VALUE, NULL) == 0)
        &&
        (strlen((char *)xv_get(MaMA_Edit_Panel->Link_Name_tf,
            PANEL_VALUE, NULL)) == 0)
       )
    {
        notice_prompt (MaMA_Edit_Panel->Edit_Panel, NULL,
            NOTICE_MESSAGE_STRINGS,
            dgettext("MaMA_labels", "You need to specify a HyperLink name!"),
            NULL,
            NOTICE_BUTTON, "OK", 100,
            NULL);
       return(0);
    } 
 
    /*
     * Check RepeatValue when the action is 'Goto Next/Prev...'
     */
    if (
        ((int)(xv_get(MaMA_Edit_Panel->Action_Settings,
            PANEL_VALUE, NULL) == 2)
        ||
        (int)(xv_get(MaMA_Edit_Panel->Action_Settings,
            PANEL_VALUE, NULL) == 3))
	&&
        (((int)xv_get(MaMA_Edit_Panel->Repeat_tf,
            PANEL_VALUE, NULL)) == 0)
       )
    {
        xv_set(MaMA_Edit_Panel->Repeat_tf, PANEL_VALUE, 1, NULL);
    }

    /*
     * Check & Set Time 
     */
    min_value = (int)(xv_get(MaMA_Edit_Panel->Minutes_tf, PANEL_VALUE));
    sec_value = (int)(xv_get(MaMA_Edit_Panel->Seconds_tf, PANEL_VALUE));

    if (min_value < 0) 
    {
        xv_set(MaMA_Edit_Panel->Minutes_tf, PANEL_VALUE, 0, NULL);
    }

    if (sec_value < 0) 
    {
        xv_set(MaMA_Edit_Panel->Seconds_tf, PANEL_VALUE, 0, NULL);
    }

    if (min_value > 59) 
    {
        xv_set(MaMA_Edit_Panel->Minutes_tf, PANEL_VALUE, 59, NULL);
    }

    if (sec_value > 59) 
    {
        xv_set(MaMA_Edit_Panel->Seconds_tf, PANEL_VALUE, 59, NULL);
    }

    return(1);
}
