/*
 * Copyright (c) 1990, 1991 Stanford University
 *
 * Permission to use, copy, modify, and distribute this software and 
 * its documentation for any purpose is hereby granted without fee, provided
 * that (i) the above copyright notices and this permission notice appear in
 * all copies of the software and related documentation, and (ii) the name
 * Stanford may not be used in any advertising or publicity relating to
 * the software without the specific, prior written permission of
 * Stanford.
 * 
 * THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND, 
 * EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY 
 * WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.  
 *
 * IN NO EVENT SHALL STANFORD BE LIABLE FOR ANY SPECIAL, INCIDENTAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY KIND, OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER OR NOT
 * ADVISED OF THE POSSIBILITY OF DAMAGE, AND ON ANY THEORY OF LIABILITY,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 */

/* $Header: /Source/Media/drapeau/NetworkProtocol/RCS/Sender.h,v 1.18 92/09/24 14:07:14 drapeau Exp $ */
/* $Log:	Sender.h,v $
 * Revision 1.18  92/09/24  14:07:14  drapeau
 * Modified code to support asynchronous messaging (actually, to be precise,
 * the code now supports one-way RPC's, but conceptually the programmer should
 * think of the messaging as non-blocking).  In Sun RPC, if an RPC call has
 * a timeout value of zero, the sender will not block waiting for a return.
 * To support this model, a new method was added to the Sender object, called
 * "SenderAsynchronousMessaging()".  It will enable or disable synchronous
 * messaging for the sender passed in as argument.  It does this by modifying
 * the Sender's "timeOut" field.  Also, each message in the Sender object
 * modifies a variable global to the NetworkProtocol library; this variable
 * is called "SenderTimeOut" and is used by each message in the Sender class.
 * When a message is sent, the responsible method first sets the SenderTimeOut
 * variable to the timeOut field of the current Sender.
 * Because of these changes, it is no longer necessary to create default
 * timeout values when a New Sender is created (via the NewSender() method).
 * Thus, the code to do so was removed.
 * A final note: the "sender->timeOut" field is new to this version of the code.
 * 
 * Revision 1.17  92/05/29  12:40:42  drapeau
 * Changed the name of the "Selection" structure to "MAESelection",
 * to avoid name conflicts with other software packages and toolkits
 * that might also define a "Selection" structure.
 * 
 * Revision 1.16  91/11/25  13:02:08  drapeau
 * Added definition of the special hostname "AnyHost".
 * 
 * Revision 1.15  91/09/26  23:25:10  drapeau
 * Added several new definitions, all of which are used by the
 * SenderGetPortFromName() method.  WaitTimeInMicroSeconds is the initial time
 * that SenderGetPortFromName() waits after it has launched an application
 * before it checks to see if the new application is alive.
 * OneSecond is one second, in microseconds, to be used by usleep().
 * LaunchNewApp is a string constant used to specify that a new copy of an
 * application should be opened, whether a copy of that application is already
 * running or not.
 * 
 * Also, defined a new function, SenderLaunchApplication(), used by
 * SenderGetPortFromName().
 * 
 * Revision 1.14  91/09/18  12:42:04  drapeau
 * - Modified the definition of the SenderGetPortFromName() method; the additional argument,
 *   PortArray** matchingPortsReturn, holds a list of matching Ports after the message has been
 *   answered.
 * - Changed the name of the major include file from LinkProtocol.h to MAEstro.h, to better reflect
 *   the project.
 * 
 * Revision 1.13  91/06/19  14:10:06  drapeau
 * Added support for five new messages:
 *  - PauseSelection
 *  - ResumeSelection
 *  - HideApplication
 *  - ShowApplication
 *  - GetAppIcon
 * Also, replaced the "PerformPartialSelection" message with
 * "HaltSelection" message.
 * 
 * Revision 1.12  91/06/17  18:17:25  drapeau
 * Added copyright notice.
 * 
 * Revision 1.11  91/02/28  07:20:01  drapeau
 * This version uses a new version numbering scheme and a new version of RCS.
 * Also, minor change to definition of SenderGetSelection(), to make this
 * method more consistent with other Sender methods that return data to the
 * caller.  The new definition allows SenderGetSelection() to allocate space
 * for a Selection* and make the caller's Selection* point to that new space.
 * 
 * Revision 1.1  1990/10/24  18:28:34  drapeau
 * Initial revision
 * */

#ifndef _Sender_
#define _Sender_

static char SenderHeaderRcsid[] = "$Header: /Source/Media/drapeau/NetworkProtocol/RCS/Sender.h,v 1.18 92/09/24 14:07:14 drapeau Exp $";

#include <MAEstro.h>

#define WaitTimeInMicroSeconds	4000000				    /* 4 million microseconds = 4 seconds */
#define OneSecond		1000000
#define LaunchNewApp		":LaunchNewApp:"		    /* Used by SenderGetPortFromName */
#define AnyHost			":AnyHost:"			    /* Used by SenderGetPortFromName */
#define On			1				    /* Used by SenderSynchronousMessaging */
#define Off			0				    /* Used by SenderSynchronousMessaging */

typedef struct _Sender
{
  CLIENT*		clientPtr;
  int			clientSocket;
  int			portNumber;
  struct timeval	timeOut;
} Sender;

extern struct timeval*	SenderTimeOut;

Sender*	NewSender(Port*);
void	DestroySender(Sender*);
int	SenderOpenDocument(Sender* sender, char* documentName);
int	SenderGetCurrentDocName(Sender* sender, char** documentNameReturn);
int	SenderGetSelection(Sender* sender, MAESelection** selectionReturn);
int	SenderSetSelection(Sender* sender, MAESelection* selection);
int	SenderPerformSelection(Sender* sender);
int	SenderConnectWithPortMgr(Sender* sender, Port* receivingPort);
int	SenderGetOpenApps(Sender* sender, PortArray** openAppsReturn);
int	SenderGetPortFromName(Sender* sender, Port* appNameAndHost, PortArray** matchingPortsReturn);
int	SenderPortNumber(Sender* sender);
int	SenderDisconnectFromPortMgr(Sender* sender, Port* appPort);
int	SenderPing(Sender* sender);
int	SenderHaltSelection(Sender* sender);
int	SenderPauseSelection(Sender* sender);
int	SenderResumeSelection(Sender* sender);
int	SenderHideApplication(Sender* sender);
int	SenderShowApplication(Sender* sender);
int	SenderGetAppIcon(Sender* sender, IconData** iconDataReturn);
int	SenderLaunchApplication(char* appName);
int	SenderSynchronousMessaging(Sender* sender, int onOff);

#endif
