/*
 * Copyright (c) 1990, 1991 Stanford University
 *
 * Permission to use, copy, modify, and distribute this software and 
 * its documentation for any purpose is hereby granted without fee, provided
 * that (i) the above copyright notices and this permission notice appear in
 * all copies of the software and related documentation, and (ii) the name
 * Stanford may not be used in any advertising or publicity relating to
 * the software without the specific, prior written permission of
 * Stanford.
 * 
 * THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND, 
 * EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY 
 * WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.  
 *
 * IN NO EVENT SHALL STANFORD BE LIABLE FOR ANY SPECIAL, INCIDENTAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY KIND, OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER OR NOT
 * ADVISED OF THE POSSIBILITY OF DAMAGE, AND ON ANY THEORY OF LIABILITY,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 */

/* $Header: /Source/Media/collab/TimeLine/RCS/mute.c,v 1.11 91/10/05 18:32:11 chua Exp Locker: drapeau $ */
/* $Log:	mute.c,v $
 * Revision 1.11  91/10/05  18:32:11  chua
 * Change the name of UnmuteAppHandler to RemoveMuteAppHandler.
 * 
 * Revision 1.0  91/09/30  16:59:03  chua
 * Update to version 1.0
 * 
 * Revision 0.66  91/09/26  18:01:49  chua
 * Added two new functions, MuteAppHandler and UnmuteAppHandler.  These have the
 * same function as the mute choice item in the edit info list.
 * 
 * Revision 0.65  91/09/25  13:51:08  chua
 * Changed the instrument field, instInfo, to editInfo.
 * Changed InstrumentInfo to EditInfo.
 * 
 * Revision 0.64  91/09/20  11:38:55  chua
 * Correct an error in CheckOpenApp which causes the function to always
 * return Error.
 * 
 * Revision 0.63  91/09/19  17:28:56  chua
 * Make sure that variables are initialized properly.  Change formatting slightly,
 * so that (if, for, while) statements with only one statement in them will not have
 * braces.
 * 
 * Revision 0.62  91/09/04  15:10:32  chua
 * Replaced the calls to notice_prompt with a call to AlertMessage.
 * 
 * Revision 0.61  91/08/16  17:00:55  chua
 * In the CheckAppOpen procedure, do a DestroySender if the NewSender call is 
 * successful.  This is to avoid having too many senders around, which will cause
 * errors.
 * 
 * Revision 0.60  91/08/08  15:13:49  chua
 * In CheckAppOpen, if the app is not alive anymore, make it permanently muted,
 * and update its hostname to "App not open".
 * 
 * Revision 0.59  91/08/08  14:26:06  chua
 * Added a new function, CheckAppOpen, which will check if an application is still open.
 * 
 * Revision 0.58  91/07/17  10:27:22  chua
 * In the InstrumentMute function, draw the mute symbol so that it is 'thicker' and
 * looks nicer.
 * 
 * Revision 0.57  91/07/09  16:58:49  chua
 * 
 * 
 * Revision 0.56  91/06/25  17:36:33  chua
 * In the InstrumentMute routine,  draw the circle and slash in red color instead of in black.
 * 
 * Revision 0.55  91/06/04  17:37:20  chua
 * Added the copyright comments in the beginning of the file.
 * 
 * Revision 0.54  91/06/03  11:11:57  chua
 * Make changes to accomodate multiple documents.  This involves identifying
 * which is the current active window, that is, the one where the last mouse
 * click was done.
 * 
 * Revision 0.53  91/05/29  14:37:44  chua
 * Delete the InstrumentMuteOff routine and replace the InstrumentMuteOn by InstrumentMute, which
 * will check if it is necessary to draw the circle and slash (mute indicator) symbol over the
 * application icon of the instrument.
 * 
 * Changes are also made to the InstrumentMuteChoice routine, so that it will call the
 * InstrumentDrawIcon routine with the proper parameters, such as whether the icon should be raised
 * or sunken.
 * 
 * Revision 0.52  91/05/28  12:12:33  chua
 * *** empty log message ***
 * 
 * Revision 0.51  91/05/28  12:04:39  chua
 * This file contains the panel choice notify procedure for the mute choice panel item in the instrument
 * info pop-up window and also 2 functions to draw/clear the mute indicator that appears besides the
 * application icons in the AppCanvas.
 *  */

static char mutercsid[] = "$Header: /Source/Media/collab/TimeLine/RCS/mute.c,v 1.11 91/10/05 18:32:11 chua Exp Locker: drapeau $";

#include "main.h"

/*
 * Notify callback function for `MuteChoice'.
 * This function checks whether the user has turned the mute function on or off and calls the appropriate drawing routine to display or turn off the
 * mute indicator besides the application icon for that instrument.
 */
int InstrumentMuteChoice(item, value, event)
     Panel_item	item;
     int		value;
     Event		*event;
{
  Instrument *instrument;
  int currentApp, type;
  TimeLineFramePtr tlFrame;
  EditInfo_EditInfoWindow_objects	*ip = (EditInfo_EditInfoWindow_objects *) xv_get(item, XV_KEY_DATA, INSTANCE);
  Window	owner = xv_get(ip->EditInfoWindow, XV_OWNER);
  TimeLine_window_objects * tlip = (TimeLine_window_objects *) xv_get(owner, XV_KEY_DATA, INSTANCE);

  tlFrame = TimeLineWindow[xv_get(tlip->controls, PANEL_CLIENT_DATA)];
  currentApp = (int) xv_get(ip->NoteInfoList, PANEL_CLIENT_DATA);
  instrument = (Instrument*) FindInstrument(currentApp, tlFrame);   /* Find the instrument for this info pop-up window */
  if (instrument == NULL) 
    return;
  type = Raised;  
  if (currentApp == tlFrame->chosenApp)				    /* Determine if the current instrument is selected */
    type = Sunken;
  InstrumentDrawIcon(instrument, type, tlFrame);
}

/*
 * This function checks to see if the mute choice is on.  If so, it draws a circle and a forward slash across the application icon in red.
 * Called by InstrumentMuteChoice (mute.c)
 */
void InstrumentMute(instrument, tlFrame)
     Instrument *instrument;
     TimeLineFramePtr tlFrame;
{
  if (xv_get(instrument->editInfo->MuteChoice, PANEL_VALUE))	    /* Check if mute is on */
  {
    XSetForeground(tlFrame->dpyApp, tlFrame->gc, (long)tlFrame->pixelTable[Red]); /* Set the foreground color to black */
    XSetBackground(tlFrame->dpyApp, tlFrame->gc, (long)tlFrame->pixelTable[Grey84]); /* Set the background color to light grey */
    SetLineAttributes(tlFrame, 4);				    /* Increase the line width */
    XDrawArc(tlFrame->dpyApp, tlFrame->xidApp, tlFrame->gc, IconXOffset + 3, instrument->cableStart - 30,
	     IconHeight - 6, IconHeight - 6, 0,0);
    XDrawLine(tlFrame->dpyApp, tlFrame->xidApp, tlFrame->gc, IconXOffset + IconHeight - 9, 
	      instrument->cableStart - 18, IconXOffset + 9, 
	      instrument->cableStart + 18);
    SetLineAttributes(tlFrame, 2);				    /* Reset the line width back to normal */
  }
}

/*
 * This function prints out an error message when the user tries to play or insert a new note in an application that is currently
 * not open.  If mode = 1, the operation is an insert, else if mode = 0, the operation is a play.  Otherwise, mode = 2.
 */
int CheckAppOpen(tlFrame, instrument, mode)
     TimeLineFramePtr tlFrame;
     Instrument *instrument;
     int mode;
{
  Sender *sender;
  char buf[100], buf1[100];
  
  if (xv_get(instrument->editInfo->MuteChoice, PANEL_VALUE) == 0 
      || mode == 1)
  {
    if (mode == 1) 
      sprintf (buf1, "You cannot insert a new note for this application.");
    else if (mode == 0)
      sprintf (buf1, "You cannot perform a play operation for this application.");
    else 
      sprintf (buf1, "");
    sprintf (buf, "Sorry, the application %s is not open.", instrument->port->appName);
    if (strcmp(instrument->port->hostName, "App not open") == 0)    /* Check if the application has never been opened before. */
    {
      AlertMessage(tlFrame, buf, buf1, NULL);
      return Error;
    }
    sender = NewSender(instrument->port);
    if (sender == NULL)						    /* Check if the application (that was opened previously) is still open. */
    {
      xv_set(instrument->editInfo->MuteChoice, PANEL_VALUE, 1, NULL);
      xv_set(instrument->editInfo->MuteChoice, PANEL_INACTIVE, TRUE, NULL);
      strcpy(instrument->port->hostName, "App not open");
      xv_set(instrument->editInfo->HostnameText,		    /* Set the hostname on the pop-up window */
	     PANEL_LABEL_STRING, instrument->port->hostName, NULL);
      AlertMessage(tlFrame, buf, buf1, NULL);
      AppCanvasRepaintHandler(tlFrame->TimeLine_window->AppCanvas, tlFrame->paintWinApp, /* Repaint the apps canvas so that the icon shows muted */
			      tlFrame->dpyApp, tlFrame->xidApp, NULL);
      return Error;
    }
    else 
      DestroySender(sender);
  }
  return OK;
}

/*
 * Menu handler for `AppMenu (Mute application)'.
 */
Menu_item MuteAppHandler(item, op)
     Menu_item	item;
     Menu_generate	op;
{
  Instrument *instrument;
  TimeLineFramePtr tlFrame;
  TimeLine_window_objects	*ip = (TimeLine_window_objects *) xv_get(item, XV_KEY_DATA, INSTANCE);

  tlFrame = TimeLineWindow[xv_get(ip->controls, PANEL_CLIENT_DATA)];
  switch (op) 
  {
   case MENU_DISPLAY:
    break;
   case MENU_DISPLAY_DONE:
    break;
   case MENU_NOTIFY:
    if (tlFrame->chosenApp >= 0) 
    {
      instrument = (Instrument *) FindInstrument(tlFrame->chosenApp, tlFrame);
      if (instrument == NULL)
	return item;
      if (xv_get(instrument->editInfo->MuteChoice, PANEL_VALUE) == 0 &&
	  strcmp(instrument->port->hostName, "App not open") != 0)
      {
	xv_set(instrument->editInfo->MuteChoice, PANEL_VALUE, 1, NULL);
	InstrumentDrawIcon(instrument, Sunken, tlFrame);
      }
    }
    break;
   case MENU_NOTIFY_DONE:
    break;
  }
  return item;
}

/*
 * Menu handler for `AppMenu (Remove mute from application)'.
 */
Menu_item RemoveMuteAppHandler(item, op)
     Menu_item	item;
     Menu_generate	op;
{
  Instrument *instrument;
  TimeLineFramePtr tlFrame;
  TimeLine_window_objects	*ip = (TimeLine_window_objects *) xv_get(item, XV_KEY_DATA, INSTANCE);

  tlFrame = TimeLineWindow[xv_get(ip->controls, PANEL_CLIENT_DATA)];
  switch (op) 
  {
   case MENU_DISPLAY:
    break;
   case MENU_DISPLAY_DONE:
    break;
   case MENU_NOTIFY:
    if (tlFrame->chosenApp >= 0) 
    {
      instrument = (Instrument *) FindInstrument(tlFrame->chosenApp, tlFrame);
      if (instrument == NULL)
	return item;
      if (xv_get(instrument->editInfo->MuteChoice, PANEL_VALUE) == 1 && 
	  strcmp(instrument->port->hostName, "App not open") != 0)
      {
	xv_set(instrument->editInfo->MuteChoice, PANEL_VALUE, 0, NULL);
	InstrumentDrawIcon(instrument, Sunken, tlFrame);
      }
    }
    break;
   case MENU_NOTIFY_DONE:
    break;
  }
  return item;
}

