/*
 * Copyright (c) 1990, 1991 Stanford University
 *
 * Permission to use, copy, modify, and distribute this software and 
 * its documentation for any purpose is hereby granted without fee, provided
 * that (i) the above copyright notices and this permission notice appear in
 * all copies of the software and related documentation, and (ii) the name
 * Stanford may not be used in any advertising or publicity relating to
 * the software without the specific, prior written permission of
 * Stanford.
 * 
 * THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND, 
 * EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY 
 * WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.  
 *
 * IN NO EVENT SHALL STANFORD BE LIABLE FOR ANY SPECIAL, INCIDENTAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY KIND, OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER OR NOT
 * ADVISED OF THE POSSIBILITY OF DAMAGE, AND ON ANY THEORY OF LIABILITY,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 */

/* $Header: /Source/Media/collab/VideoEdit/RCS/VideoEditEdit.c,v 0.31 92/05/13 00:17:32 drapeau Exp Locker: drapeau $ */
/* $Log:	VideoEditEdit.c,v $
 * Revision 0.31  92/05/13  00:17:32  drapeau
 * Extensive changes, mostly cosmetic, to the code:
 * * Cosmetic changes (mostly formatting) to conform to coding standards.
 * * Added the function UpdateHeader() to more cleanly report the
 *   current modification status of the current document.  Modified
 *   several other functions to call this function.
 * * Modified several functions to correctly update status of current
 *   document (whether one is opened, and whether one has been
 *   modified).
 * 
 * Revision 0.30  92/05/12  19:10:20  drapeau
 * Made several modifications to implement changes in the interface.
 * All of these changes are to implement the new "Document" menu
 * on the Edit panel:
 * 
 * * Added FileSaveAs(), EditCloseWindow(), EditCloseFile() functions
 *   to correspond with "Save As", "Close Window", and "Close File"
 *   menu items.
 * * Re-formatted code to conform with coding standards.
 * * Other slight changes to existing FileSave() function to reflect
 *   that it is now a callback of a menu item, not a button.
 * 
 * Revision 0.29  92/01/07  14:09:23  drapeau
 * Modified OpenHandler() to fix an error in reading files.  The error
 * occurred on a boundary condition: if an edit had a label of exactly
 * the maximum length (currently 22 characters), the function would
 * omit the NULL character at the end of the string, which would cause
 * subsequent edits to be incorrectly read.
 * Also, made cosmetic changes to make the code easier to read.
 * 
 * Revision 0.28  92/01/03  15:45:47  drapeau
 * Changed occurrances of "NULL" in calls to Browse() to use "0" instead.
 * This is due to the ANSI definition of NULL as "(void*)0".
 * Also, updated List() function to set currentAudio variable as
 * appropriate for the current edit.
 * 
 * Also made minor cosmetic changes.
 * 
 * Revision 0.27  91/09/24  21:34:55  lim
 * Changed 'vEdit' to 'VideoEdit' everywhere.
 * 
 * Revision 0.26  91/09/04  14:19:20  lim
 * *** empty log message ***
 * 
 * Revision 0.25  91/08/30  15:44:06  lim
 * Header increased to 21 char in length.
 * 
 * Revision 0.24  91/08/23  17:16:29  lim
 * Document format has been changed.
 * 
 * Revision 0.23  91/08/20  15:16:11  lim
 * Fixed deleting of old edits.
 * 
 * Revision 0.22  91/08/17  20:52:45  lim
 * OpenPanel is now Browse.
 * 
 * Revision 0.21  91/08/16  13:10:00  lim
 * 1. Interface changes. Merged editCurrSelMsg into editNewEditMsg.
 * 2. DeleteAll() causes titlebar to change to "untitled document". 
 *    Titlebar now shows filename.
 * 
 * Revision 0.20  91/08/09  17:30:19  lim
 * Included OpenPanel.
 * 
 * Revision 0.19  91/08/08  11:27:09  lim
 * Zero speed is not permitted in the edit list.
 * 
 * Revision 0.18  91/08/07  14:33:44  lim
 * *** empty log message ***
 * 
 * Revision 0.17  91/08/07  14:31:47  lim
 * Update currentAudio in FileLoad().
 * 
 * Revision 0.16  91/08/07  13:27:47  lim
 * 1. ErrorCheck() has been removed. The responsibility for notification of 
 * error has been shifted to the device drivers, which will call DisplayError()
 * in 'videoObjects.c'.
 * 2. Added instance pointer, "theObject" to all public function calls.
 * 
 * Revision 0.15  91/08/02  12:59:45  lim
 * 'currentAudio' is no longer static.
 * 
 * Revision 0.14  91/07/27  22:29:16  lim
 * Speed changed from double to int.
 * 
 * Revision 0.13  91/07/24  11:02:02  lim
 * Whenever editAudioMenu is called, a static variable "currentAudio" is set.
 * In EditAdd/EditModify, the array audioStg is set to currentAudio, instead of
 * the old way in which audioStg is always set when the editAudioMenu is called.
 * 
 * Revision 0.12  91/07/19  16:15:17  lim
 * Added check for editnum != -1 in EditSpeedStg
 * 
 * Revision 0.11  91/07/19  14:11:50  lim
 * Renamed print_onlist and print_numedits according to specs.
 * Changed code to fit in audio and speed setting options for
 * edits.
 * 
 * Revision 0.10  91/07/12  16:19:53  lim
 * Initial revision implementing VideoObject.
 *  */

#include "VideoEdit.h"

static char editrcs[] = "$Header: /Source/Media/collab/VideoEdit/RCS/VideoEditEdit.c,v 0.31 92/05/13 00:17:32 drapeau Exp Locker: drapeau $";

int currentAudio;
char currentFilename[256];					    /* Name of file currently loaded */

/* 	VIDEO EDIT FUNCTIONS (POPUP WINDOW 2) 	*/

/* format an edit entry for the panel list */
void
PrintOnList(current, buf)
     int current;
     char buf[70];
{
  char audio[7];
  
  switch (currentAudio)
  {
   case Stereo:
    strcpy(audio, "Stereo");
    break;
   case Right:
    strcpy(audio, "Right");
    break;
   case Left:
    strcpy(audio, "Left");
    break;
   case Mute:
    strcpy(audio, "Mute");
    break;
   default:
    DisplayError("Invalid audio setting", " ");
    return;
  }
  
  sprintf (buf, " %5d.  %-26.22s%5d    %5d   %-8s%d", 
	   current+1, label[current], startframe[current], endframe[current], 
	   audio, speedStg[current]);
}

/* print the number of edit entries on the list */
void
PrintNumEdits()
{
  char edit[20];
  
  sprintf(edit, "No. of Edits : %d", lines);
  xv_set(VideoEdit_editPopup->editNumEditMsg, PANEL_LABEL_STRING, 
	 edit, NULL);
}

/* file error handling procedure */
void 
FileError(optype)
     int optype;
{
  char buf[40];
  
  if (optype == SAVE) 
    strcpy(buf, "Error in writing file.");
  else 
    switch(errno) 
    {
     case ENOTDIR:
     case EFAULT : 
      case ENOENT : strcpy(buf, "File or directory does not exist.");
       break;
      case EACCES : strcpy(buf, "File path search permission denied.");
       break;
       default    : strcpy(buf, "Error in opening file.");
       break;
     }
  
  DisplayError(buf, " ");
}

/*
 * Menu handler for `DocMenu (Open)' to load an edit file.
 */
Menu_item
FileLoad(item, op)
	Menu_item	item;
	Menu_generate	op;
{
  int result;
  if (op != MENU_NOTIFY)
    return item;
 
  if ((change) && (lines > 0))					    /* check if unsaved changes exist in edit list */
  {
    result = notice_prompt(VideoEdit_editPopup->editPopup, NULL,
			   NOTICE_MESSAGE_STRINGS,
			   "Unsaved changes exist in the edit list.",
			   "Loading a file will erase all changes.",
			   "Go ahead and load the file?",
			   NULL,
			   NOTICE_BUTTON_NO,	"Yes",
			   NOTICE_BUTTON_YES,	"No",
			   NULL);
    if (result == NOTICE_YES)
      return;
  }
  
  Browse(NULL, BrowseOpen, 0, "#Video Edit Document#", "VideoEdit");
  return item;
  
}

void
DisplayEditList(oldlines, newlines)
     int oldlines;
     int newlines;
{
  int i;
  char	buf[70];
      
  if (editnum != -1)						    /* Deselect any current selection */
    xv_set(VideoEdit_editPopup->editScrollList,
	   PANEL_LIST_SELECT, editnum, FALSE,
	   NULL);
  for (i=0; (i<oldlines && newlines > 0); i++)			    /* Replace the old string by the new one */
  {
    currentAudio = audioStg[i];
    PrintOnList(i, buf);
    xv_set (VideoEdit_editPopup->editScrollList,
	    PANEL_LIST_STRING, i, buf,
	    PANEL_LIST_FONT, i, font,
	    NULL);
  }
  
  if (oldlines < newlines)					    /* Insert the additional new strings */
    for (i=oldlines; i<newlines; i++) 
    {
      currentAudio = audioStg[i];
      PrintOnList(i, buf);
      xv_set (VideoEdit_editPopup->editScrollList,
	      PANEL_LIST_INSERT, i,
	      PANEL_LIST_STRING, i, buf,
	      PANEL_LIST_FONT, i, font,
	      NULL);
    }
								    
  else								    /* delete all additional old entries in the edit list */
    for (i=oldlines-1; i>newlines-1; i--)
      xv_set(VideoEdit_editPopup->editScrollList,
	     PANEL_LIST_DELETE, i,
	     NULL);
  
  PrintNumEdits();
  xv_set(VideoEdit_editPopup->editModButton,			    /* Set modify, delete buttons to inactive */
	 PANEL_INACTIVE, TRUE, NULL);
  xv_set(VideoEdit_editPopup->editDelButton,
	 PANEL_INACTIVE, TRUE, NULL);

  sprintf(currentFilename, "Untitled");				    /* Housekeeping - clear out currentFilename, ... */
  editnum = -1;							    /* ...editnum and change */
  change = 0;
}


int 
  OpenHandler(proposedPath, id)
char* proposedPath;
int id;
{
  int	i;
  int	oldlines;
  char	header[22];
  char	tempLabel[32];
  FILE*	fp;
  
  fp = fopen(proposedPath, "r");				    /* Open file read only */
  fgets(header, 22, fp);					    /* check if file is a video edit file */
  if (strcmp(header, "#Video Edit Document#") != 0) 
  {
    DisplayError("Not a video edit file.", " ");
    fclose(fp);
    return 1;
  }
  oldlines = lines;
  fscanf(fp, "%*s%*s%*s%d", &lines);
  for (i=0; i<lines; i++) 
  {
    fscanf(fp, "%*s%*s%*d\n");					   
    fscanf(fp, "%*s%*s%d\n", &startframe[i]);
    fscanf(fp, "%*s%*s%d\n", &endframe[i]);
    fscanf(fp, "%*s%*s%d\n", &audioStg[i]);
    fscanf(fp, "%*s%*s%*s%d\n", &speedStg[i]);
    fgets(tempLabel, 32, fp);
    strncpy(label[i], &tempLabel[8], 22);
    if (strlen(label[i]) < 22)					    /* Remove newline character for label if necessary */
      label[i][strlen(label[i])-1] = '\0';
    if ((strlen(label[i]) == 22) && (label[i][21] == '\n'))
      label[i][21] = '\0';
  }
  fclose (fp);
  
  DisplayEditList(oldlines, lines);				    /* Display these new lines on edit list */
  
  xv_set(VideoEdit_editPopup->editNewEditMsg, 
	 PANEL_LABEL_STRING, "Current Selection : New Edit", 
	 NULL);
  UpdateHeader(proposedPath, 0);				    /* Print new filename on title bar */
  strcpy(currentFilename, proposedPath);			    /* Copy filename into currentFilename */
  return 0;
}								    /* end function OpenHandler */



/*
 * Menu handler for `DocMenu (Save)' to save an edit file.
 */
Menu_item
FileSave(item, op)
	Menu_item	item;
	Menu_generate	op;
{
  if ((op == MENU_NOTIFY) && lines)
  {
    if (strcmp(currentFilename, "Untitled") != 0)
      Browse(currentFilename, BrowseCheckSave, 1,		    /* ID = 1 for checksave */
	     "#Video Edit Document#", "VideoEdit");
    else							    /* There is no current file */
      Browse(NULL, BrowseSave, 0, "#Video Edit Document#", "VideoEdit");
  }
  return item;
    
}

/*
 * Menu handler for `DocMenu (Save As...)' 
   to save an edit file (Save As... from DocMenu under EditDocButton).
 */
Menu_item
FileSaveAs(item, op)
	Menu_item	item;
	Menu_generate	op;
{
  if ((op == MENU_NOTIFY) && lines)
    Browse(NULL, BrowseSave, 0, "#Video Edit Document#", "VideoEdit");
  return item;
  
}



int
  SaveHandler(char* proposedPath, int id)
{
  struct stat stbuf;
  int i;
  int result;
  char header[22];
  char buf[40];
  FILE* fp;
  
  if (stat(proposedPath, &stbuf) == 0)				    /* Does file exist?  If so, is it a VideoEdit document? */
  {
    fp = fopen(proposedPath, "r");
    fgets(header, 22, fp);
    fclose(fp);
    if (strcmp(header, "#Video Edit Document#") != 0) 
      sprintf(buf, "This file is not a video edit file.");
    else 
      sprintf(buf, "This video edit file exists.");
    if (!id)							    /* Not checksave */
    {
      result = notice_prompt(VideoEdit_editPopup->editPopup, NULL,
			     NOTICE_MESSAGE_STRINGS,
			     buf,
			     "Do you wish to overwrite it?",
			     NULL,
			     NOTICE_BUTTON_NO,	"Yes",
			     NOTICE_BUTTON_YES,	"No",
			     NULL);
      if (result == NOTICE_YES) 
	return 1;
    }
  }
  
  fp = fopen(proposedPath, "w");
  if (fp == NULL)						    /* Check if file was properly opened */
  {
    FileError(SAVE);
    return 1;
  }
  lines = xv_get(VideoEdit_editPopup->editScrollList, 
		 PANEL_LIST_NROWS);
  
  fprintf(fp, "#Video Edit Document#\n");			    /* Write contents of the edit list to the file */
  fprintf(fp, "#Number of Edits:\t%d\n\n", lines);
  for (i=0; i<lines; i++) 
  {
    fprintf(fp, "#Edit Number:\t%d\n", i+1);
    fprintf(fp, "#Start Address:\t%d\n", startframe[i]);
    fprintf(fp, "#End Address:\t%d\n", endframe[i]);
    fprintf(fp, "#Audio Setting:\t%d\n", audioStg[i]);
    fprintf(fp, "#Speed in frames/second:\t%d\n", speedStg[i]);
    fprintf(fp, "#Label:\t%s\n\n", label[i]);
  }
  fclose (fp);
  if (!id)							    /* Was a SaveAs */
  {
    bzero(currentFilename, 256);
    strcpy(currentFilename, proposedPath);
  }
  change = 0;
  UpdateHeader(proposedPath, 0);
  return 0;
}								    /* end function SaveHandler */



/*
  To close both the edit and preview popup windows.
  */
void
EditDone(item, event)
     Panel_item	item;
     Event		*event;
{
  xv_set(VideoEdit_previewPopup->previewPopup, FRAME_CMD_PUSHPIN_IN, FALSE, NULL);
  xv_set(VideoEdit_previewPopup->previewPopup, XV_SHOW, FALSE, NULL);
  xv_set(VideoEdit_editPopup->editPopup, FRAME_CMD_PUSHPIN_IN, FALSE, NULL);
  xv_set(VideoEdit_editPopup->editPopup, XV_SHOW, FALSE, NULL);
}

/*
 * Menu handler for `DocMenu (Close Window)'.
 */
Menu_item
EditCloseWindow(item, op)
	Menu_item	item;
	Menu_generate	op;
{ 
  if (op == MENU_NOTIFY)
  {
    xv_set(VideoEdit_previewPopup->previewPopup, FRAME_CMD_PUSHPIN_IN, FALSE, NULL);
    xv_set(VideoEdit_previewPopup->previewPopup, XV_SHOW, FALSE, NULL);
    xv_set(VideoEdit_editPopup->editPopup, FRAME_CMD_PUSHPIN_IN, FALSE, NULL);
    xv_set(VideoEdit_editPopup->editPopup, XV_SHOW, FALSE, NULL);
  }
  return item;
}


/*
 * Menu handler for `DocMenu (Close File)'.
 */
Menu_item
  EditCloseFile(item, op)
Menu_item	item;
Menu_generate	op;
{  
  int result;
  
  if (op != MENU_NOTIFY)
    return item;  
  if ((change) && (lines > 0)) 
  {
    result = notice_prompt(VideoEdit_editPopup->editPopup, NULL,
			   NOTICE_MESSAGE_STRINGS,
			   "Unsaved changes exist in the edit list.",
			   "Go ahead and close file without saving changes? ",
			   NULL,
			   NOTICE_BUTTON_NO,	"Yes",
			   NOTICE_BUTTON_YES,	"No",
			   NULL);
    if (result == NOTICE_YES)
      return item;
  }			
  
  DisplayEditList(lines, 0);					    /* Display new edit list which originally had 'lines'
								       and now has 0 lines. */
  lines = 0;
  xv_set(VideoEdit_editPopup->editNewEditMsg, 
	 PANEL_LABEL_STRING, "Current Selection : None", 
	 NULL);
  UpdateHeader("Untitled", 0);
  xv_set(VideoEdit_editPopup->editStartTxt, 
	 PANEL_VALUE, 0, NULL);
  xv_set(VideoEdit_editPopup->editEndTxt, 
	 PANEL_VALUE, 0, NULL);
  xv_set(VideoEdit_editPopup->editDurTxt, 
	 PANEL_VALUE, "0", NULL);
  xv_set(VideoEdit_editPopup->editLabelTxt, 
	 PANEL_VALUE, " ", NULL);
  xv_set(VideoEdit_editPopup->editSpeedTxt, 
	 PANEL_VALUE, 30, NULL);
  
  return item;
}


/*
 * Menu handler for `editAudioMenu (Stereo)'.
 */
Menu_item
EditAudioStereo(item, op)
	Menu_item	item;
	Menu_generate	op;
{
  if (op == MENU_NOTIFY)
  {
    xv_set(VideoEdit_editPopup->editAudioMenuButton, PANEL_LABEL_STRING, "Stereo", NULL);
    currentAudio = Stereo;
  }
  
  return item;
  
}


/*
 * Menu handler for `editAudioMenu (Right)'.
 */
Menu_item
EditAudioRight(item, op)
	Menu_item	item;
	Menu_generate	op;
{
  
  if (op == MENU_NOTIFY)
  {
    xv_set(VideoEdit_editPopup->editAudioMenuButton, PANEL_LABEL_STRING, "Right", NULL);
    currentAudio = Right;
  }
  
  return item;
}


/*
 * Menu handler for `editAudioMenu (Left)'.
 */
Menu_item
EditAudioLeft(item, op)
	Menu_item	item;
	Menu_generate	op;
{
  if (op == MENU_NOTIFY)
  {
    xv_set(VideoEdit_editPopup->editAudioMenuButton, PANEL_LABEL_STRING, "Left", NULL);
    currentAudio = Left;
  }
  
  return item;
  
}


/*
 * Menu handler for `editAudioMenu (Mute)'.
 */
Menu_item
EditAudioMute(item, op)
	Menu_item	item;
	Menu_generate	op;
{
  if (op == MENU_NOTIFY)
  {
    xv_set(VideoEdit_editPopup->editAudioMenuButton, PANEL_LABEL_STRING, "Mute", NULL);
    currentAudio = Mute;
  }
  
  return item;
  
}


/*
 * Notify callback function for `editSpeedTxt'.
 */
Panel_setting
EditSpeedStg(item, event)
     Panel_item	item;
     Event		*event;
{
  if ((event_action(event) != '\r') && (event_action(event) != '\n'))
    return panel_text_notify(item, event);

  if (editnum != -1)
    speedStg[editnum] =  (int) xv_get(item, PANEL_VALUE);
  SetDuration();
  
  return panel_text_notify(item, event);
}

/*
  Button notify proc to get the start frame number (editGetStartButton)
  */
void
GetStart(item, event)
     Panel_item	item;
     Event		*event;
{
  int current;
  
  current = DevQueryFrame(myVideo);
  xv_set (VideoEdit_editPopup->editStartTxt, 
	  PANEL_VALUE, current, 
	  NULL);
}

/*
  Button notify proc to get the end frame number (editGetEndButton)
  */
void
GetEnd(item, event)
     Panel_item	item;
     Event		*event;
{
  int current;
  
  current = DevQueryFrame(myVideo);

  if (current >= 0)
    xv_set (VideoEdit_editPopup->editEndTxt, 
	    PANEL_VALUE, current, 
	    NULL);
  SetDuration();
}

/*
 * Menu handler for `previewMenu (Play part of edit...)'.
 */
Menu_item
EditPreview(item, op)
     Menu_item       item;
     Menu_generate   op;
{
  switch (op) 
  {
   case MENU_DISPLAY:
    break;
    
   case MENU_DISPLAY_DONE:
    break;
    
   case MENU_NOTIFY:
    xv_set(VideoEdit_previewPopup->previewPopup, 
	   FRAME_CMD_PUSHPIN_IN, TRUE, 
	   NULL);
    xv_set(VideoEdit_previewPopup->previewPopup, 
	   XV_SHOW, TRUE, 
	   NULL);
    SetDuration();						    /* get the approximate duration */
    break;
    
   case MENU_NOTIFY_DONE:
    break;
  }
  return item;
}



/*
  Button notify proc to add the current selection to the edit list
  (editAddButton)
  */
void
  EditAdd(item, event)
Panel_item	item;
Event		*event;
{
  char buf[70];
  char s[30];
  
  if ((int) xv_get(VideoEdit_editPopup->editSpeedTxt, PANEL_VALUE) <= 0)
  {
    DisplayError("Zero speed is not acceptable.", "Please enter another value.");
    return;
  }
  
  startframe[lines] = xv_get(VideoEdit_editPopup->editStartTxt,	    /* Get start frame, ... */
			     PANEL_VALUE);
  endframe[lines] = xv_get (VideoEdit_editPopup->editEndTxt,	    /* ...end frame, label and speed&audio settings */
			    PANEL_VALUE);
  strcpy (label[lines],
	  (char*) xv_get (VideoEdit_editPopup->editLabelTxt,
			  PANEL_VALUE));
  speedStg[lines] =  (int) xv_get(VideoEdit_editPopup->editSpeedTxt,
				  PANEL_VALUE);
  audioStg[lines] = currentAudio;
  
  PrintOnList(lines, buf);					    /* Add on edit list */
  xv_set(VideoEdit_editPopup->editScrollList,
	 PANEL_LIST_INSERT, lines,
	 PANEL_LIST_STRING, lines, buf,
	 PANEL_LIST_FONT, lines, font,
	 NULL);
  if (editnum != -1)
    xv_set(VideoEdit_editPopup->editScrollList,
	   PANEL_LIST_SELECT, editnum, FALSE,
	   NULL);
  
  xv_set(VideoEdit_editPopup->editScrollList,			    /* select the newly added entry */
	 PANEL_LIST_SELECT, lines, TRUE,
	 NULL);
  editnum = lines;
  lines++;
  change = 1;
  UpdateHeader(currentFilename, 1);
  SetDuration();
  xv_set(VideoEdit_editPopup->editModButton,			    /* Make the modify and delete buttons active */
	 PANEL_INACTIVE, FALSE, NULL);
  xv_set(VideoEdit_editPopup->editDelButton,
	 PANEL_INACTIVE, FALSE, NULL);
  sprintf(s, "Current Selection : Edit #%d", lines);		    /* Update edit status info */
  xv_set(VideoEdit_editPopup->editNewEditMsg, PANEL_LABEL_STRING, s, NULL);
  PrintNumEdits();
  change = 1;
}								    /* end function EditAdd */



/*
  Button notify proc to delete the current selection from the edit list
  (editDelButton)
  */
void
  EditDelete(item, event)
Panel_item      item;
Event           *event;
{
  int i;
  char buf[70];
  
  xv_set(VideoEdit_editPopup->editScrollList,			    /* Delete the current selection */
	 PANEL_LIST_SELECT, editnum, FALSE,
	 NULL);
  if (editnum != 0) 
    xv_set(VideoEdit_editPopup->editScrollList,
	   PANEL_LIST_SELECT, editnum - 1, TRUE,
	   NULL);
  xv_set(VideoEdit_editPopup->editScrollList,
	 PANEL_LIST_DELETE, editnum,
	 NULL);
  
  lines --;							    /* Shift all the following edit entries up by one */
  for (i=editnum; i < lines;i++) 
  {
    startframe[i] = startframe[i+1];
    endframe[i] = endframe[i+1];
    strcpy(label[i], label[i+1]);
    audioStg[i] = audioStg[i+1];
    speedStg[i] = speedStg[i+1];
    currentAudio = audioStg[i];
    PrintOnList(i, buf);
    xv_set(VideoEdit_editPopup->editScrollList,
	   PANEL_LIST_STRING, i, buf,
	   PANEL_LIST_FONT, i, font,
	   NULL);
  }
  if (editnum != 0) 
    xv_set(VideoEdit_editPopup->editScrollList,
	   PANEL_LIST_SELECT, editnum - 1, FALSE,
	   NULL);
  xv_set(VideoEdit_editPopup->editNewEditMsg, 
	 PANEL_LABEL_STRING, "Current Selection : New Edit", 
	 NULL);
  xv_set(VideoEdit_editPopup->editModButton,			    /* Make the delete, modify buttons inactive */
	 PANEL_INACTIVE, TRUE, NULL);
  xv_set(VideoEdit_editPopup->editDelButton,
	 PANEL_INACTIVE, TRUE, NULL);
  PrintNumEdits();
  change = 1;
  editnum = -1;
  UpdateHeader(currentFilename, 1);
}								    /* end function EditDelete */

/*
  Button notify proc to modify the current selection (editModButton)
  */
void
EditModify(item, event)
     Panel_item	item;
     Event		*event;
{
  char buf[70];
  
  if ((int) xv_get(VideoEdit_editPopup->editSpeedTxt, PANEL_VALUE) <= 0)
  {
    DisplayError("Zero speed not acceptable", "Please enter another value");
    return;
  }
  
  startframe[editnum] = xv_get (VideoEdit_editPopup->editStartTxt, PANEL_VALUE);
  endframe[editnum] = xv_get (VideoEdit_editPopup->editEndTxt, PANEL_VALUE);
  strcpy (label[editnum], (char*) xv_get(VideoEdit_editPopup->editLabelTxt,
					  PANEL_VALUE));
  speedStg[editnum] =  (int) xv_get(VideoEdit_editPopup->editSpeedTxt, PANEL_VALUE);
  audioStg[editnum] = currentAudio;
  
  PrintOnList(editnum, buf);
  xv_set(VideoEdit_editPopup->editScrollList,
	 PANEL_LIST_STRING, editnum, buf,
	 PANEL_LIST_FONT, editnum, font,
	 NULL);
  SetDuration();
  change = 1;
  UpdateHeader(currentFilename, 1);
}								    /* end function EditModify */


/*
 * Notify callback function for `editScrollList'.
 */
int
  List(item, string, client_data, op, event)
Panel_item	item;
char		*string;
Xv_opaque	client_data;
Panel_list_op	op;
Event		*event;
{
  int selection;
  char s[30];
  char audio[7];
  
  switch(op) 
  {
   case PANEL_LIST_OP_DESELECT:
    xv_set(VideoEdit_editPopup->editNewEditMsg, PANEL_LABEL_STRING, 
	   "Current Selection : New Edit", NULL);
    xv_set(VideoEdit_editPopup->editModButton, PANEL_INACTIVE, TRUE, NULL); /* Make the modify, delete buttons inactive */
    xv_set(VideoEdit_editPopup->editDelButton, PANEL_INACTIVE, TRUE, NULL);
    break;
    
   case PANEL_LIST_OP_SELECT:
    sscanf(string, "%d", &selection);
    editnum = selection - 1;
    sprintf(s, "Current Selection : Edit #%d", selection);
    xv_set(VideoEdit_editPopup->editNewEditMsg, PANEL_LABEL_STRING, s, NULL); /* Load appropriate values into the textfields */
    xv_set(VideoEdit_editPopup->editStartTxt, PANEL_VALUE, startframe[editnum], 
	   NULL);
    xv_set(VideoEdit_editPopup->editEndTxt, PANEL_VALUE, endframe[editnum], 
	   NULL);
    xv_set(VideoEdit_editPopup->editLabelTxt, PANEL_VALUE, label[editnum], NULL);
    switch (audioStg[editnum])
    {
     case Stereo:
      strcpy(audio, "Stereo");
      currentAudio = Stereo;
      break;
     case Right:
      strcpy(audio, "Right");
      currentAudio = Right;
      break;
     case Left:
      strcpy(audio, "Left");
      currentAudio = Left;
      break;
     case Mute:
      strcpy(audio, "Mute");
      currentAudio = Mute;
      break;
    }
    xv_set(VideoEdit_editPopup->editAudioMenuButton, PANEL_LABEL_STRING, audio, NULL);
    xv_set(VideoEdit_editPopup->editSpeedTxt, PANEL_VALUE, (int) speedStg[editnum], NULL);
    SetDuration();
    xv_set(VideoEdit_editPopup->editModButton,			    /* Make the modify, delete buttons active */
	   PANEL_INACTIVE, FALSE, NULL);
    xv_set(VideoEdit_editPopup->editDelButton,
	   PANEL_INACTIVE, FALSE, NULL);
    break;
  }
  return XV_OK;
}								    /* end function List */


/*
 * Notify callback function for `editDelAllButton'.
 * Delete all entries in the edit list
 */
void
  EditDeleteAll(item, event)
Panel_item      item;
Event           *event;
{
  int i;
  int result;
  char edit[20];

  if ((change) && (lines > 0))					    /* Check if unsaved changes exist in edit list */
  {
    result = notice_prompt(VideoEdit_editPopup->editPopup, NULL,
			   NOTICE_MESSAGE_STRINGS,
			   "Unsaved changes exist in the edit list.",
			   "Go ahead and delete all entries in the edit list? ",
			   NULL,
			   NOTICE_BUTTON_NO,	"Yes",
			   NOTICE_BUTTON_YES,	"No",
			   NULL);
    if (result == NOTICE_YES)
      return;
  }
  for (i=0; i < lines; i++) 
    xv_set(VideoEdit_editPopup->editScrollList,
	   PANEL_LIST_DELETE, 0,
	   NULL);
  lines  = 0;
  editnum = -1;
  PrintNumEdits();
  sprintf(edit, "No. of Edits : 0");
  
  xv_set(VideoEdit_editPopup->editNumEditMsg, PANEL_LABEL_STRING, 
	 edit, NULL);
  xv_set(VideoEdit_editPopup->editNewEditMsg, PANEL_LABEL_STRING, "Current Selection : New Edit", NULL);
  xv_set(VideoEdit_editPopup->editModButton, PANEL_INACTIVE, TRUE, NULL);
  xv_set(VideoEdit_editPopup->editDelButton, PANEL_INACTIVE, TRUE, NULL);
  sprintf(currentFilename, "Untitled");
  change = 0;
  UpdateHeader(currentFilename, 0);
}								    /* end function EditDeleteAll */


void UpdateHeader (char* documentName, int modified)
{
  char	label[MAXPATHLEN+32];
  
  if (modified == 1)
  {
    sprintf(label, "%s (modified)",
	    documentName);
  }
  else
  {
    sprintf(label, "%s", documentName);
  }
  xv_set(VideoEdit_editPopup->editPopup,			    /* Print the new label string on the title bar */
	 XV_LABEL, label, NULL);
}								    /* end function UpdateHeader */
