/*
 * Copyright (c) 1990, 1991, 1992 Stanford University
 *
 * Permission to use, copy, modify, and distribute this software and 
 * its documentation for any purpose is hereby granted without fee, provided
 * that (i) the above copyright notices and this permission notice appear in
 * all copies of the software and related documentation, and (ii) the name
 * Stanford may not be used in any advertising or publicity relating to
 * the software without the specific, prior written permission of
 * Stanford.
 * 
 * THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND, 
 * EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY 
 * WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.  
 *
 * IN NO EVENT SHALL STANFORD BE LIABLE FOR ANY SPECIAL, INCIDENTAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY KIND, OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER OR NOT
 * ADVISED OF THE POSSIBILITY OF DAMAGE, AND ON ANY THEORY OF LIABILITY,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 */

/* $Header: /Source/Media/collab/VideoObject/RCS/videoObj.c,v 0.29 92/09/02 17:09:43 drapeau Exp $ */
/* $Log:	videoObj.c,v $
 * Revision 0.29  92/09/02  17:09:43  drapeau
 * Fixed code for determining what terminal parameters to use for serial
 * communications, especially when parity and flow control are concerned.
 * Previous versions of this code did not handle flow control and parity
 * correctly.
 * 
 * Revision 0.28  92/09/01  16:58:10  drapeau
 * Cosmetic changes to the code to make it easier to read.  Also, added
 * function prototypes to make code more ANSI-compliant.
 * 
 * Revision 0.27  92/07/30  15:14:46  drapeau
 * Removed window system-specific code, to make library more portable.
 * Also, minor cosmetic changes to enhance code readability.
 * Also, minor changes to diagnostic messages to improve readability.
 * 
 * Revision 0.26  92/06/17  01:01:58  drapeau
 * Changed termio usage, used more portable "termio" structure and
 * associated calls instead of "termios".
 * 
 * Revision 0.25  91/09/30  17:08:58  lim
 * Implemented DevPing, DefaultPing, PlayerVerified.
 * 
 * Revision 0.24  91/08/24  17:50:28  lim
 * Implemented PrintDiagnostics.
 * 
 * Revision 0.23  91/08/24  13:52:13  lim
 * CheckImplemented corrected to return an enum Boolean.
 * 
 * Revision 0.22  91/08/24  13:39:20  lim
 * 1. Clear Marker() removed as part of video Object.
 * 
 * Revision 0.21  91/08/20  10:06:41  lim
 * Removed InterruptibleSegmentPlay from BuildVideoObject.
 * 
 * Revision 0.20  91/08/17  20:42:08  lim
 * 1. Removed 'interruptibleSegmentPlay' from video object. All stops and
 * pauses must be implemented such that they can interrupt segment play.
 * 
 * Revision 0.19  91/08/16  13:14:52  lim
 * Each Dev<Function> call now checks if theObject is valid.
 * 
 * Revision 0.18  91/08/08  17:19:37  lim
 * 1. CalcSpeed() - one parameter added, 'playMode'. If playMode is 1, it is used to calculate
 *    device speed for segment play. If playMode is 0, it is used to calculate device speed for
 *    normal play.
 * 2. QueryMedium() - one parameter added, 'result'. Used to return non-integer results.
 * 
 * Revision 0.17  91/08/07  13:30:24  lim
 * 1. Added instance pointer, "theObject" to all public function calls.
 * 2. Added file descriptor field to structure 'Config'.
 * 3. ErrorDecode is now a private function, and thus removed from the
 * VideoObject.
 * 4. BuildVideoObject now returns pointer to VideoObject.
 * 
 * Revision 0.16  91/08/02  13:02:53  lim
 * 1. No more GetDevices - Device configurations stored in each device's
 * videoObject, and names of all devices stored in allDevices.
 * 2. 'myVideo' is now globally accessible.
 * 3. SetSerialLine changed to take in device configuration structure as
 * first parameter rather than device name.
 * 4. BuildVideoObject changed to include new field in videoObject - the
 * configuration.
 * 5. Pause() has been removed from videoObject's functions.
 * 6. Added : GetDeviceMaxSpeed - gets the max speed on current device.
 *            CheckImplemented - checks if particular function available on
 *                               current device.
 * 7. DevSetDefaults now takes 4 parameters.
 * 
 * Revision 0.15  91/07/29  22:31:57  lim
 * When there is no eol char, the line will be set up to 
 * 1. Ignore carriage return IGNCR
 * 2. Non-canonical input
 * 3. c_cc[VMIN] is set to default 1, and c_cc[VTIME] to 0.
 * 
 * Revision 0.14  91/07/27  22:30:05  lim
 * Changed speed from double to int.
 * 
 * Revision 0.13  91/07/23  15:59:47  lim
 * 1. Changed static variables to globals.
 * 2. Introduced a new field, interruptibleSegmentPlay, in the .VideoConfig
 * file, to denote whether device supports interruptible segment playing.
 * 
 * 
 * Revision 0.12  91/07/19  14:16:11  lim
 * Added speed field to PlayFromTo.
 * 
 * Revision 0.11  91/07/17  22:50:55  lim
 * Removed function from VideoObject, "SetSerialPort". 
 * 
 * Revision 0.10  91/07/12  16:22:36  lim
 * Initial revision implementing VideoObject.
 *  */

#include "videoObj.h"


static Config DefaultConfig = 
{
  "NOTHING",
  "na",
  0,
  -1,
  -1,
  Even,
  Yes,
  Yes,
  -1,
  -1
  }
; 


static VideoObject DefaultObj = 
{
  &DefaultConfig,
  DefaultPlay,
  DefaultPlayFromTo,
  DefaultFastForward,
  DefaultReverse,
  DefaultCalcSpeed,
  DefaultPlayAtSpeedDir,
  DefaultStep,
  DefaultStill,
  DefaultStop,
  DefaultSetDefaults,
  DefaultSetAudio,
  DefaultSetVideo,
  DefaultSetAddMode,
  DefaultSetAddressDisplay,
  DefaultEject,
  DefaultPower,
  DefaultQueryFrame,
  DefaultQueryChapter,
  DefaultQueryAudio,
  DefaultQueryVideo,
  DefaultQueryMedium,
  DefaultQueryStatus,
  DefaultPing,
  DefaultRecord,
  DefaultRecordFromTo
  }
;


/* Sets up serial line for device selected.  
 * Returns file descriptor.
 */

int
  SetSerialLine(Config*	configuration,
		char*	newSerialLine)
{
  int ibaud;
  int bits;
  int serialLine;
  char diagMsg[30];
  static struct termio lineAttributes;
  
  if (configuration->fd>=0)					    /* Close previous serial line */
    close (configuration->fd);
  
  switch (configuration->baudRate) 
  {
   case 300:
    ibaud = B300;
    break;
   case 1200:
    ibaud = B1200;
    break;
   case 2400:
    ibaud = B2400;
    break;
   case 4800:
    ibaud = B4800;
    break;
   case 9600:
    ibaud = B9600;
    break;
   default:
    PrintDiagnostics("Illegal baud rate\n");
  }
  
  switch (configuration->charSize)
  {
   case 5:
    bits = CS5;
    break;
   case 6:
    bits = CS6;
    break;
   case 7:
    bits = CS7;
    break;
   case 8:
    bits = CS8;
    break;
   default:
    PrintDiagnostics("Illegal character size\n");
  }
  
  /* Open serial line */
  
  if (newSerialLine)						    /* newSerialLine overrides all other settings */
    strcpy(configuration->serialPort, newSerialLine);
  else
    PrintDiagnostics("No entry in defaults file.\n");
  
  if ((serialLine = open(configuration->serialPort, O_RDWR)) == -1) 
    PrintDiagnostics("VideoObject: Cannot open serial line for communications.\n");
  
  lineAttributes.c_iflag = 0;
  if (configuration->flowControl == Yes)
  {
    lineAttributes.c_iflag |= IXON;
    lineAttributes.c_iflag |= IXOFF;
  }
  if (configuration->igParError == Yes)
    lineAttributes.c_iflag |= IGNPAR;
  if (configuration->parity != NoParity)
    lineAttributes.c_iflag |= INPCK;
  if (configuration->eolChar < 0)
    lineAttributes.c_iflag |= IGNCR;
  
  lineAttributes.c_oflag = 0;
  lineAttributes.c_lflag = 0;
  if (configuration->eolChar >= 0)				    /* Canonical input processing only if there is a eol char */
    lineAttributes.c_lflag |= ICANON;
  lineAttributes.c_line = 0;
  
  if (configuration->eolChar >= 0)				    /* Player has end-of-line character */
    lineAttributes.c_cc[VEOL] = configuration->eolChar;
  else								    /* Player has no end-of-line character */
  {
    lineAttributes.c_cc[VMIN] = 1;
    lineAttributes.c_cc[VTIME] = 0;
  }
  
  lineAttributes.c_cflag = 0;
  lineAttributes.c_cflag |= ibaud;
  lineAttributes.c_cflag |= bits;
  lineAttributes.c_cflag |= CLOCAL;
  lineAttributes.c_cflag |= CREAD;
  if (configuration->parity != NoParity)
    lineAttributes.c_cflag |= PARENB;
  if (configuration->parity == Odd)
    lineAttributes.c_cflag |= PARODD;
  
  if (ioctl(serialLine,TCSETA,(char*)&lineAttributes) == -1) 
    PrintDiagnostics("Can't set line attributes.\n");
  
  configuration->fd = serialLine;
  return serialLine;
}								    /* end function SetSerialLine */


/* To test if player & serial port are correctly selected.
 * A serial line is set up so that it does non-canonical input
 * processing, with a timer set to TimeOut seconds, and the read returning
 * after the first character is read. 
 * The function DevPing() is called for the object installed.
 * If DevPing() returns a positive number, it means that the device is
 * correct, and PlayerVerified returns 1.  
 * If DevPing() returns 0 or a negative number, it means that the read 
 * has timed out, and either the player is too busy to respond, or the
 * wrong player has been selected, or the wrong serial port has been 
 * selected, or the wrong line attributes (eg baud rate) have been set. 
 * In this case, PlayerVerified returns -1.
 */
int
PlayerVerified(VideoObject*	theObject,
	       char*		sLine)
{
  struct termio	lineAttributes;
  int		numRead;
  int		eolChar;							    
  int		result = 0;
  char		out[5];
  
  eolChar = theObject->DevConfig->eolChar;			    /* We must keep tmp copy of eolChar */
  theObject->DevConfig->eolChar = -1;				    /* For non-canonical processing */
  
  if (SetSerialLine(theObject->DevConfig, sLine))
  {
    result = ioctl(theObject->DevConfig->fd,TCGETA,
		   (char*)&lineAttributes);
    lineAttributes.c_cc[VMIN] = 0;
    lineAttributes.c_cc[VTIME] = TimeOut*10;			    /* VTIME is in 0.1 seconds */
    result = ioctl(theObject->DevConfig->fd,TCSETA,
		   (char*)&lineAttributes);
    if (result == -1)
      return -1;
    
    numRead = DevPing(theObject);
    close(theObject->DevConfig->fd);
    theObject->DevConfig->fd = -1;				    /* Reset file descriptor */
    theObject->DevConfig->eolChar = eolChar;			    /* Reset eolChar */
    
    if (numRead > 0)
      return 1;
    else
      return -1;
  }
  else
    return -1;
}


/* Builds the Video Object for use by application.
 * A copy of the defined functions in the new device ...
 * ... is made to theObject.
 */

VideoObject* 
  BuildVideoObject(VideoObject* videoObj)
{
  VideoObject* newObj;
  
  newObj = (VideoObject*) malloc(sizeof(VideoObject));
  newObj->DevConfig = (Config*) malloc(sizeof(Config));
  
  strcpy(newObj->DevConfig->serialPort, videoObj->DevConfig->serialPort);
  strcpy(newObj->DevConfig->modelName, videoObj->DevConfig->modelName);
  newObj->DevConfig->baudRate = videoObj->DevConfig->baudRate;
  newObj->DevConfig->eolChar = videoObj->DevConfig->eolChar;
  newObj->DevConfig->charSize = videoObj->DevConfig->charSize;
  newObj->DevConfig->parity = videoObj->DevConfig->parity;
  newObj->DevConfig->igParError = videoObj->DevConfig->igParError;
  newObj->DevConfig->maxSpeed = videoObj->DevConfig->maxSpeed;
  newObj->DevConfig->fd = videoObj->DevConfig->fd;
  
  if (videoObj->DevPlayPtr)
    newObj->DevPlayPtr =  videoObj->DevPlayPtr; 
  else
    newObj->DevPlayPtr =  DefaultObj.DevPlayPtr; 
  
  if (videoObj->DevPlayFromToPtr)
    newObj->DevPlayFromToPtr =  videoObj->DevPlayFromToPtr; 
  else
    newObj->DevPlayFromToPtr =  DefaultObj.DevPlayFromToPtr; 
  
  if (videoObj->DevFastForwardPtr)
    newObj->DevFastForwardPtr =  videoObj->DevFastForwardPtr; 
  else
    newObj->DevFastForwardPtr =  DefaultObj.DevFastForwardPtr; 
  
  if (videoObj->DevReversePtr)
    newObj->DevReversePtr =  videoObj->DevReversePtr; 
  else
    newObj->DevReversePtr =  DefaultObj.DevReversePtr; 
  
  if (videoObj->DevCalcSpeedPtr)
    newObj->DevCalcSpeedPtr =  videoObj->DevCalcSpeedPtr; 
  else
    newObj->DevCalcSpeedPtr =  DefaultObj.DevCalcSpeedPtr; 
  
  if (videoObj->DevPlayAtSpeedDirPtr)
    newObj->DevPlayAtSpeedDirPtr =  videoObj->DevPlayAtSpeedDirPtr; 
  else
    newObj->DevPlayAtSpeedDirPtr =  DefaultObj.DevPlayAtSpeedDirPtr; 
  
  if (videoObj->DevStepPtr)
    newObj->DevStepPtr =  videoObj->DevStepPtr; 
  else
    newObj->DevStepPtr =  DefaultObj.DevStepPtr; 
  
  if (videoObj->DevStillPtr)
    newObj->DevStillPtr =  videoObj->DevStillPtr; 
  else
    newObj->DevStillPtr =  DefaultObj.DevStillPtr; 
  
  if (videoObj->DevStopPtr)
    newObj->DevStopPtr =  videoObj->DevStopPtr; 
  else
    newObj->DevStopPtr =  DefaultObj.DevStopPtr; 
  
  if (videoObj->DevSetDefaultsPtr)
    newObj->DevSetDefaultsPtr =  videoObj->DevSetDefaultsPtr; 
  else
    newObj->DevSetDefaultsPtr =  DefaultObj.DevSetDefaultsPtr; 
  
  if (videoObj->DevSetAudioPtr)
    newObj->DevSetAudioPtr =  videoObj->DevSetAudioPtr; 
  else
    newObj->DevSetAudioPtr =  DefaultObj.DevSetAudioPtr; 
  
  if (videoObj->DevSetVideoPtr)
    newObj->DevSetVideoPtr =  videoObj->DevSetVideoPtr;
  else
    newObj->DevSetVideoPtr =  DefaultObj.DevSetVideoPtr;
  
  if (videoObj->DevSetAddModePtr)
    newObj->DevSetAddModePtr =  videoObj->DevSetAddModePtr;
  else
    newObj->DevSetAddModePtr =  DefaultObj.DevSetAddModePtr;
  
  if (videoObj->DevSetAddressDisplayPtr)
    newObj->DevSetAddressDisplayPtr =  videoObj->DevSetAddressDisplayPtr; 
  else
    newObj->DevSetAddressDisplayPtr =  DefaultObj.DevSetAddressDisplayPtr; 
  
  if (videoObj->DevEjectPtr)
    newObj->DevEjectPtr =  videoObj->DevEjectPtr;
  else
    newObj->DevEjectPtr =  DefaultObj.DevEjectPtr;
  
  if (videoObj->DevPowerPtr)
    newObj->DevPowerPtr =  videoObj->DevPowerPtr; 
  else
    newObj->DevPowerPtr =  DefaultObj.DevPowerPtr; 
  
  if (videoObj->DevQueryFramePtr)
    newObj->DevQueryFramePtr =  videoObj->DevQueryFramePtr; 
  else
    newObj->DevQueryFramePtr =  DefaultObj.DevQueryFramePtr; 
  
  if (videoObj->DevQueryChapterPtr)
    newObj->DevQueryChapterPtr =  videoObj->DevQueryChapterPtr; 
  else
    newObj->DevQueryChapterPtr =  DefaultObj.DevQueryChapterPtr; 
  
  if (videoObj->DevQueryAudioPtr)
    newObj->DevQueryAudioPtr =  videoObj->DevQueryAudioPtr; 
  else
    newObj->DevQueryAudioPtr =  DefaultObj.DevQueryAudioPtr; 
  
  if (videoObj->DevQueryVideoPtr)
    newObj->DevQueryVideoPtr =  videoObj->DevQueryVideoPtr; 
  else
    newObj->DevQueryVideoPtr =  DefaultObj.DevQueryVideoPtr; 
  
  if (videoObj->DevQueryMediumPtr)
    newObj->DevQueryMediumPtr =    videoObj->DevQueryMediumPtr; 
  else
    newObj->DevQueryMediumPtr =    DefaultObj.DevQueryMediumPtr; 
  
  if (videoObj->DevQueryStatusPtr)
    newObj->DevQueryStatusPtr =  videoObj->DevQueryStatusPtr;
  else
    newObj->DevQueryStatusPtr =  DefaultObj.DevQueryStatusPtr;
  
  if (videoObj->DevPingPtr)
    newObj->DevPingPtr =  videoObj->DevPingPtr;
  else
    newObj->DevPingPtr =  DefaultObj.DevPingPtr;
  
  if (videoObj->DevRecordPtr)
    newObj->DevRecordPtr =  videoObj->DevRecordPtr;
  else
    newObj->DevRecordPtr =  DefaultObj.DevRecordPtr;
  
  if (videoObj->DevRecordFromToPtr)
    newObj->DevRecordFromToPtr =  videoObj->DevRecordFromToPtr;
  else
    newObj->DevRecordFromToPtr =  DefaultObj.DevRecordFromToPtr;
  
  return (newObj);
}								    /* end function BuildVideoObject */



/* Returns the max speed for the current video device */
int
  GetDeviceMaxSpeed(VideoObject* theObject)
{
  if (theObject)
    return (theObject->DevConfig->maxSpeed);
  else 
    return -1;
}


/* Checks to see if video object function 'func' has been implemented...
 * ... in current video object */
enum Boolean
  CheckImplemented(VideoObject*	theObject,
		   int		func)
{
  switch (func)
  {
   case PlayFunction:
    if (theObject->DevPlayPtr)
      return Yes;
    else
      return No;
   case PlayFromToFunction:
    if (theObject->DevPlayFromToPtr)
      return Yes;
    else
      return No;
   case FastForwardFunction:
    if (theObject->DevFastForwardPtr)
      return Yes;
    else
      return No;
   case ReverseFunction:
    if (theObject->DevReversePtr)
      return Yes;
    else
      return No;
   case CalcSpeedFunction:
    if (theObject->DevCalcSpeedPtr)
      return Yes;
    else
      return No;
   case PlayAtSpeedDirFunction:
    if (theObject->DevPlayAtSpeedDirPtr)
      return Yes;
    else
      return No;
   case StepFunction:
    if (theObject->DevStepPtr)
      return Yes;
    else
      return No;
   case StillFunction:
    if (theObject->DevStillPtr)
      return Yes;
    else
      return No;
   case StopFunction:
    if (theObject->DevStopPtr)
      return Yes;
    else
      return No;
   case SetDefaultsFunction:
    if (theObject->DevSetDefaultsPtr)
      return Yes;
    else
      return No;
   case SetAudioFunction:
    if (theObject->DevSetAudioPtr)
      return Yes;
    else
      return No;
   case SetVideoFunction:
    if (theObject->DevSetVideoPtr)
      return Yes;
    else
      return No;
   case SetAddModeFunction:
    if (theObject->DevSetAddModePtr)
      return Yes;
    else
      return No;
   case SetAddressDisplayFunction:
    if (theObject->DevSetAddressDisplayPtr)
      return Yes;
    else
      return No;
   case EjectFunction:
    if (theObject->DevEjectPtr)
      return Yes;
    else
      return No;
   case PowerFunction:
    if (theObject->DevPowerPtr)
      return Yes;
    else
      return No;
   case QueryFrameFunction:
    if (theObject->DevQueryFramePtr)
      return Yes;
    else
      return No;
   case QueryChapterFunction:
    if (theObject->DevQueryChapterPtr)
      return Yes;
    else
      return No;
   case QueryAudioFunction:
    if (theObject->DevQueryAudioPtr)
      return Yes;
    else
      return No;
   case QueryVideoFunction:
    if (theObject->DevQueryVideoPtr)
      return Yes;
    else
      return No;
   case QueryMediumFunction:
    if (theObject->DevQueryMediumPtr)
      return Yes;
    else
      return No;
   case QueryStatusFunction:
    if (theObject->DevQueryStatusPtr)
      return Yes;
    else
      return No;
   case PingFunction:
    if (theObject->DevPingPtr)
      return Yes;
    else
      return No;    
   case RecordFunction:
    if (theObject->DevRecordPtr)
      return Yes;
    else
      return No;    
   case RecordFromToFunction:
    if (theObject->DevRecordFromToPtr)
      return Yes;
    else
      return No;    
   default:
    return -1;
  }
}




/* Functions of the Video Object */
/* All applications call one of the following generic Dev<Function>s ... 
 * ... which call the appropriate function for the specific device. */

int 
  DevPlay(VideoObject* theObject)
{
  if (theObject)
    return ((*theObject->DevPlayPtr) (theObject));  
  else
  {
    fprintf(stderr, "No device driver installed\n");
    return -1;
  }
  
}

int 
  DevPlayFromTo(VideoObject*	theObject,
		int		startFrame,
		int		endFrame,
		int		speed)
{
  if (theObject)
    return ((*theObject->DevPlayFromToPtr) (theObject, startFrame, endFrame, speed));
  else
  {
    fprintf(stderr, "No device driver installed\n");
    return -1;
  }
}


int 
  DevFastForward(VideoObject* theObject)
{
  if (theObject)
    return ((*theObject->DevFastForwardPtr) (theObject));
  else
  {
    fprintf(stderr, "No device driver installed\n");
    return -1;
  }
}



int 
  DevReverse(VideoObject* theObject)
{
  if (theObject)
    return ((*theObject->DevReversePtr) (theObject));
  else
  {
    fprintf(stderr, "No device driver installed\n");
    return -1;
  }
}


int
  DevCalcSpeed(VideoObject*	theObject,
	       int		value,
	       int		mode)
{
  if (theObject)
    return ((*theObject->DevCalcSpeedPtr) (theObject, value, mode));
  else
  {
    fprintf(stderr, "No device driver installed\n");
    return -1;
  }
}


int 
  DevPlayAtSpeedDir(VideoObject*	theObject,
		    int			framesPerSecond,
		    enum Direction	direction)
{
  if (theObject)
    return ((*theObject->DevPlayAtSpeedDirPtr) (theObject, framesPerSecond, direction));
  else
  {
    fprintf(stderr, "No device driver installed\n");
    return -1;
  }
}

int DevStep(VideoObject*	theObject,
	    enum Direction	direction)
{
  if (theObject)
    return ((*theObject->DevStepPtr) (theObject, direction));
  else
  {
    fprintf(stderr, "No device driver installed\n");
    return -1;
  }
}

int DevStill(VideoObject* theObject)
{
  if (theObject)
    return ((*theObject->DevStillPtr) (theObject));
  else
  {
    fprintf(stderr, "No device driver installed\n");
    return -1;
  }
}


int DevStop(VideoObject* theObject)
{
  if (theObject)
    return ((*theObject->DevStopPtr) (theObject));
  else
  {
    fprintf(stderr, "No device driver installed\n");
    return -1;
  }
}


int DevSetDefaults(VideoObject*	theObject,
		   int		audio,
		   int		addressingMode,
		   int		addressDisplayOnOff,
		   int		displayMode)
{
  if (theObject)
    return ((*theObject->DevSetDefaultsPtr) (theObject, audio, addressingMode, 
					     addressDisplayOnOff, displayMode));
  else
  {
    fprintf(stderr, "No device driver installed\n");
    return -1;
  }
}


int DevSetAudio(VideoObject*	theObject,
		int		setting)
{
  if (theObject)
    return ((*theObject->DevSetAudioPtr) (theObject, setting));
  else
  {
    fprintf(stderr, "No device driver installed\n");
    return -1;
  }
}


int DevSetVideo(VideoObject*	theObject,
		int		setting)
{  
  if (theObject)
    return ((*theObject->DevSetVideoPtr) (theObject, setting));
  else
  {
    fprintf(stderr, "No device driver installed\n");
    return -1;
  }
}

int DevSetAddMode(VideoObject*	theObject,
		  int		mode)
{
  if (theObject)
    return ((*theObject->DevSetAddModePtr) (theObject, mode));
  else
  {
    fprintf(stderr, "No device driver installed\n");
    return -1;
  }
}


int DevSetAddressDisplay(VideoObject*	theObject,
			 int		onOff,
			 int		mode)
{
  if (theObject)
    return ((*theObject->DevSetAddressDisplayPtr) (theObject, onOff, mode));
  else
  {
    fprintf(stderr, "No device driver installed\n");
    return -1;
  }
}


int DevEject(VideoObject* theObject)
{
  if (theObject)
    return ((*theObject->DevEjectPtr) (theObject));
  else
  {
    fprintf(stderr, "No device driver installed\n");
    return -1;
  }
}


int DevPower(VideoObject*	theObject,
	     int		mode)
{
  if (theObject)
    return ((*theObject->DevPowerPtr) (theObject, mode));
  else
  {
    fprintf(stderr, "No device driver installed\n");
    return -1;
  }
}


int DevQueryFrame(VideoObject* theObject)
{
  if (theObject)
    return ((*theObject->DevQueryFramePtr) (theObject));
  else
  {
    fprintf(stderr, "No device driver installed\n");
    return -1;
  }
}


int DevQueryChapter(VideoObject* theObject)
{
  if (theObject)
    return ((*theObject->DevQueryChapterPtr) (theObject));
  else
  {
    fprintf(stderr, "No device driver installed\n");
    return -1;
  }
}


int DevQueryAudio(VideoObject* theObject)
{
  if (theObject)
    return ((*theObject->DevQueryAudioPtr) (theObject));
  else
  {
    fprintf(stderr, "No device driver installed\n");
    return -1;
  }
}


int DevQueryVideo(VideoObject* theObject)
{
  if (theObject)
    return ((*theObject->DevQueryVideoPtr) (theObject));
  else
  {
    fprintf(stderr, "No device driver installed\n");
    return -1;
  }
}


int DevQueryMedium(VideoObject*	theObject,
		   char*	result)
{
  if (theObject)
    return ((*theObject->DevQueryMediumPtr) (theObject, result));
  else
  {
    fprintf(stderr, "No device driver installed\n");
    return -1;
  }
}


int DevQueryStatus(VideoObject*	theObject)
{
  if (theObject)
    return ((*theObject->DevQueryStatusPtr) (theObject));
  else
  {
    fprintf(stderr, "No device driver installed\n");
    return -1;
  }
}


int DevPing(VideoObject* theObject)
{
  if (theObject)
    return ((*theObject->DevPingPtr) (theObject));
  else
  {
    fprintf(stderr, "No device driver installed\n");
    return -1;
  }
}


int 
  DevRecord(VideoObject* theObject)
{
  if (theObject)
    return ((*theObject->DevRecordPtr) (theObject));
  else
  {
    fprintf(stderr, "No Record function installed.\n");
    return -1;
  }
}								    /* end function DevRecord */



int 
  DevRecordFromTo(VideoObject* theObject,
		  int startFrame,
		  int endFrame,
		  int speed)
{
  if (theObject)
    return ((*theObject->DevRecordFromToPtr) (theObject, startFrame, endFrame, speed));
  else
  {
    fprintf(stderr, "No RecordFromTo function installed.\n");
    return -1;
  }
}								    /* end function DevRecordFromTo */


/* Default Functions of the Video Object */
/* Functions that are not implemented for the driver and for ...
 * ... which no dummy function has been written by the ...
 * ... the application programmer will use the default ...
 * ... functions.
 */

int 
  DefaultPlay(VideoObject* theObject)
{
  fprintf(stderr, "DevPlay has not been implemented.\n");
  return Ok;
}

int 
  DefaultPlayFromTo(VideoObject*	theObject,
		    int			startFrame,
		    int			endFrame,
		    int			speed)
{
  fprintf(stderr, "DevPlayFromTo has not been implemented.\n");
  return Ok;
}


int 
  DefaultFastForward(VideoObject* theObject)
{
  fprintf(stderr, "DevFastForward has not been implemented.\n");
  return Ok;
}



int 
  DefaultReverse(VideoObject* theObject)
{
  fprintf(stderr, "DevReverse has not been implemented.\n");
  return Ok;
}


int
  DefaultCalcSpeed(VideoObject*	theObject,
		   int		value,
		   int		mode)
{
  fprintf(stderr, "DevCalcSpeed has not been implemented.\n");
  return Ok;
}


int 
  DefaultPlayAtSpeedDir(VideoObject*	theObject,
			int		framesPerSecond,
			enum Direction	direction)
{
  fprintf(stderr, "DevPlayAtSpeedDir has not been implemented.\n");
  return Ok;
}


int DefaultStep(VideoObject*	theObject,
		enum Direction	direction)
{
  fprintf(stderr, "DevStep has not been implemented.\n");
  return Ok;
}


int DefaultStill(VideoObject* theObject)
{
  fprintf(stderr, "DevStill has not been implemented.\n");
  return Ok;
}

int DefaultStop(VideoObject* theObject)
{
  fprintf(stderr, "DevStop has not been implemented.\n");
  return Ok;
}

int DefaultSetDefaults(VideoObject*	theObject,
		       int		audio,
		       int		addressingMode,
		       int		addressDisplayOnOff,
		       int		displayMode)
{
  fprintf(stderr, "DevSetDefaults has not been implemented.\n");
  return Ok;
}

int DefaultSetAudio(VideoObject*	theObject,
		    int			setting)
{
  fprintf(stderr, "DevSetAudio has not been implemented.\n");
  return Ok;
}

int DefaultSetVideo(VideoObject*	theObject,
		    int			setting)
{  
  fprintf(stderr, "DevSetVideo has not been implemented.\n");
  return Ok;
}

int DefaultSetAddMode(VideoObject*	theObject,
		      int 		mode)
{
  fprintf(stderr, "DevSetAddMode has not been implemented.\n");
  return Ok;
}

int DefaultSetAddressDisplay(VideoObject*	theObject,
			     int		onOff,
			     int		mode)
{
  fprintf(stderr, "DevSetAddressDisplay has not been implemented.\n");
  return Ok;
}



int DefaultEject(VideoObject* theObject)
{
  fprintf(stderr, "DevEject has not been implemented.\n");
  return Ok;
}


int DefaultPower(VideoObject*	theObject,
		 int		mode)
{
  fprintf(stderr, "DevPower has not been implemented.\n");
  return Ok;
}


int DefaultQueryFrame(VideoObject* theObject)
{
  fprintf(stderr, "DevQueryFrame has not been implemented.\n");
  return Ok;
}

int DefaultQueryChapter(VideoObject* theObject)
{
  fprintf(stderr, "DevQueryChapter has not been implemented.\n");
  return Ok;
}

int DefaultQueryAudio(VideoObject* theObject)
{
  fprintf(stderr, "DevQueryAudio has not been implemented.\n");
  return Ok;
}

int DefaultQueryVideo(VideoObject* theObject)
{
  fprintf(stderr, "DevQueryVideo has not been implemented.\n");
  return Ok;
}

int DefaultQueryMedium(VideoObject*	theObject,
		       char*		result)
{
  fprintf(stderr, "DevQueryMedium has not been implemented.\n");
  return Ok;
}

int DefaultQueryStatus(VideoObject* theObject)
{
  fprintf(stderr, "DevQueryStatus has not been implemented.\n");
  return Ok;
}

int DefaultPing(VideoObject* theObject)
{
  fprintf(stderr, "DevPing has not been implemented.\n");
  return Ok;
}



int DefaultRecord(VideoObject* theObject)
{
  fprintf(stderr, "DevRecord has not been implemented.\n");
  return Ok;
}


int 
  DefaultRecordFromTo(VideoObject* theObject,
		      int startFrame,
		      int endFrame,
		      int speed)
{
  fprintf(stderr, "DevRecordFromTo has not been implemented.\n");
  return Ok;
}
