/* Copyright (c) 1994 Gregory P. Ward.  All rights reserved.
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose, without fee, and without written
 * agreement is hereby granted, provided that the above copyright
 * notice and the following two paragraphs appear in all copies of
 * this software.  
 *
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
 * OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF THE
 * AUTHOR HAS BEEN ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * THE AUTHOR SPECIFICALLY DISCLAIMS ANY WARRANTIES, INCLUDING, BUT
 * NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER
 * IS ON AN "AS IS" BASIS, AND THE AUTHOR HAS NO OBLIGATION TO PROVIDE
 * MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 *
 */

/* ----------------------------- MNI Header -----------------------------------
@NAME       : parseargs.c
@INPUT      : 
@OUTPUT     : 
@RETURNS    : 
@DESCRIPTION: Routines for parsing the command line arguments, principally
              ParseCmdLine().
@METHOD     : 
@GLOBALS    : 
@CALLS      : 
@CREATED    : 94/6/20?, Greg Ward (ripped out of gl_mpeg.c)
@MODIFIED   : 
---------------------------------------------------------------------------- */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <malloc.h>
#include <errno.h>
#include "ParseArgv.h"
#include "mpeg.h"
#include "gl_mpeg.h"


/* ----------------------------- MNI Header -----------------------------------
@NAME       : usage
@INPUT      : progname - the name of the program (eg. from argv[0])
@OUTPUT     : (none)
@RETURNS    : (void) (does not return - calls exit())
@DESCRIPTION: Tells the luser about the "-help" option and aborts.
@METHOD     : 
@GLOBALS    : 
@CALLS      : 
@CREATED    : 
@MODIFIED   : 
---------------------------------------------------------------------------- */
void usage(char *progname)
{
   char   *nameonly;		/* strip directory from progname */

   nameonly = strrchr (progname, '/');
   if (nameonly == NULL)
   {
      nameonly = progname;
   }

   fprintf (stderr, "Usage: %s [filename] [options]\n", nameonly);
   fprintf (stderr, "    if no filename given, reads from standard input\n");
   fprintf (stderr, "    %s -help for more help\n", nameonly);
   exit (1);
}


char *SplitString (char *s, int *NumWords, char ***Words)
{
   char     delims [] = " \t\n\r";
   char    *copys;
   int      slen;
   int      i;
   
   if (s == NULL) return (NULL);

   slen = strlen (s);
   copys = (char *) malloc (slen+1);
   strcpy (copys, s);
/*   
   printf ("SplitString:\n");
   printf ("  input string = >%s<\n", copys);
*/
   /* Can't POSSIBLY be more than slen words in s, so allocate that 
    * much room for the array of char *'s
    */
   *Words = (char **) malloc (slen); 
   *NumWords = 0;

   (*Words) [0] = strtok (copys, delims);
   i = 1;
   while ((*Words) [i++] = strtok (NULL, delims))
   {
   }

   *NumWords = i-1;
/*
   printf ("  broken up (%d): ", *NumWords);
   for (i = 0; i < *NumWords; i++)
      printf (">%s< ", (*Words)[i]);
   printf ("\n");
*/
   return (copys);
   
}



/* ----------------------------- MNI Header -----------------------------------
@NAME       : SearchStringTable
@INPUT      : s - string to search for
              Table - table of strings in which to search
	      TableSize - number of strings in Table[]
	      MinUniqueLen - the minimum length, over all strings in
                             the table, required to determine uniqueness
	      ErrMsg - a format string for fprintf to print if s is not
	               found in Table.  ErrMsg should contain a single
		       "%s", for which the search string s will be
		       substituted.
@OUTPUT     : 
@RETURNS    : The position of s in Table, or -1 if not found.  Will print
              an error message (constructed from ErrMsg and s) to stderr
	      if s not found in Table.
@DESCRIPTION: Searches a list of strings for a specific string.  Makes 
              a half-hearted attempt at only needing to match enough
	      of the string to be unique, although the caller must
	      supply the minimum unique length of all strings in
	      the table.  (To override this feature, just make 
	      MinUniqueLen really big, i.e. larger than the length
	      of any string in the table.)
@METHOD     : 
@GLOBALS    : (none)
@CALLS      : 
@CREATED    : 94/6/22, Greg Ward
@MODIFIED   : 94/7/29, GW: added MinUniqueLen arg and changed strcmp
                           to strncmp
@COMMENTS   : This should be modified so that s just has to have enough
              characters to uniquely match one of the strings in Table[].
---------------------------------------------------------------------------- */
int SearchStringTable (char *s, char *Table[], int TableSize, 
		       int MinUniqueLen, char *ErrMsg)
{
   int	i;

   for (i = 0; i < TableSize; i++)
   {
      if (strncmp (s, Table [i], max (strlen (s), MinUniqueLen)) == 0)
      {
	 return (i);
      }
   }

   if (i == TableSize)
   {
      fprintf (stderr, ErrMsg, s);
      return (-1);
   }
}     /* SearchStringTable () */





/* ----------------------------- MNI Header -----------------------------------
@NAME       : ParseCmdLine
@INPUT      : pargc - pointer to number of arguments originally in argv[]
              argv - the command line arguments
@OUTPUT     : *pargc - modified to reflect the removal of switches and
                       options and whatnot
@RETURNS    : (void)
@DESCRIPTION: Calls ParseArgv() to parse the various options from the
              command line.  Checks for consistency amongst those
	      options.  If any arguments are left, assumes that the
	      first one is the name of an MPEG file, puts the
	      filename in the global MPEGfilename, and opens the file
	      as MPEGfile.  Also checks for a few inconsistencies
	      in the arguments, and prints a warning to stderr if
	      they're found.
@METHOD     : 
@GLOBALS    : DisplayName
              BufferType
	      BufferFallback
              MaxMemory
@CALLS      : ParseArgv
@CREATED    : 94/6/??, Greg Ward
@MODIFIED   : 94/6/22, GW: took out the old (rather bletcherous) argument-
                 parsing code, and replaced with a ParseArgv table and call.
              94/6/27, GW: greatly reduced use of global variables by
	         adding the Movie argument
	      94/7/3, GW: added -keeptemp option, and moved inconsistency-
	         checking from InitializeMovie() to here.
	      94/7/7, GW: many changes relating to new structure of
                 movie option flags; removed some stupid MPEG-library
		 options (dithering, etc.)
---------------------------------------------------------------------------- */
void ParseCmdLine (int *pargc, char *argv[], MovieState *Movie)
{
   WindowDesc   *WinInfo = &Movie->Window; /* just to save keystrokes... */
   int           envargc;
   char        **envargv;
   int           arg;

   static char  *buffer_name = NULL;
   static char  *fallback_name = NULL;
   static double zoom = 1.0;
   static int    framedelay = 6;
   static StyleEnum playstyle = FORWARD;
   static int    continuous=0,
                 doublebuff=0,
                 fullscreen=0,
                 preview=1;

   static ArgvInfo ArgTable [] = 
   {
      /* Buffering options: */

      { "-buffer", ARGV_STRING, NULL, (char *) &buffer_name,
	"how to buffer the movie's frames: either none, disk, or memory" },
      { "-fallback", ARGV_STRING, NULL, (char *) &fallback_name,
	"buffering method to fall back on if memory is exhausted" },
      { "-maxmemory", ARGV_INT, (char *) 1, (char *) &MaxMemory,
	"maximum allowed memory (in megabytes) to allocate when buffering to memory" },
      { "-keeptemp", ARGV_CONSTANT, (char *) 1, (char *) &KeepTempFlag,
	"keep temporary file (only for disk buffering)" },
      { "-nokeeptemp", ARGV_CONSTANT, (char *) 0, (char *) &KeepTempFlag,
	"do not keep temporary file when disk buffering (default)" },

      /* Display/playback options */

      { "-display", ARGV_STRING, NULL, (char *) &DisplayName,
        "name of GL server to display on" },
      { "-delay", ARGV_INT, (char *) 1, (char *) &framedelay,
	"hundredths of a second to delay between each frame" },

      { "-forwards", ARGV_CONSTANT, (char *) FORWARD, (char *) &playstyle,
	"play the movie forwards only (default)" },
      { "-backwards", ARGV_CONSTANT, (char *) BACKWARD, (char *) &playstyle,
	"play the movie backwards only" },
      { "-rock", ARGV_CONSTANT, (char *) ROCK, (char *) &playstyle,
	"play the movie forwards and backwards (rock it) (implies -continuous)" },

      { "-onceonly", ARGV_CONSTANT, (char *) 0, (char *) &continuous,
        "pause between each playback of the movie (default)" },
      { "-continuous", ARGV_CONSTANT, (char *) 1, (char *) &continuous,
        "show movie without pausing between repetitions" },
      { "-nocontinous", ARGV_CONSTANT, (char *) 0, (char *) &continuous,
        "equivalent to -onceonly" },

      { "-doublebuffer", ARGV_CONSTANT, (char *) 1, (char *) &doublebuff,
	"enable double buffering (default)" },
      { "-nodoublebuffer", ARGV_CONSTANT, (char *) 0, (char *) &doublebuff,
	"disable double buffering" },
      { "-singlebuffer", ARGV_CONSTANT, (char *) 0, (char *) &doublebuff,
	"equivalent to -nodoublebuffer" },

      { "-zoom", ARGV_FLOAT, (char *) 1, (char *) &zoom,
	"zoom factor to use when displaying movie (default: 1.0)" },
      { "-fullscreen", ARGV_CONSTANT, (char *) 1, (char *) &fullscreen,
	"display the movie in a full-screen, unbordered window" },
      { "-nofullscreen", ARGV_CONSTANT, (char *) 0, (char *) &fullscreen,
	"don't display the movie full-screen (default)" },

/*
      { "-preview", ARGV_CONSTANT, (char *) 1, (char *) &preview,
	"display the frames as they are decoded (default)" },
      { "-nopreview", ARGV_CONSTANT, (char *) 0, (char *) &preview,
	"do not display the frames as they are decoded" },
*/

      /* Output options */

      { "-quiet", ARGV_CONSTANT, (char *) 1, (char *) &QuietFlag,
	"don't display any useful, informative messages" },
      { "-verbose", ARGV_CONSTANT, (char *) 0, (char *) &QuietFlag,
	"display useful, informative messages" },


      { "-help", ARGV_HELP, 0, 0, "get help information" },
      { NULL, ARGV_END, 0, 0, NULL }
   };

   if (SplitString (getenv ("GL_MPEG_OPTS"), &envargc, &envargv) != NULL)
   {
      if (ParseArgv (&envargc, envargv, ArgTable, 
		     ARGV_NO_DEFAULTS|ARGV_DONT_SKIP_FIRST_ARG))
      {
	 usage (argv[0]);
      }

      if (envargc > 0)
      {
	 fprintf (stderr, "Unknown options in $GL_MPEG_OPTS: ");
	 while (*envargv)
	 {
	    fprintf (stderr, "%s ", *envargv);
	    envargv++;
	 }
	 fprintf (stderr, "\n");
      }
   }

   if (ParseArgv (pargc, argv, ArgTable, ARGV_NO_DEFAULTS))
      usage (argv[0]);

#ifdef DEBUG
   printf ("ParseCmdLine:\n");
   printf ("  Display name: %s\n", DisplayName);
   printf ("  Quiet = %d\n", QuietFlag);
   printf ("  MaxMemory = %d\n", MaxMemory);
   printf ("  Movie options: continuous=%d    doublebuff=%d   fullscreen=%d\n",
	   continuous, doublebuff, fullscreen);
   printf ("                 zoom factor = %g\n", zoom);
   printf ("                 play style=%s\n", StyleNames[playstyle]);
   
#endif

   if (buffer_name != NULL)
   {
      int  i;

      i = SearchStringTable (buffer_name, BufferNames, 
			     NUM_BUFFER_TYPES, BUFFER_UNIQUE_LEN,
			     "Unknown buffering method: %s\n");
      if (i < 0) 
	 usage (argv [0]);
      else
	 BufferType = i;
   }

   if (fallback_name != NULL)
   {
      int  i;

      i = SearchStringTable (fallback_name, BufferNames,
			     NUM_BUFFER_TYPES, BUFFER_UNIQUE_LEN,
			     "Unknown buffering fallback method: %s\n");
      if (i < 0) 
	 usage (argv [0]);
      else
	 BufferFallback = i;
   }


   /* Check for a few possible inconsistencies in the command line arguments */
   
   if (BufferType != DISK_BUFFER && KeepTempFlag)
   {
      fprintf (stderr, "Warning: -keeptemp ignored unless disk buffering is used\n");
      KeepTempFlag = FALSE;
   }

   if (BufferType == MEMORY_BUFFER && MaxMemory == 0)
   {
      fprintf (stderr, "Warning: using -buffer memory without specifying -maxmemory is not recommended\n");
   }

   if (BufferType == NO_BUFFER && playstyle != FORWARD)
   {
      fprintf (stderr, "Warning: must buffer frames to use -backwards or -rock\n");
      playstyle = FORWARD;
   }

/*
   if (BufferType == NO_BUFFER && !preview)
   {
      fprintf (stderr, "Warning: specifying -nopreview without buffering will prevent the movie from being seen! (overriding)\n");
      preview = TRUE;
   }
*/
   
   if (fullscreen && zoom != 1.0)
   {
      fprintf (stderr, "Warning: zoom factors ignored with -fullscreen option\n");
   }

   /* Now, conditions that aren't errors, but where one argument 
    * implies another that is NOT the default (e.g. -rock implies 
    * -continuous, but the default is -nocontinuous)
    */

   if (playstyle == ROCK)
   {
      continuous = TRUE;
   }

   /* Now copy the various options set by ParseArgv to Movie */

   Movie->FrameDelay = framedelay;
   Movie->PlayStyle = playstyle;

   Movie->Options.Continuous = continuous;
   Movie->Options.DoubleBuff = doublebuff;
   Movie->Options.FullScreen = fullscreen;
   Movie->Options.Preview = preview;

   WinInfo->ZoomX = WinInfo->ZoomY = (float) zoom;

   /*
    * And loop through the remaining options.  If any of them start
    * with -, that's an error -- unknown option.  The first one
    * that doesn't start with - will be the input filename; if
    * we find any more that don't start with -, that's also an
    * error because we (currently) only support one input file.
    */

   for (arg = 1; arg < *pargc; arg++)
   {
      if (argv[arg][0] == '-')
      {
	 fprintf (stderr, "Unknown option: %s\n", argv[arg]);
	 usage (argv[0]);
      }
      else
      {
	 if (Movie->Filename != NULL) /* already been assigned? */
	 {
	    fprintf (stderr, "Can't give more than one input file\n");
	    usage (argv[0]);
	 }
	 else
	 {
	    Movie->Filename = argv [1];
	    Movie->InputStream = fopen (Movie->Filename, "r");
	    if (Movie->InputStream == NULL)
	    {
	       fprintf(stderr, "Could not open file %s: %s\n", 
		       Movie->Filename, strerror (errno));
	       usage(argv[0]);
	    }
	 }	 
      }
   }
}     /* ParseCmdLine */
