/*************************************************************
Copyright (C) 1990, 1991, 1993 Andy C. Hung, all rights reserved.
PUBLIC DOMAIN LICENSE: Stanford University Portable Video Research
Group. If you use this software, you agree to the following: This
program package is purely experimental, and is licensed "as is".
Permission is granted to use, modify, and distribute this program
without charge for any purpose, provided this license/ disclaimer
notice appears in the copies.  No warranty or maintenance is given,
either expressed or implied.  In no event shall the author(s) be
liable to you or a third party for any special, incidental,
consequential, or other damages, arising out of the use or inability
to use the program for any purpose (or the loss of data), even if we
have been advised of such possibilities.  Any public reference or
advertisement of this source code should refer to it as the Portable
Video Research Group (PVRG) code, and not by any author(s) (or
Stanford University) name.
*************************************************************/
/*
************************************************************
cv.c

This program displays the luminance component of a video sequence. It
is optimized just for luminance display - no attempt has been made to
increase the user-friendliness of this program.

Usage:

cv -a StartFrame -b EndFrame [-iw ImageWidth] [-ih ImageHeight]
   [-BW] [-yg Gamma] [-rg Red_Gamma] [-gg Green_Gamma] [-bg Blue_Gamma]
   [-d Dither_Scale] [-nr] [-C] [-SF] Filename

-a StartFrame: first frame index.
-b EndFrame: last frame index.
-iw ImageWidth: specifies image width per frame (default 352).
-ih ImageHeight: specifies image height per frame (default 240).
-BW Sets black and white mode (only yg works for this).
-yg Gamma: Overall luminance gamma (default 0.8).
-rg Red_Gamma: Overall red gamma (default 1.0).
-gg Green_Gamma: Overall green gamma (default 1.0).
-bg Green_Gamma: Overall blue gamma (default 1.0).
-d Dither_Scale: scales the dither (default 1.0)  0 = no dither
-nr do not add random uniform noise less than smallest dither offset-
    useful for finely quantized images.
-C Use CCIR-601 YUV->RGB conversion. (Default JFIF)
-SF loads a single frame.
Filename - the target filename


In single frame mode (with -SF option):
Displays a single file in color or black and white:
Source: (Filename.Y Filename.U Filename.V) or
        (Filename.i Filename.j Filename.k
          where i<j<k are integers less than 20).
If BW mode is chosen then only the luminance file is loaded.

Example: foo.Y foo.U foo.V;  foo.1 foo.2 foo.3; ....


In sequence mode (default):
Displays a sequence of files in color or black and white.
The variable "Filename" becomes a sequence of filenames
with a frame number as a suffix (from StartFrame to EndFrame):
Filename(StartFrame) Filename(StartFrame+1)... Filename(EndFrame).
and they are loaded just like single frames.

Example: foo0.Y foo0.U foo0.V, foo1.Y foo1.U foo1.V, ...

Note:

The ImageWidth and ImageHeight are specified only for the luminance
files.  The index is automatically added as a sequence of digits
followed by a .Y.

Any key-press on the window starts one pass through the image, except
for return which exits the program. -NVB specifies the nonverbose
mode.

Author:

May be reached at achung@cs.stanford.edu

************************************************************
*/

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xos.h>
#include <X11/keysym.h>
#include <stdio.h>
#include <math.h>

#include "cv.h"

/* X Stuff */
#define BITMAPDEPTH 1
#define NUMBER_OF_BITS 8
#define SIDETRIM 10
#define TOPTRIM 10
#define BOTTOMTRIM 20
#define BORDERWIDTH 4
#define MINWIDTH (SIDETRIM + SIDETRIM + BORDERWIDTH + BORDERWIDTH)
#define MINHEIGHT (TOPTRIM + BOTTOMTRIM + BORDERWIDTH + BORDERWIDTH)
#define PREFERRED_X 0
#define PREFERRED_Y 0

#define STRINGBUFFER 256
#define MAX_BASENAME_LENGTH 80

#define MAX(x,y) ((x > y) ? x : y)

#define MAX_FILENAME_INDEX 20

/* For the GRAY version */
/* We use the active Rec. 601-1 region for JFIF, too...*/
/* seems like a reasonable thing to do. */

#define CV_LO_Y 16
#define CV_HI_Y 235

/* For the COLOR version */
/* Active regions per Rec. 601-1 */
/* This is  16- 235 Y; 16- 240 UV. */
/* we use this for JFIF; otherwise it is scaled. */
/* The Y-CB region is mostly unused - blue difference space. */
/* The Y-CR region is somewhat less unused - red difference space. */
/* If we don't restrict these regions, we have a substantially unused */
/* color space. Thus with the following parameters, we use almost all */
/* of the color cells, except perhaps 10 stray cells out of 240. */

#define NOMINAL_LO_Y 36
#define NOMINAL_HI_Y 214
#define NOMINAL_LO_U 60
#define NOMINAL_HI_U 196
#define NOMINAL_LO_V 56
#define NOMINAL_HI_V 200

/* Can mix the levels in the luminance/chrominance */
/* We select 240 cells - this seems to work out well */
/* as many xsessions do not use more than 15 or so cells. */
/* However, we will grab all 240 cells by creating our own */
/* colormap, if necessary. */

/* With the following levels, we see: */
/* Ave Y is 21.5  (Luminance - Green primary) */
/* Ave U is 27.2  (Blue difference) */
/* Ave V is 24.0  (Red difference) */

#define Y_LEVEL 8
#define U_LEVEL 5
#define V_LEVEL 6

/* Also note that we don't use a full dither because */
/* original signal will have background (dark/shot) noise - we just */
/* need enough ordered noise to kick it over the threshold. */
/* Warning: a) image compression tends to remove this noise and */
/* b) synthetic images - e.g. computer graphics -  are noiseless. */
/* Note that CCD cameras have lots of noise in "U" component, */
/* probably less in the "V" component and "Y" component */

#define YBIG_DITHER 10.0
#define YTINY_DITHER 5.0
#define UBIG_DITHER 13.0
#define UTINY_DITHER 6.0
#define VBIG_DITHER 11.5
#define VTINY_DITHER 6.5
#define OFFS 40

/* OFFS is the array offset. Should be greater than any of the BIG_DITHER */

/*
************************************************************
Important Global Variables for every routine to see.

************************************************************
*/

/* X Stuff */
Display *CurrentDisplay;
int CurrentScreen;
Visual *CurrentVisual;
Colormap *CurrentColormap;
Window GlobalWindow,DisplayWindow;
GC gc;
Pixmap icon_pixmap;

/* Translation matrix for colors to colorcells */
long ColorIndex[256];
int CreatedColors[256];
int CreatedCount = 0;
int MyColormap=0;       /* Set if we created a colormap */
int CCIRmode=0;

/* Stuff for random noise dither */

int DoRandom=1;
char *YRandom,*URandom,*VRandom;

/* Stuff for YUV colors */
FRAME *yframe=NULL;
FRAME *uframe1=NULL,*vframe1=NULL; 
FRAME *uframe2=NULL,*vframe2=NULL; 
FRAME *uframe4=NULL,*vframe4=NULL; 

int YIndeX[256+OFFS+OFFS],UIndeX[256+OFFS+OFFS],VIndeX[256+OFFS+OFFS];
int *Yindex=YIndeX+OFFS;
int *Uindex=UIndeX+OFFS;
int *Vindex=VIndeX+OFFS;
double YGamma=0.8,RGamma=1.0,GGamma=1.0,BGamma=1.0;

/* Dither */

double DitherScale=1.0;
int YBigDither,YTinyDither;
int UBigDither,UTinyDither;
int VBigDither,VTinyDither;

/* Dimension, filename misc flags */
unsigned int Width=0, Height=0;
char *Basename;
char Indexname[MAX_BASENAME_LENGTH];

int ColorFlag=1;
int SingleFrame=0;


/* Help() prints out the help information. */
void
Help()
{
  printf("cv -a StartFrame -b EndFrame [-iw ImageWidth] [-ih ImageHeight]\n");
  printf("   [-BW] [-yg Gamma] [-rg Red_Gamma] [-gg Green_Gamma] [-bg Blue_Gamma]\n");
  printf("   [-d Dither_Scale] [-nr] [-C] [-SF] Filename\n");
  printf("\n");
  printf("-a StartFrame: first frame index.\n");
  printf("-b EndFrame: last frame index.\n");
  printf("-iw ImageWidth: specifies image width per frame (default 352).\n");
  printf("-ih ImageHeight: specifies image height per frame (default 240).\n");
  printf("-BW Sets black and white mode (only yg works for this).\n");
  printf("-yg Gamma: Overall luminance gamma (default 0.8).\n");
  printf("-rg Red_Gamma: Overall red gamma (default 1.0).\n");
  printf("-gg Green_Gamma: Overall green gamma (default 1.0).\n");
  printf("-bg Green_Gamma: Overall blue gamma (default 1.0).\n");
  printf("-d Dither_Scale: scales the dither (default 1.0)  0 = no dither\n");
  printf("-nr do not add random uniform noise less than smallest dither offset-\n");
  printf("    useful for finely quantized images.\n");
  printf("-C Use CCIR-601 YUV->RGB conversion. (Default JFIF)\n");
  printf("-SF loads a single frame.\n");
  printf("Filename - the target filename\n");
  printf("\n\nIn single frame mode (with -SF option):\n");
  printf("Displays a single file in color or black and white:\n");
  printf("Source: (Filename.Y Filename.U Filename.V) or\n");
  printf("        (Filename.i Filename.j Filename.k \n");
  printf("          where i<j<k are integers less than 20).\n");
  printf("If BW mode is chosen then only the luminance file is loaded.\n\n");
  printf("Example: foo.Y foo.U foo.V;  foo.1 foo.2 foo.3; ....\n");
  printf("\n\nIn sequence mode (default):\n");
  printf("Displays a sequence of files in color or black and white.\n");
  printf("The variable \"Filename\" becomes a sequence of filenames\n");
  printf("with a frame number as a suffix (from StartFrame to EndFrame):\n");
  printf("Filename(StartFrame) Filename(StartFrame+1)... Filename(EndFrame).\n");
  printf("and they are loaded just like single frames.\n\n");
  printf("Example: foo0.Y foo0.U foo0.V, foo1.Y foo1.U foo1.V, ...\n");
}

/* MakeFrame() constructs the frame structure to hold all of the data.
It is only used once. */

FRAME *
MakeFrame(width, height)
     int width, height;
{
  FRAME *temp;

  temp = MakeStructure(FRAME);
  temp->width = width;
  temp->height = height;
  temp->n = width*height;
  temp->data = (unsigned char *) calloc(temp->n+2,sizeof(unsigned char));
  /* We add 2 extra characters */
  return(temp);
}

void
CVExit()
{
  int i;

  if (MyColormap)
    XFreeColormap(CurrentDisplay,*CurrentColormap); /* Uninstall, exit */
  else
    {
      for(i=0;i<CreatedCount;i++)
	XFreeColors(CurrentDisplay,*CurrentColormap,
		    CreatedColors+i,1,0);
    }
  XFreePixmap(CurrentDisplay,icon_pixmap);
  XFreeGC(CurrentDisplay,gc);
  XCloseDisplay(CurrentDisplay);
  exit(1);
}

/* Reads the input; returns 0 if okay; 1 if file not found; 2 if
filesize error; 3 if read error. */

int
ReadFrame(fm,fname)
     FRAME *fm;
     char *fname;
{
  int i,testn;
  unsigned char *ptr;
  FILE *input;

  if ((input = fopen(fname,"r")) == NULL) return(1);

  fseek(input,0,2);
  testn = ftell(input);
  rewind(input);
  if ((testn == fm->width * fm->height)||
      (testn == fm->width * (fm->height+1)))
    {
      if (!fread(fm->data,fm->width*fm->height,1,input)) return(3);
    }
  else if ((testn==(fm->width+1)*(fm->height))||
	   (testn==(fm->width+1)*(fm->height+1)))
    {
      for(ptr=fm->data,i=0;i<fm->height;i++)
	{
	  if (!fread(ptr,fm->width,1,input)) return(3);
	  (void) fgetc(input);
	  ptr+=fm->width;
	}
    }
  else if (testn == fm->width * (fm->height-1))
    {
      if (!fread(fm->data,fm->width*(fm->height-1),1,input)) return(3);
      memcpy(fm->data+fm->width*(fm->height-1),
	     fm->data+fm->width*(fm->height-2),fm->width);
    }
  else if (testn==(fm->width-1)*(fm->height))
    {
      for(ptr=fm->data,i=0;i<fm->height;i++)
	{
	  if (!fread(ptr,fm->width-1,1,input)) return(3);
	  ptr+=fm->width-1;
	  *(ptr) = *(ptr-1);
	  ptr++;
	}
    }
  else if (testn==(fm->width-1)*(fm->height-1))
    {
      for(ptr=fm->data,i=0;i<fm->height-1;i++)
	{
	  if (!fread(ptr,fm->width-1,1,input)) return(3);
	  ptr+=fm->width-1;
	  *(ptr) = *(ptr-1);
	  ptr++;
	}
      memcpy(fm->data+fm->width*(fm->height-1),
	     fm->data+fm->width*(fm->height-2),fm->width);
    }
  else
    {
      printf("File length: %d   Width,Height=%dx%d = %d\n",
	     testn,fm->width,fm->height,fm->width*fm->height);
      fclose(input);
      return(2);
    }
  fclose(input);
  return(0);
}

/* MakeGraymap() constructs a gray map out of the available colormap
cells. Returns 1 if unable to create enough colorcells.*/

int
MakeGraymap()
{
  int i,j,k;
  double temp;
  XColor xcol;
  
  CreatedCount=0;
  for(j=CV_LO_Y;j<=CV_HI_Y;j++)
    {
      if (CCIRmode)
	{
	  temp = pow((((double)(j-CV_LO_Y))/((double) CV_HI_Y-CV_LO_Y)),
		     YGamma)*256.0;
	  k =  (int) ((temp*256.0 +0.5));
	  if (k>65535) k=65535;
	  xcol.red = xcol.green = xcol.blue = k;
	}
      else
	{
	  temp = pow((((double)j)/256.0),YGamma)*256.0;
	  xcol.red = xcol.green = xcol.blue = (int) ((temp*256.0 +0.5));
	}

      if (!XAllocColor(CurrentDisplay,*CurrentColormap,&xcol))
	{
	  fprintf(stderr,
		  "Able to allocate only %d of %d colorcells\n",
		  CreatedCount,CV_HI_Y-CV_LO_Y+1);
	      for(i=0;i<CreatedCount;i++)
		{
		  XFreeColors(CurrentDisplay,*CurrentColormap,
			      CreatedColors+i,1,0);
		}
	      return(1);
	  /* for(k=j;k<=CV_HI_Y;k++)
	     ColorIndex[k] = ColorIndex[j-1]; */
	}
      ColorIndex[j] = xcol.pixel;
      CreatedColors[CreatedCount++] = xcol.pixel;
    }
  for(k=0;k<CV_LO_Y;k++)
    ColorIndex[k]=ColorIndex[CV_LO_Y];
  for(k=CV_HI_Y+1;k<256;k++)
    ColorIndex[k]=ColorIndex[CV_HI_Y];
  fprintf(stderr,"Created %d colors\n",CreatedCount);
  return(0);
}

void
Dither411(sl)
     FRAME *sl;
{
  int i,u,v,ht;
  unsigned char *cptr,*eptr,*yptr,*uptr,*vptr;
  int *Yposbig=Yindex+YBigDither,*Ypostiny=Yindex+YTinyDither;
  int *Ynegbig=Yindex-YBigDither,*Ynegtiny=Yindex-YTinyDither;
  int *Uposbig=Uindex+UBigDither,*Upostiny=Uindex+UTinyDither;
  int *Unegbig=Uindex-UBigDither,*Unegtiny=Uindex-UTinyDither;
  int *Vposbig=Vindex+VBigDither,*Vpostiny=Vindex+VTinyDither;
  int *Vnegbig=Vindex-VBigDither,*Vnegtiny=Vindex-VTinyDither;
  char *yrptr,*urptr,*vrptr;
  
  if ((!yframe)||(!uframe4)||(!vframe4))
    {
      fprintf(stderr,"Invalid frames\n");
      exit(-1);
    }

  eptr = cptr = sl->data;
  yptr = yframe->data;  uptr = uframe4->data;  vptr = vframe4->data;

  /* if width of y is odd, we assume u and v are /2, rounded up */

  ht = yframe->height>>1;

  if (DoRandom)
    {
      yrptr=YRandom; urptr=URandom; vrptr=VRandom;
      for(i=0;i<ht;i++)
	{
	  eptr += sl->width;
	  for(;cptr<eptr;)
	    {
	      *(cptr++) = ColorIndex[Yposbig[*(yptr++)+*(yrptr++)]+
				     Unegbig[(u= *(uptr++))+*(urptr++)]+
				     Vpostiny[(v= *(vptr++))+*(vrptr++)]];
	      *(cptr++) = ColorIndex[Ynegtiny[*(yptr++)+*(yrptr++)]+
				     Upostiny[u+*(urptr++)]+
				     Vnegbig[v+*(vrptr++)]];
	    }
	  if (sl->width&1) /* May have overshot */
	    {cptr--; yptr--;}
	  
	  eptr += sl->width;
	  uptr -= uframe4->width;
	  vptr -= vframe4->width;
	  for(;cptr<eptr;)
	    {
	      *(cptr++) = ColorIndex[Ypostiny[*(yptr++)+*(yrptr++)]+
				     Unegtiny[(u= *(uptr++))+*(urptr++)]+
				     Vposbig[(v= *(vptr++))+*(vrptr++)]];
	      *(cptr++) = ColorIndex[Ynegbig[*(yptr++)+*(yrptr++)]+
				     Uposbig[u+*(urptr++)]+
				     Vnegtiny[v+*(vrptr++)]];
	    }
	  cptr=eptr;
	  if (sl->width&1) /* May have overshot */
	    { cptr--; yptr--; }
	}
      if (yframe->height&1)
	{
	  eptr += sl->width;
	  for(;cptr<eptr;)
	    {
	      *(cptr++) = ColorIndex[Yposbig[*(yptr++)+*(yrptr++)]+
				     Unegbig[(u= *(uptr++))+*(urptr++)]+
				     Vpostiny[(v= *(vptr++))+*(vrptr++)]];
	      *(cptr++) = ColorIndex[Ynegtiny[*(yptr++)+*(yrptr++)]+
				     Upostiny[u+*(urptr++)]+
				     Vnegbig[v+*(vrptr++)]];
	    }
	}
    }
  else
    {
      for(i=0;i<ht;i++)
	{
	  eptr += sl->width;
	  for(;cptr<eptr;)
	    {
	      *(cptr++) = ColorIndex[Yposbig[*(yptr++)]+
				     Unegbig[(u= *(uptr++))]+
				     Vpostiny[(v= *(vptr++))]];
	      *(cptr++) = ColorIndex[Ynegtiny[*(yptr++)]+
				     Upostiny[u]+Vnegbig[v]];
	    }
	  if (sl->width&1) /* May have overshot */
	    {cptr--; yptr--;}
	  
	  eptr += sl->width;
	  uptr -= uframe4->width;
	  vptr -= vframe4->width;
	  for(;cptr<eptr;)
	    {
	      *(cptr++) = ColorIndex[Ypostiny[*(yptr++)]+
				     Unegtiny[(u= *(uptr++))]+
				     Vposbig[(v= *(vptr++))]];
	      *(cptr++) = ColorIndex[Ynegbig[*(yptr++)]+
				     Uposbig[u]+Vnegtiny[v]];
	    }
	  cptr=eptr;
	  if (sl->width&1) /* May have overshot */
	    { cptr--; yptr--; }
	}
      if (yframe->height&1)
	{
	  eptr += sl->width;
	  for(;cptr<eptr;)
	    {
	      *(cptr++) = ColorIndex[Yposbig[*(yptr++)]+
				     Unegbig[(u= *(uptr++))]+
				     Vpostiny[(v= *(vptr++))]];
	      *(cptr++) = ColorIndex[Ynegtiny[*(yptr++)]+
				     Upostiny[u]+Vnegbig[v]];
	    }
	}
    }
}

void
Dither211(sl)
     FRAME *sl;
{
  int i,u,v;
  unsigned char *cptr,*eptr,*yptr,*uptr,*vptr;
  int *Yposbig=Yindex+YBigDither,*Ypostiny=Yindex+YTinyDither;
  int *Ynegbig=Yindex-YBigDither,*Ynegtiny=Yindex-YTinyDither;
  int *Uposbig=Uindex+UBigDither,*Upostiny=Uindex+UTinyDither;
  int *Unegbig=Uindex-UBigDither,*Unegtiny=Uindex-UTinyDither;
  int *Vposbig=Vindex+VBigDither,*Vpostiny=Vindex+VTinyDither;
  int *Vnegbig=Vindex-VBigDither,*Vnegtiny=Vindex-VTinyDither;
  char *yrptr,*urptr,*vrptr;

  if ((!yframe)||(!uframe2)||(!vframe2))
    {
      fprintf(stderr,"Invalid frames\n");
      exit(-1);
    }
  
  eptr = cptr = sl->data;
  yptr = yframe->data;  uptr = uframe2->data;  vptr = vframe2->data;

  if (DoRandom)
    {
      yrptr=YRandom; urptr=URandom; vrptr=VRandom;
      for(i=0;i<sl->height;i++)
	{
	  eptr += sl->width;
	  if (!(i&1))
	    {
	      for(;cptr<eptr;)
		{
		  *(cptr++) = ColorIndex[Yposbig[*(yptr++)+*(yrptr++)]+
					 Unegbig[(u= *(uptr++))+*(urptr++)]+
					 Vpostiny[(v= *(vptr++))+*(vrptr++)]];
		  *(cptr++) = ColorIndex[Ynegtiny[*(yptr++)+*(yrptr++)]+
					 Upostiny[u+*(urptr++)]+
					 Vnegbig[v+*(vrptr++)]];
		}
	    }
	  else
	    {
	      for(;cptr<eptr;)
		{
		  *(cptr++) = ColorIndex[Ypostiny[*(yptr++)+*(yrptr++)]+
					 Unegtiny[(u= *(uptr++))+*(urptr++)]+
					 Vposbig[(v= *(vptr++))+*(vrptr++)]];
		  *(cptr++) = ColorIndex[Ynegbig[*(yptr++)+*(yrptr++)]+
					 Uposbig[u+*(urptr++)]+
					 Vnegtiny[v+*(vrptr++)]];
		}
	    }
	  if (sl->width&1) /* May have overshoot if odd */
	    {cptr--; yptr--;}
	}
    }
  else
    {
      for(i=0;i<sl->height;i++)
	{
	  eptr += sl->width;
	  if (!(i&1))
	    {
	      for(;cptr<eptr;)
		{
		  *(cptr++) = ColorIndex[Yposbig[*(yptr++)]+
					 Unegbig[(u= *(uptr++))]+
					 Vpostiny[(v= *(vptr++))]];
		  *(cptr++) = ColorIndex[Ynegtiny[*(yptr++)]+
					 Upostiny[u]+Vnegbig[v]];
		}
	    }
	  else
	    {
	      for(;cptr<eptr;)
		{
		  *(cptr++) = ColorIndex[Ypostiny[*(yptr++)]+
					 Unegtiny[(u= *(uptr++))]+
					 Vposbig[(v= *(vptr++))]];
		  *(cptr++) = ColorIndex[Ynegbig[*(yptr++)]+
					 Uposbig[u]+
					 Vnegtiny[v]];
		}
	    }
	  if (sl->width&1) /* May have overshoot if odd */
	    {cptr--; yptr--;}
	}
    }
}

void
Dither111(sl)
     FRAME *sl;
{
  int i;
  unsigned char *cptr,*eptr,*yptr,*uptr,*vptr;
  int *Yposbig=Yindex+YBigDither,*Ypostiny=Yindex+YTinyDither;
  int *Ynegbig=Yindex-YBigDither,*Ynegtiny=Yindex-YTinyDither;
  int *Uposbig=Uindex+UBigDither,*Upostiny=Uindex+UTinyDither;
  int *Unegbig=Uindex-UBigDither,*Unegtiny=Uindex-UTinyDither;
  int *Vposbig=Vindex+VBigDither,*Vpostiny=Vindex+VTinyDither;
  int *Vnegbig=Vindex-VBigDither,*Vnegtiny=Vindex-VTinyDither;
  char *yrptr,*urptr,*vrptr;
  
  if ((!yframe)||(!uframe1)||(!vframe1))
    {
      fprintf(stderr,"Invalid frames\n");
      exit(-1);
    }
  eptr = cptr = sl->data;
  yptr = yframe->data;  uptr = uframe1->data;  vptr = vframe1->data;

  if (DoRandom)
    {
      yrptr=YRandom; urptr=URandom; vrptr=VRandom;
      for(i=0;i<sl->height;i++)
	{
	  eptr += sl->width;
	  if (!(i&1))
	    {
	      for(;cptr<eptr;)
		{
		  *(cptr++) = ColorIndex[Yposbig[*(yptr++)+*(yrptr++)]+
					 Unegbig[*(uptr++)+*(urptr++)]+
					 Vpostiny[*(vptr++)+*(vrptr++)]];
		  *(cptr++) = ColorIndex[Ynegtiny[*(yptr++)+*(yrptr++)]+
					 Upostiny[*(uptr++)+*(urptr++)]+
					 Vnegbig[*(vptr++)+*(vrptr++)]];
		}
	    }
	  else
	    {
	      for(;cptr<eptr;)
		{
		  *(cptr++) = ColorIndex[Ypostiny[*(yptr++)+*(yrptr++)]+
					 Unegtiny[*(uptr++)+*(urptr++)]+
					 Vposbig[*(vptr++)]+*(vrptr++)];
		  *(cptr++) = ColorIndex[Ynegbig[*(yptr++)+*(yrptr++)]+
					 Uposbig[*(uptr++)+*(urptr++)]+
					 Vnegtiny[*(vptr++)+*(vrptr++)]];
		}
	    }
	}
    }
  else
    {
      for(i=0;i<sl->height;i++)
	{
	  eptr += sl->width;
	  if (!(i&1))
	    {
	      for(;cptr<eptr;)
		{
		  *(cptr++) = ColorIndex[Yposbig[*(yptr++)]+
					 Unegbig[*(uptr++)]+
					 Vpostiny[*(vptr++)]];
		  *(cptr++) = ColorIndex[Ynegtiny[*(yptr++)]+
					 Upostiny[*(uptr++)]+
					 Vnegbig[*(vptr++)]];
		}
	    }
	  else
	    {
	      for(;cptr<eptr;)
		{
		  *(cptr++) = ColorIndex[Ypostiny[*(yptr++)]+
					 Unegtiny[*(uptr++)]+
					 Vposbig[*(vptr++)]];
		  *(cptr++) = ColorIndex[Ynegbig[*(yptr++)]+
					 Uposbig[*(uptr++)]+
					 Vnegtiny[*(vptr++)]];
		}
	    }
	}
    }
}


void
GetUnknownBWFrame(sl,bname)
     FRAME *sl;
     char *bname;
{
  unsigned char *cptr,*eptr;
  int i,retval;
  char fname[256];
  
  sprintf(fname,"%s.Y",bname);
  retval=ReadFrame(sl,fname);
  if (retval==1)
    {
      for(i=0;i<MAX_FILENAME_INDEX;i++)/*Start searching component indices*/
	{
	  sprintf(fname,"%s.%d",bname,i);
	  retval=ReadFrame(sl,fname);
	  if (retval!=1) break;
	}
    }
  if (retval)
    {
      fprintf(stderr,"Problem (%d) with filename: %s\n",retval,fname);
      CVExit();
    }
  cptr = sl->data;
  eptr = cptr+sl->width*sl->height;
  for(;cptr<eptr;cptr++)
    *(cptr) = ColorIndex[*(cptr)];
}

void
GetUnknownYUVFrame(sl,bname)
     FRAME *sl;
     char *bname;
{
  char yuvname[3][256];
  int i,j,retval,YUV=0;
  int filesize[3];
  int dratio1,dratio2;
  FILE *ftmp;
  FRAME *yuvframe[3];

  /* Check if YUV file */
  sprintf(yuvname[0],"%s.Y",bname);
  if (ftmp=fopen(yuvname[0],"r"))
    {
      fseek(ftmp,0,2);
      filesize[0] = ftell(ftmp);
      fclose(ftmp);
      sprintf(yuvname[1],"%s.U",bname);
      if (ftmp=fopen(yuvname[1],"r"))
	{
	  fseek(ftmp,0,2);
	  filesize[1] = ftell(ftmp);
	  fclose(ftmp);
	  sprintf(yuvname[2],"%s.V",bname);
	  if (ftmp=fopen(yuvname[2],"r"))
	    {
	      fseek(ftmp,0,2);
	      filesize[2] = ftell(ftmp);
	      fclose(ftmp);
	      YUV=1;
	    }
	}
    }
  if (!YUV)
    {  /*Start searching component indices*/
      for(i=j=0;i<MAX_FILENAME_INDEX;i++)
	{
	  sprintf(yuvname[j],"%s.%d",bname,i);
	  if (ftmp=fopen(yuvname[j],"r"))
	    {
	      fseek(ftmp,0,2);
	      filesize[j] = ftell(ftmp);
	      fclose(ftmp);
	      j++; if (j==3) break;
	    }
	}
      if (j!=3)
	{
	  fprintf(stderr,"Cannot open filename (base): %s\n",bname);
	  CVExit();
	}
    }

  if (!yframe)
    yframe = MakeFrame(Width,Height);

  dratio1=((Width*Height+filesize[1]/4)/filesize[1]);
  dratio2=((Width*Height+filesize[2]/4)/filesize[2]);

  if (dratio1!=dratio2)
    {
      fprintf(stderr,"Chrominance decimation ratios do not match!\n");
      CVExit();
    }
  yuvframe[0]=yframe;
  if (dratio1==4)
    {
      /* fprintf(stderr,"4:1:1 image\n"); */
      if (!uframe4)
	uframe4 = MakeFrame((Width+1)/2,(Height+1)/2);
      if (!vframe4)
	vframe4 = MakeFrame((Width+1)/2,(Height+1)/2);
      yuvframe[1]=uframe4; yuvframe[2]=vframe4;
    }
  else if (dratio1==2)
    {
      /* fprintf(stderr,"2:1:1 image\n"); */
      if (!uframe2)
	uframe2 = MakeFrame((Width+1)/2,Height);
      if (!vframe2)
	vframe2 = MakeFrame((Width+1)/2,Height);
      yuvframe[1]=uframe2; yuvframe[2]=vframe2;
    }
  else if (dratio1==1)
    {
      /* fprintf(stderr,"1:1:1 image\n"); */
      if (!uframe1)
	uframe1 = MakeFrame(Width,Height);
      if (!vframe1)
	vframe1 = MakeFrame(Width,Height);
      yuvframe[1]=uframe1; yuvframe[2]=vframe1;
    }
  else
    {
      fprintf(stderr,"Chrominance decimation ratios not 1,2,4!\n");
      CVExit();
    }
  for(i=0;i<3;i++)
    {
      if (retval=ReadFrame(yuvframe[i],yuvname[i]))
	{
	  fprintf(stderr,"Problem (%d) with filename: %s\n",retval,yuvname[i]);
	  CVExit();
	}
    }

  if (dratio1==4)
    Dither411(sl);
  else if (dratio1==2)
    Dither211(sl);
  else if (dratio1==1)
    Dither111(sl);
}


/* MakeYUVmap() constructs a gray map out of the available colormap
cells. Returns 0 if okay, 1 if unable to create. */

int
MakeYUVmap()
{
  int i,y,u,v;
  double Y,U,V,Yd,Ud,Vd,temp;
  double Yarray[100],Uarray[100],Varray[100];
  double diff1,diff2;

  XColor xcol;
  
  /* 10 Y, 5 U, 5 V */

  Yd = ((double)(1+NOMINAL_HI_Y-NOMINAL_LO_Y))/((double)Y_LEVEL);
  Ud = ((double)(1+NOMINAL_HI_U-NOMINAL_LO_U))/((double)U_LEVEL);
  Vd = ((double)(1+NOMINAL_HI_V-NOMINAL_LO_V))/((double)V_LEVEL);

  for(y=0;y<Y_LEVEL;y++)
    Yarray[y] = (((double)y)+0.5)* Yd + NOMINAL_LO_Y;
  Yarray[Y_LEVEL]=10000;

  for(u=0;u<U_LEVEL;u++)
    Uarray[u] = (((double)u)+0.5)* Ud + NOMINAL_LO_U;
  Uarray[U_LEVEL]=10000;

  for(v=0;v<V_LEVEL;v++)
    Varray[v] = (((double)v)+0.5)* Vd + NOMINAL_LO_V;
  Varray[V_LEVEL]=10000;

  for(CreatedCount=0,y=0;y<Y_LEVEL;y++)
    {
      Y = Yarray[y];
      if (CCIRmode)
	Y = ((Y-16.0)*255.0)/219.0;

      for(u=0;u<U_LEVEL;u++)
	{
	  U = Uarray[u];
	  if (CCIRmode)
	    U = (((U-128.0)*255.0)/224.0)+128.0;

	  for(v=0;v<V_LEVEL;v++)
	    {
	      V = Varray[v];
	      if (CCIRmode)
		V = (((V-128.0)*255.0)/224.0)+128.0;

	      /* MATRIX CONVERSION */

	      /* printf("YUV-> %.2f %.2f %.2f\n",Y,U,V); */

	      temp = (Y+1.402*(V-128.0));
	      if (temp<0) temp=0;
	      if (temp>255.49) temp=255.49;
	      temp = pow((temp/256.0),RGamma*YGamma)*256.0;
	      if (temp>255.49) temp=255.49;
	      xcol.red = (int) (temp*256.0+0.5);
	      
	      temp = (Y-0.34414*(U-128.0)-0.71414*(V-128.0));
	      if (temp<0) temp=0;
	      if (temp>255.49) temp=255.49;
	      temp = pow((temp/256.0),GGamma*YGamma)*256.0;
	      if (temp>255.49) temp=255.49;
	      xcol.green = (int) (temp*256.0+0.5);

	      temp = (Y+1.772*(U-128.0));
	      if (temp<0) temp=0;
	      if (temp>255.49) temp=255.49;
	      temp = pow((temp/256.0),BGamma*YGamma)*256.0;
	      if (temp>255.49) temp=255.49;
	      xcol.blue =  (int) (temp*256.0+0.5);

	      /* printf("RGB-> %d %d %d\n",xcol.red,xcol.green,xcol.blue); */
	      if (!XAllocColor(CurrentDisplay,*CurrentColormap,&xcol))
		{
		  fprintf(stderr,
			  "Able to allocate only %d of %d colorcells\n",
			  CreatedCount,Y_LEVEL*U_LEVEL*V_LEVEL);
		  for(i=0;i<CreatedCount;i++)
		    {
		      XFreeColors(CurrentDisplay,*CurrentColormap,
				  CreatedColors+i,1,0);
		    }
		  return(1);
		}
	      ColorIndex[CreatedCount] = xcol.pixel;
	      CreatedColors[CreatedCount++] = xcol.pixel;
	    }
	}
    }

  /* Find nearest neighbors; min(L1norm)==min(L2norm) */
  for(y=0,i= -OFFS;i<256+OFFS;i++)
    {
      diff1 = (((double)i) - Yarray[y]);
      if (diff1<0) diff1 = -diff1;
      diff2 = (((double)i) - Yarray[y+1]);
      if (diff2<0) diff2 = -diff2;

      if (diff1<diff2)
	Yindex[i] = y*U_LEVEL*V_LEVEL;
      else
	{
	  y++;
	  Yindex[i] = y*U_LEVEL*V_LEVEL;
	}
      /* printf("Yindex[%d]=%d\n",i,Yindex[i]); */
    }
  for(u=0,i= -OFFS;i<256+OFFS;i++)
    {
      diff1 = (((double)i) - Uarray[u]);
      if (diff1<0) diff1 = -diff1;
      diff2 = (((double)i) - Uarray[u+1]);
      if (diff2<0) diff2 = -diff2;


      if (diff1<diff2)
	Uindex[i] = u*V_LEVEL;
      else
	{
	  u++;
	  Uindex[i] = u*V_LEVEL;
	}
      /* printf("Uindex[%d]=%d\n",i,Uindex[i]); */
    }
  for(v=0,i= -OFFS;i<256+OFFS;i++)
    {
      diff1 = (((double)i) - Varray[v]);
      if (diff1<0) diff1 = -diff1;
      diff2 = (((double)i) - Varray[v+1]);
      if (diff2<0) diff2 = -diff2;

      if (diff1<diff2)
	Vindex[i] = v;
      else
	{
	  v++;
	  Vindex[i] = v;
	}
      /* printf("Vindex[%d]=%d\n",i,Vindex[i]); */
    }
  fprintf(stderr,"Created %d colors\n",CreatedCount);
  return(0);
}

/* main() is the program that controls the display loop. */

int
main(argc,argv)
     int argc;
     char ** argv;
{
  unsigned int WindowWidth,WindowHeight;  /* Window Variables */
  unsigned int MaximumWidth,MaximumHeight;
  unsigned int InteriorWidth,InteriorHeight;
  int rval;
  char *rptr;
  char *icon_name = "CV Display";
  char *window_name = "CV Display";
  XEvent myreport;
  unsigned long valuemask = 0;
  XGCValues gcvalues;
  XSizeHints size_hints;
  char *display_name = NULL;

  FRAME *TheFrame;                        /* Sequence positional values */
  int CurrentFrame,FirstFrame,LastFrame;
  XImage *FrameImage=NULL;

  int i;                                  /* Program Variables */
  double atof();

  CurrentColormap = MakeStructure(Colormap);
  Width=352;  Height=240;
  CurrentFrame= FirstFrame= LastFrame=0;

  if (argc == 1)
    {
      Help();
      exit(-1);
    }
  for(i=1;i<argc;i++)
    {
      if (strncmp(argv[i],"-HELP",3) == 0)
	{
	  Help();
	  exit(-1);
	}
      else if (strncmp(argv[i],"-nr",3) == 0)
	DoRandom=0;
      else if (strncmp(argv[i],"-a",2) == 0)
	FirstFrame = atoi(argv[++i]);
      else if (strncmp(argv[i],"-b",2) == 0)
	LastFrame = atoi(argv[++i]);
      else if (strncmp(argv[i],"-C",2) == 0)
	CCIRmode = 1;
      else if (strncmp(argv[i],"-d",2) == 0)
	DitherScale = atof(argv[++i]);
      else if (strncmp(argv[i],"-iw",3) == 0)
	Width = atoi(argv[++i]);
      else if (strncmp(argv[i],"-ih",3) == 0)
	Height = atoi(argv[++i]);
      else if (strncmp(argv[i],"-yg",3) == 0)
	YGamma = atof(argv[++i]);
      else if (strncmp(argv[i],"-rg",3) == 0)
	RGamma = atof(argv[++i]);
      else if (strncmp(argv[i],"-gg",3) == 0)
	GGamma = atof(argv[++i]);
      else if (strncmp(argv[i],"-bg",3) == 0)
	BGamma = atof(argv[++i]);
      else if (strncmp(argv[i],"-BW",3) == 0)
	ColorFlag=0;
      else if (strncmp(argv[i],"-SF",3) == 0)
	SingleFrame=1;
      else if ((argv[i])[0] != '-')
	Basename=argv[i];
      else
	{
	  fprintf(stderr,"Don't know how to do %s\n",argv[i]);
	  exit(-1);
	}
    }

  if ((!Width)||(!Height))
    {
      fprintf(stderr,"Zero Height or Width. Exiting.\n");
      exit(-1);
    }


  YBigDither = (int) (YBIG_DITHER*DitherScale);
  YTinyDither = (int) (YTINY_DITHER*DitherScale);
  UBigDither = (int) (UBIG_DITHER*DitherScale);
  UTinyDither = (int) (UTINY_DITHER*DitherScale);
  VBigDither = (int) (VBIG_DITHER*DitherScale);
  VTinyDither = (int) (VTINY_DITHER*DitherScale);

  if (DoRandom)
    {
      YRandom = (char *) malloc((Width+1)*(Height+1));  /* bunch of chars */
      URandom = (char *) malloc((Width+1)*(Height+1));
      VRandom = (char *) malloc((Width+1)*(Height+1));
      for(rptr=YRandom,i=0;i<(Width+1)*(Height+1);i++)
	{
	  rval=random();
	  if (rval&0x100000)
	    *(rptr++) = rval%YTinyDither;
	  else
	    *(rptr++) =  - (rval%YTinyDither);
	}
      for(rptr=URandom,i=0;i<(Width+1)*(Height+1);i++)
	{
	  rval=random();
	  if (rval&0x100000)
	    *(rptr++) = rval%UTinyDither;
	  else
	    *(rptr++) =  - (rval%UTinyDither);
	}
      for(rptr=VRandom,i=0;i<(Width+1)*(Height+1);i++)
	{
	  rval=random();
	  if (rval&0x100000)
	    *(rptr++) = rval%VTinyDither;
	  else
	    *(rptr++) =  - (rval%VTinyDither);
	}
    }

  CurrentFrame=FirstFrame;
  TheFrame = MakeFrame(Width,Height); /* Need the colormap frame */

  /* Open X server */
  if ((CurrentDisplay = XOpenDisplay(display_name)) == NULL)
    {
      (void) fprintf(stderr, "Cannot connect to X Server\n");
      exit(-1);
    }
  CurrentScreen = DefaultScreen(CurrentDisplay);
  CurrentVisual = DefaultVisual(CurrentDisplay,CurrentScreen);
  switch(CurrentVisual->class)
    {
    case PseudoColor:
      fprintf(stderr,"PseudoColor\n");
      break;
    case GrayScale:
      fprintf(stderr,"GrayScale\n");
      break;
    case StaticColor:
      fprintf(stderr,"StaticColor\n");
      break;
    case StaticGray:
      fprintf(stderr,"StaticGray\n");
      break;
    default:
      fprintf(stderr,"Unknown Color\n");
      break;
    }

  MaximumWidth = DisplayWidth(CurrentDisplay,CurrentScreen);
  MaximumHeight = DisplayHeight(CurrentDisplay,CurrentScreen);
  WindowWidth = Width + MINWIDTH;
  WindowHeight = Height + MINHEIGHT;
  InteriorWidth = WindowWidth - MINWIDTH;
  InteriorHeight = WindowHeight - MINHEIGHT;
  if(MaximumWidth < WindowWidth +2*BORDERWIDTH)
    {
      fprintf(stderr,"Pixmap too large, must clip width \n");
      WindowWidth = MaximumWidth;
    }
  if (MaximumHeight < WindowHeight+ 2* BORDERWIDTH)
    {
      fprintf(stderr,"Pixmap too large, must clip height\n");
      WindowHeight = MaximumHeight;
    }
  GlobalWindow = XCreateSimpleWindow(CurrentDisplay,
				     RootWindow(CurrentDisplay,CurrentScreen),
				     PREFERRED_X, PREFERRED_Y,
				     WindowWidth,WindowHeight,BORDERWIDTH,
				     BlackPixel(CurrentDisplay,CurrentScreen),
				     WhitePixel(CurrentDisplay,CurrentScreen));
  DisplayWindow = XCreateSimpleWindow(CurrentDisplay,
				      GlobalWindow,
				      SIDETRIM, TOPTRIM,
				      InteriorWidth,
				      InteriorHeight,
				      BORDERWIDTH,
				      BlackPixel(CurrentDisplay,
						 CurrentScreen),
				      WhitePixel(CurrentDisplay,
						 CurrentScreen));

  *CurrentColormap = DefaultColormap(CurrentDisplay,CurrentScreen);

  if (ColorFlag) i=MakeYUVmap();
  else i=MakeGraymap();

  if (i)  /* Unable to use current colormap, make a new one */
    {
      *CurrentColormap = XCreateColormap(CurrentDisplay,GlobalWindow,
					 CurrentVisual,AllocNone);
      MyColormap=1;
      if (ColorFlag) i=MakeYUVmap();
      else i=MakeGraymap();
      if (i)
	{
	  fprintf(stderr,"Cannot allocate enough colors\n");
	  exit(-1);
	}
      XInstallColormap(CurrentDisplay,*CurrentColormap); /* Install map */
    }

  gc = XCreateGC(CurrentDisplay,GlobalWindow,valuemask,&gcvalues);
  if (ColorFlag)  /* Recolor screen, with grays */
    {
      XSetWindowBackground(CurrentDisplay,GlobalWindow,
			   ColorIndex[Yindex[64]+Uindex[128]+Vindex[128]]);
      XSetWindowBackground(CurrentDisplay,DisplayWindow,
			   ColorIndex[Yindex[64]+Uindex[128]+Vindex[128]]);
      XSetWindowBorder(CurrentDisplay,GlobalWindow,
		       ColorIndex[Yindex[192]+Uindex[128]+Vindex[128]]);
      XSetWindowBorder(CurrentDisplay,DisplayWindow,
		       ColorIndex[Yindex[192]+Uindex[128]+Vindex[128]]);
    }
  else
    {
      XSetWindowBackground(CurrentDisplay,GlobalWindow,
			   ColorIndex[128]);
      XSetWindowBackground(CurrentDisplay,DisplayWindow,
			   ColorIndex[128]);
      XSetWindowBorder(CurrentDisplay,GlobalWindow,
		       ColorIndex[96]);
      XSetWindowBorder(CurrentDisplay,DisplayWindow,
		       ColorIndex[96]);
    }


  icon_pixmap = XCreateBitmapFromData(CurrentDisplay,GlobalWindow,icon_bits,
				      icon_width,icon_height);
  size_hints.flags = PPosition | PSize | PMinSize;
  size_hints.x = PREFERRED_X;
  size_hints.y = PREFERRED_Y;
  size_hints.width = WindowWidth;
  size_hints.height = WindowHeight;
  size_hints.min_width = MINWIDTH;
  size_hints.min_height = MINHEIGHT;
  XSetStandardProperties(CurrentDisplay,GlobalWindow,
			 window_name,icon_name,
			 icon_pixmap,0,0,&size_hints);
  XSelectInput(CurrentDisplay,GlobalWindow,
	       EnterWindowMask|LeaveWindowMask|
	       KeyPressMask|ButtonPressMask|
	       ExposureMask|StructureNotifyMask);
  XSelectInput(CurrentDisplay,DisplayWindow,
	       EnterWindowMask|LeaveWindowMask|
	       KeyPressMask|ButtonPressMask|
	       ExposureMask|StructureNotifyMask);

  XMapWindow(CurrentDisplay,GlobalWindow);
  XMapWindow(CurrentDisplay,DisplayWindow);
  FrameImage =
    XCreateImage(CurrentDisplay,CurrentVisual,
		 8,ZPixmap,0,
		 TheFrame->data,
		 TheFrame->width,
		 TheFrame->height,
		 8,0);
  FrameImage->byte_order = MSBFirst;

  /* Main handler is here. */
  fprintf(stderr,"Press return in window to exit; any other key in window to start.\n");
  if (SingleFrame)
    {
      if (ColorFlag)
	GetUnknownYUVFrame(TheFrame,Basename);
      else
	GetUnknownBWFrame(TheFrame,Basename);
    }
  while(1)
    {
      XNextEvent(CurrentDisplay,&myreport);
      switch(myreport.type)
	{
	  /* XSetColormapWindow may not be implemented in all machines. */
	  /* Thus we use leave and enter to ensure correct colormaps. */
	case LeaveNotify:  /* Take out colormap when cursor exits screen */
	  if (MyColormap)
	    XUninstallColormap(CurrentDisplay,*CurrentColormap);
	  break;
	case EnterNotify:  /* Put colormap in when cursor enters screen */
	  if (MyColormap)
	    XInstallColormap(CurrentDisplay,*CurrentColormap);
	  break;
	case Expose:
	  while(XCheckTypedEvent(CurrentDisplay,Expose,&myreport));
	  if (!SingleFrame)
	    {
	      if (CurrentFrame<FirstFrame) CurrentFrame=FirstFrame;
	      else if (CurrentFrame>LastFrame) CurrentFrame=LastFrame;
	      sprintf(Indexname,"%s%d",Basename,CurrentFrame);
	      if (ColorFlag)
		GetUnknownYUVFrame(TheFrame,Indexname);
	      else
		GetUnknownBWFrame(TheFrame,Indexname);
	    }
	  XPutImage(CurrentDisplay,DisplayWindow,
		    gc,FrameImage,
		    0,0,0,0,
		    FrameImage->width,
		    FrameImage->height);
	  break;
	case ConfigureNotify:
	  if (myreport.xkey.window == GlobalWindow)
	    {
	      WindowWidth = myreport.xconfigure.width;
	      WindowHeight = myreport.xconfigure.height;
	      InteriorWidth = WindowWidth - MINWIDTH;
	      InteriorHeight = WindowHeight - MINHEIGHT;
	      InteriorWidth = MAX(InteriorWidth,1);
	      InteriorHeight = MAX(InteriorHeight,1);
	      XResizeWindow(CurrentDisplay,DisplayWindow,
			    InteriorWidth,InteriorHeight);
	    }
	  break;
	case KeyPress:
	  if (XLookupKeysym(&myreport,ShiftMapIndex) == XK_Return)
	    CVExit();
	  else if (!SingleFrame)
	    {
	      CurrentFrame=FirstFrame;
	      while(CurrentFrame <= LastFrame)
		{
		  sprintf(Indexname,"%s%d",Basename,CurrentFrame);
		  if (ColorFlag)
		    GetUnknownYUVFrame(TheFrame,Indexname);
		  else
		    GetUnknownBWFrame(TheFrame,Indexname);
		  XPutImage(CurrentDisplay,DisplayWindow,
			    gc,FrameImage,
			    0,0,0,0,
			    FrameImage->width,
			    FrameImage->height);
		  CurrentFrame++;
		}
	    }
	  else
	    {
	      XPutImage(CurrentDisplay,DisplayWindow,
			gc,FrameImage,
			0,0,0,0,
			FrameImage->width,
			FrameImage->height);
	    }
	  break;
	case ButtonPress:
	  break;
	default:
	  break;
	}
    }
}

