/*
$Header: list.c,v 1.8 93/08/30 11:01:31 hmallat Exp $
*/

#include <stdio.h>
#include <stdlib.h>
#include "pixel.h"
#include "block.h"
#include "list.h"
#include "X_misc.h"
#include "undo.h"
#include "tools1.h"
#include "draw.h"
#include "error.h"

/* FUNCTION:       add_view(struct picture *pic_p,
                            int x1, int y1, int x2, int y2, int scale);
   DESCRIPTION:    Add a view as the last item to the view list
   PARAMETERS:
   pic_p           Pointer to the relevant picture structure
   x1, y1          Top left coordinates of the view
   x2, y2          Bottom right coordinates of the view
   scale           Scale of the picture
   RETURNS:        Pointer to the item, or NULL if failed

   Written by Hannu Mallat, last modifications 20th August 1993.
*/
struct view *add_view(struct picture *pic_p,
		      int x1, int y1, int x2, int y2, int scale)
{
  extern struct view *head_p;
  struct view *v_p = NULL, *w_p = head_p;
  
  if(head_p == NULL)
    v_p = (struct view *) malloc(sizeof(struct view));
  else
    {
      while(w_p->next != NULL)
	w_p = w_p->next;
      v_p = (struct view *) malloc(sizeof(struct view));
      if(v_p != NULL)
	w_p->next = v_p;
    }
  if(v_p != NULL)
    {
      v_p->pic_p = pic_p;
      v_p->color1 = WHITE;
      v_p->color2 = BLACK;
      v_p->scale = scale;
      v_p->x1 = x1;
      v_p->y1 = y1;
      v_p->x2 = x2;
      v_p->y2 = y2;
      v_p->draw_area = (Widget) NULL;
      v_p->main = (Widget) NULL;
      v_p->coord_ind = (Widget) NULL;
      v_p->prim_ind = (Widget) NULL;
      v_p->second_ind = (Widget) NULL;
      v_p->h_scroll = (Widget) NULL;
      v_p->v_scroll = (Widget) NULL;
      v_p->zoom = (Widget) NULL;
      v_p->gc = (GC)NULL;
      v_p->busy_view = (Window)NULL;
      v_p->next = NULL;
      if(head_p == NULL)
	head_p = v_p;
    }
  else
    panic("Cannot allocate memory (add_view)", TRUE);

  return v_p;
}

/* FUNCTION:       del_view(struct view *v_p);
   DESCRIPTION:    Delete a view from the list. If the item is the last one, 
                   the relevant data is freed. If the view is the active one,
		   set cur_view and cur_pic to NULL.
   PARAMETERS:
   v_p             Pointer to the item to be deleted
   RETURNS:        Nothing

   Written by Hannu Mallat, last modifications 12th July 1993.
*/   
void del_view(struct view *v_p)
{
  extern struct view *cur_view, *head_p;
  extern struct picture *cur_pic;
  extern struct undo_buffer *ud;
  struct view *prev_p = head_p, *w_p = head_p;
  int nums = 0;

  while(w_p != NULL)
    {
      if(w_p->pic_p == v_p->pic_p)
	nums++;
      w_p = w_p->next;
    }
  if(v_p == head_p)
    head_p = head_p->next;
  else if(v_p != NULL)
    {
      while(prev_p->next != v_p)
	prev_p = prev_p->next;
      prev_p->next = v_p->next;
    }
  if(v_p == cur_view)
    {
      cur_view = NULL;
      cur_pic = NULL;
    }
  if(nums == 1)
    {
      if(ud->pic_p == v_p->pic_p)
	{
	  clear_undo();
	  update_menu();
	}
      delete_block(v_p->pic_p);
    }
  kill_window(v_p);
  free(v_p);
}

/* FUNCTION:       del_views(struct picture *pic_p);
   DESCRIPTION:    Delete all views of a picture from the list.
   PARAMETERS:
   pic_p           Pointer to the picture to be deleted
   RETURNS:        Nothing

   Written by Hannu Mallat, last modifications 12th July 1993.
*/   
void del_views(struct picture *pic_p)
{
  extern struct view *head_p;
  struct view *v_p = head_p, *w_p;

  while(v_p != NULL)
    {
      if(v_p->pic_p == pic_p)
	{
	  w_p = v_p->next;
	  del_view(v_p);
	  v_p = w_p;
	}
      else
	v_p = v_p->next;
    }
}

/* FUNCTION:       del_all();
   DESCRIPTION:    Delete all of the list.
   PARAMETERS:
   none
   RETURNS:        Nothing

   Written by Hannu Mallat, last modifications 12th July 1993.
*/   
void del_all()
{
  extern struct view *head_p;
  struct view *v_p;

  while(head_p != NULL)
    {
      v_p = head_p->next;
      del_view(head_p);
      head_p = v_p;
    }
}

/* FUNCTION:       new_picture(int w, int h, char *name)
   DESCRIPTION:    Create a new picture and arrange the view structure for it
   PARAMETERS:
   w               width
   h               height
   name            pointer to name string
   RETURNS:        Pointer to the view or NULL if failed

   Written by Hannu Mallat, last modifications 13th July 1993.
*/
struct view *new_picture(int w, int h, char *name)
{
  struct picture *pic_p = NULL;
  struct view *v_p = NULL;

  pic_p = create_block(name, w, h);
  if(pic_p != NULL)
    v_p = add_view(pic_p, 0, 0, w-1, h-1, 1);
  return v_p;
}

/* FUNCTION:       abandon_picture()
   DESCRIPTION:    Delete current picture and its views
   PARAMETERS:     
   none
   RETURNS:        Nothing

   Written by Hannu Mallat, last modifications 20th July 1993.
*/   
void abandon_picture()
{
  extern struct picture *cur_pic;

  if(cur_pic != NULL)
    del_views(cur_pic);
}

/* FUNCTION:       abandon_all()
   DESCRIPTION:    Abandon all pictures
   PARAMETERS:
   none
   RETURNS:        Nothing

   Written by Hannu Mallat, last modifications 12th July 1993.
*/   
void abandon_all()
{
  extern struct picture *cur_pic;
  extern struct view *head_p;

  while(head_p != NULL)
    {
      cur_pic = head_p->pic_p;
      abandon_picture();
    }
}

/* FUNCTION:       touch(struct picture *pic_p)
   DESCRIPTION:    Mark the picture as touched, that is, that it has been 
                   operated on but not yet saved.
   PARAMETERS:
   pic_p           Pointer to the picture
   RETURNS:        
   Nothing

   Written by Hannu Mallat, last modifications 28th August 1993.
*/   
void touch(struct picture *pic_p)
{
  extern struct view *head_p;
  struct view *v_p = head_p;

  if((pic_p->flags & TOUCHED) == 0)
    {
      pic_p->flags = pic_p->flags | TOUCHED;
      while(v_p != NULL)
	{
	  if(v_p->pic_p == pic_p)
	    show_zoom(v_p);
	  v_p = v_p->next;
	}
    }
}

/* FUNCTION:       untouch(struct picture *pic_p)
   DESCRIPTION:    Mark the picture as untouched, that is, no changes after
                   the last save has been done. The save function calls this.
   PARAMETERS:
   pic_p           Pointer to the picture
   RETURNS:        
   Nothing

   Written by Hannu Mallat, last modifications 28th August 1993.
*/   
void untouch(struct picture *pic_p)
{
  extern struct view *head_p;
  struct view *v_p = head_p;

  if((pic_p->flags & TOUCHED) != 0)
    {
      pic_p->flags = pic_p->flags & (!TOUCHED);
      while(v_p != NULL)
	{
	  if(v_p->pic_p == pic_p)
	    show_zoom(v_p);
	  v_p = v_p->next;
	}
    }
}

/* FUNCTION:       is_touched(struct view *v_p)
   DESCRIPTION:    Tell the X module whether the image to which the view
                   points to has been touched or not.
   PARAMETERS:
   *v_p            the view
   RETURNS:        nothing
   
   Written by Hannu Mallat, last modifications 28th August 1993.
*/   
int is_touched(struct view *v_p)
{
  return (v_p->pic_p->flags & TOUCHED);
}
