/*
 File: Sloan.c
 Authors: K.R. Sloan
 Last Modified: 2 November 1992
 Purpose: decides if a set of points is convex, 
          in O(n) time and O(1) space.
 */
#include <stdio.h>
#include <math.h>

static int VERBOSE = 0;
static int REASON = -1;
static char *RoutineName;
static void usage()
 {
  fprintf(stderr,"Usage is\n\t%s [-h][-v]\n",
               RoutineName);
 }

static void FatalError(s)
 char *s;
 {
  fprintf(stderr,"%s: FatalError(%s)\n",RoutineName,s);
  exit(-1);
 }

#define TRUE (-1)
#define FALSE (0)
#define EPSILON (0.0000000001)

typedef struct Point { double x, y; } Point;
static int NextPoint(s,P)
 FILE *s;
 Point *P;
 {
  if (feof(s)) return -1;
  if (2 != fscanf(s," %lf %lf", &(P->x), &(P->y)))
   return -1;
  if (VERBOSE) fprintf(stderr," %lf %lf\n",P->x,P->y);
  return 0;
 }

/*
  Classify the angle A,B,C into the following cases:

    -1 - turns left
     0 - straight (or close to it)
    +1 - turns right
 */

static int ClassifyAngle (A, B, C)
 Point A, B, C;
  {
   double CrossProduct;
   CrossProduct = ((A.x-B.x)*(C.y-B.y)) - ((A.y-B.y)*(C.x-B.x));
   if (CrossProduct < -EPSILON) return -1;
   if (CrossProduct >  EPSILON) return  1;
   return 0;
  }

/*
  Determine if the vectors AB and BC point in roughly the same
  direction.  Used below to catagorize situations where A,B,C are
  colinear.
 */
static int Monotonic(A,B,C)
 Point A,B,C;
 {
  double ABdx, ABdy, BCdx, BCdy;
  double DotProduct;

  ABdx = B.x - A.x; ABdy = B.y - A.y;
  BCdx = C.x - B.x; BCdy = C.y - B.y;

  DotProduct = (ABdx*BCdx) + (ABdy*BCdy);
  return (-EPSILON <= DotProduct); /* zero, if you are brave */
 }

/*
   FindTurn returns the sense of the next turn.

   sense 0 means that the first turn is a 180deg about face.

   n is 3 if a turn is found, 0,1,2 if we run out of points.

   On entry: n is the number of points already in P
 */
static int FindTurn(s,P,n)
 FILE *s;
 Point P[];
 int *n;
 {
  int i;
  int sense;
  
  /* get three points until we have 3*/
  for(i = *n;i<3;i++) 
   if (0 != NextPoint(s,&P[i]))     
    { 
     if (VERBOSE) fprintf(stderr,"FindTurn: NO MORE\n");
     *n = i;
     return 0; 
    } 
  /*
    if colinear and monotonic,
    discard middle point and get another third point
   */
  for(;0 == (sense = ClassifyAngle(P[0],P[1],P[2]));)
   {
    if (!Monotonic(P[0],P[1],P[2])) 
     {
      if (VERBOSE) fprintf(stderr,"FindTurn: REVERSE\n");
      *n = 3; 
      return 0;
     }
    P[1] = P[2];
    if (0 != NextPoint(s,&P[2]))
     {
      if (VERBOSE) fprintf(stderr,"FindTurn: NO MORE\n");
      *n = 2;
      return 0;
     }
   }

  /* we have three non-colinear points, and know which way they turn */
  if (VERBOSE) 
   {
    if (-1 == sense)     fprintf(stderr,"FindTurn:  LEFT\n");
    else if (1 == sense) fprintf(stderr,"FindTurn:  RIGHT\n");
    else                 fprintf(stderr,"FindTurn:  STRAIGHT\n");
   }
  *n = 3; return sense;
 } 

static int ConvexP(s)
 FILE *s;
 {
  int i;
  int first, more, lastleg; 
  int sense1,sense,n;
  Point V1[3], V[3];

  n = 0; 
  sense1 = FindTurn(s,V1,&n);
  if (3 > n) return TRUE;  /* if you like degenerate polygons */

  if (0 == sense1)
   {
    /*
      This polygon is either not-convex, or it is a single line.

      V1[0] is the initial point
      V1[1] is the extreme point
      V1[2] is on the same side of V1[1] as V1[0].  

      We have either:

                             0      2      1
      or 
                     2       0             1

      or even more degeneracy ...

      We can turn around (!Monotonic) either 0 or 1 more time.
      If we turn around again, we can't pass V1[0].

      Use FindTurn twice more, and check all of the cases
     
     */
     
    V[0] = V1[1]; V[1] = V1[2]; n = 2; /* prime FindTurn */
    sense = FindTurn(s,V,&n);
    if (3 > n)      return TRUE;  /* all is well - a line */
    if (0 != sense) return FALSE; /* we left the line     */

    V1[2] = V[1];   /* note the second turn */

    /* if we turned before reaching V1[0] again, we lose */
    if (!Monotonic(V1[1],V1[0],V1[2])) return FALSE;

    /*
      if we passed V1[0] and then turned, we need to search for
      yet another turn (we lose).
     */
    V[0] = V[1]; V[1] = V[2]; n = 2;
    sense = FindTurn(s,V,&n);
    if (3 == n) return FALSE; /* we left the line */

    /*
       finally, make sure that the last point didn't pass
       V1[0] (again!)
     */ 
    if (!Monotonic(V[0],V[1],V1[0])) return FALSE; 
    
    return TRUE;  /* weird, but true */
                  /* if you don't like degenerate polygons,
                      return FALSE  and throw away much of the above */
   } 

  /*
      Normal case - we have an initial turn with angle a.  0 < a < pi. 
   */

  if (VERBOSE) 
   for(i=0;i<3;i++) 
    fprintf(stderr,"    V1[%d] = (%lf,%lf)\n",i,V1[i].x,V1[i].y);


  for(i=0;i<3;i++) V[i] = V1[i];
  more = TRUE; first = TRUE; lastleg = FALSE;
  for(;more;)
   {
    /* find the next true turn */
    V[0] = V[1]; V[1] = V[2]; n = 2;
    sense = FindTurn(s,V,&n);    
    if (3 > n)
     {  /* no more points */
      if (first) return TRUE;  /* a triangle */
      V[2] = V1[0];            /* complete the polygon */
      /*
         if this final leg is straight, 
         we need only check that we didn't pass the original point
         and double back
       */
      sense = ClassifyAngle(V[0],V[1],V[2]);
      if (0 == sense) 
       {
        if (Monotonic(V[0],V[1],V[2])) return TRUE; else return FALSE; 
       }
      more = FALSE;
     }
    first = FALSE; 

    /*
      if a previous leg pointed directly at the initial point,
      and we have reached this point, then we have either
      backtracked, or we have turned off that line.
      In either case, we lose
     */ 

    if (lastleg) return FALSE; /* once on the last leg, we can't turn */

    /*
       So much for degeneracies, and end conditions
       Now, check generic vertices
     */

    if (VERBOSE) 
     for(i=0;i<3;i++) 
      fprintf(stderr,"        V[%d] = (%lf,%lf)\n",i,V[i].x,V[i].y);

    /* all turns must match the first turn */
    if (sense1 == -sense) 
     {
      if (REASON)
       {
        fprintf(stderr,"Wrong turn: (%lf, %lf)\n",V[0].x,V[0].y);
        fprintf(stderr,"            (%lf, %lf)\n",V[1].x,V[1].y);
        fprintf(stderr,"            (%lf, %lf)\n",V[2].x,V[2].y);
       }  
      return FALSE;  /* wrong turn */
     }
    if (0 == sense)       return FALSE;  /* no backtracking */

    /*
       all triangles constructed from the first vertex
       and two successive vertices must have areas with the
       same sign.  

     */
    sense = ClassifyAngle(V1[0],V[1],V[2]);
    if (sense1 == -sense) return FALSE; /* negative area triangle */  

    /*
       if the triangle has zero area, then this leg points directly
       at the initial point.  For this polygon to be convex, 
       all remaining points must lie between V[2] and V1[0] 
       and approach V1[0] monotonically
    
       FindTurn will handle all of these cases.  If the polygon
       is acceptable, then FindTurn (and the last point processing
       above) will discover no future turns.  If a future turn
       is discovered, the check above will note that turns are
       no longer allowed.  So - simple note that we're on the
       last leg, and see what happens...

       Note: we depend here on the fact that FindTurn has
             eliminated duplicate points.  All other causes
             of zero area triangles (except the last leg line)
             indicate a concavity, or non-simplicity
     */
     if (0 == sense) lastleg = TRUE; /* forbid further turns */

    /*
       all vertices must be on the same side of (or perhaps ON)
       the initial edge.

       This test ensures that the first vertex lies on the convex
       hull.

       Clearly, if the first vertex is NOT on the hull, then
       the polygon is not convex.  It is less easy to show, but
       nonetheless true, that if the first vertex IS on the hull
       then the earlier tests prove that the triangle IS convex.

       But, we don't need this test on the very last iteration, because
       V1[0] == V[2].
     */
    if (more) 
     {
      sense = ClassifyAngle(V1[0],V1[1],V[2]);
      if (sense1 == -sense) return FALSE;
     }
   } 
 
  /* seeing no objection... */
  return TRUE;
 }

int main (argc, argv)
 int argc;
 char *argv[];
 {
  int ArgsParsed=0;

  RoutineName = argv[ArgsParsed++];

  while (ArgsParsed < argc)
   {
    if ('-' == argv[ArgsParsed][0])
     switch (argv[ArgsParsed++][1])
      {
       case 'v': VERBOSE  = -1; break;
       default:
       case 'h': usage(); exit(-1);
      }
   }

  if (ConvexP(stdin))
   { fprintf(stdout,"CONVEX AND SIMPLE\n"); exit(0);}
  else
   { fprintf(stdout,"NOT CONVEX OR NOT SIMPLE.\n"); exit(-1);}

  exit(0);
 }
