/*LINTLIBRARY*/

/*  @(#)xview.c 1.3 92/04/16
 *
 *  XView dependent graphics routines used by popi.
 *  written by Rich Burridge - Sun Microsystems.
 *
 *  Popi was originally written by Gerard J. Holzmann - AT&T Bell Labs.
 *  This version is based on the code in his Prentice Hall book,
 *  "Beyond Photography - the digital darkroom," ISBN 0-13-074410-7,
 *  which is copyright (c) 1988 by Bell Telephone Laboratories, Inc. 
 *
 *  Permission is given to distribute these extensions, as long as these
 *  introductory messages are not removed, and no monies are exchanged.
 *
 *  No responsibility is taken for any errors or inaccuracies inherent
 *  either to the comments or the code of this program, but if reported
 *  (see README file) then an attempt will be made to fix them.
 */

#include "popi.h"
#include <xview/xview.h>
#include <xview/canvas.h>
#include <xview/cms.h>
#include <xview/cursor.h>
#include <xview/panel.h>
#include <xview/svrimage.h>
#include <xview/tty.h>
#include <xview/termsw.h>
#include <xview/xv_xrect.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>

#define  DUP2                           (void) dup2
#define  SIGNAL                         (void) signal

#define  NOTIFY_DISPATCH                (void) notify_dispatch
#define  NOTIFY_DO_DISPATCH             (void) notify_do_dispatch
#define  NOTIFY_INTERPOSE_DESTROY_FUNC  (void) notify_interpose_destroy_func
#define  XV_SET                         (void) xv_set

Canvas canvas ;
Canvas_paint_window cpw ;
Colormap cmap = NULL ;
Cms cms ;
Frame cframe, tframe ;
Icon popi_icon ;
Notify_value destroy_proc() ;
Panel panel ;
Panel_item slider ;
Server_image busy_cursor_pr, icon_pr ;
Tty ttysw ;
Xv_Cursor busy_cursor, main_cursor ;

Display *dpy ;                  /* Display id of popi frame. */
Drawable xid ;                  /* Xid for popi canvas. */
GC gc ;                         /* Graphics context for text and lines. */
GC pix_gc ;                     /* Graphics context for rops. */
Pixmap memarea, mpr ;
Window root ;
XColor ccol ;
XGCValues gc_val ;              /* Used to setup graphics context values. */

int colsused = 0 ;
int gc_flags ;                  /* Used to set up graphics context flags. */
int screen ;                    /* Default graphics display screen. */

unsigned long backgnd ;         /* Default background color. */
unsigned long foregnd ;         /* Default foreground color. */
unsigned long gc_mask ;         /* Mask for setting graphic context values. */
unsigned long palette[CMAPLEN] ;

static unsigned short busy_cursor_array[] = {
#include <images/hglass.cursor>
} ;

static unsigned short icon_image[] = {
#include "popi.icon"
} ;

extern int errno ;
extern char *sys_errlist[] ;

char ttybuf[MAXLINE] ;       /* Input from user in ttysw window. */
unsigned char *mptr ;        /* Pointer to scanline data. */

int iscolor ;                /* Set if this is a color screen. */
int opercent = -1 ;          /* Previous display percent done value. */
int tbufptr ;                /* Current pointer into ttybuf. */

/*ARGSUSED*/
static void
canvas_repaint(canvas, window, display, xid, xrects)
Canvas canvas ;
Xv_Window window ;
Display *display ;
Xv_Window xid ;
Xv_xrectlist *xrects ;
{
  XCopyArea(dpy, memarea, xid, gc, 0, 0, Xsize, Ysize, 0, 00) ;
}


/*  These are the exportable routines used by the popi program.
 *
 *  disp_init(argc, argv)    - called from main at the start.
 *  disp_finish()            - called from main prior to exit.
 *  disp_imgstart(w,h,c,l)   - called prior to drawing an image.
 *  disp_imgend()            - called after drawing an image.
 *  disp_putline(l,y,w,c)    - to draw an image scanline triple.
 *  disp_getchar()           - to get the next character typed.
 *  disp_prompt()            - display popi prompt and clear input buffer.
 *  disp_error(errtype, pos) - display error message.
 *  disp_percentdone(n)      - display percentage value of conversion.
 *  disp_resize(w,h)         - resize popi image window (width, height).
 *  disp_colormap(n,r,g,b)   - load new colormap.
 */


void
disp_init(argc, argv)        /* Called from main at the start. */
int argc ;
char *argv[] ;
{
  char title[MAXLINE] ;      /* Used to constructure window title lines. */
  int cht ;                  /* Height of the popi term window. */
  int cx ;                   /* X position of the popi term window. */
  int cy ;                   /* Y position of the popi term window. */
  int ttyfd ;                /* File descriptor for tty subwindow. */
  int i ;
  XSetWindowAttributes winattrs ;

  STRCPY(ttybuf, "") ;       /* Zeroise tty input buffer. */
  tbufptr = 0 ;              /* Reset tty buffer pointer. */

  xv_init(XV_INIT_ARGS, argc, argv, 0) ;
  icon_pr = xv_create(XV_NULL,            SERVER_IMAGE,
                      SERVER_IMAGE_BITS,  icon_image,
                      SERVER_IMAGE_DEPTH, 1,
                      XV_WIDTH,           64,
                      XV_HEIGHT,          64,
                      0) ;
  popi_icon = xv_create(XV_NULL,    ICON,           /* Create icon. */
                        ICON_IMAGE, icon_pr,
                        0) ;

  SPRINTF(title, "%s commands", ProgName) ;
  tframe = xv_create(XV_NULL,     FRAME,            /* Create tty frame. */
                     FRAME_ICON,  popi_icon,
                     WIN_ROWS,    10,
                     WIN_COLUMNS, 80,
                     FRAME_LABEL, title,
                     0) ;

  ttysw = xv_create(tframe,   TERMSW,
                    TTY_ARGV, TTY_ARGV_DO_NOT_FORK,
                    0) ;
  XV_SET(ttysw, TERMSW_MODE, TERMSW_MODE_TYPE, 0) ;

  cx  = (int) xv_get(tframe, XV_X) ;
  cy  = (int) xv_get(tframe, XV_Y) ;
  cht = (int) xv_get(tframe, XV_HEIGHT) ;
  SPRINTF(title, "%s image canvas", ProgName) ;
  cframe = xv_create(XV_NULL,          FRAME,       /* Create canvas frame. */
                     XV_X,             cx,
                     XV_Y,             cy + cht + 10,
                     FRAME_LABEL,      title,
                     FRAME_SHOW_LABEL, TRUE,
                     FRAME_SHOW_RESIZE_CORNER, FALSE,
                     0) ;
  panel = xv_create(cframe,  PANEL,             /* Create panel for slider. */
                    XV_X,    0,
                    XV_Y,    0,
                    XV_SHOW, FALSE,
                    0) ;
  slider = xv_create(panel,              PANEL_SLIDER,
                     PANEL_LABEL_STRING, "% done",
                     PANEL_MIN_VALUE,    0,
                     PANEL_MAX_VALUE,    100,
                     0) ;
  canvas = window_create(cframe,                CANVAS,   /* Create canvas. */
                         XV_X,                  0,
                         XV_Y,                  0,
                         XV_WIDTH,              Xsize,
                         XV_HEIGHT,             Ysize,
                         CANVAS_X_PAINT_WINDOW, TRUE,
                         CANVAS_REPAINT_PROC,   canvas_repaint,
                         CANVAS_RETAINED,       FALSE,
                         0) ;
  cpw = canvas_paint_window(canvas) ;
  window_fit(cframe) ;

  dpy = (Display *) xv_get(cframe, XV_DISPLAY) ;
  xid = (Drawable) xv_get(cpw, XV_XID) ;

  screen    = DefaultScreen(dpy) ;
  root      = RootWindow(dpy, screen) ;
  scr_depth = DefaultDepth(dpy, screen) ;

  iscolor = (scr_depth > 2) ;
  if (dtype == IS_MONO) iscolor = 0 ;
  if (!iscolor) dtype = IS_MONO ;

  foregnd   = BlackPixel(dpy, screen) ;
  backgnd   = WhitePixel(dpy, screen) ;

  gc_mask           = GCForeground | GCBackground | GCGraphicsExposures ;
  gc_val.foreground = foregnd ;
  gc_val.background = backgnd ;
  gc_val.graphics_exposures = False ;
  gc     = XCreateGC(dpy, root, gc_mask, &gc_val) ;
  pix_gc = DefaultGC(dpy, screen) ;

  memarea = XCreatePixmap(dpy, xid, Xsize, Ysize, scr_depth) ;
  gc_val.foreground = backgnd ;
  gc_val.background = foregnd ;
  XChangeGC(dpy, gc, GCBackground | GCForeground, &gc_val) ;
  XFillRectangle(dpy, memarea, gc, 0, 0,
                 (unsigned int) Xsize, (unsigned int) Ysize) ;

  SIGNAL(SIGHUP, SIG_IGN) ;
  ttyfd = (int) xv_get(ttysw, TTY_TTY_FD) ;
  DUP2(ttyfd, 0) ;
  DUP2(ttyfd, 1) ;
  DUP2(ttyfd, 2) ;

  main_cursor = xv_get(canvas, WIN_CURSOR) ;

  busy_cursor_pr = xv_create(XV_NULL,           SERVER_IMAGE,
                             XV_WIDTH,          16,
                             XV_HEIGHT,         16,
                             SERVER_IMAGE_BITS, busy_cursor_array,
                             0) ;

  busy_cursor = xv_create(XV_NULL,      CURSOR,
                          CURSOR_IMAGE, busy_cursor_pr,
                          0) ;

  NOTIFY_INTERPOSE_DESTROY_FUNC(cframe, destroy_proc) ;
  NOTIFY_INTERPOSE_DESTROY_FUNC(tframe, destroy_proc) ;
  XV_SET(tframe, XV_SHOW, TRUE, 0) ;
  XV_SET(cframe, XV_SHOW, TRUE, 0) ;
  NOTIFY_DISPATCH() ;        /* Make the windows appear. */
  NOTIFY_DO_DISPATCH() ;
  init_dither() ;    /* Initialise dither arrays and variables. */
}


void
disp_finish()    /* Called from main prior to exit - null routine. */
{}



/* Called prior to drawing an image. */

void
disp_imgstart(width, height, ncolors, len)
int width, height, ncolors, len ;
{
  XV_SET(cframe, WIN_SHOW, TRUE, 0) ;
  XClearWindow(dpy, xid) ;
  XV_SET(panel, WIN_SHOW, FALSE, 0) ;
  NOTIFY_DISPATCH() ;
}


void
disp_imgend()                /* Called after drawing an image. */
{}


void
disp_putline(lines, y, width, ncolors)      /* Draw an image scanline. */
unsigned char **lines ;
int y, width, ncolors ;
{
  XImage *image ;
  int color, i, j, len, off ;
  unsigned char *line, tmp ;

       if (scr_depth == 32) len = width * 4 ;
  else if (scr_depth == 8)  len = width ;
  else                      len = (width / 8) + 1 ;
  mptr = (unsigned char *) Emalloc(len) ;
  if ((dtype == IS_GRAY || dtype == IS_MONO) && ncolors == 3)
    line = ntsc_luma(lines, y, width) ;
  else
    line = &lines[0][y*width] ;

  if (scr_depth == 32)
    {
      off = y * width ;
      if (colors == 1)
        for (i = j = 0; i < width; i++)
          {
            tmp       = lines[0][off+i] ;
            mptr[j++] = 0 ;                           /* X */
            mptr[j++] = tmp ;                         /* B */
            mptr[j++] = tmp ;                         /* G */
            mptr[j++] = tmp ;                         /* R */
          }
        else
          for (i = j = 0; i < width; i++)
            {
              mptr[j++] = 0 ;                         /* X */
              for (color = 2; color >= 0; color--)
                mptr[j++] = lines[color][off+i] ;     /* G, B, R */
            }
    }
  else if (iscolor)
    {
      for (i = 0; i < len; i++) mptr[i] = palette[line[i]] ;
    }
  else
    { 
      halftone(line, y, width) ;
    }

  image = XCreateImage(dpy, DefaultVisual(dpy, screen), scr_depth,
                       (iscolor) ? ZPixmap : XYPixmap,
                       0, (char *) mptr, width, 1, 8, len) ;
  XPutImage(dpy, memarea, pix_gc, image, 0, 0, 0, y,
            (unsigned) image->width, (unsigned) image->height) ;
  XDestroyImage(image) ;
  XCopyArea(dpy, memarea, xid, gc, 0, y, width, 1, 0, y) ;
  FREE(mptr) ;
  NOTIFY_DISPATCH() ;
}


disp_getchar()                /* Get next user typed character. */
{
  int c ;

  if (!tbufptr) (void) gets(ttybuf) ;  /* If buffer empty, get next line. */
  c = ttybuf[tbufptr++] ;  
  if (c == '\0')                       /* Is it end of input buffer? */
    {
      c = '\n' ;
      tbufptr = 0 ;                    /* Reset to force another read. */
      set_cursor(BUSY_CUR) ;           /* We will now get busy! */
    }
  return c ;
}


disp_prompt()        /* Display popi prompt and clear input line. */
{
  char *prompt = "\r-> " ;

  set_cursor(NORMAL_CUR) ;
  PRINTF(prompt) ;
  FFLUSH(stdout) ;
  return(sizeof prompt - 1) ;
}


void
disp_error(errtype, pos)    /* Display error message. */
int errtype ;
int pos ;
{
  int i ;

  if (errtype & ERR_PARSE)
    {
      for (i = 1; i < pos; ++i) PUTC('-', stderr) ;
      PUTC('^', stderr) ;
      PUTC('\n', stderr) ;
    }

  FPRINTF(stderr, "%s\n", ErrBuf) ;

/* We assume errno hasn't been reset by the preceding output */

  if (errtype & ERR_SYS)
    FPRINTF(stderr, "\t(%s)\n", sys_errlist[errno]) ;
  FFLUSH(stderr) ;
}


void
disp_percentdone(percent)
int percent ;
{
  if (percent == opercent) return ;   /* Same as last time? */
  XV_SET(slider, PANEL_VALUE, percent, 0) ;
  if (percent == 0)   XV_SET(panel, WIN_SHOW, TRUE,  0) ;
  if (percent == 100) XV_SET(panel, WIN_SHOW, FALSE, 0) ;
  opercent = percent ;
  NOTIFY_DISPATCH() ;
}


void
disp_resize(width, height)    /* Resize popi image window. */
int width, height ;
{
  Rect *r ;
  Window w ;
  XSizeHints hints ;
  long supplied ;

  r = (Rect *) xv_get(cframe, FRAME_OPEN_RECT) ;
  r->r_width  = width ;
  r->r_height = height  ;
  XV_SET(cframe, FRAME_OPEN_RECT, r, 0) ;

  XFreePixmap(dpy, memarea) ;
  memarea = XCreatePixmap(dpy, xid, width, height, scr_depth) ;
  gc_val.foreground = backgnd ;
  gc_val.background = foregnd ;
  XChangeGC(dpy, gc, GCBackground | GCForeground, &gc_val) ;
  XFillRectangle(dpy, memarea, gc, 0, 0,
                 (unsigned int) width, (unsigned int) height) ;
}


void
disp_colormap(cmaplen, red, green, blue)      /* Load new colormap. */
int cmaplen ;
unsigned char *red, *green, *blue ;
{
  int i, newmap ;
  Visual *visual ;
  XSetWindowAttributes winattrs ;

  if (DisplayCells(dpy, screen) <= 2) return ;
  visual = DefaultVisual(dpy, screen) ;
  if (colsused) XFreeColors(dpy, cmap, palette, colsused, 0) ;
  if (cmap != NULL) XFreeColormap(dpy, cmap) ;

  newmap = 0 ;
  cmap = DefaultColormap(dpy, screen) ;

/*  Attempt to use the default colormap. If we can't allocate all the colors
 *  we need, then attempt to create a private colormap.
 */

  ccol.flags = DoRed | DoGreen | DoBlue ;
  iscolor = 1 ;
  for (i = 0; i < CMAPLEN; i++)
    {
      ccol.red   = red[i]   << 8 ;
      ccol.green = green[i] << 8 ;
      ccol.blue  = blue[i]  << 8 ;
      if (!XAllocColor(dpy, cmap, &ccol))
        if ((visual->class == StaticColor) ||
            (visual->class == StaticGray))
          {
            FPRINTF(stderr, "popi: XAllocColor failed on a static visual\n") ;
            iscolor = 0 ;
            return ;
          }
        else
          { 

/*  We can't allocate the colors shareable so free all the colors we had
 *  allocated and create a private colormap.
 */

            XFreeColors(dpy, cmap, palette, i, 0) ;
            newmap = 1 ;
            break ;
          }
        palette[i] = ccol.pixel ;
    }

  if (newmap)
    {
      if ((visual->class == PseudoColor) || (visual->class == GrayScale))
        cmap = XCreateColormap(dpy, root, visual, AllocNone) ;
      else
        cmap = XCreateColormap(dpy, root, visual, AllocAll) ;

      if (!XAllocColorCells(dpy, cmap, False, NULL, 0, palette, cmaplen))
        {
          FPRINTF(stderr, "popi: failed to allocate private colormap.\n") ;
          iscolor = 0 ;
          return ;
        } 

      ccol.flags = DoRed | DoGreen | DoBlue ;
      for (i = 0; i < cmaplen; i++)
        {
          ccol.pixel = palette[i] ;
          ccol.red   = red[i]   << 8 ;
          ccol.green = green[i] << 8 ;
          ccol.blue  = blue[i]  << 8 ;
          XStoreColor(dpy, cmap, &ccol) ;
        } 
    }

  winattrs.colormap = cmap ;
  XChangeWindowAttributes(dpy, xv_get(cframe, XV_XID),
                          CWColormap, &winattrs) ;
}


/*ARGSUSED*/
Notify_value
destroy_proc(client, status)
Notify_client client ;
Destroy_status status ;
{
  exit(0) ;
}


set_cursor(type)
enum cur_type type ;
{
  switch (type)
    {
        case BUSY_CUR  : XV_SET(canvas, WIN_CURSOR, busy_cursor, 0) ;
                         break ;
        case NORMAL_CUR: XV_SET(canvas, WIN_CURSOR, main_cursor, 0) ;
    }
}
