
/*
 * suntohips.c   converts an sun raster image to a HIPS image
 *
 * written 7/90 by Brian Tierney,  LBL cloned from suntohips.c by Mike Landy
 *
 * This program maps the sunraster colormap to grayscale based the the
 * percentages defined below
 */

#define CVT(x)  (((x)*255)/100)
int       RED = CVT(28);	/* 28% */
int       GREEN = CVT(59);	/* 59% */
int       BLUE = CVT(11);	/* 11% */

/* #define MSBFVERSION  defined in the Makefile  */

#include <stdio.h>
#include <hipl_format.h>
#include <pixrect/pixrect_hs.h>

#define GRAYDIFF 4		/* maximum difference between rgb values to
				 * be considered a gray scale image */

char     *Progname;

int       gray_flg = 0;

main(argc, argv)
    int       argc;
    char     *argv[];
{
    colormap_t colormap;
    Pixrect  *image_in, *pr_load_image();
    struct header hd;
    u_char   *image_out, *color_lut, *ptr;
    int       npix, pixel, wsize;
    register int i, j;
    int       rows, cols, depth;
    int       ras_width, skip_size;

    void      usage(), build_colormap();

    Progname = strsave(*argv);

    if (argc > 1) {
	if (strcmp(argv[1], "-g") == 0)
	    gray_flg++;
	else
	    usage();
    }
    if (!(image_in = pr_load(stdin, &colormap))) {
	fprintf(stderr, "%s: error reading rasterfile \n\n", argv[0]);
	exit(1);
    }
    rows = image_in->pr_height;
    cols = image_in->pr_width;
    depth = image_in->pr_depth;
    npix = cols * rows;

    if (depth == 1) {
#ifdef MSBFVERSION
	init_header(&hd, "", "", 1, "", rows, cols, 1, MSBFIRST, PFBYTE, "");
#else
	init_header(&hd, "", "", 1, "", rows, cols, 1, LSBFIRST, PFBYTE, "");
#endif				/* MSBFVERSION */
    } else if (depth == 8) {
	init_header(&hd, Progname, "", 1, "", rows, cols, 8, 0, PFBYTE, "");
    } else
	perr("bits per pixel must be 1 or 8");

    update_header(&hd, argc, argv);
    write_header(&hd);

    ptr = (u_char *) mpr_d(image_in)->md_image;

    /*
     * sun rasterfiles pad the width to multiples of 8, so we have to skip
     * any extra bytes at the end of each scan line
     */
    ras_width = mpr_d(image_in)->md_linebytes;
    skip_size = ras_width - cols;

    if (colormap.length > 0) {

	if ((color_lut = (u_char *) malloc(colormap.length *
					   sizeof(u_char))) == NULL) {
	    fprintf(stderr, "%s: can't allocate core\n", argv[0]);
	    exit(1);
	}
	build_colormap(&colormap, color_lut);

	if ((image_out = (u_char *) malloc(npix * sizeof(u_char))) == NULL) {
	    fprintf(stderr, "%s: can't allocate core\n", argv[0]);
	    exit(1);
	}
	/* remap image */
	j = 0;
	for (i = 0; i < npix; i++) {
	    pixel = (int) ptr[j++];
	    if (skip_size > 0 && i % cols == 0)
		j = j + skip_size;
	    if (pixel < 0 || pixel > colormap.length)
		fprintf(stderr, " Error: Invalid value in rasterfile \n");
	    image_out[i] = color_lut[pixel];
	}
    }
    if (depth == 1) {
	wsize = (cols + 7) / 8 * rows * sizeof(char);
	if (write(1, (char *) ptr, wsize) != wsize)
	    perr("error during write");

    } else {
	wsize = npix * sizeof(char);
	if (write(1, (char *) image_out, wsize) != wsize)
	    perr("error during write");

	exit(0);
    }
}

/*******************************************************/
void
build_colormap(ras_colormap, colorbuf)
    colormap_t ras_colormap;
    u_char   *colorbuf;
{
    int       i, gray;

    if (gray_flg)
	fprintf(stderr, "Gray scale input option selected.\n");
    else {
	for (i = 0; i < ras_colormap.length; i++) {
	    gray_flg = 1;	/* assume gray image to start */
	    if (abs(ras_colormap.map[0][i] - i) >= GRAYDIFF &&
		abs(ras_colormap.map[1][i] - i) >= GRAYDIFF &&
		abs(ras_colormap.map[2][i] - i) >= GRAYDIFF) {
		gray_flg = 0;
		break;
	    }
	}
	if (gray_flg)
	    fprintf(stderr, "Image appears to be gray-scale: using gray colormap.\n");
    }

    for (i = 0; i < ras_colormap.length; i++) {
	if (gray_flg)
	    gray = ((ras_colormap.map[0][i] +
		     ras_colormap.map[1][i] +
		     ras_colormap.map[2][i]) / 3.0 + .5);
	else
	    gray = ((RED * ras_colormap.map[0][i]) +
		    (BLUE * ras_colormap.map[1][i]) +
		    (GREEN * ras_colormap.map[2][i])) >> 8;
	colorbuf[i] = (u_char) gray;

#ifdef DEBUG
	fprintf(stderr, " colormap loc %d = %d \n", i, colorbuf[i]);
#endif
    }

}

/*******************************************************/
void
usage()
{
    fprintf(stderr, " Usage: suntohips2 [-g] < infile > outfile \n");
    fprintf(stderr, "    where infile is a sunraster image, and outfile is a HIPS image \n");
    fprintf(stderr, "      [-g] assume input image is a gray scale image.\n\n");
    exit(0);
}
