/* procs.c      Distribution 1.2   91/1/28   Scry */

/*   The Scry system is copyright (C) 1988-1991 Regents  of  the
University  of  California.   Anyone may reproduce ``Scry'',
the software in this distribution, in whole or in part, pro-
vided that:

(1)  Any copy  or  redistribution  of  Scry  must  show  the
     Regents  of  the  University of California, through its
     Lawrence Berkeley Laboratory, as the source,  and  must
     include this notice;

(2)  Any use of this software must reference this  distribu-
     tion,  state that the software copyright is held by the
     Regents of the University of California, and  that  the
     software is used by their permission.

     It is acknowledged that the U.S. Government has  rights
in  Scry  under  Contract DE-AC03-765F00098 between the U.S.
Department of Energy and the University of California.

     Scry is provided as a professional  academic  contribu-
tion  for  joint exchange.  Thus it is experimental, is pro-
vided ``as is'', with no warranties of any kind  whatsoever,
no  support,  promise  of updates, or printed documentation.
The Regents of the University of California  shall  have  no
liability  with respect to the infringement of copyrights by
Scry, or any part thereof. */

/* programmers:  Teck-Joo Chua, David Robertson */

/* These are notify procedures used by the panel items, such as buttons and
   slider:

create_panel_items:	create control panel
file_text_proc:       input Anima file name
gamma_proc:           set gamma setting for next frame
postscr_proc:         save current Anima frame as Postscript file
quit_proc:            quit Xanima
goto_proc:            seek to frame entered in "Number:" field
zoom_proc:            set zoom or unzoom for next frame
step_back_proc:       step backward one frame
step_forw_proc:       step forward one frame
stop_proc:            still frame
full_proc:            view sequence at full speed
speed_proc:           set speed for viewing sequence
backward_proc:        view sequence by decreasing frame number
forward_proc:         view sequence by increasing frame number
frame_monitor:        when timer expires, display a frame
set_advance:          used to unstill frame

**************************************************************/


#include <stdio.h>
#include <math.h>
#include <scry_anima.h>
#include <scry_image.h>
#include "svwin.h"

static void file_text_proc(), gamma_proc(), postscr_proc(), anima_to_sunrast() ;
static void quit_proc() ;
static void goto_proc(), zoom_proc(), step_back_proc(), step_forw_proc() ;
static void stop_proc(), full_proc(), speed_proc(), backward_proc() ;
static void forward_proc() ;
static void advance_proc() ;

Notify_value frame_monitor();

struct itimerval itimevalue;	/* timer value */



create_panel_items()
{
    Pixfont *bold;
    
        /* select bold font */
    bold = pf_open("/usr/lib/fonts/fixedwidthfonts/screen.b.12");
    
	/* Anima file name */
    a_panel.file = panel_create_item(a_panel.panel, PANEL_TEXT,
                      PANEL_LABEL_STRING, "File:",
                      PANEL_VALUE_DISPLAY_LENGTH, 28,
                      PANEL_VALUE_STORED_LENGTH, 70,
                      PANEL_VALUE, "",
                      PANEL_NOTIFY_PROC, file_text_proc,
                      NULL);
    
       /* gamma value */
   a_panel.gamma = panel_create_item(a_panel.panel, PANEL_TEXT, 
                      PANEL_LABEL_STRING, "Gamma:", 
                      PANEL_VALUE_DISPLAY_LENGTH, 4, 
                      PANEL_VALUE_STORED_LENGTH, 6,
                      PANEL_VALUE, "", 
                      PANEL_NOTIFY_PROC, gamma_proc,
                      NULL);
    
    panel_create_item(a_panel.panel, PANEL_BUTTON, 
		      PANEL_LABEL_IMAGE, panel_button_image(a_panel.panel,
							    "sunrast", 0, bold), 
		      PANEL_NOTIFY_PROC, anima_to_sunrast, 
		      0);
    
    panel_create_item(a_panel.panel, PANEL_BUTTON, 
		      PANEL_LABEL_IMAGE, panel_button_image(a_panel.panel,
							    "postscr", 0, bold), 
		      PANEL_NOTIFY_PROC, postscr_proc, 
		      0);
    
    panel_create_item(a_panel.panel, PANEL_BUTTON,
		      PANEL_LABEL_IMAGE, panel_button_image(a_panel.panel, " quit ", 0, bold),
		      PANEL_NOTIFY_PROC, quit_proc,
		      0);

    a_panel.cur_frame = panel_create_item(a_panel.panel, PANEL_TEXT, 
                      PANEL_NOTIFY_LEVEL, PANEL_NONE, 
                      PANEL_LABEL_STRING, 	" #: ", 
                      PANEL_VALUE_DISPLAY_LENGTH, 	8, 
                      PANEL_VALUE, "0", 
		      PANEL_ITEM_X, ATTR_COL(0),
		      PANEL_ITEM_Y, ATTR_ROW(2),
                      0);
    
    panel_create_item(a_panel.panel, PANEL_BUTTON, 
		      PANEL_LABEL_IMAGE, panel_button_image(a_panel.panel,
							    " goto ", 0, bold), 
		      PANEL_NOTIFY_PROC, goto_proc, 
		      0);
    
    a_panel.frame = panel_create_item(a_panel.panel, PANEL_TEXT, 
                      PANEL_NOTIFY_LEVEL, PANEL_ALL, 
                      PANEL_LABEL_STRING, "Number Input:", 
                      PANEL_VALUE_DISPLAY_LENGTH, 6, 
                      PANEL_ITEM_X, ATTR_COL(25),
                      PANEL_VALUE, "", 
                      0);
    
    panel_create_item(a_panel.panel, PANEL_BUTTON, 
		      PANEL_LABEL_IMAGE, panel_button_image(a_panel.panel,
							    " zoom ", 0, bold), 
		      PANEL_NOTIFY_PROC, zoom_proc, 
		      0);
    
    panel_create_item(a_panel.panel, PANEL_BUTTON, 
		      PANEL_LABEL_IMAGE, panel_button_image(a_panel.panel,
							    " < step ", 0, bold), 
		      PANEL_NOTIFY_PROC, step_back_proc, 
		      0);
    
    panel_create_item(a_panel.panel, PANEL_BUTTON, 
		      PANEL_LABEL_IMAGE, panel_button_image(a_panel.panel,
							    " step > ", 0, bold), 
		      PANEL_NOTIFY_PROC, step_forw_proc, 
		      0);
    
    panel_create_item(a_panel.panel, PANEL_BUTTON, 
		      PANEL_LABEL_IMAGE, panel_button_image(a_panel.panel,
							    " stop ", 0, bold), 
		      PANEL_NOTIFY_PROC, stop_proc, 
		      PANEL_ITEM_X, ATTR_COL(0), 
		      PANEL_ITEM_Y, ATTR_ROW(4), 
		      0);
    
    panel_create_item(a_panel.panel, PANEL_BUTTON, 
		      PANEL_LABEL_IMAGE, panel_button_image(a_panel.panel,
							    " full ", 0, bold), 
		      PANEL_ITEM_X, ATTR_COL(11),
		      PANEL_NOTIFY_PROC, full_proc, 
		      0);
    
    a_panel.slider = panel_create_item(a_panel.panel, PANEL_SLIDER, 
                      PANEL_SHOW_VALUE, 	FALSE, 
                      PANEL_VALUE, 		5, 
                      PANEL_MIN_VALUE, 	0, 
                      PANEL_MAX_VALUE, 	10, 
                      PANEL_SLIDER_WIDTH, 	150, 
                      PANEL_ITEM_X, ATTR_COL(22),
                      PANEL_NOTIFY_PROC, speed_proc, 
                      0);
    
	/* backwards */
    panel_create_item(a_panel.panel, PANEL_BUTTON, 
		      PANEL_LABEL_IMAGE, panel_button_image(a_panel.panel,
							    "   <    ", 0, bold), 
		      PANEL_ITEM_X, ATTR_COL(55),
		      PANEL_NOTIFY_PROC, backward_proc, 
		      0);
    
	/* forwards */
    panel_create_item(a_panel.panel, PANEL_BUTTON, 
		      PANEL_LABEL_IMAGE, panel_button_image(a_panel.panel,
							    "    >   ", 0, bold), 
		      PANEL_NOTIFY_PROC, forward_proc, 
		      0);
    
    window_fit_height(a_panel.panel);
}




/* get Anima file name */

static void
file_text_proc()
{

    strcpy (S_anima_filename, (char *) (panel_get_value(a_panel.file)));
	/* read Anima footer */
    if (!getfile())
	return ;
    panel_set_value(a_panel.file, "") ;
    window_set(a_panel.panel,
		PANEL_CARET_ITEM, a_panel.frame,
		0) ;
    S_avars.frame.last = S_anima_count -1 ;
    S_avars.start = TRUE ;
}




/* set gamma value */

static void
gamma_proc()

{
    char *value ;
    float temp_gamma ;

    strcpy(value, (char *) (panel_get_value(a_panel.gamma)));
    temp_gamma = (float)atof(value) ;
	/* only has effect if svanima in stopped mode */
    if ((S_avars.stop) && ((temp_gamma > 0.1) && (temp_gamma <= 4.0)))
    {
        scry_gamma = temp_gamma ;
        gamma_changed = 1 ;
        window_set(a_panel.panel,
		PANEL_CARET_ITEM, a_panel.frame,
		0) ;
    }
    else
        panel_set_value(a_panel.gamma, "") ;
}




/* save current Anima frame as Postscript file */

static void
postscr_proc()
{
    void anima_to_post();
    float tmp ;
	
    tmp = (float) S_avars.frame.first ;
	/* default is 60 dpi */
    if (tmp == 0)
         tmp = 60.0 ;
    anima_to_post(tmp) ;
    S_avars.frame.first = 0 ;
    panel_set_value(a_panel.frame, "");
}




/* quit */

static void
quit_proc()
{
    notify_set_itimer_func(anima_control, NOTIFY_FUNC_NULL, ITIMER_REAL,
			   NULL, NULL);
    fclose(S_anima_file);
    window_set(anima_control, FRAME_NO_CONFIRM, TRUE, 0);
    window_destroy(anima_control);
    window_set(anima_control, FRAME_NO_CONFIRM, TRUE, 0);
    window_destroy(anima_control);
    pr_close(image_win_image);
    free (S_image_info.data) ;
}




/* go to frame */

static void
goto_proc()

{
    int temp;
    
    temp = atoi((char *) panel_get_value(a_panel.frame));
    
    if((temp < 0) || (temp >= S_anima_count))	/* frame number is invalid */
	panel_set_value(a_panel.frame, "");
    else
    {
        S_avars.stop = FALSE;
        S_avars.advance = TRUE;
        S_avars.search = TRUE;
        S_avars.current = temp;
    }
}




/* zoom or unzoom by factor of 2 */

static void
zoom_proc()

{
    S_do_zoom = !S_do_zoom ;
    if (S_do_zoom)
    {
	S_image_info.s_width *= 2 ;
	S_image_info.s_height *= 2 ;
    }
    else
    {
	S_image_info.s_width /= 2 ;
	S_image_info.s_height /= 2 ;
    }
    S_zoom_changed = 1 ;
}




/* step back one frame */

static void
step_back_proc()
{
    S_avars.direction = S_BACKWARD;
    advance_proc();
}




/* step forward one frame */

static void
step_forw_proc()
{
    S_avars.direction = S_FORWARD;
    advance_proc();
}




/* stop procedure */

static void
stop_proc()
{
    S_avars.stop = TRUE;
    panel_set_value(a_panel.frame, "");
    window_set(a_panel.panel, PANEL_CARET_ITEM, a_panel.frame, 0);
}




/* display at full speed */

static void 
full_proc()
{
    S_avars.stop = FALSE;
        /* set to maximum speed value */
    S_avars.speed = 10;
    panel_set(a_panel.slider, PANEL_VALUE, 10, 0);
	/* timer value */
    itimevalue.it_value.tv_usec = S_FULL_SPEED ;
    itimevalue.it_interval.tv_usec = S_FULL_SPEED ;
    notify_set_itimer_func(anima_control, frame_monitor, ITIMER_REAL, 
			    &itimevalue, ((struct itimerval *) 0));

}




/* set speed */

static void
speed_proc()
{
    S_avars.speed = (int) panel_get_value(a_panel.slider);

    S_avars.stop = FALSE;
    if (S_avars.speed == 10)
    {
        itimevalue.it_value.tv_usec = S_FULL_SPEED ; 
        itimevalue.it_interval.tv_usec = S_FULL_SPEED ;
    }
    else if (S_avars.speed != 0)
    {
        itimevalue.it_value.tv_usec = (S_DELAY_INC * (10 - S_avars.speed)) ; 
        itimevalue.it_interval.tv_usec = (S_DELAY_INC * (10 - S_avars.speed)) ;
    }
    else
    {
        itimevalue.it_value.tv_usec = S_SLOWEST_SPEED ; 
        itimevalue.it_interval.tv_usec = S_SLOWEST_SPEED ;
    }
	/* display frame upon expiration of timer */
    notify_set_itimer_func(anima_control, frame_monitor, ITIMER_REAL, 
			    &itimevalue, ((struct itimerval *) 0));
}




/* set backward */

static void
backward_proc()
{
    S_avars.direction = S_BACKWARD;
    S_avars.stop = FALSE;

}




/* set forward */

static void
forward_proc()

{
    S_avars.direction = S_FORWARD;
    S_avars.stop = FALSE;

}




/* Notifer calls this to display farme on expiration of timer */

Notify_value
frame_monitor()

{
    if ((!S_avars.stop) || (S_avars.start))
	display_frame() ;
    return(NOTIFY_DONE);
}



/* get out of still mode */

static void
advance_proc()

{
    if(S_avars.stop == TRUE)
    {
        S_avars.advance = TRUE;
        S_avars.stop = FALSE;
    }
    else
	S_avars.stop = TRUE;
}
