#include "Assembler.hpp"

#include "Token.hpp"
#include "Scanner.hpp"
#include "Parser.hpp"
#include "Linker.hpp"
#include "Loader.hpp"
#include "Error.hpp"
#include "String.hpp"
#include "Operand.hpp"
#include "Synthesizer.hpp"
#include "InstructionSet.hpp"

#include <time.h>

namespace SoftWire
{
	InstructionSet *Assembler::instructionSet = 0;
	int Assembler::referenceCount = 0;
	bool Assembler::listingEnabled = true;

	Assembler::Assembler(const char *sourceFile)
	{
		try
		{
			errors = new char[1];
			errors[0] = '\0';

			if(sourceFile)
			{
				entryLabel = strdup(sourceFile);
				char *dot = strrchr(entryLabel, '.');

				if(dot)
				{
					*dot = '\0';
				}
				else
				{
					throw Error("Input file must have extension");
				}
			}
			else
			{
				entryLabel = 0;
			}

			if(!instructionSet)
			{
				instructionSet = new InstructionSet();
			}
			referenceCount++;

			linker = new Linker();
			loader = new Loader(*linker);
			synthesizer = new Synthesizer();

			if(sourceFile)
			{
				scanner = new Scanner();
				parser = new Parser(*scanner, *synthesizer, *instructionSet);

				scanner->scanFile(sourceFile);
				assembleFile();
			}
			else
			{
				scanner = 0;
				parser = 0;
			}
		}
		catch(const Error &error)
		{
			handleError(error.getString());
		}
	}

	Assembler::Assembler(const char *sourceString, const char *entryPoint)
	{
		try
		{
			errors = new char[1];
			errors[0] = '\0';

			if (entryPoint)
			{
				entryLabel = strdup(entryPoint);
			}
			else
			{
				entryLabel = 0;
			}

			if(!instructionSet)
			{
				instructionSet = new InstructionSet();
			}
			referenceCount++;

			linker = new Linker();
			loader = new Loader(*linker);
			synthesizer = new Synthesizer();

			scanner = new Scanner();
			parser = new Parser(*scanner, *synthesizer, *instructionSet);

			scanner->scanString(sourceString);
			assembleFile();
		}
		catch(const Error &error)
		{
			handleError(error.getString());
		}
	}

	Assembler::~Assembler()
	{
		delete[] entryLabel;
		entryLabel = 0;

		delete linker;
		linker = 0;

		delete loader;
		loader = 0;

		delete scanner;
		scanner = 0;

		delete synthesizer;
		synthesizer = 0;

		referenceCount--;
		if(!referenceCount)
		{
			delete instructionSet;
			instructionSet = 0;
		}

		delete parser;
		parser = 0;

		delete[] errors;
		errors = 0;
	}

	void Assembler::assembleFile()
	{
		if(!scanner)
		{
			throw INTERNAL_ERROR;
		}

		scanner->rewind();

		while(!scanner->isEndOfFile())
		{
			assembleLine();

			if(scanner->isEndOfLine() && !scanner->isEndOfFile())
			{
				scanner->advance();
			}
		}

		delete scanner;
		scanner = 0;

		delete parser;
		parser = 0;
	}

	void Assembler::defineExternal(void *pointer, const char *name)
	{
		Linker::defineExternal(pointer, name);
	}

	void Assembler::defineSymbol(int value, const char *name)
	{
		Scanner::defineSymbol(value, name);
	}

	void (*Assembler::callable(const char *entryLabel))()
	{
		if(!loader || errors[0] != '\0') return 0;

		if(entryLabel)
		{
			return loader->callable(entryLabel);
		}
		else
		{
			return loader->callable(this->entryLabel);
		}
	}

	void (*Assembler::finalize(const char *entryLabel))()
	{
		if(!loader) throw Error("Assembler could not be finalized (cannot re-finalize)");

		delete linker;
		linker = 0;

		delete scanner;
		scanner = 0;

		delete synthesizer;
		synthesizer = 0;

		delete parser;
		parser = 0;

		delete[] errors;
		errors = 0;

		if(entryLabel)
		{
			delete[] this->entryLabel;
			this->entryLabel = 0;

			return loader->callable(entryLabel);
		}
		else
		{
			return loader->callable(this->entryLabel);
		}
	}

	void *Assembler::acquire()
	{
		if(!loader) return 0;

		return loader->acquire();
	}

	void Assembler::assembleLine()
	{
		if(!parser || !loader) throw INTERNAL_ERROR;

		const Encoding &encoding = parser->parseLine();
		loader->appendEncoding(encoding);
	}

	const char *Assembler::getErrors() const
	{
		return errors;
	}

	void Assembler::handleError(const char *error)
	{
		const char *sourceLine = "<intrinsic>";
		if(parser) sourceLine = parser->skipLine();

		int previousLength = 0;
		if(errors) previousLength = strlen(errors);
		int newLength = previousLength + strlen(error) + strlen(sourceLine) + 16;

		delete[] errors;
		errors = new char[newLength];

		strcpy(errors, "error: ");
		strcat(errors, error);
		strcat(errors, "\n\t");
		strcat(errors, sourceLine);
		strcat(errors, "\n");
	}

	const char *Assembler::getListing() const
	{
		return loader->getListing();
	}

	void Assembler::clearListing() const
	{
		loader->clearListing();
	}

	void Assembler::reset()
	{
		if(!loader) return;

		loader->reset();
	}

	int Assembler::instructionCount()
	{
		if(!loader)
		{
			return 0;
		}

		return loader->instructionCount();
	}

	void Assembler::enableListing()
	{
		listingEnabled = true;
	}

	void Assembler::disableListing()
	{
		listingEnabled = false;
	}
};
